import { Construct } from 'constructs';
import { UserEngine } from './common';
import { IResource, Resource } from 'aws-cdk-lib/core';
/**
 * Access control configuration for ElastiCache users.
 */
export declare abstract class AccessControl {
    /**
     * Create access control from an access string.
     *
     * @param accessString The access string defining user permissions.
     */
    static fromAccessString(accessString: string): AccessControl;
    /**
     * The access string that defines user's permissions.
     */
    abstract readonly accessString: string;
}
/**
 * Properties for defining an ElastiCache base user.
 */
export interface UserBaseProps {
    /**
     * The engine type for the user.
     * Enum options: UserEngine.VALKEY, UserEngine.REDIS.
     *
     * @default UserEngine.VALKEY.
     */
    readonly engine?: UserEngine;
    /**
     * The ID of the user.
     */
    readonly userId: string;
    /**
     * Access control configuration for the user.
     */
    readonly accessControl: AccessControl;
}
/**
 * Represents an ElastiCache base user.
 */
export interface IUser extends IResource {
    /**
     * The user's ID.
     *
     * @attribute
     */
    readonly userId: string;
    /**
     * The engine for the user.
     */
    readonly engine?: UserEngine;
    /**
     * The user's name.
     *
     * @attribute
     */
    readonly userName?: string;
    /**
     * The user's ARN.
     *
     * @attribute
     */
    readonly userArn: string;
}
/**
 * Attributes for importing an existing ElastiCache user.
 */
export interface UserBaseAttributes {
    /**
     * The ID of the user.
     * One of `userId` or `userArn` is required.
     *
     * @default - derived from userArn.
     */
    readonly userId?: string;
    /**
     * The engine type for the user.
     *
     * @default - engine type is unknown.
     */
    readonly engine?: UserEngine;
    /**
     * The user's name.
     *
     * @default - name is unknown.
     */
    readonly userName?: string;
    /**
     * The ARN of the user.
     *
     * One of `userId` or `userArn` is required.
     *
     * @default - derived from userId.
     */
    readonly userArn?: string;
}
/**
 * Base class for ElastiCache users.
 */
export declare abstract class UserBase extends Resource implements IUser {
    /**
     * Import an existing user by ID.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param userId The ID of the existing user.
     */
    static fromUserId(scope: Construct, id: string, userId: string): IUser;
    /**
     * Import an existing user by ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param userArn The ARN of the existing user.
     */
    static fromUserArn(scope: Construct, id: string, userArn: string): IUser;
    /**
     * Import an existing user using attributes.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `UserBaseAttributes` object.
     */
    static fromUserAttributes(scope: Construct, id: string, attrs: UserBaseAttributes): IUser;
    /**
     * The user's ID.
     *
     * @attribute
     */
    abstract readonly userId: string;
    /**
     * The engine for the user.
     */
    abstract readonly engine?: UserEngine;
    /**
     * The user's name.
     *
     * @attribute
     */
    abstract readonly userName?: string;
    /**
     * The user's ARN.
     *
     * @attribute
     */
    abstract readonly userArn: string;
}
