import { Construct } from 'constructs';
import { IServerlessCache, ServerlessCacheBase, CacheEngine } from './serverless-cache-base';
import { IUserGroup } from './user-group';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as events from 'aws-cdk-lib/aws-events';
import * as kms from 'aws-cdk-lib/aws-kms';
import { Size } from 'aws-cdk-lib/core';
/**
 * Usage limits configuration for ServerlessCache
 */
export interface CacheUsageLimitsProperty {
    /**
     * Minimum data storage size (1 GB)
     *
     * @default - No minimum limit
     */
    readonly dataStorageMinimumSize?: Size;
    /**
     * Maximum data storage size (5000 GB)
     *
     * @default - No maximum limit
     */
    readonly dataStorageMaximumSize?: Size;
    /**
     * Minimum request rate limit (1000 ECPUs per second)
     *
     * @default - No minimum limit
     */
    readonly requestRateLimitMinimum?: number;
    /**
     * Maximum request rate limit (15000000 ECPUs per second)
     *
     * @default - No maximum limit
     */
    readonly requestRateLimitMaximum?: number;
}
/**
 * Backup configuration for ServerlessCache
 */
export interface BackupSettings {
    /**
     * Automated daily backup UTC time
     *
     * @default - No automated backups
     */
    readonly backupTime?: events.Schedule;
    /**
     * Number of days to retain backups (1-35)
     *
     * @default - Backups are not retained
     */
    readonly backupRetentionLimit?: number;
    /**
     * Name for the final backup taken before deletion
     *
     * @default - No final backup
     */
    readonly backupNameBeforeDeletion?: string;
    /**
     * ARNs of backups from which to restore data into the new cache
     *
     * @default - Create a new cache with no existing data
     */
    readonly backupArnsToRestore?: string[];
}
/**
 * Properties for defining a ServerlessCache
 */
export interface ServerlessCacheProps {
    /**
     * The cache engine combined with the version
     * Enum options: VALKEY_DEFAULT, VALKEY_7, VALKEY_8, REDIS_DEFAULT, MEMCACHED_DEFAULT
     * The default options bring the latest versions available.
     *
     * @default when not provided, the default engine would be Valkey, latest version available (VALKEY_DEFAULT)
     */
    readonly engine?: CacheEngine;
    /**
     * Name for the serverless cache
     *
     * @default automatically generated name by Resource
     */
    readonly serverlessCacheName?: string;
    /**
     * A description for the cache
     *
     * @default - No description
     */
    readonly description?: string;
    /**
     * Usage limits for the cache
     *
     * @default - No usage limits
     */
    readonly cacheUsageLimits?: CacheUsageLimitsProperty;
    /**
     * Backup configuration
     *
     * @default - No backups configured
     */
    readonly backup?: BackupSettings;
    /**
     * KMS key for encryption
     *
     * @default - Service managed encryption (AWS owned KMS key)
     */
    readonly kmsKey?: kms.IKey;
    /**
     * The VPC to place the cache in
     */
    readonly vpc: ec2.IVpc;
    /**
     * Which subnets to place the cache in
     *
     * @default - Private subnets with egress
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * Security groups for the cache
     *
     * @default - A new security group is created
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * User group for access control
     *
     * @default - No user group
     */
    readonly userGroup?: IUserGroup;
}
/**
 * Attributes that can be specified when importing a ServerlessCache
 */
export interface ServerlessCacheAttributes {
    /**
     * The cache engine used by this cache
     *
     * @default - engine type is unknown
     */
    readonly engine?: CacheEngine;
    /**
     * The name of the serverless cache
     *
     * One of `serverlessCacheName` or `serverlessCacheArn` is required.
     *
     * @default - derived from serverlessCacheArn
     */
    readonly serverlessCacheName?: string;
    /**
     * The ARN of the serverless cache
     *
     * One of `serverlessCacheName` or `serverlessCacheArn` is required.
     *
     * @default - derived from serverlessCacheName
     */
    readonly serverlessCacheArn?: string;
    /**
     * The ARNs of backups restored in the cache
     *
     * @default - backups are unknown
     */
    readonly backupArnsToRestore?: string[];
    /**
     * The KMS key used for encryption
     *
     * @default - encryption key is unknown
     */
    readonly kmsKey?: kms.IKey;
    /**
     * The VPC this cache is deployed in
     *
     * @default - VPC is unknown
     */
    readonly vpc?: ec2.IVpc;
    /**
     * The subnets this cache is deployed in
     *
     * @default - subnets are unknown
     */
    readonly subnets?: ec2.ISubnet[];
    /**
     * The security groups associated with this cache
     *
     * @default - security groups are unknown
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The user group associated with this cache
     *
     * @default - user group is unknown
     */
    readonly userGroup?: IUserGroup;
}
/**
 * A serverless ElastiCache cache
 *
 * @resource AWS::ElastiCache::ServerlessCache
 */
export declare class ServerlessCache extends ServerlessCacheBase {
    /**
     * Uniquely identifies this class.
     */
    static readonly PROPERTY_INJECTION_ID: string;
    /**
     * Return whether the given object is a `ServerlessCache`
     */
    static isServerlessCache(x: any): x is ServerlessCache;
    /**
     * Import an existing serverless cache by name
     *
     * @param scope The parent creating construct (usually `this`)
     * @param id The construct's name
     * @param serverlessCacheName The name of the existing serverless cache
     */
    static fromServerlessCacheName(scope: Construct, id: string, serverlessCacheName: string): IServerlessCache;
    /**
     * Import an existing serverless cache by ARN
     *
     * @param scope The parent creating construct (usually `this`)
     * @param id The construct's name
     * @param serverlessCacheArn The ARN of the existing serverless cache
     */
    static fromServerlessCacheArn(scope: Construct, id: string, serverlessCacheArn: string): IServerlessCache;
    /**
     * Import an existing serverless cache using attributes
     *
     * @param scope The parent creating construct (usually `this`)
     * @param id The construct's name
     * @param attrs A `ServerlessCacheAttributes` object
     */
    static fromServerlessCacheAttributes(scope: Construct, id: string, attrs: ServerlessCacheAttributes): IServerlessCache;
    readonly engine?: CacheEngine;
    readonly serverlessCacheName: string;
    readonly backupArnsToRestore?: string[];
    readonly kmsKey?: kms.IKey;
    readonly vpc?: ec2.IVpc;
    readonly subnets?: ec2.ISubnet[];
    readonly securityGroups?: ec2.ISecurityGroup[];
    readonly userGroup?: IUserGroup;
    /**
     * The ARN of the serverless cache
     *
     * @attribute
     */
    readonly serverlessCacheArn: string;
    /**
     * The endpoint address of the serverless cache
     *
     * @attribute
     */
    readonly serverlessCacheEndpointAddress: string;
    /**
     * The endpoint port of the serverless cache
     *
     * @attribute
     */
    readonly serverlessCacheEndpointPort: string;
    /**
     * The reader endpoint address of the serverless cache
     *
     * @attribute
     */
    readonly serverlessCacheReaderEndpointAddress: string;
    /**
     * The reader endpoint port of the serverless cache
     *
     * @attribute
     */
    readonly serverlessCacheReaderEndpointPort: string;
    /**
     * The current status of the serverless cache
     * Can be 'CREATING', 'AVAILABLE', 'DELETING', 'CREATE-FAILED', 'MODIFYING'
     *
     * @attribute
     */
    readonly serverlessCacheStatus: string;
    /**
     * Access to network connections
     */
    readonly connections: ec2.Connections;
    constructor(scope: Construct, id: string, props: ServerlessCacheProps);
    /**
     * Validate description meets AWS requirements
     *
     * @param description The description to validate
     */
    private validateDescription;
    /**
     * Validate data storage size limits
     *
     * @param limits The usage limits containing data storage settings
     */
    private validateDataStorageLimits;
    /**
     * Validate request rate limits
     *
     * @param limits The usage limits containing request rate settings
     */
    private validateRequestRateLimits;
    /**
     * Validate backup settings meet AWS requirements
     *
     * @param backup The backup settings to validate
     */
    private validateBackupSettings;
    /**
     * Validate user group compatibility with cache engine
     *
     * @param engine The cache engine
     * @param userGroup The user group to validate
     */
    private validateUserGroupCompatibility;
    /**
     * Render cache usage limits for CloudFormation
     *
     * @param limits The usage limits to render
     * @returns CloudFormation-compatible usage limits object or undefined
     */
    private renderCacheUsageLimits;
    /**
     * Configure subnets for the cache
     *
     * @param props The ServerlessCache properties
     * @returns Object containing subnet IDs and subnet objects
     */
    private configureSubnets;
    /**
     * Configure security groups for the cache
     *
     * @param props The ServerlessCache properties
     * @returns Object containing security group IDs and security group objects
     */
    private configureSecurityGroups;
    /**
     * Format schedule to HH:MM format for daily backups
     *
     * @param schedule The schedule to format
     * @returns Time string in HH:MM format
     */
    private formatBackupTime;
}
