"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessCacheBase = exports.CacheEngine = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const core_1 = require("aws-cdk-lib/core");
/**
 * Supported cache engines together with available versions.
 */
var CacheEngine;
(function (CacheEngine) {
    /**
     * Valkey engine, latest major version available, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    CacheEngine["VALKEY_LATEST"] = "valkey";
    /**
     * Valkey engine, major version 7, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    CacheEngine["VALKEY_7"] = "valkey_7";
    /**
     * Valkey engine, major version 8, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    CacheEngine["VALKEY_8"] = "valkey_8";
    /**
     * Redis engine, latest major version available, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    CacheEngine["REDIS_LATEST"] = "redis";
    /**
     * Redis engine, major version 7, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    CacheEngine["REDIS_7"] = "redis_7";
    /**
     * Memcached engine, latest major version available, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    CacheEngine["MEMCACHED_LATEST"] = "memcached";
})(CacheEngine || (exports.CacheEngine = CacheEngine = {}));
/**
 * Base class for ServerlessCache constructs
 */
class ServerlessCacheBase extends core_1.Resource {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@aws-cdk/aws-elasticache-alpha.ServerlessCacheBase", version: "2.223.0-alpha.0" };
    /**
     * Grant connect permissions to the cache
     *
     * @param grantee The principal to grant permissions to
     */
    grantConnect(grantee) {
        return this.grant(grantee, 'elasticache:Connect', 'elasticache:DescribeServerlessCaches');
    }
    /**
     * Grant the given identity custom permissions
     *
     * @param grantee The principal to grant permissions to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.serverlessCacheArn],
        });
    }
    /**
     * Return the given named metric for this cache
     *
     * @param metricName The name of the metric
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ElastiCache',
            metricName,
            dimensionsMap: {
                ServerlessCacheName: this.serverlessCacheName,
            },
            period: core_1.Duration.minutes(5),
            statistic: 'Average',
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for cache hit count
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Sum over 5 minutes
     */
    metricCacheHitCount(props) {
        return this.metric('CacheHits', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for cache miss count
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Sum over 5 minutes
     */
    metricCacheMissCount(props) {
        return this.metric('CacheMisses', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for cache hit rate
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metricCacheHitRate(props) {
        return this.metric('CacheHitRate', props);
    }
    /**
     * Metric for data stored in the cache
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Maximum over 5 minutes
     */
    metricDataStored(props) {
        return this.metric('BytesUsedForCache', { statistic: 'Maximum', ...props });
    }
    /**
     * Metric for ECPUs consumed
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metricProcessingUnitsConsumed(props) {
        return this.metric('ElastiCacheProcessingUnits', props);
    }
    /**
     * Metric for network bytes in
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Sum over 5 minutes
     */
    metricNetworkBytesIn(props) {
        return this.metric('NetworkBytesIn', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for network bytes out
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Sum over 5 minutes
     */
    metricNetworkBytesOut(props) {
        return this.metric('NetworkBytesOut', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for active connections
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Maximum over 5 minutes
     */
    metricActiveConnections(props) {
        return this.metric('CurrConnections', { statistic: 'Maximum', ...props });
    }
    /**
     * Metric for write request latency
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metricWriteRequestLatency(props) {
        return this.metric('SuccessfulWriteRequestLatency', props);
    }
    /**
     * Metric for read request latency
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metricReadRequestLatency(props) {
        return this.metric('SuccessfulReadRequestLatency', props);
    }
}
exports.ServerlessCacheBase = ServerlessCacheBase;
//# sourceMappingURL=data:application/json;base64,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