import { IUserGroup } from './user-group';
import * as cloudwatch from 'aws-cdk-lib/aws-cloudwatch';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as kms from 'aws-cdk-lib/aws-kms';
import { IResource, Resource } from 'aws-cdk-lib/core';
/**
 * Supported cache engines together with available versions.
 */
export declare enum CacheEngine {
    /**
     * Valkey engine, latest major version available, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    VALKEY_LATEST = "valkey",
    /**
     * Valkey engine, major version 7, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    VALKEY_7 = "valkey_7",
    /**
     * Valkey engine, major version 8, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    VALKEY_8 = "valkey_8",
    /**
     * Redis engine, latest major version available, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    REDIS_LATEST = "redis",
    /**
     * Redis engine, major version 7, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    REDIS_7 = "redis_7",
    /**
     * Memcached engine, latest major version available, minor version is selected automatically
     * For more information about the features related to this version check: https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/engine-versions.html
     */
    MEMCACHED_LATEST = "memcached"
}
/**
 * Represents a Serverless ElastiCache cache
 */
export interface IServerlessCache extends IResource, ec2.IConnectable {
    /**
     * The cache engine used by this cache
     */
    readonly engine?: CacheEngine;
    /**
     * The name of the serverless cache
     *
     * @attribute
     */
    readonly serverlessCacheName: string;
    /**
     * The ARNs of backups restored in the cache
     */
    readonly backupArnsToRestore?: string[];
    /**
     * The KMS key used for encryption
     */
    readonly kmsKey?: kms.IKey;
    /**
     * The VPC this cache is deployed in
     */
    readonly vpc?: ec2.IVpc;
    /**
     * The subnets this cache is deployed in
     */
    readonly subnets?: ec2.ISubnet[];
    /**
     * The security groups associated with this cache
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The user group associated with this cache
     */
    readonly userGroup?: IUserGroup;
    /**
     * The ARN of the serverless cache
     *
     * @attribute
     */
    readonly serverlessCacheArn: string;
    /**
     * Grant connect permissions to the cache
     */
    grantConnect(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity custom permissions
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Return the given named metric for this cache
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for cache hit count
     */
    metricCacheHitCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for cache miss count
     */
    metricCacheMissCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for cache hit rate
     */
    metricCacheHitRate(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for data stored in the cache
     */
    metricDataStored(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for ECPUs consumed
     */
    metricProcessingUnitsConsumed(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for network bytes in
     */
    metricNetworkBytesIn(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for network bytes out
     */
    metricNetworkBytesOut(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for active connections
     */
    metricActiveConnections(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for write request latency
     */
    metricWriteRequestLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for read request latency
     */
    metricReadRequestLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * Base class for ServerlessCache constructs
 */
export declare abstract class ServerlessCacheBase extends Resource implements IServerlessCache {
    abstract readonly engine?: CacheEngine;
    abstract readonly serverlessCacheName: string;
    abstract readonly backupArnsToRestore?: string[];
    abstract readonly kmsKey?: kms.IKey;
    abstract readonly vpc?: ec2.IVpc;
    abstract readonly subnets?: ec2.ISubnet[];
    abstract readonly securityGroups?: ec2.ISecurityGroup[];
    abstract readonly userGroup?: IUserGroup;
    abstract readonly serverlessCacheArn: string;
    /**
     * Access to network connections.
     */
    abstract readonly connections: ec2.Connections;
    /**
     * Grant connect permissions to the cache
     *
     * @param grantee The principal to grant permissions to
     */
    grantConnect(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity custom permissions
     *
     * @param grantee The principal to grant permissions to
     * @param actions The actions to grant
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Return the given named metric for this cache
     *
     * @param metricName The name of the metric
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for cache hit count
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Sum over 5 minutes
     */
    metricCacheHitCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for cache miss count
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Sum over 5 minutes
     */
    metricCacheMissCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for cache hit rate
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metricCacheHitRate(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for data stored in the cache
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Maximum over 5 minutes
     */
    metricDataStored(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for ECPUs consumed
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metricProcessingUnitsConsumed(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for network bytes in
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Sum over 5 minutes
     */
    metricNetworkBytesIn(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for network bytes out
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Sum over 5 minutes
     */
    metricNetworkBytesOut(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for active connections
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Maximum over 5 minutes
     */
    metricActiveConnections(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for write request latency
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metricWriteRequestLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for read request latency
     *
     * @param props Additional properties which will be merged with the default metric
     * @default Average over 5 minutes
     */
    metricReadRequestLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
