use arrow::datatypes::DataType;
use pyo3::{FromPyObject, PyAny, PyResult};

use crate::errors;

/// utility struct to convert PyObj to native DataType
#[derive(Debug, Clone)]
pub struct PyDataType {
    pub data_type: DataType,
}

impl<'source> FromPyObject<'source> for PyDataType {
    fn extract(ob: &'source PyAny) -> PyResult<Self> {
        let id = ob.getattr("id")?.extract::<i32>()?;
        let data_type = data_type_id(&id)?;
        Ok(PyDataType { data_type })
    }
}

fn data_type_id(id: &i32) -> Result<DataType, errors::DataFusionError> {
    // see https://github.com/apache/arrow/blob/3694794bdfd0677b95b8c95681e392512f1c9237/python/pyarrow/includes/libarrow.pxd
    // this is not ideal as it does not generalize for non-basic types
    // Find a way to get a unique name from the pyarrow.DataType
    Ok(match id {
        1 => DataType::Boolean,
        2 => DataType::UInt8,
        3 => DataType::Int8,
        4 => DataType::UInt16,
        5 => DataType::Int16,
        6 => DataType::UInt32,
        7 => DataType::Int32,
        8 => DataType::UInt64,
        9 => DataType::Int64,

        10 => DataType::Float16,
        11 => DataType::Float32,
        12 => DataType::Float64,

        //13 => DataType::Decimal,

        // 14 => DataType::Date32(),
        // 15 => DataType::Date64(),
        // 16 => DataType::Timestamp(),
        // 17 => DataType::Time32(),
        // 18 => DataType::Time64(),
        // 19 => DataType::Duration()
        20 => DataType::Binary,
        21 => DataType::Utf8,
        22 => DataType::LargeBinary,
        23 => DataType::LargeUtf8,

        other => {
            return Err(errors::DataFusionError::Common(format!(
                "The type {} is not valid",
                other
            )))
        }
    })
}
