"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongDataPlane = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongDataPlane extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.hostedZoneName,
                        dnsNames: props.dnsNames,
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        const kong_dp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongDpSA', {
            cluster: props.cluster,
            name: 'kong-dp-service-account',
            namespace: props.namespace,
        });
        kong_dp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_dp_service_account.node.addDependency(kong_namespace);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_dp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongDpHelm', {
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                resources: {
                    requests: {
                        cpu: '300m',
                        memory: '300Mi',
                    },
                    limits: {
                        cpu: '1200m',
                        memory: '800Mi',
                    },
                },
                autoscaling: {
                    enabled: true,
                    minReplicas: 1,
                    maxReplicas: 20,
                    metrics: [
                        {
                            type: 'Resource',
                            resource: {
                                name: 'cpu',
                                target: {
                                    type: 'Utilization',
                                    averageUtilization: 75,
                                },
                            },
                        },
                    ],
                },
                ingressController: {
                    enabled: false,
                },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    database: 'off',
                    role: 'data_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-dp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-dp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-dp-cluster-issuer-secret/ca.crt',
                    cluster_server_name: 'kong-cp.internal',
                    cluster_control_plane: `${props.cluster_dns}:8005`,
                    cluster_telemetry_endpoint: `${props.telemetry_dns}:8006`,
                },
                proxy: {
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=proxy',
                        'external-dns.alpha.kubernetes.io/hostname': 'proxy.kong-dp.internal',
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_dp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                manager: {
                    enabled: false,
                },
                portal: {
                    enabled: false,
                },
                portalapi: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-dp-cluster-issuer-secret',
                ],
            },
        });
        deploy_kong_dp_helm.node.addDependency(secrets_crd_mount);
    }
}
exports.KongDataPlane = KongDataPlane;
//# sourceMappingURL=data:application/json;base64,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