# VecGL

__*VecGL* is 3D rendering engine with vector output.__
It is inspired by OpenGL with the key difference that the rendering result is a set of points, lines, and triangles -- not a pixelated image.
These geometric primitives can be used to generate vector graphics or to drive [pen plotters](https://www.generativehut.com/post/axidraw).

## Getting started

The VecGL package is available through `pip`.

```
$ python3 -m pip install vecgl
```

Let's create and render a simple model. 
Here's the complete example for a cube.

```py
import math

import vecgl.linalg as vgll
import vecgl.model as vglm
import vecgl.rendering as vglr

# Get a predefined cube model and choose nice colors.
# The cube will span from -1.0 to 1.0 in all dimensions.
cube = vglm.get_cube_model("lightblue", "black")

# Define the view and the projection matrices.
view_mat4 = vgll.mul_mat4(
    vgll.get_translate_mat4(0.0, 0.0, -3.0),
    vgll.get_rotate_x_mat4(-0.2*math.pi),
    vgll.get_rotate_y_mat4(0.15*math.pi))
projection_mat4 = vgll.get_frustum_mat4(-1.0, 1.0, -1.0, 1.0, 1.0, 100.0)

# Transform our cube model and bring it to the clip space.
transform_mat4 = vgll.mul_mat4(projection_mat4, view_mat4)
cube.transform(transform_mat4)

# Render and display the model.
rendered = vglr.render(cube)
vglr.show(rendered)

# You can access the vector-based rendering result through the rendered model.
for ln in rendered.lines:
    print(ln)
```

VecGL will render and display the cude and print the vector-based rendering result to stdout.

![This is an image](./screenshot.png)


## Build and run tests

Clone the repository.

```
$ git clone https://github.com/frgossen/vecgl
$ cd vecgl
```

Create a virtual environment and activate it (recommended).

```
$ python3 -m venv .venv
$ source .venv/bin/activate
```

Install all requirements in the virtual environment.

```
$ python3 -m pip install -r requirements.txt
```

Intsall the `vecgl` package in editable mode. 
This makes the package (and your changes) available when running the tests.

```
$ python3 -m pip install --editable .
```

You're all set for contributing back to the project.
Run the tests with ...

```
$ python3 -m pytest --benchmark-skip
```

... and the benchmarks with ...

```sh
$ python3 -m pytest --benchmark-only
```
