﻿# Copyright 2016 The Johns Hopkins University Applied Physics Laboratory
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from intern.service.service import Service
from requests import Session


class BossService(Service):
    """Partial implementation of intern.service.service.Service for the Boss' services.

    Attributes:
        _versions (dictionary): Stores supported versions of the Boss API.
        _session (requests.Session): The HTTP session used for each service.
        _session_send_opts (dictionary): Options to use when sending requests.  See http://docs.python-requests.org/en/master/api/#sessionapi
    """

    def __init__(self):
        Service.__init__(self)
        self._versions = {}
        self._session = Session()
        self._session_send_opts = self._session.merge_environment_settings(None, {}, None, None, None)

    def __del__(self):
        if self._session is not None:
            self._session.close()

    @property
    def session(self):
        return self._session

    @property
    def session_send_opts(self):
        return self._session_send_opts

    @session_send_opts.setter
    def session_send_opts(self, opts_dict):
        self._session_send_opts = opts_dict

    def set_auth(self, token, **kwargs):
        """Set the token for authentication/authorization.

        Args:
            token (string):  Token generated by the Django Rest Framework.
        """
        self._auth = token

    def get_api_impl(self, version):
        """Get service object that implements the given version.

        Args:
            version (string): Requested version such as '0.4'.

        Returns:
            (intern.service.boss.Service): A service object that implements the
            requested version of the API.

        Raises:
            KeyError
        """
        return self._versions[version]

    @property
    def url_prefix(self):
        return self.base_protocol + '://' + self.base_url
