# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ruts', 'ruts.datasets', 'ruts.visualizers']

package_data = \
{'': ['*']}

install_requires = \
['graphviz>=0.20.1,<0.21.0',
 'matplotlib>=3.3.0,<4.0.0',
 'nltk>=3.7,<4.0',
 'numpy>=1.23.0,<2.0.0',
 'pandas>=1.4.3,<2.0.0',
 'pymorphy2>=0.9.1,<0.10.0',
 'razdel>=0.5.0,<0.6.0',
 'scipy>=1.9.0,<2.0.0',
 'spacy>=3.4.0,<4.0.0']

setup_kwargs = {
    'name': 'ruts',
    'version': '0.8.1',
    'description': 'Russian Texts Statistics',
    'long_description': '# Russian Texts Statistics (ruTS) ![README_RU](https://raw.githubusercontent.com/gosquared/flags/master/flags/flags/flat/24/Russia.png) [![README_EN](https://raw.githubusercontent.com/gosquared/flags/master/flags/flags/flat/24/United-Kingdom.png)](https://github.com/SergeyShk/ruTS/blob/master/README.en.md)\n\n![Version](https://img.shields.io/pypi/v/ruTS?logo=pypi&logoColor=FFE873)\n[![Supported Python versions](https://img.shields.io/pypi/pyversions/ruts.svg?logo=python&logoColor=FFE873)](https://pypi.org/project/ruts/)\n![Downloads](https://img.shields.io/pypi/dm/ruTS)\n[![Build Status](https://travis-ci.com/SergeyShk/ruTS.svg?branch=master)](https://travis-ci.com/SergeyShk/ruTS)\n[![codecov](https://codecov.io/gh/SergeyShk/ruTS/branch/master/graph/badge.svg)](https://codecov.io/gh/SergeyShk/ruTS)\n![Status](https://img.shields.io/pypi/status/ruts)\n[![License](https://img.shields.io/github/license/sergeyshk/ruts.svg)](LICENSE.txt)\n![Repo size](https://img.shields.io/github/repo-size/SergeyShk/ruTS)\n![Codacy grade](https://img.shields.io/codacy/grade/5e1cef0e2fa64bdc835f7bfcb7996edc.svg?logo=codacy)\n\n<p align="center"> \n<img src="https://clipartart.com/images/free-tree-roots-clipart-black-and-white-2.png">\n</p>\n\nБиблиотека для извлечения статистик из текстов на русском языке.\n\n## Установка\n\nВыполнить:\n\n```bash\n$ pip install ruts\n```\n\nЗависимости:\n\n*   python 3.8+\n*   nltk\n*   pymorphy2\n*   razdel\n*   scipy\n*   spaCy\n*   numpy\n*   pandas\n*   matplotlib\n*   graphviz\n\n## Функционал\n\nОсновной функционал базируется на адаптированных для русского языка статистиках библиотеки [textacy](https://github.com/chartbeat-labs/textacy) и позволяет работать как непосредственно с текстами, так и с подготовленными Doc-объектами библиотеки [spaCy](https://github.com/explosion/spaCy).\n\n[API](https://ruts-api.herokuapp.com/docs) для знакомства с доступными функциями.\n\n### Извлечение объектов\n\nБиблиотека позволяет создавать свои инструменты для извлечения предложений и слов из текста, которые затем можно использовать при вычислении статистик.\n\nПример:\n\n```python\nimport re\nfrom nltk.corpus import stopwords\nfrom ruts import SentsExtractor, WordsExtractor\ntext = "Не имей 100 рублей, а имей 100 друзей"\nse = SentsExtractor(tokenizer=re.compile(r\', \'))\nse.extract(text)\n\n    (\'Не имей 100 рублей\', \'а имей 100 друзей\')\n\nwe = WordsExtractor(use_lexemes=True, stopwords=stopwords.words(\'russian\'), filter_nums=True, ngram_range=(1, 2))\nwe.extract(text)\n\n    (\'иметь\', \'рубль\', \'иметь\', \'друг\', \'иметь_рубль\', \'рубль_иметь\', \'иметь_друг\')\n   \nwe.get_most_common(3)\n\n    [(\'иметь\', 2), (\'рубль\', 1), (\'друг\', 1)]\n```\n\n### Базовые статистики\n\nБиблиотека позволяет извлекать из текста следующие статистические показатели:\n\n*   количество предложений\n*   количество слов\n*   количество уникальных слов\n*   количество длинных слов\n*   количество сложных слов\n*   количество простых слов\n*   количество односложных слов\n*   количество многосложных слов\n*   количество символов\n*   количество букв\n*   количество пробелов\n*   количество слогов\n*   количество знаков препинания\n*   распределение слов по количеству букв\n*   распределение слов по количеству слогов\n\nПример:\n\n```python\nfrom ruts import BasicStats\ntext = "Существуют три вида лжи: ложь, наглая ложь и статистика"\nbs = BasicStats(text)\nbs.get_stats()\n\n    {\'c_letters\': {1: 1, 3: 2, 4: 3, 6: 1, 10: 2},\n    \'c_syllables\': {1: 5, 2: 1, 3: 1, 4: 2},\n    \'n_chars\': 55,\n    \'n_complex_words\': 2,\n    \'n_letters\': 45,\n    \'n_long_words\': 3,\n    \'n_monosyllable_words\': 5,\n    \'n_polysyllable_words\': 4,\n    \'n_punctuations\': 2,\n    \'n_sents\': 1,\n    \'n_simple_words\': 7,\n    \'n_spaces\': 8,\n    \'n_syllables\': 18,\n    \'n_unique_words\': 8,\n    \'n_words\': 9}\n\nbs.print_stats()\n\n        Статистика     | Значение \n    ------------------------------\n    Предложения         |    1     \n    Слова               |    9     \n    Уникальные слова    |    8     \n    Длинные слова       |    3     \n    Сложные слова       |    2     \n    Простые слова       |    7     \n    Односложные слова   |    5     \n    Многосложные слова  |    4     \n    Символы             |    55    \n    Буквы               |    45    \n    Пробелы             |    8     \n    Слоги               |    18\n    Знаки препинания    |    2\n```\n\n### Метрики удобочитаемости\n\nБиблиотека позволяет вычислять для текста следующие метрики удобочитаемости:\n\n*   Тест Флеша-Кинкайда\n*   Индекс удобочитаемости Флеша\n*   Индекс Колман-Лиау\n*   Индекс SMOG\n*   Автоматический индекс удобочитаемости\n*   Индекс удобочитаемости LIX\n\nКоэффициенты метрик для русского языка были взяты из работы исследователей проекта [Plain Russian Language](https://github.com/infoculture/plainrussian), которые получили их на основе специально подобранных текстов с предварительными возрастными пометками.\n\nПример:\n\n```python\nfrom ruts import ReadabilityStats\ntext = "Ног нет, а хожу, рта нет, а скажу: когда спать, когда вставать, когда работу начинать"\nrs = ReadabilityStats(text)\nrs.get_stats()\n\n    {\'automated_readability_index\': 0.2941666666666656,\n    \'coleman_liau_index\': 0.2941666666666656,\n    \'flesch_kincaid_grade\': 3.4133333333333304,\n    \'flesch_reading_easy\': 83.16166666666666,\n    \'lix\': 48.333333333333336,\n    \'smog_index\': 0.05}\n\nrs.print_stats()\n\n                    Метрика                 | Значение \n    --------------------------------------------------\n    Тест Флеша-Кинкайда                     |   3.41   \n    Индекс удобочитаемости Флеша            |  83.16   \n    Индекс Колман-Лиау                      |   0.29   \n    Индекс SMOG                             |   0.05   \n    Автоматический индекс удобочитаемости   |   0.29   \n    Индекс удобочитаемости LIX              |  48.33  \n```\n\n### Метрики лексического разнообразия\n\nБиблиотека позволяет вычислять для текста следующие метрики лексического разнообразия:\n\n*   Type-Token Ratio (TTR)\n*   Root Type-Token Ratio (RTTR)\n*   Corrected Type-Token Ratio (CTTR)\n*   Herdan Type-Token Ratio (HTTR)\n*   Summer Type-Token Ratio (STTR)\n*   Mass Type-Token Ratio (MTTR)\n*   Dugast Type-Token Ratio (DTTR)\n*   Moving Average Type-Token Ratio (MATTR)\n*   Mean Segmental Type-Token Ratio (MSTTR)\n*   Measure of Textual Lexical Diversity (MTLD)\n*   Moving Average Measure of Textual Lexical Diversity (MAMTLD)\n*   Hypergeometric Distribution D (HD-D)\n*   Индекс Симпсона\n*   Гапакс-индекс\n\nЧасть реализаций метрик взята из проекта [lexical_diversity](https://github.com/kristopherkyle/lexical_diversity).\n\nПример:\n\n```python\nfrom ruts import DiversityStats\ntext = "Ног нет, а хожу, рта нет, а скажу: когда спать, когда вставать, когда работу начинать"\nds = DiversityStats(text)\nds.get_stats()\n\n    {\'ttr\': 0.7333333333333333,\n    \'rttr\': 2.840187787218772,\n    \'cttr\': 2.008316044185609,\n    \'httr\': 0.8854692840710253,\n    \'sttr\': 0.2500605793160845,\n    \'mttr\': 0.0973825075623254,\n    \'dttr\': 10.268784661968104,\n    \'mattr\': 0.7333333333333333,\n    \'msttr\': 0.7333333333333333,\n    \'mtld\': 15.0,\n    \'mamtld\': 11.875,\n    \'hdd\': -1,\n    \'simpson_index\': 21.0,\n    \'hapax_index\': 431.2334616537499}\n\nds.print_stats()\n\n                              Метрика                           | Значение \n    ----------------------------------------------------------------------\n    Type-Token Ratio (TTR)                                      |   0.92   \n    Root Type-Token Ratio (RTTR)                                |   7.17   \n    Corrected Type-Token Ratio (CTTR)                           |   5.07   \n    Herdan Type-Token Ratio (HTTR)                              |   0.98   \n    Summer Type-Token Ratio (STTR)                              |   0.96   \n    Mass Type-Token Ratio (MTTR)                                |   0.01   \n    Dugast Type-Token Ratio (DTTR)                              |  85.82   \n    Moving Average Type-Token Ratio (MATTR)                     |   0.91   \n    Mean Segmental Type-Token Ratio (MSTTR)                     |   0.94   \n    Measure of Textual Lexical Diversity (MTLD)                 |  208.38  \n    Moving Average Measure of Textual Lexical Diversity (MTLD)  |   1.00   \n    Hypergeometric Distribution D (HD-D)                        |   0.94   \n    Индекс Симпсона                                             |  305.00  \n    Гапакс-индекс                                               | 2499.46  \n```\n\n### Морфологические статистики\n\nБиблиотека позволяет извлекать из текста следующие морфологические признаки:\n\n*   часть речи\n*   одушевленность\n*   вид\n*   падеж\n*   род\n*   совместность\n*   наклонение\n*   число\n*   лицо\n*   время\n*   переходность\n*   залог\n\nДля морфологического разбора текста используется библиотека [pymorphy2](https://github.com/kmike/pymorphy2). Описание статистик взяты из корпуса [OpenCorpora](http://opencorpora.org/dict.php?act=gram).\n\nПример:\n\n```python\nfrom ruts import MorphStats\ntext = "Постарайтесь получить то, что любите, иначе придется полюбить то, что получили"\nms = MorphStats(text)\nms.pos\n\n    (\'VERB\', \'INFN\', \'CONJ\', \'CONJ\', \'VERB\', \'ADVB\', \'VERB\', \'INFN\', \'CONJ\', \'CONJ\', \'VERB\')\n\nms.get_stats()\n\n    {\'animacy\': {None: 11},\n    \'aspect\': {None: 5, \'impf\': 1, \'perf\': 5},\n    \'case\': {None: 11},\n    \'gender\': {None: 11},\n    \'involvement\': {None: 10, \'excl\': 1},\n    \'mood\': {None: 7, \'impr\': 1, \'indc\': 3},\n    \'number\': {None: 7, \'plur\': 3, \'sing\': 1},\n    \'person\': {None: 9, \'2per\': 1, \'3per\': 1},\n    \'pos\': {\'ADVB\': 1, \'CONJ\': 4, \'INFN\': 2, \'VERB\': 4},\n    \'tense\': {None: 8, \'futr\': 1, \'past\': 1, \'pres\': 1},\n    \'transitivity\': {None: 5, \'intr\': 2, \'tran\': 4},\n    \'voice\': {None: 11}}\n\nms.explain_text(filter_none=True)\n\n    ((\'Постарайтесь\',\n        {\'aspect\': \'perf\',\n        \'involvement\': \'excl\',\n        \'mood\': \'impr\',\n        \'number\': \'plur\',\n        \'pos\': \'VERB\',\n        \'transitivity\': \'intr\'}),\n    (\'получить\', {\'aspect\': \'perf\', \'pos\': \'INFN\', \'transitivity\': \'tran\'}),\n    (\'то\', {\'pos\': \'CONJ\'}),\n    (\'что\', {\'pos\': \'CONJ\'}),\n    (\'любите\',\n        {\'aspect\': \'impf\',\n        \'mood\': \'indc\',\n        \'number\': \'plur\',\n        \'person\': \'2per\',\n        \'pos\': \'VERB\',\n        \'tense\': \'pres\',\n        \'transitivity\': \'tran\'}),\n    (\'иначе\', {\'pos\': \'ADVB\'}),\n    (\'придется\',\n        {\'aspect\': \'perf\',\n        \'mood\': \'indc\',\n        \'number\': \'sing\',\n        \'person\': \'3per\',\n        \'pos\': \'VERB\',\n        \'tense\': \'futr\',\n        \'transitivity\': \'intr\'}),\n    (\'полюбить\', {\'aspect\': \'perf\', \'pos\': \'INFN\', \'transitivity\': \'tran\'}),\n    (\'то\', {\'pos\': \'CONJ\'}),\n    (\'что\', {\'pos\': \'CONJ\'}),\n    (\'получили\',\n        {\'aspect\': \'perf\',\n        \'mood\': \'indc\',\n        \'number\': \'plur\',\n        \'pos\': \'VERB\',\n        \'tense\': \'past\',\n        \'transitivity\': \'tran\'}))\n\nms.print_stats(\'pos\', \'tense\')\n\n    ---------------Часть речи---------------\n    Глагол (личная форма)         |    4     \n    Союз                          |    4     \n    Глагол (инфинитив)            |    2     \n    Наречие                       |    1     \n\n    -----------------Время------------------\n    Неизвестно                    |    8     \n    Настоящее                     |    1     \n    Будущее                       |    1     \n    Прошедшее                     |    1 \n```\n\n### Наборы данных\n\nБиблиотека позволяет работать с несколькими заранее предобработанными наборами данных:\n\n*   sov_chrest_lit - советские хрестоматии по литературе\n*   stalin_works - полное собрание сочинений И.В. Сталина\n\nСуществует возможность работать как с чистыми текстами (без заголовочной информации), так и с записями, а также фильтровать их по различным критериям.\n\nПример:\n\n```python\nfrom ruts.datasets import SovChLit\nsc = SovChLit()\nsc.info\n\n    {\'description\': \'Корпус советских хрестоматий по литературе\',\n    \'url\': \'https://dataverse.harvard.edu/file.xhtml?fileId=3670902&version=DRAFT\',\n    \'Наименование\': \'sov_chrest_lit\'}\n\nfor i in sc.get_records(max_len=100, category=\'Весна\', limit=1):\n    pprint(i)\n\n    {\'author\': \'Е. Трутнева\',\n    \'book\': \'Родная речь. Книга для чтения в I классе начальной школы\',\n    \'category\': \'Весна\',\n    \'file\': PosixPath(\'../ruTS/ruts_data/texts/sov_chrest_lit/grade_1/155\'),\n    \'grade\': 1,\n    \'subject\': \'Дождик\',\n    \'text\': \'Дождик, дождик, поливай, будет хлеба каравай!\\n\'\n            \'Дождик, дождик, припусти, дай гороху подрасти!\',\n    \'type\': \'Стихотворение\',\n    \'year\': 1963}\n\nfor i in sc.get_texts(text_type=\'Басня\', limit=1):\n    pprint(i)\n\n    (\'— Соседка, слышала ль ты добрую молву? — вбежавши, крысе мышь сказала:\\n\'\n    \'— Ведь кошка, говорят, попалась в когти льву. Вот отдохнуть и нам пора \'\n    \'настала!\\n\'\n    \'— Не радуйся, мой свет,— ей крыса говорит в ответ,— и не надейся \'\n    \'по-пустому.\\n\'\n    \'Коль до когтей у них дойдёт, то, верно, льву не быть живому: сильнее кошки \'\n    \'зверя нет.\')\n```\n\n### Визуализация\n\nБиблиотека позволяет визуализировать тексты с помощью следующих видов графиков:\n\n*   Закон Ципфа (Zipf\'s law)\n*   Литературная дактилоскопия (Literature Fingerprinting)\n*   Дерево слов (Word Tree)\n\nПример:\n\n```python\nfrom collections import Counter\nfrom nltk.corpus import stopwords\nfrom ruts import WordsExtractor\nfrom ruts.datasets import SovChLit\nfrom ruts.visualizers import zipf\n\nsc = SovChLit()\ntext = \'\\n\'.join([text for text in sc.get_texts(limit=100)])\nwe = WordsExtractor(use_lexemes=True, stopwords=stopwords.words(\'russian\'), filter_nums=True)\ntokens_with_count = Counter(we.extract(text))\nzipf(tokens_with_count, num_words=100, num_labels=10, log=False, show_theory=True, alpha=1.1)\n```\n\n### Компоненты\n\nБиблиотека позволяет создавать компоненты spaCy для следующих классов:\n\n*   BasicStats\n*   DiversityStats\n*   MorphStats\n*   ReadabilityStats\n\nРусскоязычную модель spaCy можно скачать, выполнив команду:\n\n```bash\n$ python -m spacy download ru_core_news_sm\n```\n\nПример:\n\n```python\nimport ruts\nimport spacy\nnlp = spacy.load(\'ru_core_news_sm\')\nnlp.add_pipe(\'basic\', last=True)\ndoc = nlp("Существуют три вида лжи: ложь, наглая ложь и статистика")\ndoc._.basic.c_letters\n\n    {1: 1, 3: 2, 4: 3, 6: 1, 10: 2}\n\ndoc._.basic.get_stats()\n\n    {\'c_letters\': {1: 1, 3: 2, 4: 3, 6: 1, 10: 2},\n    \'c_syllables\': {1: 5, 2: 1, 3: 1, 4: 2},\n    \'n_chars\': 55,\n    \'n_complex_words\': 2,\n    \'n_letters\': 45,\n    \'n_long_words\': 3,\n    \'n_monosyllable_words\': 5,\n    \'n_polysyllable_words\': 4,\n    \'n_punctuations\': 2,\n    \'n_sents\': 1,\n    \'n_simple_words\': 7,\n    \'n_spaces\': 8,\n    \'n_syllables\': 18,\n    \'n_unique_words\': 8,\n    \'n_words\': 9}\n```\n\n## Структура проекта\n\n*   **docs** - документация по проекту\n*   **ruts**:\n    *   basic_stats.py - базовые текстовые статистики\n    *   components.py - компоненты spaCy\n    *   constants.py - основные используемые константы\n    *   diversity_stats.py - метрики лексического разнообразия текста\n    *   extractors.py - инструменты для извлечения объектов из текста\n    *   morph_stats.py - морфологические статистики\n    *   readability_stats.py - метрики удобочитаемости текста\n    *   utils.py - вспомогательные инструменты\n    *   **datasets** - наборы данных:\n        *   dataset.py - базовый класс для работы с наборами данных\n        *   sov_chrest_lit.py - советские хрестоматии по литературе\n        *   stalin_works.py - полное собрание сочинений И.В. Сталина\n    *   **visualizers** - инструменты для визуализации текстов:\n        *   fingerprinting.py - Литературная дактилоскопия\n        *   word_tree.py - Дерево слов\n        *   zipf.py - Закон Ципфа\n*   **tests**:\n    *   test_basic_stats.py - тесты базовых текстовых статистик\n    *   test_components.py - тесты компонентов spaCy\n    *   test_diversity_stats.py - тесты метрик лексического разнообразия текста\n    *   test_extractors.py - тесты инструментов для извлечения объектов из текста\n    *   test_morph_stats - тесты морфологических статистик\n    *   test_readability_stats.py - тесты метрик удобочитаемости текста\n    *   **datasets** - тесты наборов данных:\n        *   test_dataset.py - тесты базового класса для работы с наборами данных\n        *   test_sov_chrest_lit.py - тесты набора данных советских хрестоматий по литературе\n        *   test_stalin_works.py - тесты набора данных полного собрания сочинений И.В. Сталина\n    *   **visualizers** - тесты инструментов для визуализации текстов:\n        *   test_fingerprinting.py - тесты визуализации Литературная дактилоскопия\n        *   test_word_tree.py - тесты визуализации Дерево слов\n        *   test_zipf.py - тесты визуализации Закон Ципфа\n\n## Авторы\n\n*   Шкарин Сергей (kouki.sergey@gmail.com)\n*   Смирнова Екатерина (ekanerina@yandex.ru)\n\n## Атрибуция\n\nПожалуйста, используйте следующую BibTeX нотацию для цитирования библиотеки **ruTS**, если вы используете ее в своих исследованиях или программах.\nЦитирование является очень полезным для дальнейшей разработки и поддержки данного проекта.\n\n```\n@software{ruTS,\n  author = {Sergey Shkarin},\n  title = {{ruTS, a library for statistics extraction from texts in Russian}},\n  year = 2022,\n  publisher = {Moscow},\n  url = {https://github.com/SergeyShk/ruTS}\n}\n```',
    'author': 'Шкарин Сергей',
    'author_email': 'kouki.sergey@gmail.com',
    'maintainer': 'Шкарин Сергей',
    'maintainer_email': 'kouki.sergey@gmail.com',
    'url': 'https://github.com/SergeyShk/ruTS',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<3.12',
}


setup(**setup_kwargs)
