"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WaiterStateMachine = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("aws-cdk-lib/core");
const constructs_1 = require("constructs");
const providers_1 = require("./providers");
/**
 * A very simple StateMachine construct highly customized to the provider framework.
 * This is so that this package does not need to depend on aws-stepfunctions module.
 *
 * The state machine continuously calls the isCompleteHandler, until it succeeds or times out.
 * The handler is called `maxAttempts` times with an `interval` duration and a `backoffRate` rate.
 *
 * For example with:
 * - maxAttempts = 360 (30 minutes)
 * - interval = 5
 * - backoffRate = 1 (no backoff)
 *
 * it will make the API Call every 5 seconds and fail after 360 failures.
 *
 * If the backoff rate is changed to 2 (for example), it will
 * - make the first call
 * - wait 5 seconds
 * - make the second call
 * - wait 15 seconds
 * - etc.
 */
class WaiterStateMachine extends constructs_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_WaiterStateMachineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, WaiterStateMachine);
            }
            throw error;
        }
        const interval = props.interval || core_1.Duration.seconds(5);
        const totalTimeout = props.totalTimeout || core_1.Duration.minutes(30);
        const maxAttempts = calculateMaxRetries(totalTimeout.toSeconds(), interval.toSeconds(), props.backoffRate ?? 1);
        if (Math.round(maxAttempts) !== maxAttempts) {
            throw new Error(`Cannot determine retry count since totalTimeout=${totalTimeout.toSeconds()}s is not integrally dividable by queryInterval=${interval.toSeconds()}s`);
        }
        this.isCompleteProvider = new providers_1.AssertionsProvider(this, 'IsCompleteProvider', {
            handler: 'index.isComplete',
            uuid: '76b3e830-a873-425f-8453-eddd85c86925',
        });
        const timeoutProvider = new providers_1.AssertionsProvider(this, 'TimeoutProvider', {
            handler: 'index.onTimeout',
            uuid: '5c1898e0-96fb-4e3e-95d5-f6c67f3ce41a',
        });
        const role = new core_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'states.amazonaws.com' } }],
                },
                Policies: [
                    {
                        PolicyName: 'InlineInvokeFunctions',
                        PolicyDocument: {
                            Version: '2012-10-17',
                            Statement: [{
                                    Action: 'lambda:InvokeFunction',
                                    Effect: 'Allow',
                                    Resource: [
                                        this.isCompleteProvider.serviceToken,
                                        timeoutProvider.serviceToken,
                                    ],
                                }],
                        },
                    },
                ],
            },
        });
        const definition = core_1.Stack.of(this).toJsonString({
            StartAt: 'framework-isComplete-task',
            States: {
                'framework-isComplete-task': {
                    End: true,
                    Retry: [{
                            ErrorEquals: ['States.ALL'],
                            IntervalSeconds: interval.toSeconds(),
                            MaxAttempts: maxAttempts,
                            BackoffRate: props.backoffRate ?? 1,
                        }],
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'framework-onTimeout-task',
                        }],
                    Type: 'Task',
                    Resource: this.isCompleteProvider.serviceToken,
                },
                'framework-onTimeout-task': {
                    End: true,
                    Type: 'Task',
                    Resource: timeoutProvider.serviceToken,
                },
            },
        });
        const resource = new core_1.CfnResource(this, 'Resource', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.getAtt('Arn'),
            },
        });
        resource.node.addDependency(role);
        this.stateMachineArn = resource.ref;
        this.roleArn = role.getAtt('Arn').toString();
        this.isCompleteProvider.grantInvoke(this.roleArn);
        timeoutProvider.grantInvoke(this.roleArn);
    }
}
exports.WaiterStateMachine = WaiterStateMachine;
_a = JSII_RTTI_SYMBOL_1;
WaiterStateMachine[_a] = { fqn: "@aws-cdk/integ-tests-alpha.WaiterStateMachine", version: "2.151.1-alpha.0" };
/**
 * Calculate the max number of retries
 */
function calculateMaxRetries(maxSeconds, intervalSeconds, backoff) {
    // if backoff === 1 then we aren't really using backoff
    if (backoff === 1) {
        return Math.floor(maxSeconds / intervalSeconds);
    }
    let retries = 1;
    let nextInterval = intervalSeconds;
    let i = 0;
    while (i < maxSeconds) {
        nextInterval = nextInterval + nextInterval * backoff;
        i += nextInterval;
        if (i >= maxSeconds)
            break;
        retries++;
    }
    return retries;
}
//# sourceMappingURL=data:application/json;base64,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