"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvokeFunction = exports.InvocationType = exports.LogType = exports.AwsApiCall = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("aws-cdk-lib/core");
const api_call_base_1 = require("./api-call-base");
const providers_1 = require("./providers");
const waiter_state_machine_1 = require("./waiter-state-machine");
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
class AwsApiCall extends api_call_base_1.ApiCallBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_AwsApiCallProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AwsApiCall);
            }
            throw error;
        }
        this.provider = new providers_1.AssertionsProvider(this, 'SdkProvider', {
            logRetention: props.parameters?.RetentionDays,
        });
        this.provider.addPolicyStatementFromSdkCall(props.service, props.api);
        this.name = `${props.service}${props.api}`;
        this.api = props.api;
        this.service = props.service;
        if (props.outputPaths) {
            this.outputPaths = [...props.outputPaths];
        }
        this.apiCallResource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.provider.serviceToken,
            properties: {
                service: props.service,
                api: props.api,
                expected: core_1.Lazy.any({ produce: () => this.expectedResult }),
                actualPath: core_1.Lazy.string({ produce: () => this._assertAtPath }),
                stateMachineArn: core_1.Lazy.string({ produce: () => this.stateMachineArn }),
                parameters: this.provider.encode(props.parameters),
                flattenResponse: core_1.Lazy.string({ produce: () => this.flattenResponse }),
                outputPaths: core_1.Lazy.list({ produce: () => this.outputPaths }),
                salt: Date.now().toString(),
            },
            // Remove the slash from the resource type because when using the v3 package name as the service name,
            // the `service` props includes the slash, but the resource type name cannot contain the slash
            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html#aws-resource-cloudformation-customresource--remarks
            resourceType: `${providers_1.SDK_RESOURCE_TYPE_PREFIX}${this.name}`.substring(0, 60).replace(/[\/]/g, ''),
        });
        // Needed so that all the policies set up by the provider should be available before the custom resource is provisioned.
        this.apiCallResource.node.addDependency(this.provider);
        // if expectedResult has been configured then that means
        // we are making assertions and we should output the results
        core_1.Aspects.of(this).add({
            visit(node) {
                if (node instanceof AwsApiCall) {
                    if (node.expectedResult) {
                        const result = node.apiCallResource.getAttString('assertion');
                        new core_1.CfnOutput(node, 'AssertionResults', {
                            value: result,
                            // Remove the at sign, slash, and hyphen because when using the v3 package name or client name as the service name,
                            // the `id` includes them, but they are not allowed in the `CfnOutput` logical id
                            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html#outputs-section-syntax
                        }).overrideLogicalId(`AssertionResults${id}`.replace(/[\@\/\-]/g, ''));
                    }
                }
            },
        });
    }
    assertAtPath(path, expected) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_ExpectedResult(expected);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.assertAtPath);
            }
            throw error;
        }
        this._assertAtPath = path;
        (this.outputPaths ?? (this.outputPaths = [])).push(path);
        this.expectedResult = expected.result;
        this.flattenResponse = 'true';
        return this;
    }
    waitForAssertions(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_WaiterStateMachineOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForAssertions);
            }
            throw error;
        }
        const waiter = new waiter_state_machine_1.WaiterStateMachine(this, 'WaitFor', {
            ...options,
        });
        this.stateMachineArn = waiter.stateMachineArn;
        this.provider.addPolicyStatementFromSdkCall('states', 'StartExecution');
        waiter.isCompleteProvider.addPolicyStatementFromSdkCall(this.service, this.api);
        this.waiterProvider = waiter.isCompleteProvider;
        return this;
    }
}
exports.AwsApiCall = AwsApiCall;
_a = JSII_RTTI_SYMBOL_1;
AwsApiCall[_a] = { fqn: "@aws-cdk/integ-tests-alpha.AwsApiCall", version: "2.151.1-alpha.0" };
/**
 * Set to Tail to include the execution log in the response.
 * Applies to synchronously invoked functions only.
 */
var LogType;
(function (LogType) {
    /**
     * The log messages are not returned in the response
     */
    LogType["NONE"] = "None";
    /**
     * The log messages are returned in the response
     */
    LogType["TAIL"] = "Tail";
})(LogType || (exports.LogType = LogType = {}));
/**
 * The type of invocation. Default is REQUEST_RESPONSE
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke the function asynchronously.
     * Send events that fail multiple times to the function's
     * dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * Invoke the function synchronously.
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Validate parameter values and verify that the user
     * or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType || (exports.InvocationType = InvocationType = {}));
/**
 * An AWS Lambda Invoke function API call.
 * Use this instead of the generic AwsApiCall in order to
 * invoke a lambda function. This will automatically create
 * the correct permissions to invoke the function
 */
class LambdaInvokeFunction extends AwsApiCall {
    constructor(scope, id, props) {
        super(scope, id, {
            api: 'invoke',
            service: 'Lambda',
            parameters: {
                FunctionName: props.functionName,
                InvocationType: props.invocationType,
                LogType: props.logType,
                Payload: props.payload,
                RetentionDays: props.logRetention,
            },
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_LambdaInvokeFunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LambdaInvokeFunction);
            }
            throw error;
        }
        const stack = core_1.Stack.of(this);
        // need to give the assertion lambda permission to invoke
        new core_1.CfnResource(this, 'Invoke', {
            type: 'AWS::Lambda::Permission',
            properties: {
                Action: 'lambda:InvokeFunction',
                FunctionName: props.functionName,
                Principal: this.provider.handlerRoleArn,
            },
        });
        // the api call is 'invoke', but the permission is 'invokeFunction'
        // so need to handle it specially
        this.provider.addPolicyStatementFromSdkCall('Lambda', 'invokeFunction', [stack.formatArn({
                service: 'lambda',
                resource: 'function',
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                resourceName: props.functionName,
            })]);
        // If using `waitForAssertions`, do the same for `waiterProvider` as above.
        // Aspects are used here because we do not know if the user is using `waitForAssertions` at this point.
        core_1.Aspects.of(this).add({
            visit(node) {
                if (node instanceof AwsApiCall && node.waiterProvider) {
                    node.waiterProvider.addPolicyStatementFromSdkCall('Lambda', 'invokeFunction', [stack.formatArn({
                            service: 'lambda',
                            resource: 'function',
                            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                            resourceName: props.functionName,
                        })]);
                }
            },
        });
    }
}
exports.LambdaInvokeFunction = LambdaInvokeFunction;
_b = JSII_RTTI_SYMBOL_1;
LambdaInvokeFunction[_b] = { fqn: "@aws-cdk/integ-tests-alpha.LambdaInvokeFunction", version: "2.151.1-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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