import { CustomResource } from 'aws-cdk-lib/core';
import { Construct } from 'constructs';
import { ApiCallBase, IApiCall } from './api-call-base';
import { ExpectedResult } from './common';
import { AssertionsProvider } from './providers';
import { WaiterStateMachineOptions } from './waiter-state-machine';
import { RetentionDays } from 'aws-cdk-lib/aws-logs';
/**
 * Options to perform an AWS JavaScript V2 API call
 */
export interface AwsApiCallOptions {
    /**
     * The AWS service, i.e. S3
     */
    readonly service: string;
    /**
     * The api call to make, i.e. getBucketLifecycle
     */
    readonly api: string;
    /**
     * Any parameters to pass to the api call
     *
     * @default - no parameters
     */
    readonly parameters?: any;
    /**
     * Restrict the data returned by the API call to specific paths in
     * the API response. Use this to limit the data returned by the custom
     * resource if working with API calls that could potentially result in custom
     * response objects exceeding the hard limit of 4096 bytes.
     *
     * @default - return all data
     */
    readonly outputPaths?: string[];
}
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
export interface AwsApiCallProps extends AwsApiCallOptions {
}
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
export declare class AwsApiCall extends ApiCallBase {
    readonly provider: AssertionsProvider;
    /**
     * access the AssertionsProvider for the waiter state machine.
     * This can be used to add additional IAM policies
     * the the provider role policy
     *
     * @example
     * declare const apiCall: AwsApiCall;
     * apiCall.waiterProvider?.addToRolePolicy({
     *   Effect: 'Allow',
     *   Action: ['s3:GetObject'],
     *   Resource: ['*'],
     * });
     */
    waiterProvider?: AssertionsProvider;
    protected readonly apiCallResource: CustomResource;
    private readonly name;
    private _assertAtPath?;
    private readonly api;
    private readonly service;
    constructor(scope: Construct, id: string, props: AwsApiCallProps);
    assertAtPath(path: string, expected: ExpectedResult): IApiCall;
    waitForAssertions(options?: WaiterStateMachineOptions): IApiCall;
}
/**
 * Set to Tail to include the execution log in the response.
 * Applies to synchronously invoked functions only.
 */
export declare enum LogType {
    /**
     * The log messages are not returned in the response
     */
    NONE = "None",
    /**
     * The log messages are returned in the response
     */
    TAIL = "Tail"
}
/**
 * The type of invocation. Default is REQUEST_RESPONSE
 */
export declare enum InvocationType {
    /**
     * Invoke the function asynchronously.
     * Send events that fail multiple times to the function's
     * dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    EVENT = "Event",
    /**
     * Invoke the function synchronously.
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    REQUEST_RESPONSE = "RequestResponse",
    /**
     * Validate parameter values and verify that the user
     * or role has permission to invoke the function.
     */
    DRY_RUN = "DryRun"
}
/**
 * Options to pass to the Lambda invokeFunction API call
 */
export interface LambdaInvokeFunctionProps {
    /**
     * The name of the function to invoke
     */
    readonly functionName: string;
    /**
     * The type of invocation to use
     *
     * @default InvocationType.REQUEST_RESPONSE
     */
    readonly invocationType?: InvocationType;
    /**
     * Whether to return the logs as part of the response
     *
     * @default LogType.NONE
     */
    readonly logType?: LogType;
    /**
     * How long, in days, the log contents will be retained.
     *
     * @default - no retention days specified
     */
    readonly logRetention?: RetentionDays;
    /**
     * Payload to send as part of the invoke
     *
     * @default - no payload
     */
    readonly payload?: string;
}
/**
 * An AWS Lambda Invoke function API call.
 * Use this instead of the generic AwsApiCall in order to
 * invoke a lambda function. This will automatically create
 * the correct permissions to invoke the function
 */
export declare class LambdaInvokeFunction extends AwsApiCall {
    constructor(scope: Construct, id: string, props: LambdaInvokeFunctionProps);
}
