"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionHandler = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const helpers_internal_1 = require("aws-cdk-lib/assertions/lib/helpers-internal");
const base_1 = require("./base");
class AssertionHandler extends base_1.CustomResourceHandler {
    async processEvent(request) {
        let actual = decodeCall(request.actual);
        const expected = decodeCall(request.expected);
        let result;
        const matcher = new MatchCreator(expected).getMatcher();
        console.log(`Testing equality between ${JSON.stringify(request.actual)} and ${JSON.stringify(request.expected)}`);
        const matchResult = matcher.test(actual);
        matchResult.finished();
        if (matchResult.hasFailed()) {
            result = {
                failed: true,
                assertion: JSON.stringify({
                    status: 'fail',
                    message: matchResult.renderMismatch(),
                }),
            };
            if (request.failDeployment) {
                throw new Error(result.assertion);
            }
        }
        else {
            result = {
                assertion: JSON.stringify({
                    status: 'success',
                }),
            };
        }
        return result;
    }
}
exports.AssertionHandler = AssertionHandler;
class MatchCreator {
    constructor(obj) {
        this.parsedObj = {
            matcher: obj,
        };
    }
    /**
     * Return a Matcher that can be tested against the actual results.
     * This will convert the encoded matchers into their corresponding
     * assertions matcher.
     *
     * For example:
     *
     * ExpectedResult.objectLike({
     *   Messages: [{
     *     Body: Match.objectLike({
     *       Elements: Match.arrayWith([{ Asdf: 3 }]),
     *       Payload: Match.serializedJson({ key: 'value' }),
     *     }),
     *   }],
     * });
     *
     * Will be encoded as:
     * {
     *   $ObjectLike: {
     *     Messages: [{
     *       Body: {
     *         $ObjectLike: {
     *           Elements: {
     *             $ArrayWith: [{ Asdf: 3 }],
     *           },
     *           Payload: {
     *             $SerializedJson: { key: 'value' }
     *           }
     *         },
     *       },
     *     }],
     *   },
     * }
     *
     * Which can then be parsed by this function. For each key (recursively)
     * the parser will check if the value has one of the encoded matchers as a key
     * and if so, it will set the value as the Matcher. So,
     *
     * {
     *   Body: {
     *     $ObjectLike: {
     *       Elements: {
     *         $ArrayWith: [{ Asdf: 3 }],
     *       },
     *       Payload: {
     *         $SerializedJson: { key: 'value' }
     *       }
     *     },
     *   },
     * }
     *
     * Will be converted to
     * {
     *   Body: Match.objectLike({
     *     Elements: Match.arrayWith([{ Asdf: 3 }]),
     *     Payload: Match.serializedJson({ key: 'value' }),
     *   }),
     * }
     */
    getMatcher() {
        try {
            const final = JSON.parse(JSON.stringify(this.parsedObj), function (_k, v) {
                const nested = Object.keys(v)[0];
                switch (nested) {
                    case '$ArrayWith':
                        return helpers_internal_1.Match.arrayWith(v[nested]);
                    case '$ObjectLike':
                        return helpers_internal_1.Match.objectLike(v[nested]);
                    case '$StringLike':
                        return helpers_internal_1.Match.stringLikeRegexp(v[nested]);
                    case '$SerializedJson':
                        return helpers_internal_1.Match.serializedJson(v[nested]);
                    default:
                        return v;
                }
            });
            if (helpers_internal_1.Matcher.isMatcher(final.matcher)) {
                return final.matcher;
            }
            return helpers_internal_1.Match.exact(final.matcher);
        }
        catch {
            return helpers_internal_1.Match.exact(this.parsedObj.matcher);
        }
    }
}
function decodeCall(call) {
    if (!call) {
        return undefined;
    }
    try {
        const parsed = JSON.parse(call);
        return parsed;
    }
    catch {
        return call;
    }
}
//# sourceMappingURL=data:application/json;base64,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