"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployAssert = void 0;
const core_1 = require("aws-cdk-lib/core");
const constructs_1 = require("constructs");
const assertions_1 = require("../assertions");
const http_call_1 = require("../http-call");
const hash_1 = require("../private/hash");
const sdk_1 = require("../sdk");
const DEPLOY_ASSERT_SYMBOL = Symbol.for('@aws-cdk/integ-tests.DeployAssert');
/**
 * Construct that allows for registering a list of assertions
 * that should be performed on a construct
 */
class DeployAssert extends constructs_1.Construct {
    /**
     * Returns whether the construct is a DeployAssert construct
     */
    static isDeployAssert(x) {
        return x !== null && typeof (x) === 'object' && DEPLOY_ASSERT_SYMBOL in x;
    }
    /**
     * Finds a DeployAssert construct in the given scope
     */
    static of(construct) {
        const scopes = constructs_1.Node.of(constructs_1.Node.of(construct).root).findAll();
        const deployAssert = scopes.find(s => DeployAssert.isDeployAssert(s));
        if (!deployAssert) {
            throw new Error('No DeployAssert construct found in scopes');
        }
        return deployAssert;
    }
    constructor(scope, props) {
        super(scope, 'Default');
        this.assertionIdCounts = new Map();
        this.scope = props?.stack ?? new core_1.Stack(scope, 'DeployAssert');
        Object.defineProperty(this, DEPLOY_ASSERT_SYMBOL, { value: true });
    }
    awsApiCall(service, api, parameters, outputPaths) {
        let hash = '';
        try {
            hash = (0, hash_1.md5hash)(this.scope.resolve(parameters));
        }
        catch { }
        return new sdk_1.AwsApiCall(this.scope, this.uniqueAssertionId(`AwsApiCall${service}${api}${hash}`), {
            api,
            service,
            parameters,
            outputPaths,
        });
    }
    httpApiCall(url, options) {
        let hash = '';
        try {
            hash = (0, hash_1.md5hash)(this.scope.resolve({
                url,
                options,
            }));
        }
        catch { }
        let append = '';
        if (!core_1.Token.isUnresolved(url)) {
            const parsedUrl = new URL(url);
            append = `${parsedUrl.hostname}${parsedUrl.pathname}`;
        }
        return new http_call_1.HttpApiCall(this.scope, this.uniqueAssertionId(`HttpApiCall${append}${hash}`), {
            url,
            fetchOptions: options,
        });
    }
    invokeFunction(props) {
        const hash = (0, hash_1.md5hash)(this.scope.resolve(props));
        return new sdk_1.LambdaInvokeFunction(this.scope, this.uniqueAssertionId(`LambdaInvoke${hash}`), props);
    }
    expect(id, expected, actual) {
        new assertions_1.EqualsAssertion(this.scope, `EqualsAssertion${id}`, {
            expected,
            actual,
        });
    }
    /**
     * Gets a unique logical id based on a proposed assertion id.
     */
    uniqueAssertionId(id) {
        const count = this.assertionIdCounts.get(id);
        if (count === undefined) {
            // If we've never seen this id before, we'll return the id unchanged
            // to maintain backward compatibility.
            this.assertionIdCounts.set(id, 1);
            return id;
        }
        // Otherwise, we'll increment the counter and return a unique id.
        this.assertionIdCounts.set(id, count + 1);
        return `${id}${count}`;
    }
}
exports.DeployAssert = DeployAssert;
//# sourceMappingURL=data:application/json;base64,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