"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LoadBalancer = exports.LoadBalancerGeneration = void 0;
/**
 * The generations of AWS load balancing solutions.
 *
 * @stability stable
 */
var LoadBalancerGeneration;
(function (LoadBalancerGeneration) {
    LoadBalancerGeneration[LoadBalancerGeneration["FIRST"] = 0] = "FIRST";
    LoadBalancerGeneration[LoadBalancerGeneration["SECOND"] = 1] = "SECOND";
})(LoadBalancerGeneration = exports.LoadBalancerGeneration || (exports.LoadBalancerGeneration = {}));
/**
 * An interface of an abstract load balancer, as needed by CodeDeploy.
 *
 * Create instances using the static factory methods:
 * {@link #classic}, {@link #application} and {@link #network}.
 *
 * @stability stable
 */
class LoadBalancer {
    /**
     * Creates a new CodeDeploy load balancer from a Classic ELB Load Balancer.
     *
     * @param loadBalancer a classic ELB Load Balancer.
     * @stability stable
     */
    static classic(loadBalancer) {
        class ClassicLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.FIRST;
                this.name = loadBalancer.loadBalancerName;
            }
        }
        return new ClassicLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from an Application Load Balancer Target Group.
     *
     * @param albTargetGroup an ALB Target Group.
     * @stability stable
     */
    static application(albTargetGroup) {
        class AlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = albTargetGroup.targetGroupName;
            }
        }
        return new AlbLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from a Network Load Balancer Target Group.
     *
     * @param nlbTargetGroup an NLB Target Group.
     * @stability stable
     */
    static network(nlbTargetGroup) {
        class NlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = nlbTargetGroup.targetGroupName;
            }
        }
        return new NlbLoadBalancer();
    }
}
exports.LoadBalancer = LoadBalancer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibG9hZC1iYWxhbmNlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImxvYWQtYmFsYW5jZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7QUFNQSxJQUFZLHNCQVVYO0FBVkQsV0FBWSxzQkFBc0I7SUFJaEMscUVBQVMsQ0FBQTtJQUtULHVFQUFVLENBQUE7QUFDWixDQUFDLEVBVlcsc0JBQXNCLEdBQXRCLDhCQUFzQixLQUF0Qiw4QkFBc0IsUUFVakM7Ozs7Ozs7OztBQU9ELE1BQXNCLFlBQVk7Ozs7Ozs7SUFNekIsTUFBTSxDQUFDLE9BQU8sQ0FBQyxZQUE4QjtRQUNsRCxNQUFNLG1CQUFvQixTQUFRLFlBQVk7WUFBOUM7O2dCQUNrQixlQUFVLEdBQUcsc0JBQXNCLENBQUMsS0FBSyxDQUFDO2dCQUMxQyxTQUFJLEdBQUcsWUFBWSxDQUFDLGdCQUFnQixDQUFDO1lBQ3ZELENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxtQkFBbUIsRUFBRSxDQUFDO0lBQ25DLENBQUM7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsV0FBVyxDQUFDLGNBQTRDO1FBQ3BFLE1BQU0sZUFBZ0IsU0FBUSxZQUFZO1lBQTFDOztnQkFDa0IsZUFBVSxHQUFHLHNCQUFzQixDQUFDLE1BQU0sQ0FBQztnQkFDM0MsU0FBSSxHQUFHLGNBQWMsQ0FBQyxlQUFlLENBQUM7WUFDeEQsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLGVBQWUsRUFBRSxDQUFDO0lBQy9CLENBQUM7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsT0FBTyxDQUFDLGNBQXdDO1FBQzVELE1BQU0sZUFBZ0IsU0FBUSxZQUFZO1lBQTFDOztnQkFDa0IsZUFBVSxHQUFHLHNCQUFzQixDQUFDLE1BQU0sQ0FBQztnQkFDM0MsU0FBSSxHQUFHLGNBQWMsQ0FBQyxlQUFlLENBQUM7WUFDeEQsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLGVBQWUsRUFBRSxDQUFDO0lBQy9CLENBQUM7Q0FJRjtBQTdDRCxvQ0E2Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlbGIgZnJvbSAnQGF3cy1jZGsvYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5nJztcbmltcG9ydCAqIGFzIGVsYnYyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZ3YyJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIExvYWRCYWxhbmNlckdlbmVyYXRpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBGSVJTVCA9IDAsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFNFQ09ORCA9IDFcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIExvYWRCYWxhbmNlciB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjbGFzc2ljKGxvYWRCYWxhbmNlcjogZWxiLkxvYWRCYWxhbmNlcik6IExvYWRCYWxhbmNlciB7XG4gICAgY2xhc3MgQ2xhc3NpY0xvYWRCYWxhbmNlciBleHRlbmRzIExvYWRCYWxhbmNlciB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZ2VuZXJhdGlvbiA9IExvYWRCYWxhbmNlckdlbmVyYXRpb24uRklSU1Q7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgbmFtZSA9IGxvYWRCYWxhbmNlci5sb2FkQmFsYW5jZXJOYW1lO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgQ2xhc3NpY0xvYWRCYWxhbmNlcigpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgYXBwbGljYXRpb24oYWxiVGFyZ2V0R3JvdXA6IGVsYnYyLkFwcGxpY2F0aW9uVGFyZ2V0R3JvdXApOiBMb2FkQmFsYW5jZXIge1xuICAgIGNsYXNzIEFsYkxvYWRCYWxhbmNlciBleHRlbmRzIExvYWRCYWxhbmNlciB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZ2VuZXJhdGlvbiA9IExvYWRCYWxhbmNlckdlbmVyYXRpb24uU0VDT05EO1xuICAgICAgcHVibGljIHJlYWRvbmx5IG5hbWUgPSBhbGJUYXJnZXRHcm91cC50YXJnZXRHcm91cE5hbWU7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBBbGJMb2FkQmFsYW5jZXIoKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBuZXR3b3JrKG5sYlRhcmdldEdyb3VwOiBlbGJ2Mi5OZXR3b3JrVGFyZ2V0R3JvdXApOiBMb2FkQmFsYW5jZXIge1xuICAgIGNsYXNzIE5sYkxvYWRCYWxhbmNlciBleHRlbmRzIExvYWRCYWxhbmNlciB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZ2VuZXJhdGlvbiA9IExvYWRCYWxhbmNlckdlbmVyYXRpb24uU0VDT05EO1xuICAgICAgcHVibGljIHJlYWRvbmx5IG5hbWUgPSBubGJUYXJnZXRHcm91cC50YXJnZXRHcm91cE5hbWU7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBObGJMb2FkQmFsYW5jZXIoKTtcbiAgfVxuXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBnZW5lcmF0aW9uOiBMb2FkQmFsYW5jZXJHZW5lcmF0aW9uO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xufVxuIl19