"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomLambdaDeploymentConfig = exports.CustomLambdaDeploymentConfigType = void 0;
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const utils_1 = require("../utils");
/**
 * Lambda Deployment config type.
 *
 * @stability stable
 */
var CustomLambdaDeploymentConfigType;
(function (CustomLambdaDeploymentConfigType) {
    CustomLambdaDeploymentConfigType["CANARY"] = "Canary";
    CustomLambdaDeploymentConfigType["LINEAR"] = "Linear";
})(CustomLambdaDeploymentConfigType = exports.CustomLambdaDeploymentConfigType || (exports.CustomLambdaDeploymentConfigType = {}));
/**
 * A custom Deployment Configuration for a Lambda Deployment Group.
 *
 * @stability stable
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class CustomLambdaDeploymentConfig extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.validateParameters(props);
        // In this section we make the argument for the AWS API call
        const deploymentType = 'TimeBased' + props.type.toString();
        const intervalMinutes = props.interval.toMinutes().toString();
        const percentage = props.percentage.toString();
        let routingConfig; // The argument to the AWS API call
        if (props.type == CustomLambdaDeploymentConfigType.CANARY) {
            routingConfig = {
                type: deploymentType,
                timeBasedCanary: {
                    canaryInterval: intervalMinutes,
                    canaryPercentage: percentage,
                },
            };
        }
        else if (props.type == CustomLambdaDeploymentConfigType.LINEAR) {
            routingConfig = {
                type: deploymentType,
                timeBasedLinear: {
                    linearInterval: intervalMinutes,
                    linearPercentage: percentage,
                },
            };
        }
        // Generates the name of the deployment config. It's also what you'll see in the AWS console
        // The name of the config is <construct unique id>.Lambda<deployment type><percentage>Percent<interval>Minutes
        // Unless the user provides an explicit name
        this.deploymentConfigName = (_a = props.deploymentConfigName) !== null && _a !== void 0 ? _a : `${core_1.Names.uniqueId(this)}.Lambda${props.type}${props.percentage}Percent${props.type === CustomLambdaDeploymentConfigType.LINEAR
            ? 'Every'
            : ''}${props.interval.toMinutes()}Minutes`;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
        // The AWS Custom Resource that calls CodeDeploy to create and delete the resource
        new custom_resources_1.AwsCustomResource(this, 'DeploymentConfig', {
            onCreate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // This `resourceName` is the initial physical ID of the config
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onUpdate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // If `resourceName` is different from the last stack update (or creation),
                // the old config gets deleted and the new one is created
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onDelete: {
                service: 'CodeDeploy',
                action: 'deleteDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                },
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
    }
    // Validate the inputs. The percentage/interval limits come from CodeDeploy
    validateParameters(props) {
        if (!(1 <= props.percentage && props.percentage <= 99)) {
            throw new Error(`Invalid deployment config percentage "${props.percentage.toString()}". \
        Step percentage must be an integer between 1 and 99.`);
        }
        if (props.interval.toMinutes() > 2880) {
            throw new Error(`Invalid deployment config interval "${props.interval.toString()}". \
        Traffic shifting intervals must be positive integers up to 2880 (2 days).`);
        }
    }
}
exports.CustomLambdaDeploymentConfig = CustomLambdaDeploymentConfig;
//# sourceMappingURL=data:application/json;base64,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