`ifndef LIBSV_CODERS_ENCODER_8B10B
`define LIBSV_CODERS_ENCODER_8B10B

module encoder_8b10b (
    input  logic       i_clk,      // input clock
    input  logic       i_reset_n,  // input asynchronous active-low reset
    input  logic       i_en,       // input enable
    input  logic [7:0] i_8b,       // input 8-bit value
    input  logic       i_ctrl,     // input control symbol select
    output logic [9:0] o_10b,      // output 10-bit value
    output logic       o_code_err  // output code error
);

    logic        rd;  // running disparity
    logic [ 9:0] i_lut;  // input vector to 8b/10b encoding look-up table
    logic [11:0] o_lut;  // output vector from 8b/10b encoding look-up table

    always_ff @(posedge i_clk or negedge i_reset_n) begin

        if (!i_reset_n) begin

            // if in reset set all outputs to 0 and reset running disparity
            // to 0 (-1)
            o_10b      <= 10'b0;
            rd         <= 1'b0;
            o_code_err <= 1'b0;

        end else if (!i_en) begin

            // if not enabled, then maintain current output
            // and running disparity
            o_10b      <= o_10b;
            rd         <= rd;
            o_code_err <= o_code_err;

        end else begin

            // if enabled and not in reset, then get the 10b output and
            // updated running disparity from the 8b10b encoding look-up table
            o_10b      <= o_lut[9:0];
            rd         <= o_lut[10];
            o_code_err <= o_lut[11];

        end
    end

    // 8b/10b encoding look-up table
    // ------------------------------------------------------------
    // The mapping is XRHGFEDCBA: rjhgfiedcba, where:
    //     * X          = input control symbol select
    //     * R          = input running disparity (0 = -1, 1 = +1)
    //     * HGFEDCBA   = input 8b value
    //     * y          = output code error
    //     * r          = output running disparity (0 = -1, 1 = +1)
    //     * jhgfiedcba = output 10b value
    assign i_lut = {i_ctrl, rd, i_8b};  // Form input look-up table vector
    always_comb begin

        case (i_lut)
            // verilog_lint: waive-start line-length  // Lines are too long but this is what is wanted so disable lint checks
            // verilog_format: off // don't format this because it breaks each case into multiple lines
      //  XRHGFEDCBA:             yrjhgfiedbca
      10'b0000000000: o_lut = 12'b000010111001; // D.00.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000000001: o_lut = 12'b000010101110; // D.01.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000000010: o_lut = 12'b000010101101; // D.02.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000000011: o_lut = 12'b011101100011; // D.03.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000000100: o_lut = 12'b000010101011; // D.04.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000000101: o_lut = 12'b011101100101; // D.05.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000000110: o_lut = 12'b011101100110; // D.06.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000000111: o_lut = 12'b011101000111; // D.07.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000001000: o_lut = 12'b000010100111; // D.08.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000001001: o_lut = 12'b011101101001; // D.09.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000001010: o_lut = 12'b011101101010; // D.10.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000001011: o_lut = 12'b011101001011; // D.11.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000001100: o_lut = 12'b011101101100; // D.12.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000001101: o_lut = 12'b011101001101; // D.13.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000001110: o_lut = 12'b011101001110; // D.14.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000001111: o_lut = 12'b000010111010; // D.15.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000010000: o_lut = 12'b000010110110; // D.16.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000010001: o_lut = 12'b011101110001; // D.17.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000010010: o_lut = 12'b011101110010; // D.18.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000010011: o_lut = 12'b011101010011; // D.19.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000010100: o_lut = 12'b011101110100; // D.20.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000010101: o_lut = 12'b011101010101; // D.21.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000010110: o_lut = 12'b011101010110; // D.22.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000010111: o_lut = 12'b000010010111; // D.23.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000011000: o_lut = 12'b000010110011; // D.24.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000011001: o_lut = 12'b011101011001; // D.25.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000011010: o_lut = 12'b011101011010; // D.26.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000011011: o_lut = 12'b000010011011; // D.27.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000011100: o_lut = 12'b011101001110; // D.28.0, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000011101: o_lut = 12'b000010011101; // D.29.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000011110: o_lut = 12'b000010011110; // D.30.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000011111: o_lut = 12'b000010110101; // D.31.0, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000100000: o_lut = 12'b011001111001; // D.00.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000100001: o_lut = 12'b011001101110; // D.01.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000100010: o_lut = 12'b011001101101; // D.02.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000100011: o_lut = 12'b001001100011; // D.03.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000100100: o_lut = 12'b011001101011; // D.04.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000100101: o_lut = 12'b001001100101; // D.05.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000100110: o_lut = 12'b001001100110; // D.06.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000100111: o_lut = 12'b001001000111; // D.07.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000101000: o_lut = 12'b011001100111; // D.08.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000101001: o_lut = 12'b001001101001; // D.09.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000101010: o_lut = 12'b001001101010; // D.10.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000101011: o_lut = 12'b001001001011; // D.11.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000101100: o_lut = 12'b001001101100; // D.12.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000101101: o_lut = 12'b001001001101; // D.13.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000101110: o_lut = 12'b001001001110; // D.14.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000101111: o_lut = 12'b011001111010; // D.15.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000110000: o_lut = 12'b011001110110; // D.16.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000110001: o_lut = 12'b001001110001; // D.17.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000110010: o_lut = 12'b001001110010; // D.18.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000110011: o_lut = 12'b001001010011; // D.19.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000110100: o_lut = 12'b001001110100; // D.20.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000110101: o_lut = 12'b001001010101; // D.21.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000110110: o_lut = 12'b001001010110; // D.22.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000110111: o_lut = 12'b011001010111; // D.23.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000111000: o_lut = 12'b011001110011; // D.24.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000111001: o_lut = 12'b001001011001; // D.25.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000111010: o_lut = 12'b001001011010; // D.26.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000111011: o_lut = 12'b011001011011; // D.27.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000111100: o_lut = 12'b001001001110; // D.28.1, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0000111101: o_lut = 12'b011001011101; // D.29.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000111110: o_lut = 12'b011001011110; // D.30.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0000111111: o_lut = 12'b011001110101; // D.31.1, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001000000: o_lut = 12'b011010111001; // D.00.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001000001: o_lut = 12'b011010101110; // D.01.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001000010: o_lut = 12'b011010101101; // D.02.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001000011: o_lut = 12'b001010100011; // D.03.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001000100: o_lut = 12'b011010101011; // D.04.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001000101: o_lut = 12'b001010100101; // D.05.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001000110: o_lut = 12'b001010100110; // D.06.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001000111: o_lut = 12'b001010000111; // D.07.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001001000: o_lut = 12'b011010100111; // D.08.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001001001: o_lut = 12'b001010101001; // D.09.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001001010: o_lut = 12'b001010101010; // D.10.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001001011: o_lut = 12'b001010001011; // D.11.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001001100: o_lut = 12'b001010101100; // D.12.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001001101: o_lut = 12'b001010001101; // D.13.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001001110: o_lut = 12'b001010001110; // D.14.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001001111: o_lut = 12'b011010111010; // D.15.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001010000: o_lut = 12'b011010110110; // D.16.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001010001: o_lut = 12'b001010110001; // D.17.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001010010: o_lut = 12'b001010110010; // D.18.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001010011: o_lut = 12'b001010010011; // D.19.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001010100: o_lut = 12'b001010110100; // D.20.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001010101: o_lut = 12'b001010010101; // D.21.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001010110: o_lut = 12'b001010010110; // D.22.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001010111: o_lut = 12'b011010010111; // D.23.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001011000: o_lut = 12'b011010110011; // D.24.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001011001: o_lut = 12'b001010011001; // D.25.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001011010: o_lut = 12'b001010011010; // D.26.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001011011: o_lut = 12'b011010011011; // D.27.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001011100: o_lut = 12'b001010001110; // D.28.2, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001011101: o_lut = 12'b011010011101; // D.29.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001011110: o_lut = 12'b011010011110; // D.30.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001011111: o_lut = 12'b011010110101; // D.31.2, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001100000: o_lut = 12'b011100111001; // D.00.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001100001: o_lut = 12'b011100101110; // D.01.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001100010: o_lut = 12'b011100101101; // D.02.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001100011: o_lut = 12'b000011100011; // D.03.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001100100: o_lut = 12'b011100101011; // D.04.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001100101: o_lut = 12'b000011100101; // D.05.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001100110: o_lut = 12'b000011100110; // D.06.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001100111: o_lut = 12'b000011000111; // D.07.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001101000: o_lut = 12'b011100100111; // D.08.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001101001: o_lut = 12'b000011101001; // D.09.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001101010: o_lut = 12'b000011101010; // D.10.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001101011: o_lut = 12'b000011001011; // D.11.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001101100: o_lut = 12'b000011101100; // D.12.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001101101: o_lut = 12'b000011001101; // D.13.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001101110: o_lut = 12'b000011001110; // D.14.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001101111: o_lut = 12'b011100111010; // D.15.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001110000: o_lut = 12'b011100110110; // D.16.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001110001: o_lut = 12'b000011110001; // D.17.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001110010: o_lut = 12'b000011110010; // D.18.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001110011: o_lut = 12'b000011010011; // D.19.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001110100: o_lut = 12'b000011110100; // D.20.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001110101: o_lut = 12'b000011010101; // D.21.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001110110: o_lut = 12'b000011010110; // D.22.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001110111: o_lut = 12'b011100010111; // D.23.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001111000: o_lut = 12'b011100110011; // D.24.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001111001: o_lut = 12'b000011011001; // D.25.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001111010: o_lut = 12'b000011011010; // D.26.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001111011: o_lut = 12'b011100011011; // D.27.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001111100: o_lut = 12'b000011001110; // D.28.3, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0001111101: o_lut = 12'b011100011101; // D.29.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001111110: o_lut = 12'b011100011110; // D.30.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0001111111: o_lut = 12'b011100110101; // D.31.3, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010000000: o_lut = 12'b000100111001; // D.00.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010000001: o_lut = 12'b000100101110; // D.01.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010000010: o_lut = 12'b000100101101; // D.02.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010000011: o_lut = 12'b011011100011; // D.03.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010000100: o_lut = 12'b000100101011; // D.04.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010000101: o_lut = 12'b011011100101; // D.05.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010000110: o_lut = 12'b011011100110; // D.06.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010000111: o_lut = 12'b011011000111; // D.07.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010001000: o_lut = 12'b000100100111; // D.08.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010001001: o_lut = 12'b011011101001; // D.09.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010001010: o_lut = 12'b011011101010; // D.10.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010001011: o_lut = 12'b011011001011; // D.11.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010001100: o_lut = 12'b011011101100; // D.12.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010001101: o_lut = 12'b011011001101; // D.13.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010001110: o_lut = 12'b011011001110; // D.14.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010001111: o_lut = 12'b000100111010; // D.15.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010010000: o_lut = 12'b000100110110; // D.16.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010010001: o_lut = 12'b011011110001; // D.17.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010010010: o_lut = 12'b011011110010; // D.18.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010010011: o_lut = 12'b011011010011; // D.19.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010010100: o_lut = 12'b011011110100; // D.20.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010010101: o_lut = 12'b011011010101; // D.21.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010010110: o_lut = 12'b011011010110; // D.22.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010010111: o_lut = 12'b000100010111; // D.23.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010011000: o_lut = 12'b000100110011; // D.24.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010011001: o_lut = 12'b011011011001; // D.25.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010011010: o_lut = 12'b011011011010; // D.26.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010011011: o_lut = 12'b000100011011; // D.27.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010011100: o_lut = 12'b011011001110; // D.28.4, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010011101: o_lut = 12'b000100011101; // D.29.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010011110: o_lut = 12'b000100011110; // D.30.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010011111: o_lut = 12'b000100110101; // D.31.4, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010100000: o_lut = 12'b010101111001; // D.00.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010100001: o_lut = 12'b010101101110; // D.01.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010100010: o_lut = 12'b010101101101; // D.02.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010100011: o_lut = 12'b000101100011; // D.03.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010100100: o_lut = 12'b010101101011; // D.04.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010100101: o_lut = 12'b000101100101; // D.05.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010100110: o_lut = 12'b000101100110; // D.06.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010100111: o_lut = 12'b000101000111; // D.07.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010101000: o_lut = 12'b010101100111; // D.08.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010101001: o_lut = 12'b000101101001; // D.09.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010101010: o_lut = 12'b000101101010; // D.10.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010101011: o_lut = 12'b000101001011; // D.11.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010101100: o_lut = 12'b000101101100; // D.12.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010101101: o_lut = 12'b000101001101; // D.13.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010101110: o_lut = 12'b000101001110; // D.14.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010101111: o_lut = 12'b010101111010; // D.15.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010110000: o_lut = 12'b010101110110; // D.16.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010110001: o_lut = 12'b000101110001; // D.17.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010110010: o_lut = 12'b000101110010; // D.18.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010110011: o_lut = 12'b000101010011; // D.19.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010110100: o_lut = 12'b000101110100; // D.20.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010110101: o_lut = 12'b000101010101; // D.21.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010110110: o_lut = 12'b000101010110; // D.22.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010110111: o_lut = 12'b010101010111; // D.23.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010111000: o_lut = 12'b010101110011; // D.24.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010111001: o_lut = 12'b000101011001; // D.25.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010111010: o_lut = 12'b000101011010; // D.26.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010111011: o_lut = 12'b010101011011; // D.27.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010111100: o_lut = 12'b000101001110; // D.28.5, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0010111101: o_lut = 12'b010101011101; // D.29.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010111110: o_lut = 12'b010101011110; // D.30.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0010111111: o_lut = 12'b010101110101; // D.31.5, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011000000: o_lut = 12'b010110111001; // D.00.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011000001: o_lut = 12'b010110101110; // D.01.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011000010: o_lut = 12'b010110101101; // D.02.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011000011: o_lut = 12'b000110100011; // D.03.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011000100: o_lut = 12'b010110101011; // D.04.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011000101: o_lut = 12'b000110100101; // D.05.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011000110: o_lut = 12'b000110100110; // D.06.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011000111: o_lut = 12'b000110000111; // D.07.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011001000: o_lut = 12'b010110100111; // D.08.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011001001: o_lut = 12'b000110101001; // D.09.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011001010: o_lut = 12'b000110101010; // D.10.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011001011: o_lut = 12'b000110001011; // D.11.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011001100: o_lut = 12'b000110101100; // D.12.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011001101: o_lut = 12'b000110001101; // D.13.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011001110: o_lut = 12'b000110001110; // D.14.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011001111: o_lut = 12'b010110111010; // D.15.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011010000: o_lut = 12'b010110110110; // D.16.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011010001: o_lut = 12'b000110110001; // D.17.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011010010: o_lut = 12'b000110110010; // D.18.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011010011: o_lut = 12'b000110010011; // D.19.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011010100: o_lut = 12'b000110110100; // D.20.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011010101: o_lut = 12'b000110010101; // D.21.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011010110: o_lut = 12'b000110010110; // D.22.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011010111: o_lut = 12'b010110010111; // D.23.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011011000: o_lut = 12'b010110110011; // D.24.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011011001: o_lut = 12'b000110011001; // D.25.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011011010: o_lut = 12'b000110011010; // D.26.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011011011: o_lut = 12'b010110011011; // D.27.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011011100: o_lut = 12'b000110001110; // D.28.6, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011011101: o_lut = 12'b010110011101; // D.29.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011011110: o_lut = 12'b010110011110; // D.30.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011011111: o_lut = 12'b010110110101; // D.31.6, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011100000: o_lut = 12'b001000111001; // D.00.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011100001: o_lut = 12'b001000101110; // D.01.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011100010: o_lut = 12'b001000101101; // D.02.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011100011: o_lut = 12'b010111100011; // D.03.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011100100: o_lut = 12'b001000101011; // D.04.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011100101: o_lut = 12'b010111100101; // D.05.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011100110: o_lut = 12'b010111100110; // D.06.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011100111: o_lut = 12'b010111000111; // D.07.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011101000: o_lut = 12'b001000100111; // D.08.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011101001: o_lut = 12'b010111101001; // D.09.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011101010: o_lut = 12'b010111101010; // D.10.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011101011: o_lut = 12'b010111001011; // D.11.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011101100: o_lut = 12'b010111101100; // D.12.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011101101: o_lut = 12'b010111001101; // D.13.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011101110: o_lut = 12'b010111001110; // D.14.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011101111: o_lut = 12'b001000111010; // D.15.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011110000: o_lut = 12'b001000110110; // D.16.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011110001: o_lut = 12'b011110110001; // D.17.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011110010: o_lut = 12'b011110110010; // D.18.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011110011: o_lut = 12'b010111010011; // D.19.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011110100: o_lut = 12'b011110110100; // D.20.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011110101: o_lut = 12'b010111010101; // D.21.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011110110: o_lut = 12'b010111010110; // D.22.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011110111: o_lut = 12'b001000010111; // D.23.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011111000: o_lut = 12'b001000110011; // D.24.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011111001: o_lut = 12'b010111011001; // D.25.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011111010: o_lut = 12'b010111011010; // D.26.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011111011: o_lut = 12'b001000011011; // D.27.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011111100: o_lut = 12'b010111001110; // D.28.7, i_disp = 0, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0011111101: o_lut = 12'b001000011101; // D.29.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011111110: o_lut = 12'b001000011110; // D.30.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0011111111: o_lut = 12'b001000110101; // D.31.7, i_disp = 0, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100000000: o_lut = 12'b011101000110; // D.00.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100000001: o_lut = 12'b011101010001; // D.01.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100000010: o_lut = 12'b011101010010; // D.02.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100000011: o_lut = 12'b000010100011; // D.03.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100000100: o_lut = 12'b011101010100; // D.04.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100000101: o_lut = 12'b000010100101; // D.05.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100000110: o_lut = 12'b000010100110; // D.06.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100000111: o_lut = 12'b000010111000; // D.07.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100001000: o_lut = 12'b011101011000; // D.08.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100001001: o_lut = 12'b000010101001; // D.09.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100001010: o_lut = 12'b000010101010; // D.10.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100001011: o_lut = 12'b000010001011; // D.11.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100001100: o_lut = 12'b000010101100; // D.12.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100001101: o_lut = 12'b000010001101; // D.13.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100001110: o_lut = 12'b000010001110; // D.14.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100001111: o_lut = 12'b011101000101; // D.15.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100010000: o_lut = 12'b011101001001; // D.16.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100010001: o_lut = 12'b000010110001; // D.17.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100010010: o_lut = 12'b000010110010; // D.18.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100010011: o_lut = 12'b000010010011; // D.19.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100010100: o_lut = 12'b000010110100; // D.20.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100010101: o_lut = 12'b000010010101; // D.21.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100010110: o_lut = 12'b000010010110; // D.22.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100010111: o_lut = 12'b011101101000; // D.23.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100011000: o_lut = 12'b011101001100; // D.24.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100011001: o_lut = 12'b000010011001; // D.25.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100011010: o_lut = 12'b000010011010; // D.26.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100011011: o_lut = 12'b011101100100; // D.27.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100011100: o_lut = 12'b000010001110; // D.28.0, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100011101: o_lut = 12'b011101100010; // D.29.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100011110: o_lut = 12'b011101100001; // D.30.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100011111: o_lut = 12'b011101001010; // D.31.0, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100100000: o_lut = 12'b001001000110; // D.00.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100100001: o_lut = 12'b001001010001; // D.01.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100100010: o_lut = 12'b001001010010; // D.02.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100100011: o_lut = 12'b011001100011; // D.03.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100100100: o_lut = 12'b001001010100; // D.04.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100100101: o_lut = 12'b011001100101; // D.05.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100100110: o_lut = 12'b011001100110; // D.06.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100100111: o_lut = 12'b011001111000; // D.07.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100101000: o_lut = 12'b001001011000; // D.08.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100101001: o_lut = 12'b011001101001; // D.09.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100101010: o_lut = 12'b011001101010; // D.10.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100101011: o_lut = 12'b011001001011; // D.11.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100101100: o_lut = 12'b011001101100; // D.12.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100101101: o_lut = 12'b011001001101; // D.13.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100101110: o_lut = 12'b011001001110; // D.14.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100101111: o_lut = 12'b001001000101; // D.15.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100110000: o_lut = 12'b001001001001; // D.16.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100110001: o_lut = 12'b011001110001; // D.17.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100110010: o_lut = 12'b011001110010; // D.18.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100110011: o_lut = 12'b011001010011; // D.19.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100110100: o_lut = 12'b011001110100; // D.20.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100110101: o_lut = 12'b011001010101; // D.21.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100110110: o_lut = 12'b011001010110; // D.22.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100110111: o_lut = 12'b001001101000; // D.23.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100111000: o_lut = 12'b001001001100; // D.24.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100111001: o_lut = 12'b011001011001; // D.25.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100111010: o_lut = 12'b011001011010; // D.26.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100111011: o_lut = 12'b001001100100; // D.27.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100111100: o_lut = 12'b011001001110; // D.28.1, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0100111101: o_lut = 12'b001001100010; // D.29.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100111110: o_lut = 12'b001001100001; // D.30.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0100111111: o_lut = 12'b001001001010; // D.31.1, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101000000: o_lut = 12'b001010000110; // D.00.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101000001: o_lut = 12'b001010010001; // D.01.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101000010: o_lut = 12'b001010010010; // D.02.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101000011: o_lut = 12'b011010100011; // D.03.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101000100: o_lut = 12'b001010010100; // D.04.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101000101: o_lut = 12'b011010100101; // D.05.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101000110: o_lut = 12'b011010100110; // D.06.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101000111: o_lut = 12'b011010111000; // D.07.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101001000: o_lut = 12'b001010011000; // D.08.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101001001: o_lut = 12'b011010101001; // D.09.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101001010: o_lut = 12'b011010101010; // D.10.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101001011: o_lut = 12'b011010001011; // D.11.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101001100: o_lut = 12'b011010101100; // D.12.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101001101: o_lut = 12'b011010001101; // D.13.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101001110: o_lut = 12'b011010001110; // D.14.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101001111: o_lut = 12'b001010000101; // D.15.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101010000: o_lut = 12'b001010001001; // D.16.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101010001: o_lut = 12'b011010110001; // D.17.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101010010: o_lut = 12'b011010110010; // D.18.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101010011: o_lut = 12'b011010010011; // D.19.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101010100: o_lut = 12'b011010110100; // D.20.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101010101: o_lut = 12'b011010010101; // D.21.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101010110: o_lut = 12'b011010010110; // D.22.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101010111: o_lut = 12'b001010101000; // D.23.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101011000: o_lut = 12'b001010001100; // D.24.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101011001: o_lut = 12'b011010011001; // D.25.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101011010: o_lut = 12'b011010011010; // D.26.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101011011: o_lut = 12'b001010100100; // D.27.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101011100: o_lut = 12'b011010001110; // D.28.2, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101011101: o_lut = 12'b001010100010; // D.29.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101011110: o_lut = 12'b001010100001; // D.30.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101011111: o_lut = 12'b001010001010; // D.31.2, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101100000: o_lut = 12'b000011000110; // D.00.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101100001: o_lut = 12'b000011010001; // D.01.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101100010: o_lut = 12'b000011010010; // D.02.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101100011: o_lut = 12'b011100100011; // D.03.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101100100: o_lut = 12'b000011010100; // D.04.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101100101: o_lut = 12'b011100100101; // D.05.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101100110: o_lut = 12'b011100100110; // D.06.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101100111: o_lut = 12'b011100111000; // D.07.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101101000: o_lut = 12'b000011011000; // D.08.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101101001: o_lut = 12'b011100101001; // D.09.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101101010: o_lut = 12'b011100101010; // D.10.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101101011: o_lut = 12'b011100001011; // D.11.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101101100: o_lut = 12'b011100101100; // D.12.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101101101: o_lut = 12'b011100001101; // D.13.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101101110: o_lut = 12'b011100001110; // D.14.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101101111: o_lut = 12'b000011000101; // D.15.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101110000: o_lut = 12'b000011001001; // D.16.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101110001: o_lut = 12'b011100110001; // D.17.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101110010: o_lut = 12'b011100110010; // D.18.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101110011: o_lut = 12'b011100010011; // D.19.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101110100: o_lut = 12'b011100110100; // D.20.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101110101: o_lut = 12'b011100010101; // D.21.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101110110: o_lut = 12'b011100010110; // D.22.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101110111: o_lut = 12'b000011101000; // D.23.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101111000: o_lut = 12'b000011001100; // D.24.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101111001: o_lut = 12'b011100011001; // D.25.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101111010: o_lut = 12'b011100011010; // D.26.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101111011: o_lut = 12'b000011100100; // D.27.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101111100: o_lut = 12'b011100001110; // D.28.3, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0101111101: o_lut = 12'b000011100010; // D.29.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101111110: o_lut = 12'b000011100001; // D.30.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0101111111: o_lut = 12'b000011001010; // D.31.3, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110000000: o_lut = 12'b011011000110; // D.00.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110000001: o_lut = 12'b011011010001; // D.01.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110000010: o_lut = 12'b011011010010; // D.02.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110000011: o_lut = 12'b000100100011; // D.03.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110000100: o_lut = 12'b011011010100; // D.04.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110000101: o_lut = 12'b000100100101; // D.05.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110000110: o_lut = 12'b000100100110; // D.06.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110000111: o_lut = 12'b000100111000; // D.07.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110001000: o_lut = 12'b011011011000; // D.08.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110001001: o_lut = 12'b000100101001; // D.09.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110001010: o_lut = 12'b000100101010; // D.10.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110001011: o_lut = 12'b000100001011; // D.11.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110001100: o_lut = 12'b000100101100; // D.12.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110001101: o_lut = 12'b000100001101; // D.13.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110001110: o_lut = 12'b000100001110; // D.14.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110001111: o_lut = 12'b011011000101; // D.15.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110010000: o_lut = 12'b011011001001; // D.16.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110010001: o_lut = 12'b000100110001; // D.17.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110010010: o_lut = 12'b000100110010; // D.18.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110010011: o_lut = 12'b000100010011; // D.19.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110010100: o_lut = 12'b000100110100; // D.20.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110010101: o_lut = 12'b000100010101; // D.21.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110010110: o_lut = 12'b000100010110; // D.22.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110010111: o_lut = 12'b011011101000; // D.23.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110011000: o_lut = 12'b011011001100; // D.24.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110011001: o_lut = 12'b000100011001; // D.25.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110011010: o_lut = 12'b000100011010; // D.26.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110011011: o_lut = 12'b011011100100; // D.27.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110011100: o_lut = 12'b000100001110; // D.28.4, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110011101: o_lut = 12'b011011100010; // D.29.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110011110: o_lut = 12'b011011100001; // D.30.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110011111: o_lut = 12'b011011001010; // D.31.4, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110100000: o_lut = 12'b000101000110; // D.00.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110100001: o_lut = 12'b000101010001; // D.01.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110100010: o_lut = 12'b000101010010; // D.02.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110100011: o_lut = 12'b010101100011; // D.03.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110100100: o_lut = 12'b000101010100; // D.04.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110100101: o_lut = 12'b010101100101; // D.05.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110100110: o_lut = 12'b010101100110; // D.06.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110100111: o_lut = 12'b010101111000; // D.07.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110101000: o_lut = 12'b000101011000; // D.08.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110101001: o_lut = 12'b010101101001; // D.09.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110101010: o_lut = 12'b010101101010; // D.10.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110101011: o_lut = 12'b010101001011; // D.11.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110101100: o_lut = 12'b010101101100; // D.12.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110101101: o_lut = 12'b010101001101; // D.13.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110101110: o_lut = 12'b010101001110; // D.14.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110101111: o_lut = 12'b000101000101; // D.15.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110110000: o_lut = 12'b000101001001; // D.16.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110110001: o_lut = 12'b010101110001; // D.17.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110110010: o_lut = 12'b010101110010; // D.18.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110110011: o_lut = 12'b010101010011; // D.19.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110110100: o_lut = 12'b010101110100; // D.20.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110110101: o_lut = 12'b010101010101; // D.21.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110110110: o_lut = 12'b010101010110; // D.22.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110110111: o_lut = 12'b000101101000; // D.23.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110111000: o_lut = 12'b000101001100; // D.24.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110111001: o_lut = 12'b010101011001; // D.25.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110111010: o_lut = 12'b010101011010; // D.26.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110111011: o_lut = 12'b000101100100; // D.27.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110111100: o_lut = 12'b010101001110; // D.28.5, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0110111101: o_lut = 12'b000101100010; // D.29.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110111110: o_lut = 12'b000101100001; // D.30.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0110111111: o_lut = 12'b000101001010; // D.31.5, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111000000: o_lut = 12'b000110000110; // D.00.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111000001: o_lut = 12'b000110010001; // D.01.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111000010: o_lut = 12'b000110010010; // D.02.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111000011: o_lut = 12'b010110100011; // D.03.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111000100: o_lut = 12'b000110010100; // D.04.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111000101: o_lut = 12'b010110100101; // D.05.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111000110: o_lut = 12'b010110100110; // D.06.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111000111: o_lut = 12'b010110111000; // D.07.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111001000: o_lut = 12'b000110011000; // D.08.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111001001: o_lut = 12'b010110101001; // D.09.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111001010: o_lut = 12'b010110101010; // D.10.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111001011: o_lut = 12'b010110001011; // D.11.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111001100: o_lut = 12'b010110101100; // D.12.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111001101: o_lut = 12'b010110001101; // D.13.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111001110: o_lut = 12'b010110001110; // D.14.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111001111: o_lut = 12'b000110000101; // D.15.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111010000: o_lut = 12'b000110001001; // D.16.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111010001: o_lut = 12'b010110110001; // D.17.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111010010: o_lut = 12'b010110110010; // D.18.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111010011: o_lut = 12'b010110010011; // D.19.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111010100: o_lut = 12'b010110110100; // D.20.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111010101: o_lut = 12'b010110010101; // D.21.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111010110: o_lut = 12'b010110010110; // D.22.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111010111: o_lut = 12'b000110101000; // D.23.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111011000: o_lut = 12'b000110001100; // D.24.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111011001: o_lut = 12'b010110011001; // D.25.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111011010: o_lut = 12'b010110011010; // D.26.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111011011: o_lut = 12'b000110100100; // D.27.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111011100: o_lut = 12'b010110001110; // D.28.6, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111011101: o_lut = 12'b000110100010; // D.29.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111011110: o_lut = 12'b000110100001; // D.30.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111011111: o_lut = 12'b000110001010; // D.31.6, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111100000: o_lut = 12'b010111000110; // D.00.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111100001: o_lut = 12'b010111010001; // D.01.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111100010: o_lut = 12'b010111010010; // D.02.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111100011: o_lut = 12'b001000100011; // D.03.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111100100: o_lut = 12'b010111010100; // D.04.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111100101: o_lut = 12'b001000100101; // D.05.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111100110: o_lut = 12'b001000100110; // D.06.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111100111: o_lut = 12'b001000111000; // D.07.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111101000: o_lut = 12'b010111011000; // D.08.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111101001: o_lut = 12'b001000101001; // D.09.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111101010: o_lut = 12'b001000101010; // D.10.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111101011: o_lut = 12'b000001001011; // D.11.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111101100: o_lut = 12'b001000101100; // D.12.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111101101: o_lut = 12'b000001001101; // D.13.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111101110: o_lut = 12'b000001001110; // D.14.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111101111: o_lut = 12'b010111000101; // D.15.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111110000: o_lut = 12'b010111001001; // D.16.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111110001: o_lut = 12'b001000110001; // D.17.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111110010: o_lut = 12'b001000110010; // D.18.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111110011: o_lut = 12'b001000010011; // D.19.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111110100: o_lut = 12'b001000110100; // D.20.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111110101: o_lut = 12'b001000010101; // D.21.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111110110: o_lut = 12'b001000010110; // D.22.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111110111: o_lut = 12'b010111101000; // D.23.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111111000: o_lut = 12'b010111001100; // D.24.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111111001: o_lut = 12'b001000011001; // D.25.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111111010: o_lut = 12'b001000011010; // D.26.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111111011: o_lut = 12'b010111100100; // D.27.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111111100: o_lut = 12'b001000001110; // D.28.7, i_disp = 1, i_ctrl = 0 o_disp = 0, o_code_err = 0
      10'b0111111101: o_lut = 12'b010111100010; // D.29.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111111110: o_lut = 12'b010111100001; // D.30.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b0111111111: o_lut = 12'b010111001010; // D.31.7, i_disp = 1, i_ctrl = 0 o_disp = 1, o_code_err = 0
      10'b1000000000: o_lut = 12'b100010111001; // K.00.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000000001: o_lut = 12'b100010101110; // K.01.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000000010: o_lut = 12'b100010101101; // K.02.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000000011: o_lut = 12'b111101100011; // K.03.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000000100: o_lut = 12'b100010101011; // K.04.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000000101: o_lut = 12'b111101100101; // K.05.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000000110: o_lut = 12'b111101100110; // K.06.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000000111: o_lut = 12'b111101000111; // K.07.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000001000: o_lut = 12'b100010100111; // K.08.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000001001: o_lut = 12'b111101101001; // K.09.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000001010: o_lut = 12'b111101101010; // K.10.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000001011: o_lut = 12'b111101001011; // K.11.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000001100: o_lut = 12'b111101101100; // K.12.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000001101: o_lut = 12'b111101001101; // K.13.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000001110: o_lut = 12'b111101001110; // K.14.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000001111: o_lut = 12'b100010111010; // K.15.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000010000: o_lut = 12'b100010110110; // K.16.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000010001: o_lut = 12'b111101110001; // K.17.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000010010: o_lut = 12'b111101110010; // K.18.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000010011: o_lut = 12'b111101010011; // K.19.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000010100: o_lut = 12'b111101110100; // K.20.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000010101: o_lut = 12'b111101010101; // K.21.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000010110: o_lut = 12'b111101010110; // K.22.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000010111: o_lut = 12'b100010010111; // K.23.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000011000: o_lut = 12'b100010110011; // K.24.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000011001: o_lut = 12'b111101011001; // K.25.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000011010: o_lut = 12'b111101011010; // K.26.0, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000011011: o_lut = 12'b100010011011; // K.27.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000011100: o_lut = 12'b000010111100; // K.28.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1000011101: o_lut = 12'b100010011101; // K.29.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000011110: o_lut = 12'b100010011110; // K.30.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000011111: o_lut = 12'b100010110101; // K.31.0, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000100000: o_lut = 12'b111001111001; // K.00.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000100001: o_lut = 12'b111001101110; // K.01.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000100010: o_lut = 12'b111001101101; // K.02.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000100011: o_lut = 12'b100110100011; // K.03.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000100100: o_lut = 12'b111001101011; // K.04.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000100101: o_lut = 12'b100110100101; // K.05.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000100110: o_lut = 12'b100110100110; // K.06.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000100111: o_lut = 12'b100110000111; // K.07.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000101000: o_lut = 12'b111001100111; // K.08.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000101001: o_lut = 12'b100110101001; // K.09.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000101010: o_lut = 12'b100110101010; // K.10.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000101011: o_lut = 12'b100110001011; // K.11.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000101100: o_lut = 12'b100110101100; // K.12.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000101101: o_lut = 12'b100110001101; // K.13.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000101110: o_lut = 12'b100110001110; // K.14.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000101111: o_lut = 12'b111001111010; // K.15.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000110000: o_lut = 12'b111001110110; // K.16.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000110001: o_lut = 12'b100110110001; // K.17.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000110010: o_lut = 12'b100110110010; // K.18.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000110011: o_lut = 12'b100110010011; // K.19.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000110100: o_lut = 12'b100110110100; // K.20.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000110101: o_lut = 12'b100110010101; // K.21.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000110110: o_lut = 12'b100110010110; // K.22.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000110111: o_lut = 12'b111001010111; // K.23.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000111000: o_lut = 12'b111001110011; // K.24.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000111001: o_lut = 12'b100110011001; // K.25.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000111010: o_lut = 12'b100110011010; // K.26.1, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1000111011: o_lut = 12'b111001011011; // K.27.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000111100: o_lut = 12'b011001111100; // K.28.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1000111101: o_lut = 12'b111001011101; // K.29.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000111110: o_lut = 12'b111001011110; // K.30.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1000111111: o_lut = 12'b111001110101; // K.31.1, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001000000: o_lut = 12'b111010111001; // K.00.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001000001: o_lut = 12'b111010101110; // K.01.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001000010: o_lut = 12'b111010101101; // K.02.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001000011: o_lut = 12'b100101100011; // K.03.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001000100: o_lut = 12'b111010101011; // K.04.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001000101: o_lut = 12'b100101100101; // K.05.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001000110: o_lut = 12'b100101100110; // K.06.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001000111: o_lut = 12'b100101000111; // K.07.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001001000: o_lut = 12'b111010100111; // K.08.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001001001: o_lut = 12'b100101101001; // K.09.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001001010: o_lut = 12'b100101101010; // K.10.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001001011: o_lut = 12'b100101001011; // K.11.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001001100: o_lut = 12'b100101101100; // K.12.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001001101: o_lut = 12'b100101001101; // K.13.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001001110: o_lut = 12'b100101001110; // K.14.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001001111: o_lut = 12'b111010111010; // K.15.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001010000: o_lut = 12'b111010110110; // K.16.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001010001: o_lut = 12'b100101110001; // K.17.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001010010: o_lut = 12'b100101110010; // K.18.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001010011: o_lut = 12'b100101010011; // K.19.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001010100: o_lut = 12'b100101110100; // K.20.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001010101: o_lut = 12'b100101010101; // K.21.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001010110: o_lut = 12'b100101010110; // K.22.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001010111: o_lut = 12'b111010010111; // K.23.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001011000: o_lut = 12'b111010110011; // K.24.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001011001: o_lut = 12'b100101011001; // K.25.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001011010: o_lut = 12'b100101011010; // K.26.2, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001011011: o_lut = 12'b111010011011; // K.27.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001011100: o_lut = 12'b011010111100; // K.28.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1001011101: o_lut = 12'b111010011101; // K.29.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001011110: o_lut = 12'b111010011110; // K.30.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001011111: o_lut = 12'b111010110101; // K.31.2, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001100000: o_lut = 12'b111100111001; // K.00.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001100001: o_lut = 12'b111100101110; // K.01.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001100010: o_lut = 12'b111100101101; // K.02.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001100011: o_lut = 12'b100011100011; // K.03.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001100100: o_lut = 12'b111100101011; // K.04.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001100101: o_lut = 12'b100011100101; // K.05.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001100110: o_lut = 12'b100011100110; // K.06.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001100111: o_lut = 12'b100011000111; // K.07.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001101000: o_lut = 12'b111100100111; // K.08.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001101001: o_lut = 12'b100011101001; // K.09.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001101010: o_lut = 12'b100011101010; // K.10.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001101011: o_lut = 12'b100011001011; // K.11.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001101100: o_lut = 12'b100011101100; // K.12.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001101101: o_lut = 12'b100011001101; // K.13.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001101110: o_lut = 12'b100011001110; // K.14.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001101111: o_lut = 12'b111100111010; // K.15.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001110000: o_lut = 12'b111100110110; // K.16.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001110001: o_lut = 12'b100011110001; // K.17.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001110010: o_lut = 12'b100011110010; // K.18.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001110011: o_lut = 12'b100011010011; // K.19.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001110100: o_lut = 12'b100011110100; // K.20.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001110101: o_lut = 12'b100011010101; // K.21.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001110110: o_lut = 12'b100011010110; // K.22.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001110111: o_lut = 12'b111100010111; // K.23.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001111000: o_lut = 12'b111100110011; // K.24.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001111001: o_lut = 12'b100011011001; // K.25.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001111010: o_lut = 12'b100011011010; // K.26.3, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1001111011: o_lut = 12'b111100011011; // K.27.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001111100: o_lut = 12'b011100111100; // K.28.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1001111101: o_lut = 12'b111100011101; // K.29.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001111110: o_lut = 12'b111100011110; // K.30.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1001111111: o_lut = 12'b111100110101; // K.31.3, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010000000: o_lut = 12'b100100111001; // K.00.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010000001: o_lut = 12'b100100101110; // K.01.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010000010: o_lut = 12'b100100101101; // K.02.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010000011: o_lut = 12'b111011100011; // K.03.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010000100: o_lut = 12'b100100101011; // K.04.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010000101: o_lut = 12'b111011100101; // K.05.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010000110: o_lut = 12'b111011100110; // K.06.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010000111: o_lut = 12'b111011000111; // K.07.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010001000: o_lut = 12'b100100100111; // K.08.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010001001: o_lut = 12'b111011101001; // K.09.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010001010: o_lut = 12'b111011101010; // K.10.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010001011: o_lut = 12'b111011001011; // K.11.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010001100: o_lut = 12'b111011101100; // K.12.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010001101: o_lut = 12'b111011001101; // K.13.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010001110: o_lut = 12'b111011001110; // K.14.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010001111: o_lut = 12'b100100111010; // K.15.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010010000: o_lut = 12'b100100110110; // K.16.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010010001: o_lut = 12'b111011110001; // K.17.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010010010: o_lut = 12'b111011110010; // K.18.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010010011: o_lut = 12'b111011010011; // K.19.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010010100: o_lut = 12'b111011110100; // K.20.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010010101: o_lut = 12'b111011010101; // K.21.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010010110: o_lut = 12'b111011010110; // K.22.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010010111: o_lut = 12'b100100010111; // K.23.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010011000: o_lut = 12'b100100110011; // K.24.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010011001: o_lut = 12'b111011011001; // K.25.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010011010: o_lut = 12'b111011011010; // K.26.4, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010011011: o_lut = 12'b100100011011; // K.27.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010011100: o_lut = 12'b000100111100; // K.28.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1010011101: o_lut = 12'b100100011101; // K.29.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010011110: o_lut = 12'b100100011110; // K.30.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010011111: o_lut = 12'b100100110101; // K.31.4, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010100000: o_lut = 12'b110101111001; // K.00.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010100001: o_lut = 12'b110101101110; // K.01.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010100010: o_lut = 12'b110101101101; // K.02.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010100011: o_lut = 12'b101010100011; // K.03.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010100100: o_lut = 12'b110101101011; // K.04.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010100101: o_lut = 12'b101010100101; // K.05.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010100110: o_lut = 12'b101010100110; // K.06.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010100111: o_lut = 12'b101010000111; // K.07.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010101000: o_lut = 12'b110101100111; // K.08.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010101001: o_lut = 12'b101010101001; // K.09.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010101010: o_lut = 12'b101010101010; // K.10.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010101011: o_lut = 12'b101010001011; // K.11.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010101100: o_lut = 12'b101010101100; // K.12.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010101101: o_lut = 12'b101010001101; // K.13.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010101110: o_lut = 12'b101010001110; // K.14.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010101111: o_lut = 12'b110101111010; // K.15.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010110000: o_lut = 12'b110101110110; // K.16.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010110001: o_lut = 12'b101010110001; // K.17.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010110010: o_lut = 12'b101010110010; // K.18.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010110011: o_lut = 12'b101010010011; // K.19.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010110100: o_lut = 12'b101010110100; // K.20.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010110101: o_lut = 12'b101010010101; // K.21.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010110110: o_lut = 12'b101010010110; // K.22.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010110111: o_lut = 12'b110101010111; // K.23.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010111000: o_lut = 12'b110101110011; // K.24.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010111001: o_lut = 12'b101010011001; // K.25.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010111010: o_lut = 12'b101010011010; // K.26.5, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1010111011: o_lut = 12'b110101011011; // K.27.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010111100: o_lut = 12'b010101111100; // K.28.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1010111101: o_lut = 12'b110101011101; // K.29.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010111110: o_lut = 12'b110101011110; // K.30.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1010111111: o_lut = 12'b110101110101; // K.31.5, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011000000: o_lut = 12'b110110111001; // K.00.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011000001: o_lut = 12'b110110101110; // K.01.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011000010: o_lut = 12'b110110101101; // K.02.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011000011: o_lut = 12'b101001100011; // K.03.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011000100: o_lut = 12'b110110101011; // K.04.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011000101: o_lut = 12'b101001100101; // K.05.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011000110: o_lut = 12'b101001100110; // K.06.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011000111: o_lut = 12'b101001000111; // K.07.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011001000: o_lut = 12'b110110100111; // K.08.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011001001: o_lut = 12'b101001101001; // K.09.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011001010: o_lut = 12'b101001101010; // K.10.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011001011: o_lut = 12'b101001001011; // K.11.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011001100: o_lut = 12'b101001101100; // K.12.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011001101: o_lut = 12'b101001001101; // K.13.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011001110: o_lut = 12'b101001001110; // K.14.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011001111: o_lut = 12'b110110111010; // K.15.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011010000: o_lut = 12'b110110110110; // K.16.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011010001: o_lut = 12'b101001110001; // K.17.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011010010: o_lut = 12'b101001110010; // K.18.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011010011: o_lut = 12'b101001010011; // K.19.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011010100: o_lut = 12'b101001110100; // K.20.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011010101: o_lut = 12'b101001010101; // K.21.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011010110: o_lut = 12'b101001010110; // K.22.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011010111: o_lut = 12'b110110010111; // K.23.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011011000: o_lut = 12'b110110110011; // K.24.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011011001: o_lut = 12'b101001011001; // K.25.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011011010: o_lut = 12'b101001011010; // K.26.6, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011011011: o_lut = 12'b110110011011; // K.27.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011011100: o_lut = 12'b010110111100; // K.28.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1011011101: o_lut = 12'b110110011101; // K.29.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011011110: o_lut = 12'b110110011110; // K.30.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011011111: o_lut = 12'b110110110101; // K.31.6, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011100000: o_lut = 12'b100001111001; // K.00.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011100001: o_lut = 12'b100001101110; // K.01.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011100010: o_lut = 12'b100001101101; // K.02.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011100011: o_lut = 12'b111110100011; // K.03.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011100100: o_lut = 12'b100001101011; // K.04.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011100101: o_lut = 12'b111110100101; // K.05.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011100110: o_lut = 12'b111110100110; // K.06.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011100111: o_lut = 12'b111110000111; // K.07.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011101000: o_lut = 12'b100001100111; // K.08.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011101001: o_lut = 12'b111110101001; // K.09.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011101010: o_lut = 12'b111110101010; // K.10.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011101011: o_lut = 12'b111110001011; // K.11.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011101100: o_lut = 12'b111110101100; // K.12.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011101101: o_lut = 12'b111110001101; // K.13.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011101110: o_lut = 12'b111110001110; // K.14.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011101111: o_lut = 12'b100001111010; // K.15.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011110000: o_lut = 12'b100001110110; // K.16.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011110001: o_lut = 12'b111110110001; // K.17.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011110010: o_lut = 12'b111110110010; // K.18.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011110011: o_lut = 12'b111110010011; // K.19.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011110100: o_lut = 12'b111110110100; // K.20.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011110101: o_lut = 12'b111110010101; // K.21.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011110110: o_lut = 12'b111110010110; // K.22.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011110111: o_lut = 12'b000001010111; // K.23.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1011111000: o_lut = 12'b100001110011; // K.24.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1011111001: o_lut = 12'b111110011001; // K.25.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011111010: o_lut = 12'b111110011010; // K.26.7, i_disp = 0, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1011111011: o_lut = 12'b000001011011; // K.27.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1011111100: o_lut = 12'b000001111100; // K.28.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1011111101: o_lut = 12'b000001011101; // K.29.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1011111110: o_lut = 12'b000001011110; // K.30.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1011111111: o_lut = 12'b100001110101; // K.31.7, i_disp = 0, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100000000: o_lut = 12'b111101000110; // K.00.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100000001: o_lut = 12'b111101010001; // K.01.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100000010: o_lut = 12'b111101010010; // K.02.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100000011: o_lut = 12'b100010100011; // K.03.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100000100: o_lut = 12'b111101010100; // K.04.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100000101: o_lut = 12'b100010100101; // K.05.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100000110: o_lut = 12'b100010100110; // K.06.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100000111: o_lut = 12'b100010111000; // K.07.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100001000: o_lut = 12'b111101011000; // K.08.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100001001: o_lut = 12'b100010101001; // K.09.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100001010: o_lut = 12'b100010101010; // K.10.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100001011: o_lut = 12'b100010001011; // K.11.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100001100: o_lut = 12'b100010101100; // K.12.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100001101: o_lut = 12'b100010001101; // K.13.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100001110: o_lut = 12'b100010001110; // K.14.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100001111: o_lut = 12'b111101000101; // K.15.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100010000: o_lut = 12'b111101001001; // K.16.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100010001: o_lut = 12'b100010110001; // K.17.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100010010: o_lut = 12'b100010110010; // K.18.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100010011: o_lut = 12'b100010010011; // K.19.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100010100: o_lut = 12'b100010110100; // K.20.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100010101: o_lut = 12'b100010010101; // K.21.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100010110: o_lut = 12'b100010010110; // K.22.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100010111: o_lut = 12'b111101101000; // K.23.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100011000: o_lut = 12'b111101001100; // K.24.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100011001: o_lut = 12'b100010011001; // K.25.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100011010: o_lut = 12'b100010011010; // K.26.0, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100011011: o_lut = 12'b111101100100; // K.27.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100011100: o_lut = 12'b011101000011; // K.28.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1100011101: o_lut = 12'b111101100010; // K.29.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100011110: o_lut = 12'b111101100001; // K.30.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100011111: o_lut = 12'b111101001010; // K.31.0, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100100000: o_lut = 12'b100110000110; // K.00.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100100001: o_lut = 12'b100110010001; // K.01.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100100010: o_lut = 12'b100110010010; // K.02.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100100011: o_lut = 12'b111001100011; // K.03.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100100100: o_lut = 12'b100110010100; // K.04.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100100101: o_lut = 12'b111001100101; // K.05.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100100110: o_lut = 12'b111001100110; // K.06.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100100111: o_lut = 12'b111001111000; // K.07.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100101000: o_lut = 12'b100110011000; // K.08.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100101001: o_lut = 12'b111001101001; // K.09.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100101010: o_lut = 12'b111001101010; // K.10.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100101011: o_lut = 12'b111001001011; // K.11.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100101100: o_lut = 12'b111001101100; // K.12.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100101101: o_lut = 12'b111001001101; // K.13.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100101110: o_lut = 12'b111001001110; // K.14.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100101111: o_lut = 12'b100110000101; // K.15.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100110000: o_lut = 12'b100110001001; // K.16.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100110001: o_lut = 12'b111001110001; // K.17.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100110010: o_lut = 12'b111001110010; // K.18.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100110011: o_lut = 12'b111001010011; // K.19.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100110100: o_lut = 12'b111001110100; // K.20.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100110101: o_lut = 12'b111001010101; // K.21.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100110110: o_lut = 12'b111001010110; // K.22.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100110111: o_lut = 12'b100110101000; // K.23.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100111000: o_lut = 12'b100110001100; // K.24.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100111001: o_lut = 12'b111001011001; // K.25.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100111010: o_lut = 12'b111001011010; // K.26.1, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1100111011: o_lut = 12'b100110100100; // K.27.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100111100: o_lut = 12'b000110000011; // K.28.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1100111101: o_lut = 12'b100110100010; // K.29.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100111110: o_lut = 12'b100110100001; // K.30.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1100111111: o_lut = 12'b100110001010; // K.31.1, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101000000: o_lut = 12'b100101000110; // K.00.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101000001: o_lut = 12'b100101010001; // K.01.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101000010: o_lut = 12'b100101010010; // K.02.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101000011: o_lut = 12'b111010100011; // K.03.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101000100: o_lut = 12'b100101010100; // K.04.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101000101: o_lut = 12'b111010100101; // K.05.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101000110: o_lut = 12'b111010100110; // K.06.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101000111: o_lut = 12'b111010111000; // K.07.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101001000: o_lut = 12'b100101011000; // K.08.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101001001: o_lut = 12'b111010101001; // K.09.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101001010: o_lut = 12'b111010101010; // K.10.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101001011: o_lut = 12'b111010001011; // K.11.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101001100: o_lut = 12'b111010101100; // K.12.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101001101: o_lut = 12'b111010001101; // K.13.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101001110: o_lut = 12'b111010001110; // K.14.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101001111: o_lut = 12'b100101000101; // K.15.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101010000: o_lut = 12'b100101001001; // K.16.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101010001: o_lut = 12'b111010110001; // K.17.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101010010: o_lut = 12'b111010110010; // K.18.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101010011: o_lut = 12'b111010010011; // K.19.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101010100: o_lut = 12'b111010110100; // K.20.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101010101: o_lut = 12'b111010010101; // K.21.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101010110: o_lut = 12'b111010010110; // K.22.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101010111: o_lut = 12'b100101101000; // K.23.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101011000: o_lut = 12'b100101001100; // K.24.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101011001: o_lut = 12'b111010011001; // K.25.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101011010: o_lut = 12'b111010011010; // K.26.2, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101011011: o_lut = 12'b100101100100; // K.27.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101011100: o_lut = 12'b000101000011; // K.28.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1101011101: o_lut = 12'b100101100010; // K.29.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101011110: o_lut = 12'b100101100001; // K.30.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101011111: o_lut = 12'b100101001010; // K.31.2, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101100000: o_lut = 12'b100011000110; // K.00.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101100001: o_lut = 12'b100011010001; // K.01.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101100010: o_lut = 12'b100011010010; // K.02.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101100011: o_lut = 12'b111100100011; // K.03.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101100100: o_lut = 12'b100011010100; // K.04.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101100101: o_lut = 12'b111100100101; // K.05.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101100110: o_lut = 12'b111100100110; // K.06.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101100111: o_lut = 12'b111100111000; // K.07.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101101000: o_lut = 12'b100011011000; // K.08.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101101001: o_lut = 12'b111100101001; // K.09.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101101010: o_lut = 12'b111100101010; // K.10.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101101011: o_lut = 12'b111100001011; // K.11.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101101100: o_lut = 12'b111100101100; // K.12.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101101101: o_lut = 12'b111100001101; // K.13.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101101110: o_lut = 12'b111100001110; // K.14.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101101111: o_lut = 12'b100011000101; // K.15.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101110000: o_lut = 12'b100011001001; // K.16.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101110001: o_lut = 12'b111100110001; // K.17.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101110010: o_lut = 12'b111100110010; // K.18.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101110011: o_lut = 12'b111100010011; // K.19.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101110100: o_lut = 12'b111100110100; // K.20.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101110101: o_lut = 12'b111100010101; // K.21.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101110110: o_lut = 12'b111100010110; // K.22.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101110111: o_lut = 12'b100011101000; // K.23.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101111000: o_lut = 12'b100011001100; // K.24.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101111001: o_lut = 12'b111100011001; // K.25.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101111010: o_lut = 12'b111100011010; // K.26.3, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1101111011: o_lut = 12'b100011100100; // K.27.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101111100: o_lut = 12'b000011000011; // K.28.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1101111101: o_lut = 12'b100011100010; // K.29.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101111110: o_lut = 12'b100011100001; // K.30.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1101111111: o_lut = 12'b100011001010; // K.31.3, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110000000: o_lut = 12'b111011000110; // K.00.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110000001: o_lut = 12'b111011010001; // K.01.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110000010: o_lut = 12'b111011010010; // K.02.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110000011: o_lut = 12'b100100100011; // K.03.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110000100: o_lut = 12'b111011010100; // K.04.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110000101: o_lut = 12'b100100100101; // K.05.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110000110: o_lut = 12'b100100100110; // K.06.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110000111: o_lut = 12'b100100111000; // K.07.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110001000: o_lut = 12'b111011011000; // K.08.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110001001: o_lut = 12'b100100101001; // K.09.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110001010: o_lut = 12'b100100101010; // K.10.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110001011: o_lut = 12'b100100001011; // K.11.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110001100: o_lut = 12'b100100101100; // K.12.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110001101: o_lut = 12'b100100001101; // K.13.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110001110: o_lut = 12'b100100001110; // K.14.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110001111: o_lut = 12'b111011000101; // K.15.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110010000: o_lut = 12'b111011001001; // K.16.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110010001: o_lut = 12'b100100110001; // K.17.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110010010: o_lut = 12'b100100110010; // K.18.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110010011: o_lut = 12'b100100010011; // K.19.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110010100: o_lut = 12'b100100110100; // K.20.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110010101: o_lut = 12'b100100010101; // K.21.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110010110: o_lut = 12'b100100010110; // K.22.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110010111: o_lut = 12'b111011101000; // K.23.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110011000: o_lut = 12'b111011001100; // K.24.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110011001: o_lut = 12'b100100011001; // K.25.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110011010: o_lut = 12'b100100011010; // K.26.4, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110011011: o_lut = 12'b111011100100; // K.27.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110011100: o_lut = 12'b011011000011; // K.28.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1110011101: o_lut = 12'b111011100010; // K.29.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110011110: o_lut = 12'b111011100001; // K.30.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110011111: o_lut = 12'b111011001010; // K.31.4, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110100000: o_lut = 12'b101010000110; // K.00.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110100001: o_lut = 12'b101010010001; // K.01.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110100010: o_lut = 12'b101010010010; // K.02.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110100011: o_lut = 12'b110101100011; // K.03.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110100100: o_lut = 12'b101010010100; // K.04.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110100101: o_lut = 12'b110101100101; // K.05.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110100110: o_lut = 12'b110101100110; // K.06.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110100111: o_lut = 12'b110101111000; // K.07.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110101000: o_lut = 12'b101010011000; // K.08.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110101001: o_lut = 12'b110101101001; // K.09.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110101010: o_lut = 12'b110101101010; // K.10.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110101011: o_lut = 12'b110101001011; // K.11.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110101100: o_lut = 12'b110101101100; // K.12.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110101101: o_lut = 12'b110101001101; // K.13.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110101110: o_lut = 12'b110101001110; // K.14.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110101111: o_lut = 12'b101010000101; // K.15.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110110000: o_lut = 12'b101010001001; // K.16.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110110001: o_lut = 12'b110101110001; // K.17.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110110010: o_lut = 12'b110101110010; // K.18.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110110011: o_lut = 12'b110101010011; // K.19.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110110100: o_lut = 12'b110101110100; // K.20.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110110101: o_lut = 12'b110101010101; // K.21.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110110110: o_lut = 12'b110101010110; // K.22.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110110111: o_lut = 12'b101010101000; // K.23.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110111000: o_lut = 12'b101010001100; // K.24.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110111001: o_lut = 12'b110101011001; // K.25.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110111010: o_lut = 12'b110101011010; // K.26.5, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1110111011: o_lut = 12'b101010100100; // K.27.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110111100: o_lut = 12'b001010000011; // K.28.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1110111101: o_lut = 12'b101010100010; // K.29.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110111110: o_lut = 12'b101010100001; // K.30.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1110111111: o_lut = 12'b101010001010; // K.31.5, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111000000: o_lut = 12'b101001000110; // K.00.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111000001: o_lut = 12'b101001010001; // K.01.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111000010: o_lut = 12'b101001010010; // K.02.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111000011: o_lut = 12'b110110100011; // K.03.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111000100: o_lut = 12'b101001010100; // K.04.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111000101: o_lut = 12'b110110100101; // K.05.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111000110: o_lut = 12'b110110100110; // K.06.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111000111: o_lut = 12'b110110111000; // K.07.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111001000: o_lut = 12'b101001011000; // K.08.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111001001: o_lut = 12'b110110101001; // K.09.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111001010: o_lut = 12'b110110101010; // K.10.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111001011: o_lut = 12'b110110001011; // K.11.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111001100: o_lut = 12'b110110101100; // K.12.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111001101: o_lut = 12'b110110001101; // K.13.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111001110: o_lut = 12'b110110001110; // K.14.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111001111: o_lut = 12'b101001000101; // K.15.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111010000: o_lut = 12'b101001001001; // K.16.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111010001: o_lut = 12'b110110110001; // K.17.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111010010: o_lut = 12'b110110110010; // K.18.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111010011: o_lut = 12'b110110010011; // K.19.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111010100: o_lut = 12'b110110110100; // K.20.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111010101: o_lut = 12'b110110010101; // K.21.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111010110: o_lut = 12'b110110010110; // K.22.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111010111: o_lut = 12'b101001101000; // K.23.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111011000: o_lut = 12'b101001001100; // K.24.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111011001: o_lut = 12'b110110011001; // K.25.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111011010: o_lut = 12'b110110011010; // K.26.6, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111011011: o_lut = 12'b101001100100; // K.27.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111011100: o_lut = 12'b001001000011; // K.28.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 0
      10'b1111011101: o_lut = 12'b101001100010; // K.29.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111011110: o_lut = 12'b101001100001; // K.30.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111011111: o_lut = 12'b101001001010; // K.31.6, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111100000: o_lut = 12'b111110000110; // K.00.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111100001: o_lut = 12'b111110010001; // K.01.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111100010: o_lut = 12'b111110010010; // K.02.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111100011: o_lut = 12'b100001100011; // K.03.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111100100: o_lut = 12'b111110010100; // K.04.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111100101: o_lut = 12'b100001100101; // K.05.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111100110: o_lut = 12'b100001100110; // K.06.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111100111: o_lut = 12'b100001111000; // K.07.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111101000: o_lut = 12'b111110011000; // K.08.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111101001: o_lut = 12'b100001101001; // K.09.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111101010: o_lut = 12'b100001101010; // K.10.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111101011: o_lut = 12'b100001001011; // K.11.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111101100: o_lut = 12'b100001101100; // K.12.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111101101: o_lut = 12'b100001001101; // K.13.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111101110: o_lut = 12'b100001001110; // K.14.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111101111: o_lut = 12'b111110000101; // K.15.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111110000: o_lut = 12'b111110001001; // K.16.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111110001: o_lut = 12'b100001110001; // K.17.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111110010: o_lut = 12'b100001110010; // K.18.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111110011: o_lut = 12'b100001010011; // K.19.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111110100: o_lut = 12'b100001110100; // K.20.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111110101: o_lut = 12'b100001010101; // K.21.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111110110: o_lut = 12'b100001010110; // K.22.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111110111: o_lut = 12'b011110101000; // K.23.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1111111000: o_lut = 12'b111110001100; // K.24.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      10'b1111111001: o_lut = 12'b100001011001; // K.25.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111111010: o_lut = 12'b100001011010; // K.26.7, i_disp = 1, i_ctrl = 1 o_disp = 0, o_code_err = 1
      10'b1111111011: o_lut = 12'b011110100100; // K.27.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1111111100: o_lut = 12'b011110000011; // K.28.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1111111101: o_lut = 12'b011110100010; // K.29.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1111111110: o_lut = 12'b011110100001; // K.30.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 0
      10'b1111111111: o_lut = 12'b111110001010; // K.31.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      default: o_lut = 12'b111110001010; // K.31.7, i_disp = 1, i_ctrl = 1 o_disp = 1, o_code_err = 1
      // verilog_format: on
            // verilog_lint: waive-stop line-length
        endcase

    end

endmodule

`endif  /* LIBSV_CODERS_ENCODER_8B10B */
