`ifndef LIBSV_CODERS_DECODER_8B10B
`define LIBSV_CODERS_DECODER_8B10B

module decoder_8b10b (
    input  logic       i_clk,
    input  logic       i_reset_n,
    input  logic       i_en,
    input  logic [9:0] i_10b,
    output logic [7:0] o_8b,
    output logic       o_ctrl,
    output logic       o_code_err,
    output logic       o_disp_err
);

    logic        rd;  // running disparity
    logic [10:0] i_lut;  // input vector to 8b/10b decoding look-up table
    logic [11:0] o_lut;  // output vector from 8b/10b encoding look-up table

    always_ff @(posedge i_clk or negedge i_reset_n) begin

        if (!i_reset_n) begin

            // if in reset set all outputs to 0 and reset running disparity
            // to 0 (-1)
            o_8b       <= 8'b0;
            o_ctrl     <= 1'b0;
            o_code_err <= 1'b0;
            o_disp_err <= 1'b0;
            rd         <= 1'b0;

        end else if (!i_en) begin

            // if not enabled, then maintain current output
            // and running disparity
            o_8b       <= o_8b;
            o_ctrl     <= o_ctrl;
            o_code_err <= o_code_err;
            o_disp_err <= o_disp_err;
            rd         <= rd;

        end else begin

            // if enabled and not in reset, then get the 8b output and
            // updated running disparity from the 8b10b decoding look-up table
            o_8b       <= o_lut[7:0];
            rd         <= o_lut[8];
            o_ctrl     <= o_lut[9];
            o_code_err <= o_lut[10];
            o_disp_err <= o_lut[11];

        end

    end

    // 8b/10b decoding look-up table
    // ------------------------------------------------------------
    // The mapping is rjhgfiedcba: ZYXRHGFEDCBA, where:
    //     * r          = input running disparity (0 = -1, 1 = +1)
    //     * jhgfiedcba = input 10b value
    //     * Z          = output disparity error
    //     * Y          = output code error
    //     * X          = output control symbol flag
    //     * R          = output disparity (0 = -1, 1 = +1)
    //     * HGFEDCBA   = output 8b value
    assign i_lut = {rd, i_10b};  // Form input look-up table vector
    always_comb begin
        case (i_lut)
            // verilog_lint: waive-start line-length  // Lines are too long but this is what is wanted so disable lint checks
            // verilog_format: off // don't format this because it breaks each case into multiple lines
      //  rjhgfiedbca:             ZYXRHGFEDCBA
      11'b00000000000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000000001: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000000010: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000000011: o_lut = 12'b110000011100; // D.28.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000000100: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000000101: o_lut = 12'b110000001111; // D.15.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000000110: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000000111: o_lut = 12'b110100000111; // D.07.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000001000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000001001: o_lut = 12'b110000010000; // D.16.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000001010: o_lut = 12'b110000011111; // D.31.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000001011: o_lut = 12'b110100001011; // D.11.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000001100: o_lut = 12'b110000011000; // D.24.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000001101: o_lut = 12'b110100001101; // D.13.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000001110: o_lut = 12'b110100001110; // D.14.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000001111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000010000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000010001: o_lut = 12'b110000000001; // D.01.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000010010: o_lut = 12'b110000000010; // D.02.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000010011: o_lut = 12'b110100010011; // D.19.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000010100: o_lut = 12'b110000000100; // D.04.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000010101: o_lut = 12'b110100010101; // D.21.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000010110: o_lut = 12'b110100010110; // D.22.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000010111: o_lut = 12'b110000010111; // D.23.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000011000: o_lut = 12'b110000001000; // D.08.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000011001: o_lut = 12'b110100011001; // D.25.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000011010: o_lut = 12'b110100011010; // D.26.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000011011: o_lut = 12'b110000011011; // D.27.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000011100: o_lut = 12'b110100011100; // D.28.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000011101: o_lut = 12'b110000011101; // D.29.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000011110: o_lut = 12'b110000011110; // D.30.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000011111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000100000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000100001: o_lut = 12'b110000011110; // D.30.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000100010: o_lut = 12'b110000011101; // D.29.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000100011: o_lut = 12'b110100000011; // D.03.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000100100: o_lut = 12'b110000011011; // D.27.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000100101: o_lut = 12'b110100000101; // D.05.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000100110: o_lut = 12'b110100000110; // D.06.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000100111: o_lut = 12'b110000001000; // D.08.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000101000: o_lut = 12'b110000010111; // D.23.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000101001: o_lut = 12'b110100001001; // D.09.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000101010: o_lut = 12'b110100001010; // D.10.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000101011: o_lut = 12'b110000000100; // D.04.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000101100: o_lut = 12'b110100001100; // D.12.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000101101: o_lut = 12'b110000000010; // D.02.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000101110: o_lut = 12'b110000000001; // D.01.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000101111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000110000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000110001: o_lut = 12'b110100010001; // D.17.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000110010: o_lut = 12'b110100010010; // D.18.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000110011: o_lut = 12'b110000011000; // D.24.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000110100: o_lut = 12'b110100010100; // D.20.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000110101: o_lut = 12'b110000011111; // D.31.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000110110: o_lut = 12'b110000010000; // D.16.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000110111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000111000: o_lut = 12'b110100000111; // D.07.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000111001: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000111010: o_lut = 12'b110000001111; // D.15.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000111011: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000111100: o_lut = 12'b110000011100; // D.28.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000111101: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000111110: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00000111111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001000000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001000001: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001000010: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001000011: o_lut = 12'b100011111100; // D.28.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001000100: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001000101: o_lut = 12'b100011101111; // D.15.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001000110: o_lut = 12'b100011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001000111: o_lut = 12'b100111100111; // D.07.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001001000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001001001: o_lut = 12'b100011110000; // D.16.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001001010: o_lut = 12'b100011111111; // D.31.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001001011: o_lut = 12'b100111101011; // D.11.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001001100: o_lut = 12'b100011111000; // D.24.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001001101: o_lut = 12'b100111101101; // D.13.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001001110: o_lut = 12'b100111101110; // D.14.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001001111: o_lut = 12'b010011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b00001010000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001010001: o_lut = 12'b100011100001; // D.01.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001010010: o_lut = 12'b100011100010; // D.02.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001010011: o_lut = 12'b100111110011; // D.19.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001010100: o_lut = 12'b100011100100; // D.04.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001010101: o_lut = 12'b100111110101; // D.21.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001010110: o_lut = 12'b100111110110; // D.22.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001010111: o_lut = 12'b001011110111; // K.23.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b00001011000: o_lut = 12'b100011101000; // D.08.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001011001: o_lut = 12'b100111111001; // D.25.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001011010: o_lut = 12'b100111111010; // D.26.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001011011: o_lut = 12'b001011111011; // K.27.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b00001011100: o_lut = 12'b100111111100; // D.28.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001011101: o_lut = 12'b001011111101; // K.29.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b00001011110: o_lut = 12'b001011111110; // K.30.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b00001011111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001100000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001100001: o_lut = 12'b100011111110; // D.30.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001100010: o_lut = 12'b100011111101; // D.29.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001100011: o_lut = 12'b100111100011; // D.03.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001100100: o_lut = 12'b100011111011; // D.27.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001100101: o_lut = 12'b100111100101; // D.05.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001100110: o_lut = 12'b100111100110; // D.06.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001100111: o_lut = 12'b000011101000; // D.08.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00001101000: o_lut = 12'b100011110111; // D.23.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001101001: o_lut = 12'b100111101001; // D.09.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001101010: o_lut = 12'b100111101010; // D.10.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001101011: o_lut = 12'b000011100100; // D.04.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00001101100: o_lut = 12'b100111101100; // D.12.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001101101: o_lut = 12'b000011100010; // D.02.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00001101110: o_lut = 12'b000011100001; // D.01.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00001101111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001110000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001110001: o_lut = 12'b100111110001; // D.17.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001110010: o_lut = 12'b100111110010; // D.18.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001110011: o_lut = 12'b000011111000; // D.24.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00001110100: o_lut = 12'b100111110100; // D.20.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001110101: o_lut = 12'b000011111111; // D.31.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00001110110: o_lut = 12'b000011110000; // D.16.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00001110111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001111000: o_lut = 12'b100111100111; // D.07.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00001111001: o_lut = 12'b000011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00001111010: o_lut = 12'b000011101111; // D.15.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00001111011: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001111100: o_lut = 12'b001011111100; // K.28.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b00001111101: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001111110: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00001111111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010000000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010000001: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010000010: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010000011: o_lut = 12'b100000011100; // D.28.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010000100: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010000101: o_lut = 12'b100000001111; // D.15.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010000110: o_lut = 12'b100000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010000111: o_lut = 12'b100100000111; // D.07.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010001000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010001001: o_lut = 12'b100000010000; // D.16.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010001010: o_lut = 12'b100000011111; // D.31.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010001011: o_lut = 12'b100100001011; // D.11.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010001100: o_lut = 12'b100000011000; // D.24.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010001101: o_lut = 12'b100100001101; // D.13.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010001110: o_lut = 12'b100100001110; // D.14.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010001111: o_lut = 12'b010000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b00010010000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010010001: o_lut = 12'b100000000001; // D.01.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010010010: o_lut = 12'b100000000010; // D.02.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010010011: o_lut = 12'b100100010011; // D.19.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010010100: o_lut = 12'b100000000100; // D.04.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010010101: o_lut = 12'b100100010101; // D.21.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010010110: o_lut = 12'b100100010110; // D.22.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010010111: o_lut = 12'b000000010111; // D.23.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010011000: o_lut = 12'b100000001000; // D.08.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010011001: o_lut = 12'b100100011001; // D.25.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010011010: o_lut = 12'b100100011010; // D.26.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010011011: o_lut = 12'b000000011011; // D.27.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010011100: o_lut = 12'b100100011100; // D.28.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010011101: o_lut = 12'b000000011101; // D.29.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010011110: o_lut = 12'b000000011110; // D.30.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010011111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010100000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010100001: o_lut = 12'b100000011110; // D.30.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010100010: o_lut = 12'b100000011101; // D.29.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010100011: o_lut = 12'b100100000011; // D.03.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010100100: o_lut = 12'b100000011011; // D.27.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010100101: o_lut = 12'b100100000101; // D.05.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010100110: o_lut = 12'b100100000110; // D.06.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010100111: o_lut = 12'b000000001000; // D.08.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010101000: o_lut = 12'b100000010111; // D.23.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010101001: o_lut = 12'b100100001001; // D.09.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010101010: o_lut = 12'b100100001010; // D.10.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010101011: o_lut = 12'b000000000100; // D.04.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010101100: o_lut = 12'b100100001100; // D.12.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010101101: o_lut = 12'b000000000010; // D.02.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010101110: o_lut = 12'b000000000001; // D.01.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010101111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010110000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010110001: o_lut = 12'b100100010001; // D.17.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010110010: o_lut = 12'b100100010010; // D.18.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010110011: o_lut = 12'b000000011000; // D.24.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010110100: o_lut = 12'b100100010100; // D.20.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010110101: o_lut = 12'b000000011111; // D.31.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010110110: o_lut = 12'b000000010000; // D.16.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010110111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010111000: o_lut = 12'b100100000111; // D.07.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00010111001: o_lut = 12'b000000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010111010: o_lut = 12'b000000001111; // D.15.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00010111011: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010111100: o_lut = 12'b001000011100; // K.28.0, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b00010111101: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010111110: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00010111111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011000000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011000001: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011000010: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011000011: o_lut = 12'b101101111100; // K.28.3, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b00011000100: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011000101: o_lut = 12'b100101101111; // D.15.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011000110: o_lut = 12'b100101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011000111: o_lut = 12'b000001100111; // D.07.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011001000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011001001: o_lut = 12'b100101110000; // D.16.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011001010: o_lut = 12'b100101111111; // D.31.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011001011: o_lut = 12'b000001101011; // D.11.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011001100: o_lut = 12'b100101111000; // D.24.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011001101: o_lut = 12'b000001101101; // D.13.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011001110: o_lut = 12'b000001101110; // D.14.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011001111: o_lut = 12'b010101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b00011010000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011010001: o_lut = 12'b100101100001; // D.01.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011010010: o_lut = 12'b100101100010; // D.02.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011010011: o_lut = 12'b000001110011; // D.19.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011010100: o_lut = 12'b100101100100; // D.04.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011010101: o_lut = 12'b000001110101; // D.21.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011010110: o_lut = 12'b000001110110; // D.22.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011010111: o_lut = 12'b000101110111; // D.23.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011011000: o_lut = 12'b100101101000; // D.08.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011011001: o_lut = 12'b000001111001; // D.25.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011011010: o_lut = 12'b000001111010; // D.26.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011011011: o_lut = 12'b000101111011; // D.27.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011011100: o_lut = 12'b000001111100; // D.28.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011011101: o_lut = 12'b000101111101; // D.29.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011011110: o_lut = 12'b000101111110; // D.30.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011011111: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011100000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011100001: o_lut = 12'b100101111110; // D.30.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011100010: o_lut = 12'b100101111101; // D.29.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011100011: o_lut = 12'b000001100011; // D.03.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011100100: o_lut = 12'b100101111011; // D.27.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011100101: o_lut = 12'b000001100101; // D.05.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011100110: o_lut = 12'b000001100110; // D.06.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011100111: o_lut = 12'b000101101000; // D.08.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011101000: o_lut = 12'b100101110111; // D.23.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00011101001: o_lut = 12'b000001101001; // D.09.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011101010: o_lut = 12'b000001101010; // D.10.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011101011: o_lut = 12'b000101100100; // D.04.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011101100: o_lut = 12'b000001101100; // D.12.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011101101: o_lut = 12'b000101100010; // D.02.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011101110: o_lut = 12'b000101100001; // D.01.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011101111: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011110000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011110001: o_lut = 12'b000001110001; // D.17.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011110010: o_lut = 12'b000001110010; // D.18.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011110011: o_lut = 12'b000101111000; // D.24.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011110100: o_lut = 12'b000001110100; // D.20.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011110101: o_lut = 12'b000101111111; // D.31.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011110110: o_lut = 12'b000101110000; // D.16.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011110111: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011111000: o_lut = 12'b000001100111; // D.07.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011111001: o_lut = 12'b000101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011111010: o_lut = 12'b000101101111; // D.15.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011111011: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011111100: o_lut = 12'b000101111100; // D.28.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00011111101: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011111110: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00011111111: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100000000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100000001: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100000010: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100000011: o_lut = 12'b100010011100; // D.28.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100000100: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100000101: o_lut = 12'b100010001111; // D.15.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100000110: o_lut = 12'b100010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100000111: o_lut = 12'b100110000111; // D.07.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100001000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100001001: o_lut = 12'b100010010000; // D.16.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100001010: o_lut = 12'b100010011111; // D.31.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100001011: o_lut = 12'b100110001011; // D.11.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100001100: o_lut = 12'b100010011000; // D.24.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100001101: o_lut = 12'b100110001101; // D.13.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100001110: o_lut = 12'b100110001110; // D.14.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100001111: o_lut = 12'b010010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b00100010000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100010001: o_lut = 12'b100010000001; // D.01.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100010010: o_lut = 12'b100010000010; // D.02.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100010011: o_lut = 12'b100110010011; // D.19.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100010100: o_lut = 12'b100010000100; // D.04.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100010101: o_lut = 12'b100110010101; // D.21.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100010110: o_lut = 12'b100110010110; // D.22.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100010111: o_lut = 12'b000010010111; // D.23.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100011000: o_lut = 12'b100010001000; // D.08.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100011001: o_lut = 12'b100110011001; // D.25.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100011010: o_lut = 12'b100110011010; // D.26.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100011011: o_lut = 12'b000010011011; // D.27.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100011100: o_lut = 12'b100110011100; // D.28.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100011101: o_lut = 12'b000010011101; // D.29.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100011110: o_lut = 12'b000010011110; // D.30.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100011111: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100100000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100100001: o_lut = 12'b100010011110; // D.30.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100100010: o_lut = 12'b100010011101; // D.29.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100100011: o_lut = 12'b100110000011; // D.03.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100100100: o_lut = 12'b100010011011; // D.27.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100100101: o_lut = 12'b100110000101; // D.05.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100100110: o_lut = 12'b100110000110; // D.06.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100100111: o_lut = 12'b000010001000; // D.08.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100101000: o_lut = 12'b100010010111; // D.23.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100101001: o_lut = 12'b100110001001; // D.09.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100101010: o_lut = 12'b100110001010; // D.10.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100101011: o_lut = 12'b000010000100; // D.04.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100101100: o_lut = 12'b100110001100; // D.12.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100101101: o_lut = 12'b000010000010; // D.02.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100101110: o_lut = 12'b000010000001; // D.01.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100101111: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100110000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100110001: o_lut = 12'b100110010001; // D.17.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100110010: o_lut = 12'b100110010010; // D.18.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100110011: o_lut = 12'b000010011000; // D.24.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100110100: o_lut = 12'b100110010100; // D.20.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100110101: o_lut = 12'b000010011111; // D.31.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100110110: o_lut = 12'b000010010000; // D.16.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100110111: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100111000: o_lut = 12'b100110000111; // D.07.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00100111001: o_lut = 12'b000010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100111010: o_lut = 12'b000010001111; // D.15.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00100111011: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100111100: o_lut = 12'b001010011100; // K.28.4, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b00100111101: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100111110: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00100111111: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101000000: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101000001: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101000010: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101000011: o_lut = 12'b101110111100; // K.28.5, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b00101000100: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101000101: o_lut = 12'b100110101111; // D.15.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101000110: o_lut = 12'b100110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101000111: o_lut = 12'b000010100111; // D.07.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101001000: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101001001: o_lut = 12'b100110110000; // D.16.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101001010: o_lut = 12'b100110111111; // D.31.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101001011: o_lut = 12'b000010101011; // D.11.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101001100: o_lut = 12'b100110111000; // D.24.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101001101: o_lut = 12'b000010101101; // D.13.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101001110: o_lut = 12'b000010101110; // D.14.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101001111: o_lut = 12'b010110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b00101010000: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101010001: o_lut = 12'b100110100001; // D.01.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101010010: o_lut = 12'b100110100010; // D.02.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101010011: o_lut = 12'b000010110011; // D.19.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101010100: o_lut = 12'b100110100100; // D.04.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101010101: o_lut = 12'b000010110101; // D.21.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101010110: o_lut = 12'b000010110110; // D.22.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101010111: o_lut = 12'b000110110111; // D.23.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101011000: o_lut = 12'b100110101000; // D.08.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101011001: o_lut = 12'b000010111001; // D.25.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101011010: o_lut = 12'b000010111010; // D.26.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101011011: o_lut = 12'b000110111011; // D.27.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101011100: o_lut = 12'b000010111100; // D.28.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101011101: o_lut = 12'b000110111101; // D.29.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101011110: o_lut = 12'b000110111110; // D.30.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101011111: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101100000: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101100001: o_lut = 12'b100110111110; // D.30.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101100010: o_lut = 12'b100110111101; // D.29.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101100011: o_lut = 12'b000010100011; // D.03.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101100100: o_lut = 12'b100110111011; // D.27.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101100101: o_lut = 12'b000010100101; // D.05.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101100110: o_lut = 12'b000010100110; // D.06.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101100111: o_lut = 12'b000110101000; // D.08.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101101000: o_lut = 12'b100110110111; // D.23.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00101101001: o_lut = 12'b000010101001; // D.09.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101101010: o_lut = 12'b000010101010; // D.10.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101101011: o_lut = 12'b000110100100; // D.04.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101101100: o_lut = 12'b000010101100; // D.12.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101101101: o_lut = 12'b000110100010; // D.02.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101101110: o_lut = 12'b000110100001; // D.01.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101101111: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101110000: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101110001: o_lut = 12'b000010110001; // D.17.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101110010: o_lut = 12'b000010110010; // D.18.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101110011: o_lut = 12'b000110111000; // D.24.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101110100: o_lut = 12'b000010110100; // D.20.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101110101: o_lut = 12'b000110111111; // D.31.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101110110: o_lut = 12'b000110110000; // D.16.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101110111: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101111000: o_lut = 12'b000010100111; // D.07.5, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101111001: o_lut = 12'b000110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101111010: o_lut = 12'b000110101111; // D.15.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00101111011: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101111100: o_lut = 12'b001110111100; // K.28.5, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b00101111101: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101111110: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00101111111: o_lut = 12'b110110100000; // D.00.5, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110000000: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110000001: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110000010: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110000011: o_lut = 12'b101111011100; // K.28.6, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b00110000100: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110000101: o_lut = 12'b100111001111; // D.15.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110000110: o_lut = 12'b100111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110000111: o_lut = 12'b000011000111; // D.07.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110001000: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110001001: o_lut = 12'b100111010000; // D.16.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110001010: o_lut = 12'b100111011111; // D.31.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110001011: o_lut = 12'b000011001011; // D.11.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110001100: o_lut = 12'b100111011000; // D.24.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110001101: o_lut = 12'b000011001101; // D.13.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110001110: o_lut = 12'b000011001110; // D.14.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110001111: o_lut = 12'b010111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b00110010000: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110010001: o_lut = 12'b100111000001; // D.01.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110010010: o_lut = 12'b100111000010; // D.02.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110010011: o_lut = 12'b000011010011; // D.19.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110010100: o_lut = 12'b100111000100; // D.04.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110010101: o_lut = 12'b000011010101; // D.21.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110010110: o_lut = 12'b000011010110; // D.22.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110010111: o_lut = 12'b000111010111; // D.23.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110011000: o_lut = 12'b100111001000; // D.08.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110011001: o_lut = 12'b000011011001; // D.25.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110011010: o_lut = 12'b000011011010; // D.26.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110011011: o_lut = 12'b000111011011; // D.27.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110011100: o_lut = 12'b000011011100; // D.28.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110011101: o_lut = 12'b000111011101; // D.29.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110011110: o_lut = 12'b000111011110; // D.30.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110011111: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110100000: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110100001: o_lut = 12'b100111011110; // D.30.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110100010: o_lut = 12'b100111011101; // D.29.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110100011: o_lut = 12'b000011000011; // D.03.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110100100: o_lut = 12'b100111011011; // D.27.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110100101: o_lut = 12'b000011000101; // D.05.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110100110: o_lut = 12'b000011000110; // D.06.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110100111: o_lut = 12'b000111001000; // D.08.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110101000: o_lut = 12'b100111010111; // D.23.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00110101001: o_lut = 12'b000011001001; // D.09.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110101010: o_lut = 12'b000011001010; // D.10.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110101011: o_lut = 12'b000111000100; // D.04.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110101100: o_lut = 12'b000011001100; // D.12.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110101101: o_lut = 12'b000111000010; // D.02.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110101110: o_lut = 12'b000111000001; // D.01.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110101111: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110110000: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110110001: o_lut = 12'b000011010001; // D.17.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110110010: o_lut = 12'b000011010010; // D.18.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110110011: o_lut = 12'b000111011000; // D.24.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110110100: o_lut = 12'b000011010100; // D.20.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110110101: o_lut = 12'b000111011111; // D.31.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110110110: o_lut = 12'b000111010000; // D.16.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110110111: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110111000: o_lut = 12'b000011000111; // D.07.6, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110111001: o_lut = 12'b000111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110111010: o_lut = 12'b000111001111; // D.15.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00110111011: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110111100: o_lut = 12'b001111011100; // K.28.6, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b00110111101: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110111110: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00110111111: o_lut = 12'b110111000000; // D.00.6, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111000000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111000001: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111000010: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111000011: o_lut = 12'b100011111100; // D.28.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111000100: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111000101: o_lut = 12'b100011101111; // D.15.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111000110: o_lut = 12'b100011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111000111: o_lut = 12'b000111100111; // D.07.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111001000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111001001: o_lut = 12'b100011110000; // D.16.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111001010: o_lut = 12'b100011111111; // D.31.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111001011: o_lut = 12'b000111101011; // D.11.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111001100: o_lut = 12'b100011111000; // D.24.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111001101: o_lut = 12'b000111101101; // D.13.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111001110: o_lut = 12'b000111101110; // D.14.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111001111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111010000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111010001: o_lut = 12'b100011100001; // D.01.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111010010: o_lut = 12'b100011100010; // D.02.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111010011: o_lut = 12'b000111110011; // D.19.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111010100: o_lut = 12'b100011100100; // D.04.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111010101: o_lut = 12'b000111110101; // D.21.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111010110: o_lut = 12'b000111110110; // D.22.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111010111: o_lut = 12'b100011110111; // D.23.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111011000: o_lut = 12'b100011101000; // D.08.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111011001: o_lut = 12'b000111111001; // D.25.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111011010: o_lut = 12'b000111111010; // D.26.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111011011: o_lut = 12'b100011111011; // D.27.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111011100: o_lut = 12'b000111111100; // D.28.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111011101: o_lut = 12'b100011111101; // D.29.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111011110: o_lut = 12'b100011111110; // D.30.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111011111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111100000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111100001: o_lut = 12'b100011111110; // D.30.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111100010: o_lut = 12'b100011111101; // D.29.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111100011: o_lut = 12'b000111100011; // D.03.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111100100: o_lut = 12'b100011111011; // D.27.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111100101: o_lut = 12'b000111100101; // D.05.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111100110: o_lut = 12'b000111100110; // D.06.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111100111: o_lut = 12'b100011101000; // D.08.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111101000: o_lut = 12'b100011110111; // D.23.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111101001: o_lut = 12'b000111101001; // D.09.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111101010: o_lut = 12'b000111101010; // D.10.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111101011: o_lut = 12'b100011100100; // D.04.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111101100: o_lut = 12'b000111101100; // D.12.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111101101: o_lut = 12'b100011100010; // D.02.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111101110: o_lut = 12'b100011100001; // D.01.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111101111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111110000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111110001: o_lut = 12'b000111110001; // D.17.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111110010: o_lut = 12'b000111110010; // D.18.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111110011: o_lut = 12'b100011111000; // D.24.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111110100: o_lut = 12'b000111110100; // D.20.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111110101: o_lut = 12'b100011111111; // D.31.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111110110: o_lut = 12'b100011110000; // D.16.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111110111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111111000: o_lut = 12'b000111100111; // D.07.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b00111111001: o_lut = 12'b100011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111111010: o_lut = 12'b100011101111; // D.15.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111111011: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111111100: o_lut = 12'b100011111100; // D.28.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b00111111101: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111111110: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b00111111111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000000000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000000001: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000000010: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000000011: o_lut = 12'b100011111100; // D.28.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000000100: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000000101: o_lut = 12'b100011101111; // D.15.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000000110: o_lut = 12'b100011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000000111: o_lut = 12'b100111100111; // D.07.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000001000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000001001: o_lut = 12'b100011110000; // D.16.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000001010: o_lut = 12'b100011111111; // D.31.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000001011: o_lut = 12'b100111101011; // D.11.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000001100: o_lut = 12'b100011111000; // D.24.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000001101: o_lut = 12'b100111101101; // D.13.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000001110: o_lut = 12'b100111101110; // D.14.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000001111: o_lut = 12'b010011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b01000010000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000010001: o_lut = 12'b100011100001; // D.01.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000010010: o_lut = 12'b100011100010; // D.02.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000010011: o_lut = 12'b100111110011; // D.19.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000010100: o_lut = 12'b100011100100; // D.04.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000010101: o_lut = 12'b100111110101; // D.21.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000010110: o_lut = 12'b100111110110; // D.22.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000010111: o_lut = 12'b000011110111; // D.23.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000011000: o_lut = 12'b100011101000; // D.08.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000011001: o_lut = 12'b100111111001; // D.25.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000011010: o_lut = 12'b100111111010; // D.26.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000011011: o_lut = 12'b000011111011; // D.27.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000011100: o_lut = 12'b100111111100; // D.28.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000011101: o_lut = 12'b000011111101; // D.29.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000011110: o_lut = 12'b000011111110; // D.30.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000011111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000100000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000100001: o_lut = 12'b100011111110; // D.30.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000100010: o_lut = 12'b100011111101; // D.29.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000100011: o_lut = 12'b100111100011; // D.03.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000100100: o_lut = 12'b100011111011; // D.27.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000100101: o_lut = 12'b100111100101; // D.05.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000100110: o_lut = 12'b100111100110; // D.06.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000100111: o_lut = 12'b000011101000; // D.08.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000101000: o_lut = 12'b100011110111; // D.23.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000101001: o_lut = 12'b100111101001; // D.09.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000101010: o_lut = 12'b100111101010; // D.10.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000101011: o_lut = 12'b000011100100; // D.04.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000101100: o_lut = 12'b100111101100; // D.12.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000101101: o_lut = 12'b000011100010; // D.02.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000101110: o_lut = 12'b000011100001; // D.01.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000101111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000110000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000110001: o_lut = 12'b100111110001; // D.17.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000110010: o_lut = 12'b100111110010; // D.18.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000110011: o_lut = 12'b000011111000; // D.24.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000110100: o_lut = 12'b100111110100; // D.20.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000110101: o_lut = 12'b000011111111; // D.31.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000110110: o_lut = 12'b000011110000; // D.16.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000110111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000111000: o_lut = 12'b100111100111; // D.07.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01000111001: o_lut = 12'b000011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000111010: o_lut = 12'b000011101111; // D.15.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000111011: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000111100: o_lut = 12'b000011111100; // D.28.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01000111101: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000111110: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01000111111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001000000: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001000001: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001000010: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001000011: o_lut = 12'b101100111100; // K.28.1, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b01001000100: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001000101: o_lut = 12'b100100101111; // D.15.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001000110: o_lut = 12'b100100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001000111: o_lut = 12'b000000100111; // D.07.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001001000: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001001001: o_lut = 12'b100100110000; // D.16.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001001010: o_lut = 12'b100100111111; // D.31.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001001011: o_lut = 12'b000000101011; // D.11.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001001100: o_lut = 12'b100100111000; // D.24.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001001101: o_lut = 12'b000000101101; // D.13.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001001110: o_lut = 12'b000000101110; // D.14.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001001111: o_lut = 12'b010100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b01001010000: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001010001: o_lut = 12'b100100100001; // D.01.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001010010: o_lut = 12'b100100100010; // D.02.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001010011: o_lut = 12'b000000110011; // D.19.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001010100: o_lut = 12'b100100100100; // D.04.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001010101: o_lut = 12'b000000110101; // D.21.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001010110: o_lut = 12'b000000110110; // D.22.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001010111: o_lut = 12'b000100110111; // D.23.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001011000: o_lut = 12'b100100101000; // D.08.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001011001: o_lut = 12'b000000111001; // D.25.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001011010: o_lut = 12'b000000111010; // D.26.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001011011: o_lut = 12'b000100111011; // D.27.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001011100: o_lut = 12'b000000111100; // D.28.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001011101: o_lut = 12'b000100111101; // D.29.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001011110: o_lut = 12'b000100111110; // D.30.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001011111: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001100000: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001100001: o_lut = 12'b100100111110; // D.30.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001100010: o_lut = 12'b100100111101; // D.29.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001100011: o_lut = 12'b000000100011; // D.03.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001100100: o_lut = 12'b100100111011; // D.27.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001100101: o_lut = 12'b000000100101; // D.05.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001100110: o_lut = 12'b000000100110; // D.06.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001100111: o_lut = 12'b000100101000; // D.08.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001101000: o_lut = 12'b100100110111; // D.23.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01001101001: o_lut = 12'b000000101001; // D.09.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001101010: o_lut = 12'b000000101010; // D.10.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001101011: o_lut = 12'b000100100100; // D.04.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001101100: o_lut = 12'b000000101100; // D.12.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001101101: o_lut = 12'b000100100010; // D.02.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001101110: o_lut = 12'b000100100001; // D.01.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001101111: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001110000: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001110001: o_lut = 12'b000000110001; // D.17.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001110010: o_lut = 12'b000000110010; // D.18.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001110011: o_lut = 12'b000100111000; // D.24.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001110100: o_lut = 12'b000000110100; // D.20.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001110101: o_lut = 12'b000100111111; // D.31.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001110110: o_lut = 12'b000100110000; // D.16.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001110111: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001111000: o_lut = 12'b000000100111; // D.07.1, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001111001: o_lut = 12'b000100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001111010: o_lut = 12'b000100101111; // D.15.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01001111011: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001111100: o_lut = 12'b001100111100; // K.28.1, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b01001111101: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001111110: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01001111111: o_lut = 12'b110100100000; // D.00.1, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010000000: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010000001: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010000010: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010000011: o_lut = 12'b101101011100; // K.28.2, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b01010000100: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010000101: o_lut = 12'b100101001111; // D.15.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010000110: o_lut = 12'b100101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010000111: o_lut = 12'b000001000111; // D.07.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010001000: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010001001: o_lut = 12'b100101010000; // D.16.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010001010: o_lut = 12'b100101011111; // D.31.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010001011: o_lut = 12'b000001001011; // D.11.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010001100: o_lut = 12'b100101011000; // D.24.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010001101: o_lut = 12'b000001001101; // D.13.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010001110: o_lut = 12'b000001001110; // D.14.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010001111: o_lut = 12'b010101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b01010010000: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010010001: o_lut = 12'b100101000001; // D.01.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010010010: o_lut = 12'b100101000010; // D.02.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010010011: o_lut = 12'b000001010011; // D.19.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010010100: o_lut = 12'b100101000100; // D.04.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010010101: o_lut = 12'b000001010101; // D.21.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010010110: o_lut = 12'b000001010110; // D.22.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010010111: o_lut = 12'b000101010111; // D.23.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010011000: o_lut = 12'b100101001000; // D.08.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010011001: o_lut = 12'b000001011001; // D.25.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010011010: o_lut = 12'b000001011010; // D.26.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010011011: o_lut = 12'b000101011011; // D.27.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010011100: o_lut = 12'b000001011100; // D.28.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010011101: o_lut = 12'b000101011101; // D.29.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010011110: o_lut = 12'b000101011110; // D.30.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010011111: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010100000: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010100001: o_lut = 12'b100101011110; // D.30.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010100010: o_lut = 12'b100101011101; // D.29.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010100011: o_lut = 12'b000001000011; // D.03.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010100100: o_lut = 12'b100101011011; // D.27.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010100101: o_lut = 12'b000001000101; // D.05.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010100110: o_lut = 12'b000001000110; // D.06.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010100111: o_lut = 12'b000101001000; // D.08.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010101000: o_lut = 12'b100101010111; // D.23.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01010101001: o_lut = 12'b000001001001; // D.09.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010101010: o_lut = 12'b000001001010; // D.10.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010101011: o_lut = 12'b000101000100; // D.04.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010101100: o_lut = 12'b000001001100; // D.12.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010101101: o_lut = 12'b000101000010; // D.02.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010101110: o_lut = 12'b000101000001; // D.01.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010101111: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010110000: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010110001: o_lut = 12'b000001010001; // D.17.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010110010: o_lut = 12'b000001010010; // D.18.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010110011: o_lut = 12'b000101011000; // D.24.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010110100: o_lut = 12'b000001010100; // D.20.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010110101: o_lut = 12'b000101011111; // D.31.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010110110: o_lut = 12'b000101010000; // D.16.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010110111: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010111000: o_lut = 12'b000001000111; // D.07.2, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010111001: o_lut = 12'b000101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010111010: o_lut = 12'b000101001111; // D.15.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01010111011: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010111100: o_lut = 12'b001101011100; // K.28.2, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b01010111101: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010111110: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01010111111: o_lut = 12'b110101000000; // D.00.2, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011000000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011000001: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011000010: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011000011: o_lut = 12'b101010011100; // K.28.4, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b01011000100: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011000101: o_lut = 12'b100010001111; // D.15.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011000110: o_lut = 12'b100010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011000111: o_lut = 12'b000110000111; // D.07.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011001000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011001001: o_lut = 12'b100010010000; // D.16.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011001010: o_lut = 12'b100010011111; // D.31.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011001011: o_lut = 12'b000110001011; // D.11.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011001100: o_lut = 12'b100010011000; // D.24.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011001101: o_lut = 12'b000110001101; // D.13.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011001110: o_lut = 12'b000110001110; // D.14.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011001111: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011010000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011010001: o_lut = 12'b100010000001; // D.01.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011010010: o_lut = 12'b100010000010; // D.02.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011010011: o_lut = 12'b000110010011; // D.19.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011010100: o_lut = 12'b100010000100; // D.04.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011010101: o_lut = 12'b000110010101; // D.21.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011010110: o_lut = 12'b000110010110; // D.22.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011010111: o_lut = 12'b100010010111; // D.23.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011011000: o_lut = 12'b100010001000; // D.08.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011011001: o_lut = 12'b000110011001; // D.25.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011011010: o_lut = 12'b000110011010; // D.26.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011011011: o_lut = 12'b100010011011; // D.27.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011011100: o_lut = 12'b000110011100; // D.28.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011011101: o_lut = 12'b100010011101; // D.29.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011011110: o_lut = 12'b100010011110; // D.30.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011011111: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011100000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011100001: o_lut = 12'b100010011110; // D.30.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011100010: o_lut = 12'b100010011101; // D.29.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011100011: o_lut = 12'b000110000011; // D.03.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011100100: o_lut = 12'b100010011011; // D.27.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011100101: o_lut = 12'b000110000101; // D.05.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011100110: o_lut = 12'b000110000110; // D.06.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011100111: o_lut = 12'b100010001000; // D.08.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011101000: o_lut = 12'b100010010111; // D.23.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011101001: o_lut = 12'b000110001001; // D.09.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011101010: o_lut = 12'b000110001010; // D.10.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011101011: o_lut = 12'b100010000100; // D.04.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011101100: o_lut = 12'b000110001100; // D.12.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011101101: o_lut = 12'b100010000010; // D.02.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011101110: o_lut = 12'b100010000001; // D.01.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011101111: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011110000: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011110001: o_lut = 12'b000110010001; // D.17.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011110010: o_lut = 12'b000110010010; // D.18.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011110011: o_lut = 12'b100010011000; // D.24.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011110100: o_lut = 12'b000110010100; // D.20.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011110101: o_lut = 12'b100010011111; // D.31.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011110110: o_lut = 12'b100010010000; // D.16.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011110111: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011111000: o_lut = 12'b000110000111; // D.07.4, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01011111001: o_lut = 12'b100010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011111010: o_lut = 12'b100010001111; // D.15.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011111011: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011111100: o_lut = 12'b100010011100; // D.28.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01011111101: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011111110: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01011111111: o_lut = 12'b110010000000; // D.00.4, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100000000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100000001: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100000010: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100000011: o_lut = 12'b100101111100; // D.28.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100000100: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100000101: o_lut = 12'b100101101111; // D.15.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100000110: o_lut = 12'b100101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100000111: o_lut = 12'b000001100111; // D.07.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100001000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100001001: o_lut = 12'b100101110000; // D.16.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100001010: o_lut = 12'b100101111111; // D.31.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100001011: o_lut = 12'b000001101011; // D.11.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100001100: o_lut = 12'b100101111000; // D.24.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100001101: o_lut = 12'b000001101101; // D.13.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100001110: o_lut = 12'b000001101110; // D.14.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100001111: o_lut = 12'b010101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b01100010000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100010001: o_lut = 12'b100101100001; // D.01.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100010010: o_lut = 12'b100101100010; // D.02.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100010011: o_lut = 12'b000001110011; // D.19.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100010100: o_lut = 12'b100101100100; // D.04.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100010101: o_lut = 12'b000001110101; // D.21.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100010110: o_lut = 12'b000001110110; // D.22.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100010111: o_lut = 12'b000101110111; // D.23.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100011000: o_lut = 12'b100101101000; // D.08.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100011001: o_lut = 12'b000001111001; // D.25.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100011010: o_lut = 12'b000001111010; // D.26.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100011011: o_lut = 12'b000101111011; // D.27.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100011100: o_lut = 12'b000001111100; // D.28.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100011101: o_lut = 12'b000101111101; // D.29.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100011110: o_lut = 12'b000101111110; // D.30.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100011111: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100100000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100100001: o_lut = 12'b100101111110; // D.30.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100100010: o_lut = 12'b100101111101; // D.29.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100100011: o_lut = 12'b000001100011; // D.03.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100100100: o_lut = 12'b100101111011; // D.27.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100100101: o_lut = 12'b000001100101; // D.05.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100100110: o_lut = 12'b000001100110; // D.06.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100100111: o_lut = 12'b000101101000; // D.08.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100101000: o_lut = 12'b100101110111; // D.23.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01100101001: o_lut = 12'b000001101001; // D.09.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100101010: o_lut = 12'b000001101010; // D.10.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100101011: o_lut = 12'b000101100100; // D.04.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100101100: o_lut = 12'b000001101100; // D.12.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100101101: o_lut = 12'b000101100010; // D.02.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100101110: o_lut = 12'b000101100001; // D.01.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100101111: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100110000: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100110001: o_lut = 12'b000001110001; // D.17.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100110010: o_lut = 12'b000001110010; // D.18.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100110011: o_lut = 12'b000101111000; // D.24.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100110100: o_lut = 12'b000001110100; // D.20.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100110101: o_lut = 12'b000101111111; // D.31.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100110110: o_lut = 12'b000101110000; // D.16.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100110111: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100111000: o_lut = 12'b000001100111; // D.07.3, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100111001: o_lut = 12'b000101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100111010: o_lut = 12'b000101101111; // D.15.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01100111011: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100111100: o_lut = 12'b001101111100; // K.28.3, i_disp = 0, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b01100111101: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100111110: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01100111111: o_lut = 12'b110101100000; // D.00.3, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101000000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101000001: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101000010: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101000011: o_lut = 12'b101000011100; // K.28.0, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b01101000100: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101000101: o_lut = 12'b100000001111; // D.15.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101000110: o_lut = 12'b100000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101000111: o_lut = 12'b000100000111; // D.07.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101001000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101001001: o_lut = 12'b100000010000; // D.16.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101001010: o_lut = 12'b100000011111; // D.31.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101001011: o_lut = 12'b000100001011; // D.11.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101001100: o_lut = 12'b100000011000; // D.24.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101001101: o_lut = 12'b000100001101; // D.13.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101001110: o_lut = 12'b000100001110; // D.14.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101001111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101010000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101010001: o_lut = 12'b100000000001; // D.01.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101010010: o_lut = 12'b100000000010; // D.02.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101010011: o_lut = 12'b000100010011; // D.19.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101010100: o_lut = 12'b100000000100; // D.04.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101010101: o_lut = 12'b000100010101; // D.21.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101010110: o_lut = 12'b000100010110; // D.22.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101010111: o_lut = 12'b100000010111; // D.23.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101011000: o_lut = 12'b100000001000; // D.08.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101011001: o_lut = 12'b000100011001; // D.25.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101011010: o_lut = 12'b000100011010; // D.26.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101011011: o_lut = 12'b100000011011; // D.27.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101011100: o_lut = 12'b000100011100; // D.28.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101011101: o_lut = 12'b100000011101; // D.29.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101011110: o_lut = 12'b100000011110; // D.30.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101011111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101100000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101100001: o_lut = 12'b100000011110; // D.30.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101100010: o_lut = 12'b100000011101; // D.29.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101100011: o_lut = 12'b000100000011; // D.03.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101100100: o_lut = 12'b100000011011; // D.27.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101100101: o_lut = 12'b000100000101; // D.05.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101100110: o_lut = 12'b000100000110; // D.06.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101100111: o_lut = 12'b100000001000; // D.08.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101101000: o_lut = 12'b100000010111; // D.23.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101101001: o_lut = 12'b000100001001; // D.09.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101101010: o_lut = 12'b000100001010; // D.10.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101101011: o_lut = 12'b100000000100; // D.04.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101101100: o_lut = 12'b000100001100; // D.12.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101101101: o_lut = 12'b100000000010; // D.02.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101101110: o_lut = 12'b100000000001; // D.01.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101101111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101110000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101110001: o_lut = 12'b000100010001; // D.17.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101110010: o_lut = 12'b000100010010; // D.18.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101110011: o_lut = 12'b100000011000; // D.24.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101110100: o_lut = 12'b000100010100; // D.20.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101110101: o_lut = 12'b100000011111; // D.31.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101110110: o_lut = 12'b100000010000; // D.16.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101110111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101111000: o_lut = 12'b000100000111; // D.07.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01101111001: o_lut = 12'b100000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101111010: o_lut = 12'b100000001111; // D.15.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101111011: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101111100: o_lut = 12'b100000011100; // D.28.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01101111101: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101111110: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01101111111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110000000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110000001: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110000010: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110000011: o_lut = 12'b101011111100; // K.28.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b01110000100: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110000101: o_lut = 12'b100011101111; // D.15.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110000110: o_lut = 12'b100011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110000111: o_lut = 12'b000111100111; // D.07.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110001000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110001001: o_lut = 12'b100011110000; // D.16.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110001010: o_lut = 12'b100011111111; // D.31.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110001011: o_lut = 12'b000111101011; // D.11.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110001100: o_lut = 12'b100011111000; // D.24.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110001101: o_lut = 12'b000111101101; // D.13.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110001110: o_lut = 12'b000111101110; // D.14.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110001111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110010000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110010001: o_lut = 12'b100011100001; // D.01.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110010010: o_lut = 12'b100011100010; // D.02.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110010011: o_lut = 12'b000111110011; // D.19.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110010100: o_lut = 12'b100011100100; // D.04.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110010101: o_lut = 12'b000111110101; // D.21.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110010110: o_lut = 12'b000111110110; // D.22.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110010111: o_lut = 12'b100011110111; // D.23.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110011000: o_lut = 12'b100011101000; // D.08.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110011001: o_lut = 12'b000111111001; // D.25.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110011010: o_lut = 12'b000111111010; // D.26.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110011011: o_lut = 12'b100011111011; // D.27.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110011100: o_lut = 12'b000111111100; // D.28.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110011101: o_lut = 12'b100011111101; // D.29.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110011110: o_lut = 12'b100011111110; // D.30.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110011111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110100000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110100001: o_lut = 12'b101011111110; // K.30.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b01110100010: o_lut = 12'b101011111101; // K.29.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b01110100011: o_lut = 12'b000111100011; // D.03.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110100100: o_lut = 12'b101011111011; // K.27.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b01110100101: o_lut = 12'b000111100101; // D.05.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110100110: o_lut = 12'b000111100110; // D.06.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110100111: o_lut = 12'b100011101000; // D.08.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110101000: o_lut = 12'b101011110111; // K.23.7, i_disp = 0, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b01110101001: o_lut = 12'b000111101001; // D.09.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110101010: o_lut = 12'b000111101010; // D.10.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110101011: o_lut = 12'b100011100100; // D.04.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110101100: o_lut = 12'b000111101100; // D.12.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110101101: o_lut = 12'b100011100010; // D.02.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110101110: o_lut = 12'b100011100001; // D.01.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110101111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110110000: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110110001: o_lut = 12'b000111110001; // D.17.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110110010: o_lut = 12'b000111110010; // D.18.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110110011: o_lut = 12'b100011111000; // D.24.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110110100: o_lut = 12'b000111110100; // D.20.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110110101: o_lut = 12'b100011111111; // D.31.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110110110: o_lut = 12'b100011110000; // D.16.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110110111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110111000: o_lut = 12'b000111100111; // D.07.7, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b01110111001: o_lut = 12'b100011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110111010: o_lut = 12'b100011101111; // D.15.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110111011: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110111100: o_lut = 12'b100011111100; // D.28.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b01110111101: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110111110: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01110111111: o_lut = 12'b110011100000; // D.00.7, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111000000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111000001: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111000010: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111000011: o_lut = 12'b110000011100; // D.28.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111000100: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111000101: o_lut = 12'b110000001111; // D.15.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111000110: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111000111: o_lut = 12'b110100000111; // D.07.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111001000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111001001: o_lut = 12'b110000010000; // D.16.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111001010: o_lut = 12'b110000011111; // D.31.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111001011: o_lut = 12'b110100001011; // D.11.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111001100: o_lut = 12'b110000011000; // D.24.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111001101: o_lut = 12'b110100001101; // D.13.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111001110: o_lut = 12'b110100001110; // D.14.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111001111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111010000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111010001: o_lut = 12'b110000000001; // D.01.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111010010: o_lut = 12'b110000000010; // D.02.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111010011: o_lut = 12'b110100010011; // D.19.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111010100: o_lut = 12'b110000000100; // D.04.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111010101: o_lut = 12'b110100010101; // D.21.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111010110: o_lut = 12'b110100010110; // D.22.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111010111: o_lut = 12'b110000010111; // D.23.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111011000: o_lut = 12'b110000001000; // D.08.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111011001: o_lut = 12'b110100011001; // D.25.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111011010: o_lut = 12'b110100011010; // D.26.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111011011: o_lut = 12'b110000011011; // D.27.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111011100: o_lut = 12'b110100011100; // D.28.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111011101: o_lut = 12'b110000011101; // D.29.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111011110: o_lut = 12'b110000011110; // D.30.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111011111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111100000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111100001: o_lut = 12'b110000011110; // D.30.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111100010: o_lut = 12'b110000011101; // D.29.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111100011: o_lut = 12'b110100000011; // D.03.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111100100: o_lut = 12'b110000011011; // D.27.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111100101: o_lut = 12'b110100000101; // D.05.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111100110: o_lut = 12'b110100000110; // D.06.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111100111: o_lut = 12'b110000001000; // D.08.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111101000: o_lut = 12'b110000010111; // D.23.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111101001: o_lut = 12'b110100001001; // D.09.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111101010: o_lut = 12'b110100001010; // D.10.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111101011: o_lut = 12'b110000000100; // D.04.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111101100: o_lut = 12'b110100001100; // D.12.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111101101: o_lut = 12'b110000000010; // D.02.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111101110: o_lut = 12'b110000000001; // D.01.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111101111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111110000: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111110001: o_lut = 12'b110100010001; // D.17.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111110010: o_lut = 12'b110100010010; // D.18.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111110011: o_lut = 12'b110000011000; // D.24.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111110100: o_lut = 12'b110100010100; // D.20.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111110101: o_lut = 12'b110000011111; // D.31.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111110110: o_lut = 12'b110000010000; // D.16.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111110111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111111000: o_lut = 12'b110100000111; // D.07.0, i_disp = 0, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111111001: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111111010: o_lut = 12'b110000001111; // D.15.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111111011: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111111100: o_lut = 12'b110000011100; // D.28.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111111101: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111111110: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b01111111111: o_lut = 12'b110000000000; // D.00.0, i_disp = 0, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000000000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000000001: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000000010: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000000011: o_lut = 12'b110100011100; // D.28.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000000100: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000000101: o_lut = 12'b110100001111; // D.15.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000000110: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000000111: o_lut = 12'b110000000111; // D.07.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000001000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000001001: o_lut = 12'b110100010000; // D.16.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000001010: o_lut = 12'b110100011111; // D.31.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000001011: o_lut = 12'b110000001011; // D.11.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000001100: o_lut = 12'b110100011000; // D.24.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000001101: o_lut = 12'b110000001101; // D.13.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000001110: o_lut = 12'b110000001110; // D.14.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000001111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000010000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000010001: o_lut = 12'b110100000001; // D.01.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000010010: o_lut = 12'b110100000010; // D.02.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000010011: o_lut = 12'b110000010011; // D.19.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000010100: o_lut = 12'b110100000100; // D.04.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000010101: o_lut = 12'b110000010101; // D.21.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000010110: o_lut = 12'b110000010110; // D.22.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000010111: o_lut = 12'b110100010111; // D.23.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000011000: o_lut = 12'b110100001000; // D.08.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000011001: o_lut = 12'b110000011001; // D.25.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000011010: o_lut = 12'b110000011010; // D.26.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000011011: o_lut = 12'b110100011011; // D.27.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000011100: o_lut = 12'b110000011100; // D.28.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000011101: o_lut = 12'b110100011101; // D.29.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000011110: o_lut = 12'b110100011110; // D.30.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000011111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000100000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000100001: o_lut = 12'b110100011110; // D.30.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000100010: o_lut = 12'b110100011101; // D.29.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000100011: o_lut = 12'b110000000011; // D.03.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000100100: o_lut = 12'b110100011011; // D.27.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000100101: o_lut = 12'b110000000101; // D.05.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000100110: o_lut = 12'b110000000110; // D.06.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000100111: o_lut = 12'b110100001000; // D.08.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000101000: o_lut = 12'b110100010111; // D.23.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000101001: o_lut = 12'b110000001001; // D.09.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000101010: o_lut = 12'b110000001010; // D.10.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000101011: o_lut = 12'b110100000100; // D.04.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000101100: o_lut = 12'b110000001100; // D.12.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000101101: o_lut = 12'b110100000010; // D.02.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000101110: o_lut = 12'b110100000001; // D.01.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000101111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000110000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000110001: o_lut = 12'b110000010001; // D.17.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000110010: o_lut = 12'b110000010010; // D.18.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000110011: o_lut = 12'b110100011000; // D.24.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000110100: o_lut = 12'b110000010100; // D.20.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000110101: o_lut = 12'b110100011111; // D.31.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000110110: o_lut = 12'b110100010000; // D.16.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000110111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000111000: o_lut = 12'b110000000111; // D.07.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000111001: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000111010: o_lut = 12'b110100001111; // D.15.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000111011: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000111100: o_lut = 12'b110100011100; // D.28.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000111101: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000111110: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10000111111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001000000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001000001: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001000010: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001000011: o_lut = 12'b100111111100; // D.28.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001000100: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001000101: o_lut = 12'b100111101111; // D.15.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001000110: o_lut = 12'b100111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001000111: o_lut = 12'b000011100111; // D.07.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001001000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001001001: o_lut = 12'b100111110000; // D.16.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001001010: o_lut = 12'b100111111111; // D.31.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001001011: o_lut = 12'b000011101011; // D.11.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001001100: o_lut = 12'b100111111000; // D.24.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001001101: o_lut = 12'b000011101101; // D.13.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001001110: o_lut = 12'b000011101110; // D.14.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001001111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001010000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001010001: o_lut = 12'b100111100001; // D.01.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001010010: o_lut = 12'b100111100010; // D.02.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001010011: o_lut = 12'b000011110011; // D.19.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001010100: o_lut = 12'b100111100100; // D.04.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001010101: o_lut = 12'b000011110101; // D.21.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001010110: o_lut = 12'b000011110110; // D.22.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001010111: o_lut = 12'b101111110111; // K.23.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b10001011000: o_lut = 12'b100111101000; // D.08.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001011001: o_lut = 12'b000011111001; // D.25.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001011010: o_lut = 12'b000011111010; // D.26.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001011011: o_lut = 12'b101111111011; // K.27.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b10001011100: o_lut = 12'b000011111100; // D.28.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001011101: o_lut = 12'b101111111101; // K.29.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b10001011110: o_lut = 12'b101111111110; // K.30.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b10001011111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001100000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001100001: o_lut = 12'b100111111110; // D.30.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001100010: o_lut = 12'b100111111101; // D.29.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001100011: o_lut = 12'b000011100011; // D.03.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001100100: o_lut = 12'b100111111011; // D.27.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001100101: o_lut = 12'b000011100101; // D.05.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001100110: o_lut = 12'b000011100110; // D.06.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001100111: o_lut = 12'b100111101000; // D.08.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001101000: o_lut = 12'b100111110111; // D.23.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001101001: o_lut = 12'b000011101001; // D.09.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001101010: o_lut = 12'b000011101010; // D.10.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001101011: o_lut = 12'b100111100100; // D.04.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001101100: o_lut = 12'b000011101100; // D.12.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001101101: o_lut = 12'b100111100010; // D.02.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001101110: o_lut = 12'b100111100001; // D.01.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001101111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001110000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001110001: o_lut = 12'b000011110001; // D.17.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001110010: o_lut = 12'b000011110010; // D.18.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001110011: o_lut = 12'b100111111000; // D.24.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001110100: o_lut = 12'b000011110100; // D.20.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001110101: o_lut = 12'b100111111111; // D.31.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001110110: o_lut = 12'b100111110000; // D.16.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001110111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001111000: o_lut = 12'b000011100111; // D.07.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10001111001: o_lut = 12'b100111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001111010: o_lut = 12'b100111101111; // D.15.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10001111011: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001111100: o_lut = 12'b101111111100; // K.28.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b10001111101: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001111110: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10001111111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010000000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010000001: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010000010: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010000011: o_lut = 12'b100100011100; // D.28.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010000100: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010000101: o_lut = 12'b100100001111; // D.15.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010000110: o_lut = 12'b100100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010000111: o_lut = 12'b000000000111; // D.07.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010001000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010001001: o_lut = 12'b100100010000; // D.16.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010001010: o_lut = 12'b100100011111; // D.31.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010001011: o_lut = 12'b000000001011; // D.11.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010001100: o_lut = 12'b100100011000; // D.24.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010001101: o_lut = 12'b000000001101; // D.13.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010001110: o_lut = 12'b000000001110; // D.14.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010001111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010010000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010010001: o_lut = 12'b100100000001; // D.01.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010010010: o_lut = 12'b100100000010; // D.02.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010010011: o_lut = 12'b000000010011; // D.19.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010010100: o_lut = 12'b100100000100; // D.04.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010010101: o_lut = 12'b000000010101; // D.21.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010010110: o_lut = 12'b000000010110; // D.22.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010010111: o_lut = 12'b100100010111; // D.23.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010011000: o_lut = 12'b100100001000; // D.08.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010011001: o_lut = 12'b000000011001; // D.25.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010011010: o_lut = 12'b000000011010; // D.26.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010011011: o_lut = 12'b100100011011; // D.27.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010011100: o_lut = 12'b000000011100; // D.28.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010011101: o_lut = 12'b100100011101; // D.29.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010011110: o_lut = 12'b100100011110; // D.30.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010011111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010100000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010100001: o_lut = 12'b100100011110; // D.30.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010100010: o_lut = 12'b100100011101; // D.29.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010100011: o_lut = 12'b000000000011; // D.03.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010100100: o_lut = 12'b100100011011; // D.27.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010100101: o_lut = 12'b000000000101; // D.05.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010100110: o_lut = 12'b000000000110; // D.06.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010100111: o_lut = 12'b100100001000; // D.08.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010101000: o_lut = 12'b100100010111; // D.23.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010101001: o_lut = 12'b000000001001; // D.09.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010101010: o_lut = 12'b000000001010; // D.10.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010101011: o_lut = 12'b100100000100; // D.04.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010101100: o_lut = 12'b000000001100; // D.12.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010101101: o_lut = 12'b100100000010; // D.02.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010101110: o_lut = 12'b100100000001; // D.01.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010101111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010110000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010110001: o_lut = 12'b000000010001; // D.17.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010110010: o_lut = 12'b000000010010; // D.18.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010110011: o_lut = 12'b100100011000; // D.24.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010110100: o_lut = 12'b000000010100; // D.20.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010110101: o_lut = 12'b100100011111; // D.31.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010110110: o_lut = 12'b100100010000; // D.16.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010110111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010111000: o_lut = 12'b000000000111; // D.07.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10010111001: o_lut = 12'b100100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010111010: o_lut = 12'b100100001111; // D.15.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10010111011: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010111100: o_lut = 12'b101100011100; // K.28.0, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b10010111101: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010111110: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10010111111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011000000: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011000001: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011000010: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011000011: o_lut = 12'b001001111100; // K.28.3, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b10011000100: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011000101: o_lut = 12'b000001101111; // D.15.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011000110: o_lut = 12'b000001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011000111: o_lut = 12'b000101100111; // D.07.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011001000: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011001001: o_lut = 12'b000001110000; // D.16.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011001010: o_lut = 12'b000001111111; // D.31.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011001011: o_lut = 12'b000101101011; // D.11.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011001100: o_lut = 12'b000001111000; // D.24.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011001101: o_lut = 12'b000101101101; // D.13.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011001110: o_lut = 12'b000101101110; // D.14.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011001111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011010000: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011010001: o_lut = 12'b000001100001; // D.01.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011010010: o_lut = 12'b000001100010; // D.02.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011010011: o_lut = 12'b000101110011; // D.19.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011010100: o_lut = 12'b000001100100; // D.04.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011010101: o_lut = 12'b000101110101; // D.21.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011010110: o_lut = 12'b000101110110; // D.22.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011010111: o_lut = 12'b100001110111; // D.23.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011011000: o_lut = 12'b000001101000; // D.08.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011011001: o_lut = 12'b000101111001; // D.25.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011011010: o_lut = 12'b000101111010; // D.26.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011011011: o_lut = 12'b100001111011; // D.27.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011011100: o_lut = 12'b000101111100; // D.28.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011011101: o_lut = 12'b100001111101; // D.29.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011011110: o_lut = 12'b100001111110; // D.30.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011011111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011100000: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011100001: o_lut = 12'b000001111110; // D.30.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011100010: o_lut = 12'b000001111101; // D.29.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011100011: o_lut = 12'b000101100011; // D.03.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011100100: o_lut = 12'b000001111011; // D.27.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011100101: o_lut = 12'b000101100101; // D.05.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011100110: o_lut = 12'b000101100110; // D.06.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011100111: o_lut = 12'b100001101000; // D.08.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011101000: o_lut = 12'b000001110111; // D.23.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011101001: o_lut = 12'b000101101001; // D.09.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011101010: o_lut = 12'b000101101010; // D.10.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011101011: o_lut = 12'b100001100100; // D.04.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011101100: o_lut = 12'b000101101100; // D.12.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011101101: o_lut = 12'b100001100010; // D.02.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011101110: o_lut = 12'b100001100001; // D.01.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011101111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011110000: o_lut = 12'b010001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b10011110001: o_lut = 12'b000101110001; // D.17.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011110010: o_lut = 12'b000101110010; // D.18.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011110011: o_lut = 12'b100001111000; // D.24.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011110100: o_lut = 12'b000101110100; // D.20.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011110101: o_lut = 12'b100001111111; // D.31.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011110110: o_lut = 12'b100001110000; // D.16.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011110111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011111000: o_lut = 12'b000101100111; // D.07.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10011111001: o_lut = 12'b100001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011111010: o_lut = 12'b100001101111; // D.15.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011111011: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011111100: o_lut = 12'b100001111100; // D.28.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10011111101: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011111110: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10011111111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100000000: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100000001: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100000010: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100000011: o_lut = 12'b100110011100; // D.28.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100000100: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100000101: o_lut = 12'b100110001111; // D.15.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100000110: o_lut = 12'b100110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100000111: o_lut = 12'b000010000111; // D.07.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100001000: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100001001: o_lut = 12'b100110010000; // D.16.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100001010: o_lut = 12'b100110011111; // D.31.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100001011: o_lut = 12'b000010001011; // D.11.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100001100: o_lut = 12'b100110011000; // D.24.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100001101: o_lut = 12'b000010001101; // D.13.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100001110: o_lut = 12'b000010001110; // D.14.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100001111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100010000: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100010001: o_lut = 12'b100110000001; // D.01.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100010010: o_lut = 12'b100110000010; // D.02.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100010011: o_lut = 12'b000010010011; // D.19.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100010100: o_lut = 12'b100110000100; // D.04.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100010101: o_lut = 12'b000010010101; // D.21.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100010110: o_lut = 12'b000010010110; // D.22.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100010111: o_lut = 12'b100110010111; // D.23.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100011000: o_lut = 12'b100110001000; // D.08.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100011001: o_lut = 12'b000010011001; // D.25.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100011010: o_lut = 12'b000010011010; // D.26.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100011011: o_lut = 12'b100110011011; // D.27.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100011100: o_lut = 12'b000010011100; // D.28.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100011101: o_lut = 12'b100110011101; // D.29.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100011110: o_lut = 12'b100110011110; // D.30.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100011111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100100000: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100100001: o_lut = 12'b100110011110; // D.30.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100100010: o_lut = 12'b100110011101; // D.29.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100100011: o_lut = 12'b000010000011; // D.03.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100100100: o_lut = 12'b100110011011; // D.27.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100100101: o_lut = 12'b000010000101; // D.05.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100100110: o_lut = 12'b000010000110; // D.06.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100100111: o_lut = 12'b100110001000; // D.08.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100101000: o_lut = 12'b100110010111; // D.23.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100101001: o_lut = 12'b000010001001; // D.09.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100101010: o_lut = 12'b000010001010; // D.10.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100101011: o_lut = 12'b100110000100; // D.04.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100101100: o_lut = 12'b000010001100; // D.12.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100101101: o_lut = 12'b100110000010; // D.02.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100101110: o_lut = 12'b100110000001; // D.01.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100101111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100110000: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100110001: o_lut = 12'b000010010001; // D.17.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100110010: o_lut = 12'b000010010010; // D.18.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100110011: o_lut = 12'b100110011000; // D.24.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100110100: o_lut = 12'b000010010100; // D.20.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100110101: o_lut = 12'b100110011111; // D.31.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100110110: o_lut = 12'b100110010000; // D.16.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100110111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100111000: o_lut = 12'b000010000111; // D.07.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10100111001: o_lut = 12'b100110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100111010: o_lut = 12'b100110001111; // D.15.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10100111011: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100111100: o_lut = 12'b101110011100; // K.28.4, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b10100111101: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100111110: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10100111111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101000000: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101000001: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101000010: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101000011: o_lut = 12'b001001011100; // K.28.2, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b10101000100: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101000101: o_lut = 12'b000010101111; // D.15.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101000110: o_lut = 12'b000010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101000111: o_lut = 12'b000110100111; // D.07.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101001000: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101001001: o_lut = 12'b000010110000; // D.16.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101001010: o_lut = 12'b000010111111; // D.31.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101001011: o_lut = 12'b000110101011; // D.11.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101001100: o_lut = 12'b000010111000; // D.24.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101001101: o_lut = 12'b000110101101; // D.13.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101001110: o_lut = 12'b000110101110; // D.14.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101001111: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101010000: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101010001: o_lut = 12'b000010100001; // D.01.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101010010: o_lut = 12'b000010100010; // D.02.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101010011: o_lut = 12'b000110110011; // D.19.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101010100: o_lut = 12'b000010100100; // D.04.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101010101: o_lut = 12'b000110110101; // D.21.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101010110: o_lut = 12'b000110110110; // D.22.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101010111: o_lut = 12'b100010110111; // D.23.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101011000: o_lut = 12'b000010101000; // D.08.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101011001: o_lut = 12'b000110111001; // D.25.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101011010: o_lut = 12'b000110111010; // D.26.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101011011: o_lut = 12'b100010111011; // D.27.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101011100: o_lut = 12'b000110111100; // D.28.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101011101: o_lut = 12'b100010111101; // D.29.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101011110: o_lut = 12'b100010111110; // D.30.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101011111: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101100000: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101100001: o_lut = 12'b000010111110; // D.30.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101100010: o_lut = 12'b000010111101; // D.29.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101100011: o_lut = 12'b000110100011; // D.03.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101100100: o_lut = 12'b000010111011; // D.27.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101100101: o_lut = 12'b000110100101; // D.05.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101100110: o_lut = 12'b000110100110; // D.06.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101100111: o_lut = 12'b100010101000; // D.08.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101101000: o_lut = 12'b000010110111; // D.23.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101101001: o_lut = 12'b000110101001; // D.09.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101101010: o_lut = 12'b000110101010; // D.10.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101101011: o_lut = 12'b100010100100; // D.04.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101101100: o_lut = 12'b000110101100; // D.12.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101101101: o_lut = 12'b100010100010; // D.02.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101101110: o_lut = 12'b100010100001; // D.01.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101101111: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101110000: o_lut = 12'b010010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b10101110001: o_lut = 12'b000110110001; // D.17.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101110010: o_lut = 12'b000110110010; // D.18.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101110011: o_lut = 12'b100010111000; // D.24.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101110100: o_lut = 12'b000110110100; // D.20.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101110101: o_lut = 12'b100010111111; // D.31.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101110110: o_lut = 12'b100010110000; // D.16.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101110111: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101111000: o_lut = 12'b000110100111; // D.07.5, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10101111001: o_lut = 12'b100010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101111010: o_lut = 12'b100010101111; // D.15.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10101111011: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101111100: o_lut = 12'b101001011100; // K.28.2, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b10101111101: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101111110: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10101111111: o_lut = 12'b110010100000; // D.00.5, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110000000: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110000001: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110000010: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110000011: o_lut = 12'b001000111100; // K.28.1, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b10110000100: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110000101: o_lut = 12'b000011001111; // D.15.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110000110: o_lut = 12'b000011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110000111: o_lut = 12'b000111000111; // D.07.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110001000: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110001001: o_lut = 12'b000011010000; // D.16.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110001010: o_lut = 12'b000011011111; // D.31.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110001011: o_lut = 12'b000111001011; // D.11.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110001100: o_lut = 12'b000011011000; // D.24.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110001101: o_lut = 12'b000111001101; // D.13.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110001110: o_lut = 12'b000111001110; // D.14.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110001111: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110010000: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110010001: o_lut = 12'b000011000001; // D.01.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110010010: o_lut = 12'b000011000010; // D.02.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110010011: o_lut = 12'b000111010011; // D.19.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110010100: o_lut = 12'b000011000100; // D.04.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110010101: o_lut = 12'b000111010101; // D.21.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110010110: o_lut = 12'b000111010110; // D.22.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110010111: o_lut = 12'b100011010111; // D.23.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110011000: o_lut = 12'b000011001000; // D.08.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110011001: o_lut = 12'b000111011001; // D.25.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110011010: o_lut = 12'b000111011010; // D.26.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110011011: o_lut = 12'b100011011011; // D.27.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110011100: o_lut = 12'b000111011100; // D.28.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110011101: o_lut = 12'b100011011101; // D.29.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110011110: o_lut = 12'b100011011110; // D.30.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110011111: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110100000: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110100001: o_lut = 12'b000011011110; // D.30.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110100010: o_lut = 12'b000011011101; // D.29.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110100011: o_lut = 12'b000111000011; // D.03.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110100100: o_lut = 12'b000011011011; // D.27.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110100101: o_lut = 12'b000111000101; // D.05.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110100110: o_lut = 12'b000111000110; // D.06.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110100111: o_lut = 12'b100011001000; // D.08.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110101000: o_lut = 12'b000011010111; // D.23.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110101001: o_lut = 12'b000111001001; // D.09.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110101010: o_lut = 12'b000111001010; // D.10.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110101011: o_lut = 12'b100011000100; // D.04.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110101100: o_lut = 12'b000111001100; // D.12.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110101101: o_lut = 12'b100011000010; // D.02.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110101110: o_lut = 12'b100011000001; // D.01.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110101111: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110110000: o_lut = 12'b010011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b10110110001: o_lut = 12'b000111010001; // D.17.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110110010: o_lut = 12'b000111010010; // D.18.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110110011: o_lut = 12'b100011011000; // D.24.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110110100: o_lut = 12'b000111010100; // D.20.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110110101: o_lut = 12'b100011011111; // D.31.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110110110: o_lut = 12'b100011010000; // D.16.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110110111: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110111000: o_lut = 12'b000111000111; // D.07.6, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10110111001: o_lut = 12'b100011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110111010: o_lut = 12'b100011001111; // D.15.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10110111011: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110111100: o_lut = 12'b101000111100; // K.28.1, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b10110111101: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110111110: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10110111111: o_lut = 12'b110011000000; // D.00.6, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111000000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111000001: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111000010: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111000011: o_lut = 12'b000111111100; // D.28.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111000100: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111000101: o_lut = 12'b000111101111; // D.15.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111000110: o_lut = 12'b000111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111000111: o_lut = 12'b100011100111; // D.07.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111001000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111001001: o_lut = 12'b000111110000; // D.16.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111001010: o_lut = 12'b000111111111; // D.31.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111001011: o_lut = 12'b100011101011; // D.11.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111001100: o_lut = 12'b000111111000; // D.24.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111001101: o_lut = 12'b100011101101; // D.13.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111001110: o_lut = 12'b100011101110; // D.14.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111001111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111010000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111010001: o_lut = 12'b000111100001; // D.01.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111010010: o_lut = 12'b000111100010; // D.02.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111010011: o_lut = 12'b100011110011; // D.19.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111010100: o_lut = 12'b000111100100; // D.04.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111010101: o_lut = 12'b100011110101; // D.21.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111010110: o_lut = 12'b100011110110; // D.22.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111010111: o_lut = 12'b100111110111; // D.23.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111011000: o_lut = 12'b000111101000; // D.08.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111011001: o_lut = 12'b100011111001; // D.25.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111011010: o_lut = 12'b100011111010; // D.26.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111011011: o_lut = 12'b100111111011; // D.27.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111011100: o_lut = 12'b100011111100; // D.28.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111011101: o_lut = 12'b100111111101; // D.29.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111011110: o_lut = 12'b100111111110; // D.30.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111011111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111100000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111100001: o_lut = 12'b000111111110; // D.30.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111100010: o_lut = 12'b000111111101; // D.29.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111100011: o_lut = 12'b100011100011; // D.03.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111100100: o_lut = 12'b000111111011; // D.27.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111100101: o_lut = 12'b100011100101; // D.05.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111100110: o_lut = 12'b100011100110; // D.06.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111100111: o_lut = 12'b100111101000; // D.08.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111101000: o_lut = 12'b000111110111; // D.23.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b10111101001: o_lut = 12'b100011101001; // D.09.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111101010: o_lut = 12'b100011101010; // D.10.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111101011: o_lut = 12'b100111100100; // D.04.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111101100: o_lut = 12'b100011101100; // D.12.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111101101: o_lut = 12'b100111100010; // D.02.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111101110: o_lut = 12'b100111100001; // D.01.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111101111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111110000: o_lut = 12'b010111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b10111110001: o_lut = 12'b100011110001; // D.17.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111110010: o_lut = 12'b100011110010; // D.18.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111110011: o_lut = 12'b100111111000; // D.24.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111110100: o_lut = 12'b100011110100; // D.20.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111110101: o_lut = 12'b100111111111; // D.31.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111110110: o_lut = 12'b100111110000; // D.16.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111110111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111111000: o_lut = 12'b100011100111; // D.07.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111111001: o_lut = 12'b100111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111111010: o_lut = 12'b100111101111; // D.15.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111111011: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111111100: o_lut = 12'b100111111100; // D.28.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b10111111101: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111111110: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b10111111111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000000000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000000001: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000000010: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000000011: o_lut = 12'b100111111100; // D.28.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000000100: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000000101: o_lut = 12'b100111101111; // D.15.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000000110: o_lut = 12'b100111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000000111: o_lut = 12'b000011100111; // D.07.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000001000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000001001: o_lut = 12'b100111110000; // D.16.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000001010: o_lut = 12'b100111111111; // D.31.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000001011: o_lut = 12'b000011101011; // D.11.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000001100: o_lut = 12'b100111111000; // D.24.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000001101: o_lut = 12'b000011101101; // D.13.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000001110: o_lut = 12'b000011101110; // D.14.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000001111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000010000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000010001: o_lut = 12'b100111100001; // D.01.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000010010: o_lut = 12'b100111100010; // D.02.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000010011: o_lut = 12'b000011110011; // D.19.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000010100: o_lut = 12'b100111100100; // D.04.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000010101: o_lut = 12'b000011110101; // D.21.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000010110: o_lut = 12'b000011110110; // D.22.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000010111: o_lut = 12'b100111110111; // D.23.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000011000: o_lut = 12'b100111101000; // D.08.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000011001: o_lut = 12'b000011111001; // D.25.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000011010: o_lut = 12'b000011111010; // D.26.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000011011: o_lut = 12'b100111111011; // D.27.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000011100: o_lut = 12'b000011111100; // D.28.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000011101: o_lut = 12'b100111111101; // D.29.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000011110: o_lut = 12'b100111111110; // D.30.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000011111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000100000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000100001: o_lut = 12'b100111111110; // D.30.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000100010: o_lut = 12'b100111111101; // D.29.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000100011: o_lut = 12'b000011100011; // D.03.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000100100: o_lut = 12'b100111111011; // D.27.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000100101: o_lut = 12'b000011100101; // D.05.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000100110: o_lut = 12'b000011100110; // D.06.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000100111: o_lut = 12'b100111101000; // D.08.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000101000: o_lut = 12'b100111110111; // D.23.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000101001: o_lut = 12'b000011101001; // D.09.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000101010: o_lut = 12'b000011101010; // D.10.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000101011: o_lut = 12'b100111100100; // D.04.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000101100: o_lut = 12'b000011101100; // D.12.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000101101: o_lut = 12'b100111100010; // D.02.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000101110: o_lut = 12'b100111100001; // D.01.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000101111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000110000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000110001: o_lut = 12'b000011110001; // D.17.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000110010: o_lut = 12'b000011110010; // D.18.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000110011: o_lut = 12'b100111111000; // D.24.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000110100: o_lut = 12'b000011110100; // D.20.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000110101: o_lut = 12'b100111111111; // D.31.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000110110: o_lut = 12'b100111110000; // D.16.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000110111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000111000: o_lut = 12'b000011100111; // D.07.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11000111001: o_lut = 12'b100111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000111010: o_lut = 12'b100111101111; // D.15.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000111011: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000111100: o_lut = 12'b100111111100; // D.28.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11000111101: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000111110: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11000111111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001000000: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001000001: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001000010: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001000011: o_lut = 12'b001011011100; // K.28.6, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b11001000100: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001000101: o_lut = 12'b000000101111; // D.15.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001000110: o_lut = 12'b000000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001000111: o_lut = 12'b000100100111; // D.07.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001001000: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001001001: o_lut = 12'b000000110000; // D.16.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001001010: o_lut = 12'b000000111111; // D.31.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001001011: o_lut = 12'b000100101011; // D.11.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001001100: o_lut = 12'b000000111000; // D.24.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001001101: o_lut = 12'b000100101101; // D.13.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001001110: o_lut = 12'b000100101110; // D.14.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001001111: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001010000: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001010001: o_lut = 12'b000000100001; // D.01.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001010010: o_lut = 12'b000000100010; // D.02.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001010011: o_lut = 12'b000100110011; // D.19.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001010100: o_lut = 12'b000000100100; // D.04.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001010101: o_lut = 12'b000100110101; // D.21.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001010110: o_lut = 12'b000100110110; // D.22.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001010111: o_lut = 12'b100000110111; // D.23.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001011000: o_lut = 12'b000000101000; // D.08.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001011001: o_lut = 12'b000100111001; // D.25.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001011010: o_lut = 12'b000100111010; // D.26.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001011011: o_lut = 12'b100000111011; // D.27.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001011100: o_lut = 12'b000100111100; // D.28.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001011101: o_lut = 12'b100000111101; // D.29.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001011110: o_lut = 12'b100000111110; // D.30.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001011111: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001100000: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001100001: o_lut = 12'b000000111110; // D.30.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001100010: o_lut = 12'b000000111101; // D.29.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001100011: o_lut = 12'b000100100011; // D.03.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001100100: o_lut = 12'b000000111011; // D.27.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001100101: o_lut = 12'b000100100101; // D.05.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001100110: o_lut = 12'b000100100110; // D.06.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001100111: o_lut = 12'b100000101000; // D.08.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001101000: o_lut = 12'b000000110111; // D.23.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001101001: o_lut = 12'b000100101001; // D.09.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001101010: o_lut = 12'b000100101010; // D.10.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001101011: o_lut = 12'b100000100100; // D.04.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001101100: o_lut = 12'b000100101100; // D.12.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001101101: o_lut = 12'b100000100010; // D.02.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001101110: o_lut = 12'b100000100001; // D.01.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001101111: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001110000: o_lut = 12'b010000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b11001110001: o_lut = 12'b000100110001; // D.17.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001110010: o_lut = 12'b000100110010; // D.18.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001110011: o_lut = 12'b100000111000; // D.24.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001110100: o_lut = 12'b000100110100; // D.20.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001110101: o_lut = 12'b100000111111; // D.31.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001110110: o_lut = 12'b100000110000; // D.16.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001110111: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001111000: o_lut = 12'b000100100111; // D.07.1, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11001111001: o_lut = 12'b100000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001111010: o_lut = 12'b100000101111; // D.15.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11001111011: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001111100: o_lut = 12'b101011011100; // K.28.6, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b11001111101: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001111110: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11001111111: o_lut = 12'b110000100000; // D.00.1, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010000000: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010000001: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010000010: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010000011: o_lut = 12'b001010111100; // K.28.5, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b11010000100: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010000101: o_lut = 12'b000001001111; // D.15.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010000110: o_lut = 12'b000001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010000111: o_lut = 12'b000101000111; // D.07.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010001000: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010001001: o_lut = 12'b000001010000; // D.16.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010001010: o_lut = 12'b000001011111; // D.31.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010001011: o_lut = 12'b000101001011; // D.11.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010001100: o_lut = 12'b000001011000; // D.24.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010001101: o_lut = 12'b000101001101; // D.13.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010001110: o_lut = 12'b000101001110; // D.14.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010001111: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010010000: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010010001: o_lut = 12'b000001000001; // D.01.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010010010: o_lut = 12'b000001000010; // D.02.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010010011: o_lut = 12'b000101010011; // D.19.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010010100: o_lut = 12'b000001000100; // D.04.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010010101: o_lut = 12'b000101010101; // D.21.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010010110: o_lut = 12'b000101010110; // D.22.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010010111: o_lut = 12'b100001010111; // D.23.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010011000: o_lut = 12'b000001001000; // D.08.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010011001: o_lut = 12'b000101011001; // D.25.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010011010: o_lut = 12'b000101011010; // D.26.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010011011: o_lut = 12'b100001011011; // D.27.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010011100: o_lut = 12'b000101011100; // D.28.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010011101: o_lut = 12'b100001011101; // D.29.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010011110: o_lut = 12'b100001011110; // D.30.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010011111: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010100000: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010100001: o_lut = 12'b000001011110; // D.30.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010100010: o_lut = 12'b000001011101; // D.29.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010100011: o_lut = 12'b000101000011; // D.03.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010100100: o_lut = 12'b000001011011; // D.27.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010100101: o_lut = 12'b000101000101; // D.05.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010100110: o_lut = 12'b000101000110; // D.06.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010100111: o_lut = 12'b100001001000; // D.08.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010101000: o_lut = 12'b000001010111; // D.23.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010101001: o_lut = 12'b000101001001; // D.09.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010101010: o_lut = 12'b000101001010; // D.10.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010101011: o_lut = 12'b100001000100; // D.04.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010101100: o_lut = 12'b000101001100; // D.12.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010101101: o_lut = 12'b100001000010; // D.02.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010101110: o_lut = 12'b100001000001; // D.01.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010101111: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010110000: o_lut = 12'b010001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b11010110001: o_lut = 12'b000101010001; // D.17.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010110010: o_lut = 12'b000101010010; // D.18.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010110011: o_lut = 12'b100001011000; // D.24.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010110100: o_lut = 12'b000101010100; // D.20.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010110101: o_lut = 12'b100001011111; // D.31.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010110110: o_lut = 12'b100001010000; // D.16.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010110111: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010111000: o_lut = 12'b000101000111; // D.07.2, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11010111001: o_lut = 12'b100001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010111010: o_lut = 12'b100001001111; // D.15.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11010111011: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010111100: o_lut = 12'b101010111100; // K.28.5, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b11010111101: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010111110: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11010111111: o_lut = 12'b110001000000; // D.00.2, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011000000: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011000001: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011000010: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011000011: o_lut = 12'b001110011100; // K.28.4, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b11011000100: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011000101: o_lut = 12'b000110001111; // D.15.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011000110: o_lut = 12'b000110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011000111: o_lut = 12'b100010000111; // D.07.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011001000: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011001001: o_lut = 12'b000110010000; // D.16.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011001010: o_lut = 12'b000110011111; // D.31.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011001011: o_lut = 12'b100010001011; // D.11.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011001100: o_lut = 12'b000110011000; // D.24.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011001101: o_lut = 12'b100010001101; // D.13.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011001110: o_lut = 12'b100010001110; // D.14.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011001111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011010000: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011010001: o_lut = 12'b000110000001; // D.01.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011010010: o_lut = 12'b000110000010; // D.02.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011010011: o_lut = 12'b100010010011; // D.19.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011010100: o_lut = 12'b000110000100; // D.04.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011010101: o_lut = 12'b100010010101; // D.21.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011010110: o_lut = 12'b100010010110; // D.22.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011010111: o_lut = 12'b100110010111; // D.23.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011011000: o_lut = 12'b000110001000; // D.08.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011011001: o_lut = 12'b100010011001; // D.25.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011011010: o_lut = 12'b100010011010; // D.26.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011011011: o_lut = 12'b100110011011; // D.27.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011011100: o_lut = 12'b100010011100; // D.28.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011011101: o_lut = 12'b100110011101; // D.29.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011011110: o_lut = 12'b100110011110; // D.30.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011011111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011100000: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011100001: o_lut = 12'b000110011110; // D.30.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011100010: o_lut = 12'b000110011101; // D.29.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011100011: o_lut = 12'b100010000011; // D.03.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011100100: o_lut = 12'b000110011011; // D.27.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011100101: o_lut = 12'b100010000101; // D.05.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011100110: o_lut = 12'b100010000110; // D.06.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011100111: o_lut = 12'b100110001000; // D.08.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011101000: o_lut = 12'b000110010111; // D.23.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11011101001: o_lut = 12'b100010001001; // D.09.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011101010: o_lut = 12'b100010001010; // D.10.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011101011: o_lut = 12'b100110000100; // D.04.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011101100: o_lut = 12'b100010001100; // D.12.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011101101: o_lut = 12'b100110000010; // D.02.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011101110: o_lut = 12'b100110000001; // D.01.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011101111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011110000: o_lut = 12'b010110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b11011110001: o_lut = 12'b100010010001; // D.17.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011110010: o_lut = 12'b100010010010; // D.18.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011110011: o_lut = 12'b100110011000; // D.24.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011110100: o_lut = 12'b100010010100; // D.20.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011110101: o_lut = 12'b100110011111; // D.31.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011110110: o_lut = 12'b100110010000; // D.16.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011110111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011111000: o_lut = 12'b100010000111; // D.07.4, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011111001: o_lut = 12'b100110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011111010: o_lut = 12'b100110001111; // D.15.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011111011: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011111100: o_lut = 12'b100110011100; // D.28.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11011111101: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011111110: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11011111111: o_lut = 12'b110110000000; // D.00.4, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100000000: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100000001: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100000010: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100000011: o_lut = 12'b000001111100; // D.28.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100000100: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100000101: o_lut = 12'b000001101111; // D.15.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100000110: o_lut = 12'b000001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100000111: o_lut = 12'b000101100111; // D.07.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100001000: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100001001: o_lut = 12'b000001110000; // D.16.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100001010: o_lut = 12'b000001111111; // D.31.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100001011: o_lut = 12'b000101101011; // D.11.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100001100: o_lut = 12'b000001111000; // D.24.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100001101: o_lut = 12'b000101101101; // D.13.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100001110: o_lut = 12'b000101101110; // D.14.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100001111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100010000: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100010001: o_lut = 12'b000001100001; // D.01.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100010010: o_lut = 12'b000001100010; // D.02.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100010011: o_lut = 12'b000101110011; // D.19.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100010100: o_lut = 12'b000001100100; // D.04.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100010101: o_lut = 12'b000101110101; // D.21.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100010110: o_lut = 12'b000101110110; // D.22.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100010111: o_lut = 12'b100001110111; // D.23.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100011000: o_lut = 12'b000001101000; // D.08.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100011001: o_lut = 12'b000101111001; // D.25.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100011010: o_lut = 12'b000101111010; // D.26.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100011011: o_lut = 12'b100001111011; // D.27.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100011100: o_lut = 12'b000101111100; // D.28.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100011101: o_lut = 12'b100001111101; // D.29.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100011110: o_lut = 12'b100001111110; // D.30.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100011111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100100000: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100100001: o_lut = 12'b000001111110; // D.30.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100100010: o_lut = 12'b000001111101; // D.29.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100100011: o_lut = 12'b000101100011; // D.03.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100100100: o_lut = 12'b000001111011; // D.27.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100100101: o_lut = 12'b000101100101; // D.05.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100100110: o_lut = 12'b000101100110; // D.06.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100100111: o_lut = 12'b100001101000; // D.08.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100101000: o_lut = 12'b000001110111; // D.23.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100101001: o_lut = 12'b000101101001; // D.09.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100101010: o_lut = 12'b000101101010; // D.10.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100101011: o_lut = 12'b100001100100; // D.04.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100101100: o_lut = 12'b000101101100; // D.12.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100101101: o_lut = 12'b100001100010; // D.02.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100101110: o_lut = 12'b100001100001; // D.01.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100101111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100110000: o_lut = 12'b010001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b11100110001: o_lut = 12'b000101110001; // D.17.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100110010: o_lut = 12'b000101110010; // D.18.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100110011: o_lut = 12'b100001111000; // D.24.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100110100: o_lut = 12'b000101110100; // D.20.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100110101: o_lut = 12'b100001111111; // D.31.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100110110: o_lut = 12'b100001110000; // D.16.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100110111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100111000: o_lut = 12'b000101100111; // D.07.3, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11100111001: o_lut = 12'b100001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100111010: o_lut = 12'b100001101111; // D.15.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11100111011: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100111100: o_lut = 12'b101001111100; // K.28.3, i_disp = 1, o_disp = 0, o_ctrl = 1, o_code_err = 0, o_disp_err = 1
      11'b11100111101: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100111110: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11100111111: o_lut = 12'b110001100000; // D.00.3, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101000000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101000001: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101000010: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101000011: o_lut = 12'b001100011100; // K.28.0, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b11101000100: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101000101: o_lut = 12'b000100001111; // D.15.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101000110: o_lut = 12'b000100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101000111: o_lut = 12'b100000000111; // D.07.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101001000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101001001: o_lut = 12'b000100010000; // D.16.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101001010: o_lut = 12'b000100011111; // D.31.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101001011: o_lut = 12'b100000001011; // D.11.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101001100: o_lut = 12'b000100011000; // D.24.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101001101: o_lut = 12'b100000001101; // D.13.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101001110: o_lut = 12'b100000001110; // D.14.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101001111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101010000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101010001: o_lut = 12'b000100000001; // D.01.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101010010: o_lut = 12'b000100000010; // D.02.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101010011: o_lut = 12'b100000010011; // D.19.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101010100: o_lut = 12'b000100000100; // D.04.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101010101: o_lut = 12'b100000010101; // D.21.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101010110: o_lut = 12'b100000010110; // D.22.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101010111: o_lut = 12'b100100010111; // D.23.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101011000: o_lut = 12'b000100001000; // D.08.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101011001: o_lut = 12'b100000011001; // D.25.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101011010: o_lut = 12'b100000011010; // D.26.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101011011: o_lut = 12'b100100011011; // D.27.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101011100: o_lut = 12'b100000011100; // D.28.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101011101: o_lut = 12'b100100011101; // D.29.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101011110: o_lut = 12'b100100011110; // D.30.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101011111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101100000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101100001: o_lut = 12'b000100011110; // D.30.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101100010: o_lut = 12'b000100011101; // D.29.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101100011: o_lut = 12'b100000000011; // D.03.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101100100: o_lut = 12'b000100011011; // D.27.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101100101: o_lut = 12'b100000000101; // D.05.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101100110: o_lut = 12'b100000000110; // D.06.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101100111: o_lut = 12'b100100001000; // D.08.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101101000: o_lut = 12'b000100010111; // D.23.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11101101001: o_lut = 12'b100000001001; // D.09.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101101010: o_lut = 12'b100000001010; // D.10.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101101011: o_lut = 12'b100100000100; // D.04.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101101100: o_lut = 12'b100000001100; // D.12.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101101101: o_lut = 12'b100100000010; // D.02.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101101110: o_lut = 12'b100100000001; // D.01.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101101111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101110000: o_lut = 12'b010100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b11101110001: o_lut = 12'b100000010001; // D.17.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101110010: o_lut = 12'b100000010010; // D.18.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101110011: o_lut = 12'b100100011000; // D.24.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101110100: o_lut = 12'b100000010100; // D.20.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101110101: o_lut = 12'b100100011111; // D.31.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101110110: o_lut = 12'b100100010000; // D.16.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101110111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101111000: o_lut = 12'b100000000111; // D.07.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101111001: o_lut = 12'b100100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101111010: o_lut = 12'b100100001111; // D.15.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101111011: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101111100: o_lut = 12'b100100011100; // D.28.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11101111101: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101111110: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11101111111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110000000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110000001: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110000010: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110000011: o_lut = 12'b001111111100; // K.28.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b11110000100: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110000101: o_lut = 12'b000111101111; // D.15.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11110000110: o_lut = 12'b000111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11110000111: o_lut = 12'b100011100111; // D.07.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110001000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110001001: o_lut = 12'b000111110000; // D.16.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11110001010: o_lut = 12'b000111111111; // D.31.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11110001011: o_lut = 12'b100011101011; // D.11.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110001100: o_lut = 12'b000111111000; // D.24.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11110001101: o_lut = 12'b100011101101; // D.13.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110001110: o_lut = 12'b100011101110; // D.14.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110001111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110010000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110010001: o_lut = 12'b000111100001; // D.01.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11110010010: o_lut = 12'b000111100010; // D.02.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11110010011: o_lut = 12'b100011110011; // D.19.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110010100: o_lut = 12'b000111100100; // D.04.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11110010101: o_lut = 12'b100011110101; // D.21.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110010110: o_lut = 12'b100011110110; // D.22.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110010111: o_lut = 12'b100111110111; // D.23.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110011000: o_lut = 12'b000111101000; // D.08.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 0
      11'b11110011001: o_lut = 12'b100011111001; // D.25.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110011010: o_lut = 12'b100011111010; // D.26.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110011011: o_lut = 12'b100111111011; // D.27.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110011100: o_lut = 12'b100011111100; // D.28.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110011101: o_lut = 12'b100111111101; // D.29.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110011110: o_lut = 12'b100111111110; // D.30.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110011111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110100000: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110100001: o_lut = 12'b001111111110; // K.30.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b11110100010: o_lut = 12'b001111111101; // K.29.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b11110100011: o_lut = 12'b100011100011; // D.03.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110100100: o_lut = 12'b001111111011; // K.27.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b11110100101: o_lut = 12'b100011100101; // D.05.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110100110: o_lut = 12'b100011100110; // D.06.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110100111: o_lut = 12'b100111101000; // D.08.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110101000: o_lut = 12'b001111110111; // K.23.7, i_disp = 1, o_disp = 1, o_ctrl = 1, o_code_err = 0, o_disp_err = 0
      11'b11110101001: o_lut = 12'b100011101001; // D.09.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110101010: o_lut = 12'b100011101010; // D.10.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110101011: o_lut = 12'b100111100100; // D.04.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110101100: o_lut = 12'b100011101100; // D.12.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110101101: o_lut = 12'b100111100010; // D.02.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110101110: o_lut = 12'b100111100001; // D.01.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110101111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110110000: o_lut = 12'b010111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 0
      11'b11110110001: o_lut = 12'b100011110001; // D.17.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110110010: o_lut = 12'b100011110010; // D.18.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110110011: o_lut = 12'b100111111000; // D.24.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110110100: o_lut = 12'b100011110100; // D.20.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110110101: o_lut = 12'b100111111111; // D.31.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110110110: o_lut = 12'b100111110000; // D.16.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110110111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110111000: o_lut = 12'b100011100111; // D.07.7, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110111001: o_lut = 12'b100111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110111010: o_lut = 12'b100111101111; // D.15.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110111011: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110111100: o_lut = 12'b100111111100; // D.28.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 0, o_disp_err = 1
      11'b11110111101: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110111110: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11110111111: o_lut = 12'b110111100000; // D.00.7, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111000000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111000001: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111000010: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111000011: o_lut = 12'b110100011100; // D.28.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111000100: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111000101: o_lut = 12'b110100001111; // D.15.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111000110: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111000111: o_lut = 12'b110000000111; // D.07.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111001000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111001001: o_lut = 12'b110100010000; // D.16.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111001010: o_lut = 12'b110100011111; // D.31.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111001011: o_lut = 12'b110000001011; // D.11.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111001100: o_lut = 12'b110100011000; // D.24.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111001101: o_lut = 12'b110000001101; // D.13.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111001110: o_lut = 12'b110000001110; // D.14.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111001111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111010000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111010001: o_lut = 12'b110100000001; // D.01.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111010010: o_lut = 12'b110100000010; // D.02.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111010011: o_lut = 12'b110000010011; // D.19.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111010100: o_lut = 12'b110100000100; // D.04.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111010101: o_lut = 12'b110000010101; // D.21.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111010110: o_lut = 12'b110000010110; // D.22.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111010111: o_lut = 12'b110100010111; // D.23.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111011000: o_lut = 12'b110100001000; // D.08.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111011001: o_lut = 12'b110000011001; // D.25.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111011010: o_lut = 12'b110000011010; // D.26.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111011011: o_lut = 12'b110100011011; // D.27.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111011100: o_lut = 12'b110000011100; // D.28.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111011101: o_lut = 12'b110100011101; // D.29.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111011110: o_lut = 12'b110100011110; // D.30.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111011111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111100000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111100001: o_lut = 12'b110100011110; // D.30.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111100010: o_lut = 12'b110100011101; // D.29.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111100011: o_lut = 12'b110000000011; // D.03.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111100100: o_lut = 12'b110100011011; // D.27.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111100101: o_lut = 12'b110000000101; // D.05.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111100110: o_lut = 12'b110000000110; // D.06.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111100111: o_lut = 12'b110100001000; // D.08.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111101000: o_lut = 12'b110100010111; // D.23.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111101001: o_lut = 12'b110000001001; // D.09.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111101010: o_lut = 12'b110000001010; // D.10.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111101011: o_lut = 12'b110100000100; // D.04.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111101100: o_lut = 12'b110000001100; // D.12.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111101101: o_lut = 12'b110100000010; // D.02.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111101110: o_lut = 12'b110100000001; // D.01.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111101111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111110000: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111110001: o_lut = 12'b110000010001; // D.17.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111110010: o_lut = 12'b110000010010; // D.18.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111110011: o_lut = 12'b110100011000; // D.24.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111110100: o_lut = 12'b110000010100; // D.20.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111110101: o_lut = 12'b110100011111; // D.31.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111110110: o_lut = 12'b110100010000; // D.16.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111110111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111111000: o_lut = 12'b110000000111; // D.07.0, i_disp = 1, o_disp = 0, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111111001: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111111010: o_lut = 12'b110100001111; // D.15.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111111011: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111111100: o_lut = 12'b110100011100; // D.28.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111111101: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111111110: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      11'b11111111111: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      default: o_lut = 12'b110100000000; // D.00.0, i_disp = 1, o_disp = 1, o_ctrl = 0, o_code_err = 1, o_disp_err = 1
      // verilog_format: on
            // verilog_lint: waive-stop line-length
        endcase
    end

endmodule

`endif  /* LIBSV_CODERS_DECODER_8B10B */
