# Facial Recognition

Recognize and manipulate faces from Python. This is a fork from https://github.com/ageitgey/face_recognition.

Built using [dlib](http://dlib.net/)'s state-of-the-art face recognition
built with deep learning. The model has an accuracy of 99.38% on the
[Labeled Faces in the Wild](http://vis-www.cs.umass.edu/lfw/) benchmark.


## Features

#### Find faces in pictures

Find all the faces that appear in a picture:

![](https://cloud.githubusercontent.com/assets/896692/23625227/42c65360-025d-11e7-94ea-b12f28cb34b4.png)

```python
import face_biometric_recognition
image = face_biometric_recognition.load_image_file("your_file.jpg")
face_locations = face_biometric_recognition.face_locations(image)
```

#### Find and manipulate facial features in pictures

Get the locations and outlines of each person's eyes, nose, mouth and chin.

![](https://cloud.githubusercontent.com/assets/896692/23625282/7f2d79dc-025d-11e7-8728-d8924596f8fa.png)

```python
import face_biometric_recognition
image = face_biometric_recognition.load_image_file("your_file.jpg")
face_landmarks_list = face_biometric_recognition.face_landmarks(image)
```

#### Identify faces in pictures

Recognize who appears in each photo.

![](https://cloud.githubusercontent.com/assets/896692/23625229/45e049b6-025d-11e7-89cc-8a71cf89e713.png)

```python
import face_biometric_recognition
known_image = face_biometric_recognition.load_image_file("biden.jpg")
unknown_image = face_biometric_recognition.load_image_file("unknown.jpg")

biden_encoding = face_biometric_recognition.face_encodings(known_image)[0]
unknown_encoding = face_biometric_recognition.face_encodings(unknown_image)[0]

results = face_biometric_recognition.compare_faces([biden_encoding], unknown_encoding)
```

## Installation

### Requirements

  * Python 3.3+ or Python 2.7
  * macOS or Linux (Windows not officially supported, but might work)

### Installation Options:

#### Installing on Mac or Linux

First, make sure you have dlib already installed with Python bindings:

  * [How to install dlib from source on macOS or Ubuntu](https://gist.github.com/ageitgey/629d75c1baac34dfa5ca2a1928a7aeaf)
  
Then, make sure you have cmake installed:  
 
```brew install cmake```

Finally, install this module from pypi using `pip3` (or `pip2` for Python 2):

```bash
pip3 install face_biometric_recognition --no-index --find-links file:/path/to/face_biometric_recognition
```

## Usage

#### Python Module

You can import the `face_biometric_recognition` module and then easily manipulate
faces with just a couple of lines of code. It's super easy!

##### Automatically find all the faces in an image

```python
import face_biometric_recognition

image = face_biometric_recognition.load_image_file("my_picture.jpg")
face_locations = face_biometric_recognition.face_locations(image)

# face_locations is now an array listing the co-ordinates of each face!
```

You can also opt-in to a somewhat more accurate deep-learning-based face detection model.

Note: GPU acceleration (via NVidia's CUDA library) is required for good
performance with this model. You'll also want to enable CUDA support
when compliling `dlib`.

```python
import face_biometric_recognition

image = face_biometric_recognition.load_image_file("my_picture.jpg")
face_locations = face_biometric_recognition.face_locations(image, model="cnn")

# face_locations is now an array listing the co-ordinates of each face!
```

##### Automatically locate the facial features of a person in an image

```python
import face_biometric_recognition

image = face_biometric_recognition.load_image_file("my_picture.jpg")
face_landmarks_list = face_biometric_recognition.face_landmarks(image)

# face_landmarks_list is now an array with the locations of each facial feature in each face.
# face_landmarks_list[0]['left_eye'] would be the location and outline of the first person's left eye.
```

##### Recognize faces in images and identify who they are

```python
import face_biometric_recognition

picture_of_me = face_biometric_recognition.load_image_file("me.jpg")
my_face_encoding = face_biometric_recognition.face_encodings(picture_of_me)[0]

# my_face_encoding now contains a universal 'encoding' of my facial features
# that can be compared to any other picture of a face!

unknown_picture = face_biometric_recognition.load_image_file("unknown.jpg")
unknown_face_encoding = face_biometric_recognition.face_encodings(unknown_picture)[0]

# Now we can see the two face encodings
# are of the same person with `compare_faces`!

results = face_biometric_recognition.compare_faces(
    [my_face_encoding], unknown_face_encoding
)

if results[0] is True:
    print("It's a picture of me!")
else:
    print("It's not a picture of me!")
```

## Articles and Guides that cover `face_biometric_recognition`

- Adam Geitgey's article on how Face Recognition works: [Modern Face Recognition with Deep Learning](https://medium.com/@ageitgey/machine-learning-is-fun-part-4-modern-face-recognition-with-deep-learning-c3cffc121d78)
  - Covers the algorithms and how they generally work
- [Face recognition with OpenCV, Python, and deep learning](https://www.pyimagesearch.com/2018/06/18/face-recognition-with-opencv-python-and-deep-learning/) by Adrian Rosebrock
  - Covers how to use face recognition in practice
- [Raspberry Pi Face Recognition](https://www.pyimagesearch.com/2018/06/25/raspberry-pi-face-recognition/) by Adrian Rosebrock
  - Covers how to use this on a Raspberry Pi
- [Face clustering with Python](https://www.pyimagesearch.com/2018/07/09/face-clustering-with-python/) by Adrian Rosebrock
  - Covers how to automatically cluster photos based on who appears in each photo using unsupervised learning

## How Face Recognition Works

If you want to learn how face location and recognition work instead of
depending on a black box library, [read Adam Geitgey's article](https://medium.com/@ageitgey/machine-learning-is-fun-part-4-modern-face-recognition-with-deep-learning-c3cffc121d78).

## Caveats

* The facial recognition model is trained on adults and does not work very well on children. It tends to mix
  up children quite easy using the default comparison threshold of 0.6.
* Accuracy may vary between ethnic groups. Please see [this wiki page](https://github.com/ageitgey/face_recognition/wiki/Face-Recognition-Accuracy-Problems#question-face-recognition-works-well-with-european-individuals-but-overall-accuracy-is-lower-with-asian-individuals) for more details.

## Thanks

* Many thanks to [Adam Geitgey](https://github.com/ageitgey) for putting together such an amazing project and allowing public access to it.
