import numpy as np
from scipy.constants import mu_0
from scipy.interpolate import interp1d
from scipy.constants import pi
from geoapps.simpegPF import Utils


def EvalDigitalFilt(base, weight, fun, r):
    """
        Evaluating Digital filtering based on given base and weight

    """
    return np.dot(fun(base / r), weight) / r


def setFrequency(time):
    wt = np.array(
        [
            7.214369775966785e-20,
            5.997984537445829e-20,
            1.383536819510307e-20,
            6.127201193993877e-20,
            2.735622069700930e-20,
            6.567948836420383e-20,
            4.144963335850363e-20,
            7.316414067200350e-20,
            5.682375914662966e-20,
            8.391977074915078e-20,
            7.418756524583309e-20,
            9.829637687190485e-20,
            9.430643800653847e-20,
            1.168146262188112e-19,
            1.180370735968097e-19,
            1.401723019040171e-19,
            1.463726071463266e-19,
            1.692722072070252e-19,
            1.804796158499069e-19,
            2.052560499147526e-19,
            2.217507732438609e-19,
            2.495469564846162e-19,
            2.718603842873614e-19,
            3.039069705922034e-19,
            3.328334008394297e-19,
            3.705052796297763e-19,
            4.071277819975917e-19,
            4.520053409594589e-19,
            4.977334107366132e-19,
            5.516707191291291e-19,
            6.082931168675559e-19,
            6.734956703766505e-19,
            7.432489554623685e-19,
            8.223651399147256e-19,
            9.080210233648037e-19,
            1.004250388267800e-18,
            1.109225156214032e-18,
            1.226448534750949e-18,
            1.354938655056596e-18,
            1.497875155579711e-18,
            1.655024636692164e-18,
            1.829422009902478e-18,
            2.021527957180686e-18,
            2.234394042862191e-18,
            2.469158736824458e-18,
            2.729043278909879e-18,
            3.015882778812807e-18,
            3.333221019045560e-18,
            3.683642665131121e-18,
            4.071174485366807e-18,
            4.499238428427072e-18,
            4.972519918024098e-18,
            5.495403162992602e-18,
            6.073431145514256e-18,
            6.712116746365455e-18,
            7.418091347704607e-18,
            8.198210388921290e-18,
            9.060466264497684e-18,
            1.001332641867938e-17,
            1.106647001686341e-17,
            1.223031194783507e-17,
            1.351661046246575e-17,
            1.493814249254853e-17,
            1.650922025025269e-17,
            1.824549287949245e-17,
            2.016440324953847e-17,
            2.228509875325462e-17,
            2.462885473506622e-17,
            2.721908372832262e-17,
            3.008174877960754e-17,
            3.324546598231868e-17,
            3.674192913569353e-17,
            4.060610542324258e-17,
            4.487669220181069e-17,
            4.959641037849226e-17,
            5.481251456381401e-17,
            6.057719336989671e-17,
            6.694815564512041e-17,
            7.398915178848498e-17,
            8.177066132132114e-17,
            9.037055462918574e-17,
            9.987491078055815e-17,
            1.103788451159722e-16,
            1.219874911140742e-16,
            1.348170262066998e-16,
            1.489958578076007e-16,
            1.646658879212839e-16,
            1.819839514458913e-16,
            2.011233698894207e-16,
            2.222757000537238e-16,
            2.456526388749016e-16,
            2.714881529754608e-16,
            3.000408107960083e-16,
            3.315963787425073e-16,
            3.664706739627943e-16,
            4.050127315080793e-16,
            4.476082920363670e-16,
            4.946836672898304e-16,
            5.467100025245505e-16,
            6.042079955957903e-16,
            6.677531050397348e-16,
            7.379813122861424e-16,
            8.155954842977402e-16,
            9.013724102689123e-16,
            9.961705740887021e-16,
            1.100938748010566e-15,
            1.216725486808607e-15,
            1.344689623369201e-15,
            1.486111865526057e-15,
            1.642407614840039e-15,
            1.815141131499014e-15,
            2.006041190779248e-15,
            2.217018384471440e-15,
            2.450184243392977e-15,
            2.707872369692257e-15,
            2.992661792874233e-15,
            3.307402781094011e-15,
            3.655245368051253e-15,
            4.039670879180488e-15,
            4.464526774284602e-15,
            4.934065153895433e-15,
            5.452985315986473e-15,
            6.026480787914038e-15,
            6.660291305149181e-15,
            7.360760256360466e-15,
            8.134898170257041e-15,
            8.990452879276204e-15,
            9.935987062502841e-15,
            1.098096394385775e-14,
            1.213584200318437e-14,
            1.341217964828528e-14,
            1.482275089528562e-14,
            1.638167321535499e-14,
            1.810454882702344e-14,
            2.000862084851265e-14,
            2.211294587257239e-14,
            2.443858469135401e-14,
            2.700881307980678e-14,
            2.984935474755050e-14,
            3.298863879030854e-14,
            3.645808421795958e-14,
            4.029241440643229e-14,
            4.453000462105175e-14,
            4.921326608894885e-14,
            5.438907046503769e-14,
            6.010921893911273e-14,
            6.643096067976429e-14,
            7.341756580308676e-14,
            8.113895860149252e-14,
            8.967241736929777e-14,
            9.910334783010448e-14,
            1.095261379057530e-13,
            1.210451023825933e-13,
            1.337755269287210e-13,
            1.478448219118764e-13,
            1.633937975650728e-13,
            1.805780732628623e-13,
            1.995696350122467e-13,
            2.205585567465074e-13,
            2.437549026489779e-13,
            2.693908295460095e-13,
            2.977229104105259e-13,
            3.290347022305518e-13,
            3.636395839428896e-13,
            4.018838928348062e-13,
            4.441503908040617e-13,
            4.908620951685787e-13,
            5.424865123659980e-13,
            5.995403169151822e-13,
            6.625945224685207e-13,
            7.322801967084261e-13,
            8.092947772848716e-13,
            8.944090520057436e-13,
            9.884748731403624e-13,
            1.092433683043238e-12,
            1.207325936425662e-12,
            1.334301513576084e-12,
            1.474631228748613e-12,
            1.629719548899119e-12,
            1.801118650062676e-12,
            1.990543952052933e-12,
            2.199891286960273e-12,
            2.431255873276498e-12,
            2.686953285545802e-12,
            2.969542629413028e-12,
            3.281852154013172e-12,
            3.627007558039277e-12,
            4.008463272785582e-12,
            4.430037035256956e-12,
            4.895948097364050e-12,
            5.410859453614547e-12,
            5.979924509929487e-12,
            6.608838660661838e-12,
            7.303896290017477e-12,
            8.072053768367932e-12,
            8.920999073943177e-12,
            9.859228736701785e-12,
            1.089613287445852e-11,
            1.204208917233957e-11,
            1.330856674614333e-11,
            1.470824092910627e-11,
            1.625512013089818e-11,
            1.796468603849469e-11,
            1.985404856210394e-11,
            2.194211707689892e-11,
            2.424978967439970e-11,
            2.680016231759770e-11,
            2.961875999311579e-11,
            3.273379217385409e-11,
            3.617643514887572e-11,
            3.998114404618718e-11,
            4.418599767123930e-11,
            4.883307961241208e-11,
            5.396889942771051e-11,
            5.964485812805529e-11,
            6.591776261587440e-11,
            7.285039422767879e-11,
            8.051213707077629e-11,
            8.897967244274265e-11,
            9.833774628361575e-11,
            1.086800173417544e-10,
            1.201099945420632e-10,
            1.327420729381141e-10,
            1.467026786162787e-10,
            1.621315340105112e-10,
            1.791830562914075e-10,
            1.980279028251780e-10,
            2.188546791698937e-10,
            2.418718267033471e-10,
            2.673097087743666e-10,
            2.954229162567076e-10,
            3.264928155800021e-10,
            3.608303647396648e-10,
            3.987792254688925e-10,
            4.407192027209688e-10,
            4.870700458846789e-10,
            5.382956497775456e-10,
            5.949086974607432e-10,
            6.574757913439202e-10,
            7.266231239320192e-10,
            8.030427449710128e-10,
            8.874994877135167e-10,
            9.808386236281220e-10,
            1.083994322159010e-09,
            1.197999000209434e-09,
            1.323993654914953e-09,
            1.463239283128961e-09,
            1.617129501899646e-09,
            1.787204496262075e-09,
            1.975166433922344e-09,
            2.182896501130837e-09,
            2.412473730218034e-09,
            2.666195807259519e-09,
            2.946602068077095e-09,
            3.256498912782063e-09,
            3.598987893149563e-09,
            3.977496754017933e-09,
            4.395813739277522e-09,
            4.858125505931142e-09,
            5.369059025511281e-09,
            5.933727892433384e-09,
            6.557783502483194e-09,
            7.247471613991360e-09,
            8.009694857348590e-09,
            8.852081819018630e-09,
            9.783063390784292e-09,
            1.081195714921208e-08,
            1.194906060875559e-08,
            1.320575428316232e-08,
            1.459461558495058e-08,
            1.612954470504804e-08,
            1.782590372973567e-08,
            1.970067039062624e-08,
            2.177260798218037e-08,
            2.406245315273551e-08,
            2.659312344174916e-08,
            2.938994664888302e-08,
            3.248091431980495e-08,
            3.589696189917651e-08,
            3.967227833770833e-08,
            4.384464827330457e-08,
            4.845583018407081e-08,
            5.355197433170284e-08,
            5.918408463559961e-08,
            6.540852915386353e-08,
            7.228760421284378e-08,
            7.989015791604288e-08,
            8.829227916594097e-08,
            9.757805922900159e-08,
            1.078404332968648e-07,
            1.191821106789995e-07,
            1.317166026689236e-07,
            1.455693587079098e-07,
            1.608790217936311e-07,
            1.777988162313823e-07,
            1.964980809461758e-07,
            2.171639645456637e-07,
            2.400032980365736e-07,
            2.652446652738443e-07,
            2.931406901825997e-07,
            3.239705657602287e-07,
            3.580428475071237e-07,
            3.956985425939703e-07,
            4.373145214673157e-07,
            4.833072913425415e-07,
            5.341371626757850e-07,
            5.903128587132423e-07,
            6.523966036832935e-07,
            7.210097538541495e-07,
            7.968390110811429e-07,
            8.806433020866372e-07,
            9.732613658282036e-07,
            1.075620158230134e-06,
            1.188744116446123e-06,
            1.313765428158270e-06,
            1.451935342270991e-06,
            1.604636717777632e-06,
            1.773397831228256e-06,
            1.959907713317686e-06,
            2.166033001576880e-06,
            2.393836687356070e-06,
            2.645598681084377e-06,
            2.923838733370935e-06,
            3.231341523918154e-06,
            3.571184694601016e-06,
            3.946769446344899e-06,
            4.361854837678969e-06,
            4.820595081762782e-06,
            5.327581531949061e-06,
            5.887888119174313e-06,
            6.507122780830562e-06,
            7.191482772393097e-06,
            7.947817716468041e-06,
            8.783696866498923e-06,
            9.707486485040472e-06,
            1.072843153422521e-05,
            1.185675077161778e-05,
            1.310373578995573e-05,
            1.448186809502301e-05,
            1.600493890578862e-05,
            1.768819362222417e-05,
            1.954847630132444e-05,
            2.160440843572022e-05,
            2.387656249074371e-05,
            2.638768394666778e-05,
            2.916289862392297e-05,
            3.222998971512441e-05,
            3.561964367629314e-05,
            3.936579782365431e-05,
            4.350592904974602e-05,
            4.808149299156779e-05,
            5.313825827671661e-05,
            5.872686606041739e-05,
            6.490320915255368e-05,
            7.172915206849267e-05,
            7.927294798468421e-05,
            8.761017620761336e-05,
            9.682417843295337e-05,
            1.070072955978771e-04,
            1.182612851235724e-04,
            1.306989769939818e-04,
            1.444446003274482e-04,
            1.596360362963627e-04,
            1.764249271609239e-04,
            1.949797924244976e-04,
            2.154857030671910e-04,
            2.381486646105023e-04,
            2.631944925246626e-04,
            2.908750792099106e-04,
            3.214658697246949e-04,
            3.552749625435381e-04,
            3.926382043270680e-04,
            4.339325952975191e-04,
            4.795674127479124e-04,
            5.300042093562213e-04,
            5.857414026355948e-04,
            6.473444397414629e-04,
            7.154197401707392e-04,
            7.906606243262904e-04,
            8.738040302727717e-04,
            9.657009935888906e-04,
            1.067245638145834e-03,
            1.179484028621435e-03,
            1.303498707764836e-03,
            1.440577691237741e-03,
            1.592027938865682e-03,
            1.759438818176274e-03,
            1.944382214020240e-03,
            2.148824632015574e-03,
            2.374646777242952e-03,
            2.624289840901410e-03,
            2.899987938462482e-03,
            3.204783728012370e-03,
            3.541304571287609e-03,
            3.913361077715114e-03,
            4.323998734848948e-03,
            4.778017035442578e-03,
            5.278871213895021e-03,
            5.832645828904957e-03,
            6.443132211847618e-03,
            7.118100704687155e-03,
            7.861484687059508e-03,
            8.683286454219962e-03,
            9.587172959576953e-03,
            1.058612645311708e-02,
            1.168276512339872e-02,
            1.289407692301174e-02,
            1.422020567085629e-02,
            1.568354709989395e-02,
            1.727924763496293e-02,
            1.903701004445868e-02,
            2.094259894090355e-02,
            2.303555498203885e-02,
            2.528473397535577e-02,
            2.774280095909549e-02,
            3.034889679856765e-02,
            3.317292189089636e-02,
            3.610269051747732e-02,
            3.923023471609136e-02,
            4.235591398256915e-02,
            4.559945470018810e-02,
            4.861418172220856e-02,
            5.155399423688033e-02,
            5.382905665985834e-02,
            5.563737547309198e-02,
            5.599656739496778e-02,
            5.517328802198061e-02,
            5.157565446188783e-02,
            4.561585237274122e-02,
            3.481744626013846e-02,
            1.997678484763328e-02,
            -2.511444299727086e-03,
            -3.078890380569448e-02,
            -6.952663437748715e-02,
            -1.140926319655417e-01,
            -1.692861783153246e-01,
            -2.240265004914591e-01,
            -2.809223452446239e-01,
            -3.165386782849084e-01,
            -3.295050746499982e-01,
            -2.805919713655642e-01,
            -1.744060875765448e-01,
            2.722628846693606e-02,
            2.668949880744598e-01,
            5.262102231394616e-01,
            6.256684356927903e-01,
            4.995016301447683e-01,
            -1.002368152582941e-02,
            -6.114010724740713e-01,
            -9.727382503860407e-01,
            -3.838420705230950e-01,
            7.198704705669955e-01,
            1.262041888009595e00,
            -2.998397076312483e-01,
            -1.479978761932394e00,
            1.886890549669046e-01,
            1.961538671802124e00,
            -2.104506074490929e00,
            7.701373097387101e-01,
            4.062497351127477e-01,
            -8.229740504000808e-01,
            7.307456920106093e-01,
            -4.903037312539515e-01,
            2.839808721720737e-01,
            -1.517915989046718e-01,
            7.860615976683388e-02,
            -4.139435902417716e-02,
            2.340179865400356e-02,
            -1.488928090494461e-02,
            1.080204283974104e-02,
            -8.695630540330540e-03,
            7.448774255862835e-03,
            -6.571253694245813e-03,
            5.859704720178251e-03,
            -5.235154219023063e-03,
            4.669537109654244e-03,
            -4.153880559277143e-03,
            3.685278478886407e-03,
            -3.262012231674279e-03,
            2.882025619739767e-03,
            -2.542670610556139e-03,
            2.240859550470028e-03,
            -1.973292341858488e-03,
            1.736649256291777e-03,
            -1.527725614465373e-03,
            1.343513590939351e-03,
            -1.181244115916277e-03,
            1.038401885876272e-03,
            -9.127236961818876e-04,
            8.021869803583510e-04,
            -7.049929363136232e-04,
            6.195471678105551e-04,
            -5.444398377266471e-04,
            4.784265058211163e-04,
            -4.204101656165671e-04,
            3.694246665626042e-04,
            -3.246196272200836e-04,
            2.852468930079681e-04,
            -2.506484828993674e-04,
            2.202458813636377e-04,
            -1.935305291014704e-04,
            1.700554065180346e-04,
            -1.494276181460851e-04,
            1.313018693894386e-04,
            -1.153747197310416e-04,
            1.013795159657149e-04,
            -8.908193308740761e-05,
            7.827605834070905e-05,
            -6.878095175364698e-05,
            6.043762035968366e-05,
            -5.310635544925448e-05,
            4.666439257514449e-05,
            -4.100385733848758e-05,
            3.602996086454923e-05,
            -3.165941281357187e-05,
            2.781902585783293e-05,
            -2.444448983707277e-05,
            2.147929539928508e-05,
            -1.887378820764930e-05,
            1.658433732127529e-05,
            -1.457260421932255e-05,
            1.280490076321861e-05,
            -1.125162529600942e-05,
            9.886767060922143e-06,
            -8.687470528401389e-06,
            7.633652544743499e-06,
            -6.707666049011719e-06,
            5.894004649462370e-06,
            -5.179042976527735e-06,
            4.550808463155193e-06,
            -3.998780821517730e-06,
            3.513715894699684e-06,
            -3.087490887193285e-06,
            2.712968339794580e-06,
            -2.383876585436306e-06,
            2.094704718364733e-06,
            -1.840610321658703e-06,
            1.617338386451554e-06,
            -1.421150054108675e-06,
            1.248759998169063e-06,
            -1.097281408784448e-06,
            9.641776560691427e-07,
            -8.472198144303782e-07,
            7.444493348026655e-07,
            -6.541452446960252e-07,
            5.747953299561648e-07,
            -5.050708135463763e-07,
            4.438041043830051e-07,
            -3.899692438669907e-07,
            3.426647247474678e-07,
            -3.010983952488172e-07,
            2.645741945310525e-07,
            -2.324804964806138e-07,
            2.042798670572760e-07,
            -1.795000644284691e-07,
            1.577261311964345e-07,
            -1.385934457978807e-07,
            1.217816165104314e-07,
            -1.070091160694850e-07,
            9.402856728542481e-08,
            -8.262260063861433e-08,
            7.260021429510923e-08,
            -6.379357556568284e-08,
            5.605521036390507e-08,
            -4.925553366022761e-08,
            4.328067952283362e-08,
            -3.803059434473390e-08,
            3.341736133872807e-08,
            -2.936372828943425e-08,
            2.580181391877248e-08,
            -2.267197117462836e-08,
            1.992178838990867e-08,
            -1.750521159700888e-08,
            1.538177331647477e-08,
            -1.351591490408599e-08,
            1.187639109711096e-08,
            -1.043574678473461e-08,
            9.169857246988264e-09,
            -8.057524168015392e-09,
            7.080120656351000e-09,
            -6.221279323637567e-09,
            5.466618198282681e-09,
            -4.803499887007487e-09,
            4.220819952418786e-09,
            -3.708820961440479e-09,
            3.258929089378474e-09,
            -2.863610543832675e-09,
            2.516245405206862e-09,
            -2.211016771314404e-09,
            1.942813349072686e-09,
            -1.707143861761559e-09,
            1.500061838825614e-09,
            -1.318099529115311e-09,
            1.158209830829835e-09,
            -1.017715265474667e-09,
            8.942631413031228e-10,
            -7.857861555682931e-10,
            6.904677759378704e-10,
            -6.067118212948776e-10,
            5.331157324405220e-10,
            -4.684470851019305e-10,
            4.116229519995125e-10,
            -3.616917683963526e-10,
            3.178173974200357e-10,
            -2.792651282909316e-10,
            2.453893729983475e-10,
            -2.156228554187546e-10,
            1.894671118403349e-10,
            -1.664841438026135e-10,
            1.462890834648734e-10,
            -1.285437486867098e-10,
            1.129509799028343e-10,
            -9.924966395649082e-11,
            8.721036155589632e-11,
            -7.663146513121262e-11,
            6.733582275589511e-11,
            -5.916777159927192e-11,
            5.199053123157253e-11,
            -4.568391312831626e-11,
            4.014230801823632e-11,
            -3.527291737272422e-11,
            3.099419942203976e-11,
            -2.723450367478780e-11,
            2.393087107406673e-11,
            -2.102797969829416e-11,
            1.847721835214320e-11,
            -1.623587253411553e-11,
            1.426640914879302e-11,
            -1.253584798559500e-11,
            1.101520943914518e-11,
            -9.679029223044069e-12,
            8.504931950483330e-12,
            -7.473256440847272e-12,
            6.566726477759826e-12,
            -5.770161505244737e-12,
            5.070222417415932e-12,
            -4.455188184705872e-12,
            3.914759576021876e-12,
            -3.439886690015256e-12,
            3.022617407370815e-12,
            -2.655964226336168e-12,
            2.333787251527801e-12,
            -2.050691376558676e-12,
            1.801935938738539e-12,
            -1.583355332954674e-12,
            1.391289255348560e-12,
            -1.222521408656179e-12,
            1.074225642782795e-12,
            -9.439186286983481e-13,
            8.294182731437524e-13,
            -7.288071777679118e-13,
            6.404005307872462e-13,
            -5.627178934884672e-13,
            4.944584091193514e-13,
            -4.344790190215893e-13,
            3.817753212167385e-13,
            -3.354647509064286e-13,
            2.947718012314345e-13,
            -2.590150368003291e-13,
            2.275956825191015e-13,
            -1.999875966322778e-13,
            1.757284600660376e-13,
            -1.544120345321836e-13,
            1.356813597490631e-13,
            -1.192227758615040e-13,
            1.047606709603187e-13,
            -9.205286574443716e-14,
            8.088655803832021e-14,
            -7.107475925247343e-14,
            6.245316311274798e-14,
            -5.487739422278246e-14,
            4.822059038461202e-14,
            -4.237127819154857e-14,
            3.723150631847816e-14,
            -3.271520525003134e-14,
            2.874674597896990e-14,
            -2.525967353907224e-14,
            2.219559416454687e-14,
            -1.950319744058413e-14,
            1.713739707017873e-14,
            -1.505857586868545e-14,
            1.323192234295437e-14,
            -1.162684774554722e-14,
            1.021647384214807e-14,
            -8.977182814427699e-15,
            7.888221761131355e-15,
            -6.931355174452619e-15,
            6.090559572138626e-15,
            -5.351755171700232e-15,
            4.702570113399845e-15,
            -4.132133283746073e-15,
            3.630892270162917e-15,
            -3.190453398341683e-15,
            2.803441173574897e-15,
            -2.463374772306652e-15,
            2.164559515653694e-15,
            -1.901991507536473e-15,
            1.671273840510759e-15,
            -1.468542966100438e-15,
            1.290403996644843e-15,
            -1.133873855239388e-15,
            9.963313217707399e-16,
            -8.754731385280971e-16,
            7.692754403444974e-16,
            -6.759598633855865e-16,
            5.939637650509565e-16,
            -5.219140562969427e-16,
            4.586042081825544e-16,
            -4.029740475950767e-16,
            3.540920038189227e-16,
            -3.111395086526072e-16,
            2.733972888415509e-16,
            -2.402333212827215e-16,
            2.110922493015534e-16,
            -1.854860827684001e-16,
            1.629860263206755e-16,
            -1.432152988478505e-16,
            1.258428239959391e-16,
            -1.105776860340182e-16,
            9.716425824191095e-17,
            -8.537792224005708e-17,
            7.502130657839168e-17,
            -6.592098159645411e-17,
            5.792455520756561e-17,
            -5.089812097369260e-17,
            4.472401573699795e-17,
            -3.929884925786503e-17,
            3.453177286415005e-17,
            -3.034295811884372e-17,
            2.666226003023219e-17,
            -2.342804241895063e-17,
            2.058614577177349e-17,
            -1.808898029804633e-17,
            1.589472900128195e-17,
            -1.396664742072978e-17,
            1.227244831653922e-17,
            -1.078376099458355e-17,
            9.475656216910048e-18,
            -8.326228742065685e-18,
            7.316230504610306e-18,
            -6.428748291129759e-18,
            5.648920515191333e-18,
            -4.963688348418389e-18,
            4.361577040171507e-18,
            -3.832503763852548e-18,
            3.367608772061774e-18,
            -2.959107033168789e-18,
            2.600157864838119e-18,
            -2.284750381424515e-18,
            2.007602836968745e-18,
            -1.764074178215185e-18,
            1.550086326535024e-18,
            -1.362055887301009e-18,
            1.196834143131819e-18,
            -1.051654326148005e-18,
            9.240852862763527e-19,
            -8.119907797435101e-19,
            7.134936960083838e-19,
            -6.269446240781204e-19,
            5.508942318228495e-19,
            -4.840689957627215e-19,
            4.253498749090647e-19,
            -3.737535715383304e-19,
            3.284160650943604e-19,
            -2.885781434802982e-19,
            2.535726894517719e-19,
            -2.228135092144265e-19,
            1.957855161528666e-19,
            -1.720361053077579e-19,
            1.511675741544441e-19,
            -1.328304627571508e-19,
            1.167177017717951e-19,
            -1.025594703000911e-19,
            9.011867747602604e-20,
            -7.918699208456320e-20,
            6.958135363559505e-20,
            -6.114090626414241e-20,
            5.372430364847189e-20,
            -4.720733874362162e-20,
            4.148085614846149e-20,
            -3.644890635898519e-20,
            3.202709755606534e-20,
            -2.814108611035396e-20,
            2.472510802483146e-20,
            -2.172035832750181e-20,
            1.907280017594962e-20,
            -7.276969157651721e-21,
        ]
    )
    ab = 0.7866057737580476e0
    # ------- Compute Frequency components reqired for transform -------#
    # This is for Digital filtering and here we evalute frequency domain responses
    # ritght at this bases.
    # a. Generate time based
    n = np.ceil(-10 * np.log(time.min() / time.max()))
    tbase = time.max() * np.exp(-0.1 * np.arange(0, n + 1))

    # b. Determine required frequencies
    omega_int = (ab / tbase[0]) * np.exp(
        0.1 * (np.r_[1 : 786 + tbase.size : (786 + tbase.size) * 1j] - 425)
    )

    return wt, tbase, omega_int


def transFilt(hz, wt, tbase, omega_int, t, tol=1e-12):
    """
        Compute Step-off responses by Fast Hankel Transform (FHT) with cosine filters
    """
    # Define the filter coeffs
    # Generate time base
    # Determine required frequencies
    # Calculate the frequency domain data

    hziw = Utils.mkvc(hz.imag) / omega_int

    # Clean the low frequency results
    idKeep = [idx for idx in range(len(hz)) if abs(hz.imag)[idx] > tol]
    idKeep = np.array(idKeep)

    idRemove = [idx for idx in range(len(hz)) if abs(hz.imag)[idx] < tol]
    idRemove = np.array(idRemove)

    hziwc = hziw.copy()
    # if idRemove.size != 0:
    #     lastKeep = idKeep.max()

    #     idRemove[idRemove > lastKeep] = 0
    #     lastRemove = idRemove.max()

    #     hziwc[:lastRemove] = hziw[lastRemove+1]

    # Apply filter
    dt_filt = np.zeros_like(tbase)
    for i in np.arange(tbase.size):
        F = np.r_[np.zeros(i), 2 * wt / (pi * tbase[i]), np.zeros(tbase.size - i - 1)]
        dt_filt[i] = -np.dot(F, hziwc)

    # Interpolate result
    fhz = interp1d(tbase[::-1], dt_filt[::-1])
    hz_out = fhz(t)

    return hz_out, np.r_[tbase[-1], dt_filt[-1]]


def transFiltInterp(hz, wt, tbase, omega, omega_int, t, tol=1e-12):
    """
        Compute Step-off responses by Fast Hankel Transform (FHT) with cosine filters
    """
    # Define the filter coeffs
    # Generate time base
    # Determine required frequencies
    # Calculate the frequency domain data

    hziw = Utils.mkvc(hz.imag) / omega

    # Clean the low frequency results
    idKeep = [idx for idx in range(len(hz)) if abs(hz.imag)[idx] > tol]
    idKeep = np.array(idKeep)

    idRemove = [idx for idx in range(len(hz)) if abs(hz.imag)[idx] < tol]
    idRemove = np.array(idRemove)
    hziwint = interp1d(omega, hziw)
    hziwc = np.zeros(omega_int.shape)
    ind = (omega_int > omega.min()) & (omega_int < omega.max())

    hziwc[ind] = hziwint(omega_int[ind])

    # if idRemove.size != 0:
    #     lastKeep = idKeep.max()

    #     idRemove[idRemove > lastKeep] = 0
    #     lastRemove = idRemove.max()

    #     hziwc[:lastRemove] = hziw[lastRemove+1]

    # Apply filter
    dt_filt = np.zeros_like(tbase)
    for i in np.arange(tbase.size):
        F = np.r_[np.zeros(i), 2 * wt / (pi * tbase[i]), np.zeros(tbase.size - i - 1)]
        dt_filt[i] = -np.dot(F, hziwc)

    # Interpolate result
    fhz = interp1d(tbase[::-1], dt_filt[::-1])
    hz_out = fhz(t)

    return hz_out, np.r_[tbase[-1], dt_filt[-1]]


def transFiltImpulse(hz, wt, tbase, omega_int, t, tol=1e-12):
    """
        Compute Impulse responses by Fast Hankel Transform (FHT) with cosine filters
    """
    # Define the filter coeffs
    # Generate time base
    # Determine required frequencies
    # Calculate the frequency domain data

    hzr = -Utils.mkvc(hz.real)

    # Clean the low frequency results
    idKeep = [idx for idx in range(len(hz)) if abs(hz.imag)[idx] > tol]
    idKeep = np.array(idKeep)

    idRemove = [idx for idx in range(len(hz)) if abs(hz.imag)[idx] < tol]
    idRemove = np.array(idRemove)

    hzrc = hzr.copy()
    # if idRemove.size != 0:
    #     lastKeep = idKeep.max()

    #     idRemove[idRemove > lastKeep] = 0
    #     lastRemove = idRemove.max()

    #     hzrc[:lastRemove] = hzr[lastRemove+1]

    # Apply filter
    dt_filt = np.zeros_like(tbase)
    for i in np.arange(tbase.size):
        F = np.r_[np.zeros(i), 2 * wt / (pi * tbase[i]), np.zeros(tbase.size - i - 1)]
        dt_filt[i] = -np.dot(F, hzrc)

    # Interpolate result
    fhz = interp1d(tbase[::-1], dt_filt[::-1])
    hz_out = fhz(t)

    return hz_out


def transFiltImpulseInterp(hz, wt, tbase, omega, omega_int, t, tol=1e-12):
    """
        Compute Impulse responses by Fast Hankel Transform (FHT) with cosine filters
    """
    # Define the filter coeffs
    # Generate time base
    # Determine required frequencies
    # Calculate the frequency domain data

    hzr = -Utils.mkvc(hz.real)

    # Clean the low frequency results
    # idKeep = [idx for idx in range(len(hz)) if abs(hz.imag)[idx] > tol]
    # idKeep = np.array(idKeep)

    # idRemove = [idx for idx in range(len(hz)) if abs(hz.imag)[idx] < tol]
    # idRemove = np.array(idRemove)

    hzrint = interp1d(omega, hzr, kind="quadratic")
    hzrc = np.zeros(omega_int.shape)
    ind = (omega_int > omega.min()) & (omega_int < omega.max())

    hzrc[ind] = hzrint(omega_int[ind])

    # if idRemove.size != 0:
    #     lastKeep = idKeep.max()

    #     idRemove[idRemove > lastKeep] = 0
    #     lastRemove = idRemove.max()

    #     hzrc[:lastRemove] = hzr[lastRemove+1]

    # Apply filter
    dt_filt = np.zeros_like(tbase)
    for i in np.arange(tbase.size):
        F = np.r_[np.zeros(i), 2 * wt / (pi * tbase[i]), np.zeros(tbase.size - i - 1)]
        dt_filt[i] = -np.dot(F, hzrc)

    # Interpolate result
    fhz = interp1d(tbase[::-1], dt_filt[::-1])
    hz_out = fhz(t)

    return hz_out


def LoadWeights():
    """
    """
    YBASE = np.array(
        [
            8.9170998013274418e-14,
            9.8549193740052245e-14,
            1.0891370292130841e-13,
            1.2036825704856076e-13,
            1.3302749714952345e-13,
            1.4701812115404443e-13,
            1.6248015192957209e-13,
            1.7956833867707590e-13,
            1.9845370571306282e-13,
            2.1932526413842005e-13,
            2.4239190352504162e-13,
            2.6788448255287407e-13,
            2.9605813952117967e-13,
            3.2719484585839035e-13,
            3.6160622818693727e-13,
            3.9963668718722961e-13,
            4.4166684447542096e-13,
            4.8811735199247527e-13,
            5.3945310203017795e-13,
            5.9618788002944785e-13,
            6.5888950671771899e-13,
            7.2818552104963217e-13,
            8.0476946082781583e-13,
            8.8940780386232124e-13,
            9.8294763913816720e-13,
            1.0863251447666186e-12,
            1.2005749575703847e-12,
            1.3268405280766938e-12,
            1.4663855645544968e-12,
            1.6206066806315702e-12,
            1.7910473730731204e-12,
            1.9794134696161976e-12,
            2.1875902014670364e-12,
            2.4176610713286156e-12,
            2.6719287057960000e-12,
            2.9529379008172425e-12,
            3.2635010908665675e-12,
            3.6067264967338821e-12,
            3.9860492336431492e-12,
            4.4052656910401313e-12,
            4.8685715281339743e-12,
            5.3806036654647833e-12,
            5.9464866927629101e-12,
            6.5718841575654077e-12,
            7.2630552479033658e-12,
            8.0269174363595134e-12,
            8.8711157124588670e-12,
            9.8040990962934688e-12,
            1.0835205199155280e-11,
            1.1974753677488471e-11,
            1.3234149515479672e-11,
            1.4625997169973058e-11,
            1.6164226720110948e-11,
            1.7864233284247930e-11,
            1.9743031099469828e-11,
            2.1819423805797136e-11,
            2.4114192639334462e-11,
            2.6650304417866294e-11,
            2.9453141400488783e-11,
            3.2550755321790057e-11,
            3.5974148143038491e-11,
            3.9757582330231206e-11,
            4.3938923764369768e-11,
            4.8560020715924429e-11,
            5.3667122676390673e-11,
            5.9311343238745088e-11,
            6.5549171659463761e-11,
            7.2443038221987792e-11,
            8.0061939059983487e-11,
            8.8482126693838504e-11,
            9.7787873191515278e-11,
            1.0807231359173195e-10,
            1.1943837803073370e-10,
            1.3199982190169224e-10,
            1.4588236435691521e-10,
            1.6122494654737812e-10,
            1.7818112219246310e-10,
            1.9692059439719362e-10,
            2.1763091409794871e-10,
            2.4051935713527238e-10,
            2.6581499874015355e-10,
            2.9377100619593265e-10,
            3.2466717262156566e-10,
            3.5881271723520049e-10,
            3.9654938012404428e-10,
            4.3825484249401899e-10,
            4.8434650663021337e-10,
            5.3528567339924565e-10,
            5.9158215910338558e-10,
            6.5379939789346253e-10,
            7.2256008080722362e-10,
            7.9855238787053352e-10,
            8.8253687563437822e-10,
            9.7535408908045947e-10,
            1.0779329740778886e-09,
            1.1913001745856734e-09,
            1.3165903076505279e-09,
            1.4550573190356333e-09,
            1.6080870331313015e-09,
            1.7772110227512649e-09,
            1.9641219376281762e-09,
            2.1706904450210516e-09,
            2.3989839519819520e-09,
            2.6512872966606394e-09,
            2.9301256157327411e-09,
            3.2382896168163261e-09,
            3.5788635088117363e-09,
            3.9552558697009004e-09,
            4.3712337607414382e-09,
            4.8309604284818817e-09,
            5.3390369719324459e-09,
            5.9005483919104070e-09,
            6.5211144834374116e-09,
            7.2069460805369272e-09,
            7.9649072163486862e-09,
            8.8025838206794284e-09,
            9.7283596425381269e-09,
            1.0751500157513941e-08,
            1.1882245299770154e-08,
            1.3131911946747030e-08,
            1.4513007182274982e-08,
            1.6039353471673310e-08,
            1.7726227001629019e-08,
            1.9590510569407677e-08,
            2.1650862551562963e-08,
            2.3927903643240499e-08,
            2.6444423237025737e-08,
            2.9225607506844726e-08,
            3.2299291479658129e-08,
            3.5696237617766722e-08,
            3.9450443699873716e-08,
            4.3599483082281089e-08,
            4.8184880745668262e-08,
            5.3252528891055793e-08,
            5.8853146244378082e-08,
            6.5042785666539686e-08,
            7.1883395149287240e-08,
            7.9443437811532343e-08,
            8.7798577101256825e-08,
            9.7032434060731539e-08,
            1.0723742423401342e-07,
            1.1851568259277232e-07,
            1.3098008573741623e-07,
            1.4475538160404734e-07,
            1.5997943798373573e-07,
            1.7680462234971138e-07,
            1.9539932680224871e-07,
            2.1594965339340473e-07,
            2.3866127669890702e-07,
            2.6376150227843728e-07,
            2.9150154162607256e-07,
            3.2215902637935327e-07,
            3.5604078695002664e-07,
            3.9348592338593700e-07,
            4.3486919919827997e-07,
            4.8060479212078480e-07,
            5.3115043933968360e-07,
            5.8701201868132178e-07,
            6.4874861160747570e-07,
            7.1697809869053571e-07,
            7.9238334356995170e-07,
            8.7571902728105488e-07,
            9.6781920135651654e-07,
            1.0696056352944216e-06,
            1.1820970419372223e-06,
            1.3064192730922673e-06,
            1.4438165874351013e-06,
            1.5956641034684996e-06,
            1.7634815621706371e-06,
            1.9489485370736003e-06,
            2.1539212439998211e-06,
            2.3804511186939235e-06,
            2.6308053482811660e-06,
            2.9074895620382204e-06,
            3.2132729085731429e-06,
            3.5512157703953872e-06,
            3.9247003932525885e-06,
            4.3374647367828189e-06,
            4.7936398852710157e-06,
            5.2977913929290109e-06,
            5.8549649774966190e-06,
            6.4707370194807025e-06,
            7.1512703724455683e-06,
            7.9033760429228480e-06,
            8.7345813572541237e-06,
            9.6532052976029776e-06,
            1.0668441761124589e-05,
            1.1790451575578641e-05,
            1.3030464192308714e-05,
            1.4400890074365673e-05,
            1.5915444904593194e-05,
            1.7589286856791647e-05,
            1.9439168303816350e-05,
            2.1483603480955747e-05,
            2.3743053782621043e-05,
            2.6240132546858776e-05,
            2.8999831377238596e-05,
            3.2049770267221755e-05,
            3.5420474030339066e-05,
            3.9145677802784465e-05,
            4.3262664675996812e-05,
            4.7812638838370288e-05,
            5.2841137960621061e-05,
            5.8398488952101537e-05,
            6.4540311649424626e-05,
            7.1328075478483034e-05,
            7.8829714661124184e-05,
            8.7120308123675967e-05,
            9.6282830912076270e-05,
            1.0640898463402168e-04,
            1.1760011523947924e-04,
            1.2996822732501724e-04,
            1.4363710511345378e-04,
            1.5874355132796403e-04,
            1.7543875635971472e-04,
            1.9388981143211579e-04,
            2.1428138090594562e-04,
            2.3681755046234149e-04,
            2.6172386966089197e-04,
            2.8924960931543914e-04,
            3.1967025628016629e-04,
            3.5329027061462893e-04,
            3.9044613272236350e-04,
            4.3150971095986065e-04,
            4.7689198342006658e-04,
            5.2704715113927156e-04,
            5.8247718389374341e-04,
            6.4373684408196633e-04,
            7.1143923897318674e-04,
            7.8626195689103691e-04,
            8.6895384874522254e-04,
            9.6034252278312515e-04,
            1.0613426275713101e-03,
            1.1729650061058051e-03,
            1.2963268126685603e-03,
            1.4326626936829910e-03,
            1.5833371444703617e-03,
            1.7498581655775839e-03,
            1.9338923553535471e-03,
            2.1372815898255564e-03,
            2.3620614568136901e-03,
            2.6104816287778878e-03,
            2.8850283782960702e-03,
            3.1884494615157647e-03,
            3.5237816186211822e-03,
            3.8943809665496639e-03,
            4.3039565881380203e-03,
            4.7566076538702283e-03,
            5.2568644477534125e-03,
            5.8097337079228714e-03,
            6.4207487357601564e-03,
            7.0960247750331065e-03,
            7.8423202153108801e-03,
            8.6671042321983371e-03,
            9.5786315413559676e-03,
            1.0586025014468731e-02,
            1.1699366984012178e-02,
            1.2929800150624660e-02,
            1.4289639103000504e-02,
            1.5792493566432742e-02,
            1.7453404613518231e-02,
            1.9288995200267688e-02,
            2.1317636534236600e-02,
            2.3559631939745231e-02,
            2.6037420060372587e-02,
            2.8775799432443259e-02,
            3.1802176677114019e-02,
            3.5146840795050052e-02,
            3.8843266308924096e-02,
            4.2928448287690518e-02,
            4.7443272605669898e-02,
            5.2432925142121424e-02,
            5.7947344016710048e-02,
            6.4041719386992838e-02,
            7.0777045810065886e-02,
            7.8220732696592687e-02,
            8.6447278966843177e-02,
            9.5539018660927705e-02,
            1.0558694496554391e-01,
            1.1669162090437306e-01,
            1.2896418580662142e-01,
            1.4252746762678220e-01,
            1.5751721224808804e-01,
            1.7408344207293611e-01,
            1.9239195749751564e-01,
            2.1262599629790035e-01,
            2.3498806753529980e-01,
            2.5970197833480957e-01,
            2.8701507382234348e-01,
            3.1720071263778915e-01,
            3.5056100280015512e-01,
            3.8742982530616715e-01,
            4.2817617572350458e-01,
            4.7320785722246539e-01,
            5.2297556200716211e-01,
            5.7797738199458315e-01,
            6.3876379388591276e-01,
            7.0594316852237804e-01,
            7.8018785966510817e-01,
            8.6224093313756223e-01,
            9.5292360367804285e-01,
            1.0531434539328173e00,
            1.1639035178482904e00,
            1.2863123193718711e00,
            1.4215949669322265e00,
            1.5711054147362089e00,
            1.7363400135976372e00,
            1.9189524869191834e00,
            2.1207704817120212e00,
            2.3438138603014083e00,
            2.5903149157877352e00,
            2.8627407135861755e00,
            3.1638177826465683e00,
            3.4965594034715681e00,
            3.8642957660407120e00,
            4.2707072994710522e00,
            4.7198615069887930e00,
            5.2162536748687147e00,
            5.7648518627701284e00,
            6.3711466257477705e00,
            7.0412059655722290e00,
            7.7817360613311877e00,
            8.6001483871237632e00,
            9.5046338885843706e00,
            1.0504244960619703e01,
            1.1608986046819572e01,
            1.2829913767290970e01,
            1.4179247577028352e01,
            1.5670492062326328e01,
            1.7318572099218340e01,
            1.9139982226652432e01,
            2.1152951729381048e01,
            2.3377627082769912e01,
            2.5836273585494951e01,
            2.8553498198135060e01,
            3.1556495817904278e01,
            3.4875321454323611e01,
            3.8543191029858157e01,
            4.2596813816033411e01,
            4.7076759832163077e01,
            5.2027865883738443e01,
            5.7499684304247886e01,
            6.3546978891585546e01,
            7.0230273002547406e01,
            7.7616455290928698e01,
            8.5779449151653139e01,
            9.4800952570955843e01,
            1.0477125578728921e02,
            1.1579014494637693e02,
            1.2796790079449971e02,
            1.4142640240527066e02,
            1.5630034698636896e02,
            1.7273859797446769e02,
            1.9090567491054267e02,
            2.1098340000673559e02,
            2.3317271788416559e02,
            2.5769570669423729e02,
            2.8479780075142304e02,
            3.1475024692237560e02,
            3.4785281935573863e02,
            3.8443681972258412e02,
            4.2486839299489054e02,
            4.6955219194748827e02,
            5.1893542705903837e02,
            5.7351234234481581e02,
            6.3382916191693528e02,
            7.0048955677885772e02,
            7.7416068656769369e02,
            8.5557987671209173e02,
            9.4556199783295187e02,
            1.0450076212424869e03,
            1.1549120321646080e03,
            1.2763751908839718e03,
            1.4106127415182191e03,
            1.5589681785928965e03,
            1.7229262931862318e03,
            1.9041280332173003e03,
            2.1043869266043412e03,
            2.3257072316617105e03,
            2.5703039963907459e03,
            2.8406252274246667e03,
            3.1393763905017645e03,
            3.4695474876758481e03,
            3.8344429822617740e03,
            4.2377148710149695e03,
            4.6833992345424385e03,
            5.1759566317540530e03,
            5.7203167426353639e03,
            6.3219277061418234e03,
            6.9868106470046323e03,
            7.7216199371708199e03,
            8.5337097949943000e03,
            9.4312078887249972e03,
            1.0423096680944496e04,
            1.1519303328070666e04,
            1.2730799034675721e04,
            1.4069708856989137e04,
            1.5549433054535755e04,
            1.7184781204437102e04,
            1.8992120420636886e04,
            2.0989539161478522e04,
            2.3197028265075980e04,
            2.5636681024340771e04,
            2.8332914304083228e04,
            3.1312712913202311e04,
            3.4605899677722984e04,
            3.8245433917662871e04,
            4.2267741314984989e04,
            4.6713078474065944e04,
            5.1625935823323234e04,
            5.7055482890376610e04,
            6.3056060407206925e04,
            6.9687724170466376e04,
            7.7016846100076829e04,
            8.5116778511712779e04,
            9.4068588251431182e04,
            1.0396186803991429e05,
            1.1489563314653141e05,
            1.2697931236743495e05,
            1.4033384322573253e05,
            1.5509288235486683e05,
            1.7140414317912661e05,
            1.8943087427924512e05,
            2.0935349323906592e05,
            2.3137139232536239e05,
            2.5570493407266162e05,
            2.8259765674555639e05,
            3.1231871175151330e05,
            3.4516555739862355e05,
            3.8146693595832947e05,
            4.2158616382857127e05,
            4.6592476772641251e05,
            5.1492650330238225e05,
            5.6908179639617680e05,
            6.2893265138330159e05,
            6.9507807573703467e05,
            7.6818007509655319e05,
            8.4897027884187771e05,
            9.3825726248661662e05,
            1.0369346401734781e06,
            1.1459900082649642e06,
            1.2665148295397097e06,
            1.3997153569188234e06,
            1.5469247060505589e06,
            1.7096161975797976e06,
            1.8894181026362628e06,
            2.0881299391192668e06,
            2.3077404818776865e06,
            2.5504476670371005e06,
            2.8186805896832864e06,
            3.1151238150622859e06,
            3.4427442466117009e06,
            3.8048208197275074e06,
            4.2049773184515880e06,
            4.6472186435204167e06,
            5.1359708947577253e06,
            5.6761256689692009e06,
            6.2730890166874416e06,
            6.9328355477427216e06,
            7.6619682271663100e06,
            8.4677844598838333e06,
            9.3583491255965196e06,
            1.0342575294807941e07,
            1.1430313433829404e07,
            1.2632449991557652e07,
            1.3961016354714479e07,
            1.5429309262008933e07,
            1.7052023882367507e07,
            1.8845400889123969e07,
            2.0827389002136763e07,
            2.3017824624610133e07,
            2.5438630372484632e07,
            2.8114034483345896e07,
            3.1070813300769802e07,
            3.4338559260968812e07,
            3.7949977063839935e07,
            4.1941210992593758e07,
            4.6352206657889292e07,
            5.1227110786931656e07,
            5.6614713058756173e07,
            6.2568934407734923e07,
            6.9149366682411388e07,
            7.6421869060750201e07,
            8.4459227190926239e07,
            9.3341881654555663e07,
            1.0315873304307374e08,
            1.1400803170473446e08,
            1.2599836106711893e08,
            1.3924972437657478e08,
            1.5389474573104006e08,
            1.7007999742659190e08,
            1.8796746690225038e08,
            2.0773617796471399e08,
            2.2958398251878911e08,
            2.5372954073575363e08,
            2.8041450947784531e08,
            3.0990596088136274e08,
            3.4249905530437142e08,
            3.7851999539077419e08,
            4.1832929081601185e08,
            4.6232536638906646e08,
            5.1094854962186480e08,
            5.6468547767501700e08,
            6.2407396778608418e08,
            6.8970839992525887e08,
            7.6224566554988432e08,
            8.4241174199494874e08,
            9.3100895829826319e08,
            1.0289240251791439e09,
            1.1371369095373254e09,
            1.2567306422910707e09,
            1.3889021577146211e09,
            1.5349742727587159e09,
            1.6964089262472496e09,
            1.8748218104523966e09,
            2.0719985414859231e09,
            2.2899125303454008e09,
            2.5307447334747562e09,
            2.7969054805094066e09,
            3.0910585976653914e09,
            3.4161480682074847e09,
            3.7754274968232164e09,
            4.1724926727921586e09,
            4.6113175578536234e09,
            5.0962940589514427e09,
            5.6322759839148350e09,
            6.2246276199985800e09,
            6.8792774214728651e09,
            7.6027773435862408e09,
            8.4023684167359400e09,
            9.2860532171338844e09,
            1.0262675959279177e10,
            1.1342011011829447e10,
            1.2534860722767656e10,
            1.3853163532931507e10,
            1.5310113459941998e10,
            1.6920292148366428e10,
            1.8699814807718300e10,
            2.0666491498890625e10,
            2.2840005383231522e10,
            2.5242109718238716e10,
            2.7896845571472111e10,
            3.0830782431638401e10,
            3.4073284124964363e10,
            3.7656802698239258e10,
            4.1617203209806610e10,
            4.5994122679122765e10,
            5.0831366787370079e10,
            5.6177348299437775e10,
            6.2085571595145073e10,
            6.8615168159057838e10,
            7.5831488388260895e10,
            8.3806755641097107e10,
            9.2620789072812759e10,
            1.0236180249249139e11,
            1.1312728723650484e11,
            1.2502498789457556e11,
            1.3817398065384482e11,
            1.5270586505337646e11,
            1.6876608107657605e11,
            1.8651536476342874e11,
            2.0613135691081287e11,
            2.2781038096130206e11,
            2.5176940787416525e11,
            2.7824822764365344e11,
            3.0751184919785828e11,
            3.3985315269713715e11,
            3.7559582077719836e11,
            4.1509757807371277e11,
            4.5875377145070300e11,
            5.0700132676483929e11,
            5.6032312176626892e11,
            6.1925281890144031e11,
            6.8438020638623755e11,
            7.5635710100467944e11,
            8.3590387171037695e11,
            9.2381664932114575e11,
            1.0209752944638193e12,
            1.1283522035151340e12,
            1.2470220406715007e12,
            1.3781724935494902e12,
            1.5231161599626948e12,
            1.6833036848418264e12,
            1.8603382787767620e12,
            2.0559917634869844e12,
            2.2722223048088804e12,
            2.5111940106775947e12,
            2.7752985902466255e12,
            3.0671792909169141e12,
            3.3897573528452603e12,
            3.7462612456976733e12,
            4.1402589802589175e12,
            4.5756938182836924e12,
            5.0569237379856543e12,
            5.5887650501481416e12,
            6.1765406013813154e12,
            6.8261330469601016e12,
            7.5440437264154141e12,
            8.3374577311253535e12,
            9.2143158151247129e12,
            1.0183393868840340e13,
            1.1254390751152201e13,
            1.2438025358832957e13,
            1.3746143904869607e13,
            1.5191838479344713e13,
            1.6789578079474348e13,
            1.8555353420195434e13,
            2.0506836974615496e13,
            2.2663559846063445e13,
            2.5047107241936324e13,
            2.7681334505709973e13,
            3.0592605869234598e13,
            3.3810058314828449e13,
            3.7365893187990141e13,
            4.1295698479287656e13,
            4.5638805000929469e13,
            5.0438680022752680e13,
            5.5743362307269414e13,
            6.1605942897748391e13,
            6.8085096471220516e13,
            7.5245668574367812e13,
            8.3159324619549984e13,
            9.1905267136338875e13,
            1.0157102845705528e14,
            1.1225334676977153e14,
            1.2405913430661245e14,
            1.3710654735730897e14,
            1.5152616881705944e14,
            1.6746231510403516e14,
            1.8507448052659994e14,
            2.0453893355595603e14,
            2.2605048098024984e14,
            2.4982441759638447e14,
            2.7609868095271022e14,
            3.0513623270798212e14,
            3.3722769044002506e14,
            3.7269423624413281e14,
            4.1189083123143062e14,
            4.5520976809898188e14,
            5.0308459732695450e14,
            5.5599446629754788e14,
            6.1446891476304075e14,
            6.7909317465761662e14,
            7.5051402729526438e14,
            8.2944627657455900e14,
            9.1667990297633300e14,
            1.0130879699538496e15,
            1.1196353618452902e15,
            1.2373884407605195e15,
            1.3675257190914975e15,
            1.5113496544604105e15,
            1.6702996851533248e15,
            1.8459666365023652e15,
            2.0401086424003345e15,
            2.2546687412956410e15,
            2.4917943227741685e15,
            2.7538586193560145e15,
            3.0434844586042220e15,
            3.3635705132645935e15,
            3.7173203121568085e15,
            4.1082743021675935e15,
            4.5403452822331500e15,
            5.0178575639460460e15,
            5.5455902507190850e15,
            6.1288250686585730e15,
            6.7733992278544400e15,
            7.4857638431407750e15,
            8.2730484990213790e15,
            9.1431326049478160e15,
            1.0104724255097566e16,
            1.1167447381907442e16,
            1.2341938075624136e16,
            1.3639951033870320e16,
            1.5074477206609342e16,
            1.6659873813938872e16,
            1.8412008037975264e16,
            2.0348415826945328e16,
            2.2488477400850208e16,
            2.4853611215221080e16,
            2.7467488324221096e16,
            3.0356269288511564e16,
            3.3548865998935916e16,
            3.7077231036440888e16,
            4.0976677464246272e16,
            4.5286232252850760e16,
            5.0049026875070080e16,
            5.5312728980313968e16,
            6.1130019468443072e16,
            6.7559119737921448e16,
            7.4664374385141264e16,
            8.2516895186770448e16,
            9.1195272810315088e16,
            1.0078636337593507e17,
            1.1138615774168800e17,
            1.2310074221230024e17,
            1.3604736028656150e17,
            1.5035558606966758e17,
            1.6616862109441658e17,
            1.8364472753028080e17,
            2.0295881212439261e17,
            2.2430417672705789e17,
            2.4789445292164490e17,
            2.7396574012127472e17,
            3.0277896853110349e17,
            3.3462251062551731e17,
            3.6981506727678112e17,
            4.0870885742048762e17,
            4.5169314318104928e17,
            4.9919812573787520e17,
            5.5169925092337018e17,
            6.0972196764462810e17,
            6.7384698675270400e17,
            7.4471609299199475e17,
            8.2303856819767232e17,
            9.0959829002668813e17,
            1.0052615772688342e18,
            1.1109858602563712e18,
            1.2278292631485970e18,
            1.3569611939940810e18,
            1.4996740485594657e18,
            1.6573961450606881e18,
            1.8317060192517601e18,
            2.0243482229411579e18,
            2.2372507840526853e18,
            2.4725445029769687e18,
            2.7325842783379528e18,
            3.0199726756098365e18,
            3.3375859744670935e18,
            3.6886029555582029e18,
            4.0765367148108068e18,
            4.5052698236765440e18,
            4.9790931872111176e18,
            5.5027489888943135e18,
            6.0814781519961702e18,
            6.7210727924986010e18,
            7.4279341885389363e18,
            8.2091368465530675e18,
            9.0724993053136814e18,
            1.0026662386494198e19,
            1.1081175674916358e19,
            1.2246593094004847e19,
            1.3534578533000223e19,
            1.4958022583082809e19,
            1.6531171550741899e19,
            1.8269770039599454e19,
            2.0191218527695090e19,
            2.2314747517318812e19,
            2.4661610000341512e19,
            2.7255294165301002e19,
            3.0121758475087553e19,
            3.3289691467965432e19,
            3.6790798882106413e19,
            4.0660120977274061e19,
            4.4936383229520880e19,
            4.9662383908768727e19,
            5.4885422418279211e19,
            6.0657772682979369e19,
            6.7037206324472250e19,
            7.4087570858843619e19,
            8.1879428704062800e19,
            9.0490763392378634e19,
            1.0000776005572131e20,
            1.1052566799547061e20,
            1.2214975396947848e20,
            1.3499635573716302e20,
            1.4919404640690720e20,
            1.6488492123894242e20,
            1.8222601978247286e20,
            2.0139089758026668e20,
            2.2257136317086207e20,
            2.4597939777289005e20,
            2.7184927686435983e20,
            3.0043991489038549e20,
            3.3203745656597683e20,
            3.6695814070852361e20,
            4.0555146526217175e20,
            4.4820368519071852e20,
            4.9534167824711496e20,
            5.4743721730949612e20,
            6.0501169204271369e20,
            6.6864132714134700e20,
            7.3896294938012195e20,
            8.1668036119031775e20,
            9.0257138455105503e20,
            9.9749564569309793e20,
            1.1024031785271021e21,
            1.2183439329023096e21,
            1.3464782828575407e21,
            1.4880886400345899e21,
            1.6445922884849697e21,
            1.8175555693250643e21,
            2.0087095572044880e21,
            2.2199673854830119e21,
            2.4534433935122555e21,
            2.7114742876545719e21,
            2.9966425278257163e21,
            3.3118021736216768e21,
            3.6601074487063940e21,
            4.0450443093423623e21,
            4.4704653330125727e21,
            4.9406282763108614e21,
        ]
    )
    #
    # Next, setup the weights for 0-th order.
    #
    WT0 = np.array(
        [
            0.21035620538389819885e-28,
            -0.12644693616088940552e-13,
            0.46157312567885668321e-13,
            -0.27987033742576678494e-13,
            0.54657649654108409156e-13,
            -0.26529331099287291499e-13,
            0.56749134340673213135e-13,
            -0.21572768289772080733e-13,
            0.58318460867739760925e-13,
            -0.15465892848687829700e-13,
            0.60573024556529743179e-13,
            -0.85025312590830646706e-14,
            0.63880180611476449908e-13,
            -0.56596576350102877128e-15,
            0.68485006047914070374e-13,
            0.85728977321682762439e-14,
            0.74650681546818133979e-13,
            0.19208372932613381433e-13,
            0.82693454289757706437e-13,
            0.31701165629228998860e-13,
            0.93000040396952081623e-13,
            0.46490696394179916916e-13,
            0.10604419444905640479e-12,
            0.64112165895974571186e-13,
            0.12240608340017008854e-12,
            0.85217767515070225126e-13,
            0.14279579404871719178e-12,
            0.11060266069684630524e-12,
            0.16808202030984049793e-12,
            0.14123670281595459178e-12,
            0.19932710117763080694e-12,
            0.17830320429641190908e-12,
            0.23782981967994220694e-12,
            0.22324626027650970672e-12,
            0.28517768171070337866e-12,
            0.27782855757859949941e-12,
            0.34331077352335570356e-12,
            0.34420197641399489491e-12,
            0.41459976123278392377e-12,
            0.42499381982249688902e-12,
            0.50194116319499513568e-12,
            0.52341213107734220974e-12,
            0.60887371932475596308e-12,
            0.64337432538102752823e-12,
            0.73972052807969328662e-12,
            0.78966429788659195358e-12,
            0.89976265596232089469e-12,
            0.96812431295714248033e-12,
            0.10954511874715580138e-11,
            0.11858893754914550946e-11,
            0.13346662261643230004e-11,
            0.14516734901185850330e-11,
            0.16270332417806469835e-11,
            0.17761192965262209981e-11,
            0.19843094598657728049e-11,
            0.21722251127126358340e-11,
            0.24208558013562638359e-11,
            0.26558665246211218806e-11,
            0.29542133130007581834e-11,
            0.32464334551104291636e-11,
            0.36058072230543221354e-11,
            0.39676082798212998573e-11,
            0.44018068787207206107e-11,
            0.48483162182209038194e-11,
            0.53741760778847699533e-11,
            0.59238861421284167256e-11,
            0.65619559488550109069e-11,
            0.72374683888302430790e-11,
            0.80128318647926483616e-11,
            0.88417664804021047188e-11,
            0.97850472788000974558e-11,
            0.10801152249024990298e-10,
            0.11949741288775620211e-10,
            0.13194249255521529763e-10,
            0.14593803746893311587e-10,
            0.16117088182600741411e-10,
            0.17823362499440759998e-10,
            0.19686960839662181492e-10,
            0.21768042712348459703e-10,
            0.24047127453754399616e-10,
            0.26586169224245549843e-10,
            0.29372566166660639848e-10,
            0.32471120715874496926e-10,
            0.35876995485484108464e-10,
            0.39659090711123993131e-10,
            0.43821451522206353096e-10,
            0.48438566886024388833e-10,
            0.53524764256840081139e-10,
            0.59161909123775483693e-10,
            0.65376353273289546079e-10,
            0.72259490983916676274e-10,
            0.79851856505621554776e-10,
            0.88256972132554113981e-10,
            0.97532219231111761839e-10,
            0.10779639493701439540e-09,
            0.11912700941828918848e-09,
            0.13166195190543400137e-09,
            0.14550289515667290553e-09,
            0.16081145810919829686e-09,
            0.17771842706736448843e-09,
            0.19641479168712890847e-09,
            0.21706652163468219836e-09,
            0.23990084518390418007e-09,
            0.26512635046403231588e-09,
            0.29301487204485379059e-09,
            0.32382671796405940321e-09,
            0.35788852978338560143e-09,
            0.39552347102192978760e-09,
            0.43712543089936042400e-09,
            0.48309404739375797825e-09,
            0.53390563500721587301e-09,
            0.59005295736900699180e-09,
            0.65211327580989620296e-09,
            0.72069283339348376395e-09,
            0.79649244503723412979e-09,
            0.88025670846750671949e-09,
            0.97283758951862838429e-09,
            0.10751484374562240226e-08,
            0.11882260626931209546e-08,
            0.13131897062580609792e-08,
            0.14513021636655630472e-08,
            0.16039339435115999074e-08,
            0.17726240632935599375e-08,
            0.19590497332198700268e-08,
            0.21650875406672578739e-08,
            0.23927891159868709863e-08,
            0.26444435360147638239e-08,
            0.29225595734392348364e-08,
            0.32299302912485969586e-08,
            0.35696226515762041770e-08,
            0.39450454481729238111e-08,
            0.43599472612559631445e-08,
            0.48184890913636877388e-08,
            0.53252519017629532640e-08,
            0.58853155833436851597e-08,
            0.65042776355473761329e-08,
            0.71883404192425699700e-08,
            0.79443429030829021263e-08,
            0.87798585629591097164e-08,
            0.97032425780220928695e-08,
            0.10723743227689399320e-07,
            0.11851567478400949647e-07,
            0.13098009332253010086e-07,
            0.14475537424021579590e-07,
            0.15997944513720238914e-07,
            0.17680461540553181382e-07,
            0.19539933354871179973e-07,
            0.21594964684505000780e-07,
            0.23866128306162058448e-07,
            0.26376149610345229076e-07,
            0.29150154762698309478e-07,
            0.32215902055657983021e-07,
            0.35604079260985092872e-07,
            0.39348591789544360374e-07,
            0.43486920453657868292e-07,
            0.48060478694380069026e-07,
            0.53115044437493142888e-07,
            0.58701201380017479006e-07,
            0.64874861635712681668e-07,
            0.71697809408859342572e-07,
            0.79238334805050121668e-07,
            0.87571902294266750180e-07,
            0.96781920558355529704e-07,
            0.10696056312048640378e-06,
            0.11820970459254829996e-06,
            0.13064192692376749733e-06,
            0.14438165911984879199e-06,
            0.15956640998357920017e-06,
            0.17634815657222329376e-06,
            0.19489485336503760544e-06,
            0.21539212473518470874e-06,
            0.23804511154682969141e-06,
            0.26308053514448537707e-06,
            0.29074895589985508713e-06,
            0.32132729115584617374e-06,
            0.35512157675298091583e-06,
            0.39247003960676882480e-06,
            0.43374647340783931675e-06,
            0.47936398879211592319e-06,
            0.52977913903701991577e-06,
            0.58549649799821895163e-06,
            0.64707370170465826468e-06,
            0.71512703747583724585e-06,
            0.79033760405820345763e-06,
            0.87345813593705873095e-06,
            0.96532052953043588492e-06,
            0.10668441762993300587e-05,
            0.11790451573234470162e-05,
            0.13030464193827640161e-05,
            0.14400890071820729325e-05,
            0.15915444905568059534e-05,
            0.17589286853766969218e-05,
            0.19439168303884759854e-05,
            0.21483603476946079820e-05,
            0.23743053781113989137e-05,
            0.26240132540945578888e-05,
            0.28999831372928098690e-05,
            0.32049770257732388059e-05,
            0.35420474020978068237e-05,
            0.39145677786671023699e-05,
            0.43262664657480666650e-05,
            0.47812638810077579977e-05,
            0.52841137925459449193e-05,
            0.58398488901504216978e-05,
            0.64540311583956847791e-05,
            0.71328075387126483640e-05,
            0.78829714540444842174e-05,
            0.87120307957926337232e-05,
            0.96282830690788715692e-05,
            0.10640898433258409191e-04,
            0.11760011483484689510e-04,
            0.12996822677619209683e-04,
            0.14363710437469910021e-04,
            0.15874355032820411062e-04,
            0.17543875501207770228e-04,
            0.19388980961051210830e-04,
            0.21428137844875080257e-04,
            0.23681754714302638776e-04,
            0.26172386518181301401e-04,
            0.28924960326686671307e-04,
            0.31967024811679323171e-04,
            0.35329025959273332330e-04,
            0.39044611784553647914e-04,
            0.43150969087563300697e-04,
            0.47689195631023510063e-04,
            0.52704711454198507913e-04,
            0.58247713449355698436e-04,
            0.64373677739542765810e-04,
            0.71143914895687183756e-04,
            0.78626183537770216584e-04,
            0.86895368472094227166e-04,
            0.96034230136811580777e-04,
            0.10613423286950590218e-03,
            0.11729646026571620215e-03,
            0.12963262680752149015e-03,
            0.14326619585462989793e-03,
            0.15833361521503999387e-03,
            0.17498568260644121238e-03,
            0.19338905472215898854e-03,
            0.21372791490647819607e-03,
            0.23620581621746139576e-03,
            0.26104771814107751186e-03,
            0.28850223750643329973e-03,
            0.31884413578705622606e-03,
            0.35237706800135917308e-03,
            0.38943662007281590462e-03,
            0.43039366566958368833e-03,
            0.47565807487700027594e-03,
            0.52568281303167982161e-03,
            0.58096846836193664822e-03,
            0.64206825611437718954e-03,
            0.70959354469423893946e-03,
            0.78421996374912559657e-03,
            0.86669414659246905939e-03,
            0.95784118347590099622e-03,
            0.10585728435434579864e-02,
            0.11698966653840010659e-02,
            0.12929259749833240008e-02,
            0.14288909657393959393e-02,
            0.15791508896276298946e-02,
            0.17452075486231769744e-02,
            0.19287201026563420645e-02,
            0.21315214730933669356e-02,
            0.23556362776736309780e-02,
            0.26033007312297349808e-02,
            0.28769842725422829777e-02,
            0.31794136300501061286e-02,
            0.35135987233342810820e-02,
            0.38828616256736492134e-02,
            0.42908672540316900729e-02,
            0.47416579734868603835e-02,
            0.52396893384671194144e-02,
            0.57898709851889693795e-02,
            0.63976070720003401851e-02,
            0.70688437831027441105e-02,
            0.78101127966257229834e-02,
            0.86285849757206656285e-02,
            0.95321125348557418644e-02,
            0.10529286971052299882e-01,
            0.11629470426845669312e-01,
            0.12842853010275019979e-01,
            0.14180454004084899408e-01,
            0.15654168426849128515e-01,
            0.17276700258609969246e-01,
            0.19061578784670331344e-01,
            0.21022951738450229575e-01,
            0.23175536220878610594e-01,
            0.25534136813168170632e-01,
            0.28113470567056909888e-01,
            0.30927161344728969217e-01,
            0.33987341082207328524e-01,
            0.37302669043183162012e-01,
            0.40877565846017260842e-01,
            0.44708454582277110112e-01,
            0.48782456555642082774e-01,
            0.53070463888373879680e-01,
            0.57525214796560822372e-01,
            0.62068889249144526543e-01,
            0.66590986905823254527e-01,
            0.70926870977039285782e-01,
            0.74857621553299974471e-01,
            0.78074664211727651253e-01,
            0.80188872113380424422e-01,
            0.80676406709186576638e-01,
            0.78917673064227769619e-01,
            0.74124063016304961304e-01,
            0.65458647531413310938e-01,
            0.51957717257333460581e-01,
            0.32847972741848592559e-01,
            0.74970763258312700383e-02,
            -0.23866128698945488634e-01,
            -0.60174943784761181220e-01,
            -0.98178997988506697125e-01,
            -0.13281477972726110637e00,
            -0.15546285697725620301e00,
            -0.15639821579874499391e00,
            -0.12430498665290500016e00,
            -0.54868159863436967438e-01,
            0.46862558991703072431e-01,
            0.15112182958062059246e00,
            0.21193155344105990556e00,
            0.16951341358877961008e00,
            0.13861974203314799889e-01,
            -0.18693504518381348634e00,
            -0.24558896069253360883e00,
            -0.53092694018998139172e-01,
            0.25199984157985949595e00,
            0.19682248760574280744e00,
            -0.20143336189691599114e00,
            -0.24584508272586030886e00,
            0.34335593140766357267e00,
            -0.47700665106262918336e-01,
            -0.20966284735074519618e00,
            0.25090851237942479734e00,
            -0.16764126613351920669e00,
            0.74951997868967626393e-01,
            -0.16951351027511458308e-01,
            -0.88285242013749955919e-02,
            0.16167873561835501006e-01,
            -0.15564117559925290737e-01,
            0.12513111252163700016e-01,
            -0.92993152469780984010e-02,
            0.66505259967794931597e-02,
            -0.46677074086718837662e-02,
            0.32488855455568548675e-02,
            -0.22555291085446750252e-02,
            0.15668752111009150857e-02,
            -0.10911324863680910667e-02,
            0.76253830490409491537e-03,
            -0.53525405778204085232e-03,
            0.37771045672146697511e-03,
            -0.26825282115352968703e-03,
            0.19202652749967551236e-03,
            -0.13882137232089440237e-03,
            0.10159989178100049974e-03,
            -0.75497250626312589804e-04,
            0.57141729200529359656e-04,
            -0.44191609676806338589e-04,
            0.35017761995785832974e-04,
            -0.28484969541399899885e-04,
            0.23801048173241150913e-04,
            -0.20412749240321660898e-04,
            0.17933634897918600856e-04,
            -0.16093677807779768610e-04,
            0.14703946920177649808e-04,
            -0.13632085400878389976e-04,
            0.12785408514801159958e-04,
            -0.12099103232765920005e-04,
            0.11527807463720849231e-04,
            -0.11039642499677090445e-04,
            0.10612147244787119201e-04,
            -0.10229546722558040478e-04,
            0.98808019172690629615e-05,
            -0.95581362683731644660e-05,
            0.92559894628748428139e-05,
            -0.89703680398349605195e-05,
            0.86984392262176360162e-05,
            -0.84381998993781744882e-05,
            0.81881850397080960968e-05,
            -0.79472759767526677708e-05,
            0.77146204892770847038e-05,
            -0.74895903346701209313e-05,
            0.72717128455631313132e-05,
            -0.70605952027200397989e-05,
            0.68558909586342899260e-05,
            -0.66573078263653524112e-05,
            0.64646106359474119065e-05,
            -0.62775967052488337714e-05,
            0.60960690045641779074e-05,
            -0.59198350149128057049e-05,
            0.57487213891732378171e-05,
            -0.55825764283683409239e-05,
            0.54212557290731716876e-05,
            -0.52646114832336199488e-05,
            0.51124978277574427215e-05,
            -0.49647806501954759280e-05,
            0.48213365261592663842e-05,
            -0.46820435304229967753e-05,
            0.45467775878200409083e-05,
            -0.44154180000760036035e-05,
            0.42878525031129105819e-05,
            -0.41639746681098469138e-05,
            0.40436784078298272207e-05,
            -0.39268575265170754164e-05,
            0.38134098513603608910e-05,
            -0.37032392398348510301e-05,
            0.35962529271829669603e-05,
            -0.34923586009061261710e-05,
            0.33914651799141980567e-05,
            -0.32934854312419398030e-05,
            0.31983363564463991755e-05,
            -0.31059371103484061739e-05,
            0.30162076790696608685e-05,
            -0.29290699115622908088e-05,
            0.28444489540948809787e-05,
            -0.27622729358204111026e-05,
            0.26824715684158770123e-05,
            -0.26049757091405561401e-05,
            0.25297182336640620443e-05,
            -0.24566346909850541750e-05,
            0.23856627938558707926e-05,
            -0.23167415846242317899e-05,
            0.22498114158806991699e-05,
            -0.21848145526023768863e-05,
            0.21216953789870668849e-05,
            -0.20603999269499001719e-05,
            0.20008754358939798722e-05,
            -0.19430704833474811314e-05,
            0.18869353395796189952e-05,
            -0.18324219538318030139e-05,
            0.17794835990689859392e-05,
            -0.17280746718615620423e-05,
            0.16781508408187099176e-05,
            -0.16296692208678659180e-05,
            0.15825882749397249515e-05,
            -0.15368675777174579453e-05,
            0.14924677441823879966e-05,
            -0.14493505440316709914e-05,
            0.14074789632159439540e-05,
            -0.13668170916979060623e-05,
            0.13273299804037580486e-05,
            -0.12889836283545319635e-05,
            0.12517450527435769451e-05,
            -0.12155822896230419416e-05,
            0.11804642977313919266e-05,
            -0.11463608782321180614e-05,
            0.11132426817551759487e-05,
            -0.10810812413606430626e-05,
            0.10498489463079790445e-05,
            -0.10195189699197359345e-05,
            0.99006522696480728454e-06,
            -0.96146238238294861780e-06,
            0.93368585903962674737e-06,
            -0.90671180391877650539e-06,
            0.88051703794158972103e-06,
            -0.85507903355065637431e-06,
            0.83037591856657379114e-06,
            -0.80638646950736030707e-06,
            0.78309007959684342440e-06,
            -0.76046672516600126341e-06,
            0.73849695280747625374e-06,
            -0.71716187755802708277e-06,
            0.69644316956135183135e-06,
            -0.67632302736954520768e-06,
            0.65678415552512063648e-06,
            -0.63780975556830963862e-06,
            0.61938351997573715339e-06,
            -0.60148961697207375561e-06,
            0.58411266955661727328e-06,
            -0.56723774005096348097e-06,
            0.55085032232163946943e-06,
            -0.53493633333997110944e-06,
            0.51948209868244907623e-06,
            -0.50447433644131324069e-06,
            0.48990014588904722006e-06,
            -0.47574699997612418437e-06,
            0.46200273605516768571e-06,
            -0.44865554313913170035e-06,
            0.43569394953682061395e-06,
            -0.42310681390672771474e-06,
            0.41088331789010578938e-06,
            -0.39901295702377770089e-06,
            0.38748552999527151342e-06,
            -0.37629112895780040410e-06,
            0.36542013202237861199e-06,
            -0.35486319617658597678e-06,
            0.34461124894551899091e-06,
            -0.33465547948734137742e-06,
            0.32498733079349469753e-06,
            -0.31559849314212400870e-06,
            0.30648089749286697872e-06,
            -0.29762670812339488137e-06,
            0.28902831526969758359e-06,
            -0.28067832866533619338e-06,
            0.27256957173628210936e-06,
            -0.26469507560502209121e-06,
            0.25704807272840818962e-06,
            -0.24962199077868072385e-06,
            0.24241044716871418470e-06,
            -0.23540724388966429376e-06,
            0.22860636218196940360e-06,
            -0.22200195709699679458e-06,
            0.21558835236008589946e-06,
            -0.20936003566018609291e-06,
            0.20331165407816460064e-06,
            -0.19743800942055008854e-06,
            0.19173405358761880565e-06,
            -0.18619488421725469657e-06,
            0.18081574059851621267e-06,
            -0.17559199964945570756e-06,
            0.17051917187001359107e-06,
            -0.16559289739464161294e-06,
            0.16080894226704971095e-06,
            -0.15616319488355259655e-06,
            0.15165166247713490835e-06,
            -0.14727046762624959541e-06,
            0.14301584488191908927e-06,
            -0.13888413756276189610e-06,
            0.13487179465891340034e-06,
            -0.13097536777476139733e-06,
            0.12719150812468619494e-06,
            -0.12351696364172060964e-06,
            0.11994857620999400758e-06,
            -0.11648327897320799497e-06,
            0.11311809368621049480e-06,
            -0.10985012813125349456e-06,
            0.10667657363213940060e-06,
            -0.10359470265995370341e-06,
            0.10060186649751670646e-06,
            -0.97695492950586510549e-07,
            0.94873084124749734212e-07,
            -0.92132214283314405060e-07,
            0.89470527774724840543e-07,
            -0.86885737009431156146e-07,
            0.84375620484431720137e-07,
            -0.81938020868743667602e-07,
            0.79570843154679285033e-07,
            -0.77272052863928245642e-07,
            0.75039674297478696595e-07,
            -0.72871788831488877790e-07,
            0.70766533266929795021e-07,
            -0.68722098232649864085e-07,
            0.66736726633350816695e-07,
            -0.64808712137167796518e-07,
            0.62936397705674278282e-07,
            -0.61118174170089166613e-07,
            0.59352478851276259248e-07,
            -0.57637794217708362227e-07,
            0.55972646579199561646e-07,
            -0.54355604818630417116e-07,
            0.52785279162883091201e-07,
            -0.51260319990189909104e-07,
            0.49779416670227040800e-07,
            -0.48341296436233499673e-07,
            0.46944723290471018939e-07,
            -0.45588496942848318593e-07,
            0.44271451780258727462e-07,
            -0.42992455864451837250e-07,
            0.41750409958276353087e-07,
            -0.40544246580819631899e-07,
            0.39372929090650920317e-07,
            -0.38235450795273419860e-07,
            0.37130834085523768675e-07,
            -0.36058129594866006882e-07,
            0.35016415383563508384e-07,
            -0.34004796146768310693e-07,
            0.33022402445156628786e-07,
            -0.32068389957362020693e-07,
            0.31141938754091167181e-07,
            -0.30242252593599009938e-07,
            0.29368558237615870861e-07,
            -0.28520104786762129308e-07,
            0.27696163034962460481e-07,
            -0.26896024842649531045e-07,
            0.26119002528299780677e-07,
            -0.25364428277531399452e-07,
            0.24631653569082881086e-07,
            -0.23920048617305701087e-07,
            0.23229001830889051246e-07,
            -0.22557919287333441164e-07,
            0.21906224222551861542e-07,
            -0.21273356535101449785e-07,
            0.20658772304732728872e-07,
            -0.20061943324939738367e-07,
            0.19482356649058811450e-07,
            -0.18919514149424371449e-07,
            0.18372932089200169455e-07,
            -0.17842140706600119178e-07,
            0.17326683811178871537e-07,
            -0.16826118391794460252e-07,
            0.16340014235853439520e-07,
            -0.15867953559529469194e-07,
            0.15409530648689289756e-07,
            -0.14964351510224189348e-07,
            0.14532033533448680711e-07,
            -0.14112205161252269515e-07,
            0.13704505570744190592e-07,
            -0.13308584363144380637e-07,
            0.12924101262648650824e-07,
            -0.12550725823983400048e-07,
            0.12188137148392460070e-07,
            -0.11836023607829919776e-07,
            0.11494082577135149689e-07,
            -0.11162020173950579471e-07,
            0.10839551006143939185e-07,
            -0.10526397926519000691e-07,
            0.10222291794616780561e-07,
            -0.99269712454067339727e-08,
            0.96401824646609211080e-08,
            -0.93616789708108062008e-08,
            0.90912214031028235422e-08,
            -0.88285773158782117222e-08,
            0.85735209788006156203e-08,
            -0.83258331828536520172e-08,
            0.80853010519388850005e-08,
            -0.78517178599159764754e-08,
            0.76248828529317467947e-08,
            -0.74046010768841071059e-08,
            0.71906832098687622728e-08,
            -0.69829453994641022983e-08,
            0.67812091047173317459e-08,
            -0.65853009426977313290e-08,
            0.63950525394835678792e-08,
            -0.62103003854524726628e-08,
            0.60308856947513333758e-08,
            -0.58566542688268000505e-08,
            0.56874563638996400563e-08,
            -0.55231465622676621705e-08,
            0.53635836473256503502e-08,
            -0.52086304821955339059e-08,
            0.50581538918636123925e-08,
            -0.49120245487234261852e-08,
            0.47701168614250102390e-08,
            -0.46323088669346301918e-08,
            0.44984821257128528939e-08,
            -0.43685216199214189850e-08,
            0.42423156545711511770e-08,
            -0.41197557615253826557e-08,
            0.40007366062763786183e-08,
            -0.38851558974150727748e-08,
            0.37729142987165276171e-08,
            -0.36639153437652961865e-08,
            0.35580653530469988486e-08,
            -0.34552733534349709264e-08,
            0.33554510000030559535e-08,
            -0.32585125000973940249e-08,
            0.31643745396017359674e-08,
            -0.30729562113327868796e-08,
            0.29841789455041530200e-08,
            -0.28979664421992758989e-08,
            0.28142446057953132032e-08,
            -0.27329414812814390995e-08,
            0.26539871924168161407e-08,
            -0.25773138816751668664e-08,
            0.25028556519244291169e-08,
            -0.24305485097912959499e-08,
            0.23603303106619080563e-08,
            -0.22921407052714058628e-08,
            0.22259210878365399183e-08,
            -0.21616145456867578372e-08,
            0.20991658103504500059e-08,
            -0.20385212100542671023e-08,
            0.19796286235947108123e-08,
            -0.19224374355423869606e-08,
            0.18668984927404269626e-08,
            -0.18129640620596530537e-08,
            0.17605877893741759270e-08,
            -0.17097246597221680675e-08,
            0.16603309586176070024e-08,
            -0.16123642344797300183e-08,
            0.15657832621478869464e-08,
            -0.15205480074504270167e-08,
            0.14766195927971890460e-08,
            -0.14339602637660350412e-08,
            0.13925333566546899287e-08,
            -0.13523032669700050132e-08,
            0.13132354188275479691e-08,
            -0.12752962352352419048e-08,
            0.12384531092355140920e-08,
            -0.12026743758811379070e-08,
            0.11679292850206920657e-08,
            -0.11341879748702300079e-08,
            0.11014214463484779337e-08,
            -0.10696015381534739059e-08,
            0.10387009025592480307e-08,
            -0.10086929819117270958e-08,
            0.97955198580366660312e-09,
            -0.95125286890900208679e-09,
            0.92377130945756002788e-09,
            -0.89708368833163302057e-09,
            0.87116706876644941479e-09,
            -0.84599917663709281674e-09,
            0.82155838131493233922e-09,
            -0.79782367707710818756e-09,
            0.77477466505309300292e-09,
            -0.75239153569281363935e-09,
            0.73065505174126690031e-09,
            -0.70954653170499806391e-09,
            0.68904783379622916597e-09,
            -0.66914134034083904798e-09,
            0.64980994263679380996e-09,
            -0.63103702625001653087e-09,
            0.61280645673505631634e-09,
            -0.59510256576828648337e-09,
            0.57791013768171107199e-09,
            -0.56121439638580773724e-09,
            0.54500099267016873900e-09,
            -0.52925599187102496775e-09,
            0.51396586189505248004e-09,
            -0.49911746158916965897e-09,
            0.48469802944632955612e-09,
            -0.47069517263760014005e-09,
            0.45709685636110621144e-09,
            -0.44389139349867930081e-09,
            0.43106743457132480471e-09,
            -0.41861395798487421376e-09,
            0.40652026055743952578e-09,
            -0.39477594832052842209e-09,
            0.38337092758591272230e-09,
            -0.37229539627057491552e-09,
            0.36153983547227521766e-09,
            -0.35109500128849910154e-09,
            0.34095191687175519511e-09,
            -0.33110186471439289704e-09,
            0.32153637915631239045e-09,
            -0.31224723910912398970e-09,
            0.30322646099050710905e-09,
            -0.29446629186269331473e-09,
            0.28595920276917881651e-09,
            -0.27769788226393698380e-09,
            0.26967523012757238777e-09,
            -0.26188435126501301881e-09,
            0.25431854977949852574e-09,
            -0.24697132321776888896e-09,
            0.23983635698150977934e-09,
            -0.23290751890024982077e-09,
            0.22617885396104611209e-09,
            -0.21964457919042880688e-09,
            0.21329907868420528750e-09,
            -0.20713689878085250208e-09,
            0.20115274337434959303e-09,
            -0.19534146936242159446e-09,
            0.18969808222628370222e-09,
            -0.18421773173808480806e-09,
            0.17889570779236339873e-09,
            -0.17372743635793169822e-09,
            0.16870847554670939670e-09,
            -0.16383451179612808855e-09,
            0.15910135616182540748e-09,
            -0.15450494071744330869e-09,
            0.15004131505843449035e-09,
            -0.14570664290687508993e-09,
            0.14149719881436189859e-09,
            -0.13740936496016368823e-09,
            0.13343962804187350973e-09,
            -0.12958457625588848896e-09,
            0.12584089636512460714e-09,
            -0.12220537085144369393e-09,
            0.11867487515034671199e-09,
            -0.11524637496555659739e-09,
            0.11191692366118209830e-09,
            -0.10868365972922199935e-09,
            0.10554380433023220449e-09,
            -0.10249465890504320142e-09,
            0.99533602855475026004e-10,
            -0.96658091292055187607e-10,
            0.93865652846806834872e-10,
            -0.91153887549225778384e-10,
            0.88520464763624569567e-10,
            -0.85963121186075071965e-10,
            0.83479658899239781350e-10,
            -0.81067943483446593655e-10,
            0.78725902182445575131e-10,
            -0.76451522122414838220e-10,
            0.74242848583018786633e-10,
            -0.72097983319807716229e-10,
            0.70015082938317186447e-10,
            -0.67992357322650702613e-10,
            0.66028068126905105916e-10,
            -0.64120527350693063962e-10,
            0.62268096049913068230e-10,
            -0.60469183303296924744e-10,
            0.58722245716347976830e-10,
            -0.57025788118346391824e-10,
            0.55378366976800887211e-10,
            -0.53778600071222512844e-10,
            0.52225190653632309517e-10,
            -0.50716985205207501023e-10,
            0.49253109171572777739e-10,
            -0.47833283755322262690e-10,
            0.46458563164427748566e-10,
            -0.45133048314954522821e-10,
            0.43867868281155817508e-10,
            -0.42690428488605550129e-10,
            0.41665890740660430845e-10,
            -0.40947061319725302450e-10,
            0.40890256062850869212e-10,
            -0.42324395200868817252e-10,
            0.47175970286618397865e-10,
            -0.59920514722972034865e-10,
            0.90953607290146280299e-10,
        ]
    )

    # Next, setup the weights for 1-th order.
    WT1 = np.array(
        [
            -0.23779001100582381051e-28,
            0.79721192934236443185e-27,
            0.38325661863384939290e-27,
            0.84934033213920264104e-27,
            0.82321643716951481883e-27,
            0.11003687170198490196e-26,
            0.13254287328216880532e-26,
            0.15922464286287840307e-26,
            0.19986019650982509534e-26,
            0.23685465814544270716e-26,
            0.29798793392175001283e-26,
            0.35431019643707156492e-26,
            0.44257411307894436460e-26,
            0.53180794470588352748e-26,
            0.65556190066651513897e-26,
            0.79947762199059232857e-26,
            0.97071583223006191650e-26,
            0.12004979935771289997e-25,
            0.14406653509631588960e-25,
            0.17970467112699249374e-25,
            0.21453903977135419404e-25,
            0.26816147371136119450e-25,
            0.32037836290813990881e-25,
            0.39922197526685471353e-25,
            0.47937803770632117330e-25,
            0.59351247450921383090e-25,
            0.71773343544187673606e-25,
            0.88252267695870270015e-25,
            0.10736460080008319237e-24,
            0.13139234241813519897e-24,
            0.16038389953077929496e-24,
            0.19586330559120450202e-24,
            0.23934895950562681781e-24,
            0.29216907136422889152e-24,
            0.35704555610974980825e-24,
            0.43591578004654766279e-24,
            0.53258407091337795497e-24,
            0.65037749121846816863e-24,
            0.79446117848346841533e-24,
            0.97029807615169898533e-24,
            0.11851621862641970763e-23,
            0.14475349036735769831e-23,
            0.17680470446219830604e-23,
            0.21594646724325061636e-23,
            0.26376279226241200337e-23,
            0.32215255903424431280e-23,
            0.39348989123644471831e-23,
            0.48059305631057603546e-23,
            0.58701957447819091530e-23,
            0.71696014006070338140e-23,
            0.87572874338578454769e-23,
            0.10695830329268070127e-22,
            0.13064267975684400067e-22,
            0.15956396898683340405e-22,
            0.19489490174852410262e-22,
            0.23804258576458750974e-22,
            0.29074795736995329016e-22,
            0.35511877512642321550e-22,
            0.43374414815306912102e-22,
            0.52977566601218580972e-22,
            0.64706963771419739714e-22,
            0.79033290084770208947e-22,
            0.96531416601155305002e-22,
            0.11790381653809730581e-21,
            0.14400798270141680420e-21,
            0.17589176532470089827e-21,
            0.21483473765595870697e-21,
            0.26239961572838840870e-21,
            0.32049580752851479664e-21,
            0.39145421602105668850e-21,
            0.47812354861088417904e-21,
            0.58398109523939098113e-21,
            0.71327648305877594511e-21,
            0.87119745891754148321e-21,
            0.10640834348623049967e-20,
            0.12996739243852600375e-20,
            0.15874259111262610786e-20,
            0.19388856933392509112e-20,
            0.23681611511416871136e-20,
            0.28924775863608318981e-20,
            0.35328812735771197862e-20,
            0.43150695199493149714e-20,
            0.52704395186827729765e-20,
            0.64373272953957132288e-20,
            0.78625718444181387840e-20,
            0.96033638142877597977e-20,
            0.11729578932087649353e-19,
            0.14326535217816349936e-19,
            0.17498475666460750925e-19,
            0.21372678938714691346e-19,
            0.26104658300704859397e-19,
            0.31884290164425472060e-19,
            0.38943574121348422642e-19,
            0.47565771349593827251e-19,
            0.58096985951299433442e-19,
            0.70959792070530976791e-19,
            0.86670519080596190619e-19,
            0.10585956951243269563e-18,
            0.12929722209660508847e-18,
            0.15792391867645680267e-18,
            0.19288879140366910337e-18,
            0.23559479941190520470e-18,
            0.28775626657086499509e-18,
            0.35146613570216158086e-18,
            0.42928191135495900673e-18,
            0.52432585404586203783e-18,
            0.64041336717328802188e-18,
            0.78220224663792857958e-18,
            0.95538449298914256672e-18,
            0.11669086110519650175e-17,
            0.14252662063096260770e-17,
            0.17408230557831051757e-17,
            0.21262473650918901610e-17,
            0.25970027813123101601e-17,
            0.31719883922980728330e-17,
            0.38742728138556379626e-17,
            0.47320507187765319669e-17,
            0.57797357502250896216e-17,
            0.70593902823169582028e-17,
            0.86223523498694457235e-17,
            0.10531373010190409793e-16,
            0.12863037889413019372e-16,
            0.15710962731510451527e-16,
            0.19189397138461720101e-16,
            0.23438002820075089885e-16,
            0.28627215837238828004e-16,
            0.34965392409988710645e-16,
            0.42706786428448497367e-16,
            0.52162237447471309093e-16,
            0.63711036879213345092e-16,
            0.77816916463307065235e-16,
            0.95045695365657204522e-16,
            0.11608920160259351019e-15,
            0.14179151105966549619e-15,
            0.17318474398326101217e-15,
            0.21152807068685210363e-15,
            0.25836128767539467929e-15,
            0.31556278833430849848e-15,
            0.38542976466718232176e-15,
            0.47076434267244616602e-15,
            0.57499366556778888035e-15,
            0.70229784368391582701e-15,
            0.85778978838088164829e-15,
            0.10477052216252109330e-14,
            0.12796720504321799823e-14,
            0.15629924515539289125e-14,
            0.19090464627227088193e-14,
            0.23317106244241540845e-14,
            0.28479628092949809297e-14,
            0.34785033119698000940e-14,
            0.42486614900323137393e-14,
            0.51893168582342936762e-14,
            0.63382585119205047931e-14,
            0.77415505884680514824e-14,
            0.94555711726406619067e-14,
            0.11549035630408280097e-13,
            0.14106055531142479946e-13,
            0.17229135422830471373e-13,
            0.21043763493017538471e-13,
            0.25702847898914450407e-13,
            0.31393608430354023054e-13,
            0.38344140375861572417e-13,
            0.46833764078042948155e-13,
            0.57202731000069601939e-13,
            0.69867771751590042453e-13,
            0.85336439559542886244e-13,
            0.10423047667756279584e-12,
            0.12730699655547051243e-12,
            0.15549361393599358810e-12,
            0.18991970329598081032e-12,
            0.23196923668250597755e-12,
            0.28332687489467591536e-12,
            0.34605747293074408206e-12,
            0.42267398347606569328e-12,
            0.51625714279965715259e-12,
            0.63055541477094275274e-12,
            0.77016524760249830236e-12,
            0.94067802567402927053e-12,
            0.11489516827349049439e-11,
            0.14033265286503660695e-11,
            0.17140347264354340417e-11,
            0.20935168852125301327e-11,
            0.25570397012560628610e-11,
            0.31231597365723921107e-11,
            0.38146555551079558782e-11,
            0.46592061049755500161e-11,
            0.56907982819762827155e-11,
            0.69507175913176593709e-11,
            0.84896748701239594140e-11,
            0.10369250352182410092e-10,
            0.12665108932133729727e-10,
            0.15469100894640800720e-10,
            0.18894125922585180356e-10,
            0.23077182209924898792e-10,
            0.28186729422033271362e-10,
            0.34427103160373659936e-10,
            0.42049668170180476841e-10,
            0.51359191342712271588e-10,
            0.62730747867977636297e-10,
            0.76618892012631225296e-10,
            0.93583301711963271940e-10,
            0.11430192677538819367e-09,
            0.13960991678261530046e-09,
            0.17051839200598010096e-09,
            0.20827357888452780935e-09,
            0.25438347678424908063e-09,
            0.31070775906120548243e-09,
            0.37949543960166480335e-09,
            0.46352165150554629058e-09,
            0.56614048928061041213e-09,
            0.69149327357632214648e-09,
            0.84458207882202713872e-09,
            0.10315871013042910049e-08,
            0.12599679520777700732e-08,
            0.15389476810193000824e-08,
            0.18796505921747109347e-08,
            0.22958410530823679168e-08,
            0.28041080392694730677e-08,
            0.34249937978079458891e-08,
            0.41832358233125644067e-08,
            0.51094925678131643906e-08,
            0.62406516508546298375e-08,
            0.76224707236306606749e-08,
            0.93099537030146089046e-08,
            0.11371395622837669325e-07,
            0.13888811603603460790e-07,
            0.16964137540446880094e-07,
            0.20719660567011048447e-07,
            0.25307532985450009557e-07,
            0.30910082760196839954e-07,
            0.37754423983369756739e-07,
            0.46112394911463966981e-07,
            0.56323015017988116740e-07,
            0.68791561040010018689e-07,
            0.84024115221375004322e-07,
            0.10262487036321569423e-06,
            0.12534932630323100331e-06,
            0.15309818493173440690e-06,
            0.18699933285775400780e-06,
            0.22839543319719978897e-06,
            0.27897038209650021238e-06,
            0.34072557173474452451e-06,
            0.41617511631609012044e-06,
            0.50830216571405257032e-06,
            0.62086055851273829343e-06,
            0.75829654716288451693e-06,
            0.92621529324795927674e-06,
            0.11312433484295780322e-05,
            0.13817506779893170808e-05,
            0.16876126549713120575e-05,
            0.20613285048301949073e-05,
            0.25176142217059249495e-05,
            0.30751364937151691114e-05,
            0.37558230263550338492e-05,
            0.45875527884742903516e-05,
            0.56029966733669012944e-05,
            0.68437947583602416914e-05,
            0.83586201447068474255e-05,
            0.10209670082926559548e-04,
            0.12469450254224749995e-04,
            0.15230868871022190138e-04,
            0.18601920832727638478e-04,
            0.22721395800711319485e-04,
            0.27750126132155010419e-04,
            0.33895447798826631189e-04,
            0.41396841136470921454e-04,
            0.50564046316342118052e-04,
            0.61753575721436104002e-04,
            0.75428141719150856360e-04,
            0.92118338007583430976e-04,
            0.11251534681498950364e-03,
            0.13740856459760000547e-03,
            0.16783029354650548710e-03,
            0.20495440270807768974e-03,
            0.25032223970649432505e-03,
            0.30567824291028637771e-03,
            0.37332275750272562662e-03,
            0.45584569317397240131e-03,
            0.55667758728412204596e-03,
            0.67965842762967730497e-03,
            0.82989825334180589590e-03,
            0.10130793883812590737e-02,
            0.12367998790718090669e-02,
            0.15094441288791659965e-02,
            0.18422782918639890679e-02,
            0.22476208557991408411e-02,
            0.27421062075955880862e-02,
            0.33437164053099180556e-02,
            0.40768527847886001578e-02,
            0.49675073471938113734e-02,
            0.60511176276551595712e-02,
            0.73646815739407444149e-02,
            0.89588637538056889453e-02,
            0.10885052122857430054e-01,
            0.13213887638701850638e-01,
            0.16013879136554731025e-01,
            0.19379495481742049084e-01,
            0.23395566125555520426e-01,
            0.28179341735834879251e-01,
            0.33820790606197338735e-01,
            0.40444641201110451134e-01,
            0.48109746447546619663e-01,
            0.56899341828119280595e-01,
            0.66751684235759015107e-01,
            0.77591144073115236868e-01,
            0.89047612974465659152e-01,
            0.10065405918556659393e00,
            0.11140105420732690500e00,
            0.12006932932879070541e00,
            0.12458597468885210247e00,
            0.12268765549548429905e00,
            0.11115942050671910601e00,
            0.87319954786645864742e-01,
            0.48465000302246638575e-01,
            -0.49561566373757083917e-02,
            -0.69471081820978636268e-01,
            -0.13351269533458298699e00,
            -0.17957507285653320772e00,
            -0.18191106295048839470e00,
            -0.12098476462017759836e00,
            0.57914272269164483384e-02,
            0.15348128813633118983e00,
            0.23522097310698558692e00,
            0.15150400266403860061e00,
            -0.82790857821451305232e-01,
            -0.27115431040159521103e00,
            -0.13312798063762890100e00,
            0.22420582796659790925e00,
            0.23042922487820899358e00,
            -0.25728138031301517330e00,
            -0.14755494381924760816e00,
            0.36119355267312441837e00,
            -0.24353715522887228739e00,
            0.38114547256147109089e-01,
            0.82736503214593373756e-01,
            -0.10914677811197480428e00,
            0.89643285228612118454e-01,
            -0.60977888159322063411e-01,
            0.37801231818370660542e-01,
            -0.22438621211053739329e-01,
            0.13156278189915569798e-01,
            -0.77896933116771488112e-02,
            0.47370324617059750630e-02,
            -0.29965293695965681403e-02,
            0.19878544860469768082e-02,
            -0.13869827135561199356e-02,
            0.10157400415733200925e-02,
            -0.77631830897451124893e-03,
            0.61460402632747919080e-03,
            -0.50020690522478944255e-03,
            0.41570616801530989705e-03,
            -0.35086447759715891749e-03,
            0.29949364033671969747e-03,
            -0.25773573414907620556e-03,
            0.22310488448276601017e-03,
            -0.19394286742651010405e-03,
            0.16910356690987170454e-03,
            -0.14776630535110279563e-03,
            0.12932306435161670878e-03,
            -0.11330886340153000630e-03,
            0.99357870873694533213e-04,
            -0.87175171695487733894e-04,
            0.76518203465147482511e-04,
            -0.67184226920608184730e-04,
            0.59001625485470532949e-04,
            -0.51823700437172301393e-04,
            0.45524131447897800190e-04,
            -0.39993558219308811687e-04,
            0.35136924058234570150e-04,
            -0.30871356264610351938e-04,
            0.27124444492712818783e-04,
            -0.23832821622649580026e-04,
            0.20940974022723349952e-04,
            -0.18400228343346008335e-04,
            0.16167881311147351016e-04,
            -0.14206450345841819494e-04,
            0.12483025923096489624e-04,
            -0.10968708114345040680e-04,
            0.96381137476371433619e-05,
            -0.84689451209231461783e-05,
            0.74416131426310713740e-05,
            -0.65389076895947963034e-05,
            0.57457083047103120349e-05,
            -0.50487298647161201493e-05,
            0.44362992864155191501e-05,
            -0.38981597741651851768e-05,
            0.34252990553006400639e-05,
            -0.30097983612462208859e-05,
            0.26446996057275441348e-05,
            -0.23238887351302340716e-05,
            0.20419933647589090787e-05,
            -0.17942928642745300170e-05,
            0.15766392837610099303e-05,
            -0.13853878318735780200e-05,
            0.12173358208997559817e-05,
            -0.10696690610117151049e-05,
            0.93991475334313001620e-06,
            -0.82590006822179900410e-06,
            0.72571573986650604055e-06,
            -0.63768409395308406728e-06,
            0.56033096589579911757e-06,
            -0.49236101932854606775e-06,
            0.43263604794155821972e-06,
            -0.38015590945392312373e-06,
            0.33404177953802387996e-06,
            -0.29352143989559692218e-06,
            0.25791634724263857612e-06,
            -0.22663026755387299218e-06,
            0.19913929019334100136e-06,
            -0.17498305605429440553e-06,
            0.15375705002049011184e-06,
            -0.13510582667935480436e-06,
            0.11871705693399870160e-06,
            -0.10431629758749350038e-06,
            0.91662396276475694184e-07,
            -0.80543453607028647290e-07,
            0.70773274428481856046e-07,
            -0.62188249446989189968e-07,
            0.54644615505335029221e-07,
            -0.48016048443646231508e-07,
            0.42191547784261369808e-07,
            -0.37073577780952398937e-07,
            0.32576433988135847120e-07,
            -0.28624808133909398953e-07,
            0.25152527127147581427e-07,
            -0.22101444936595110588e-07,
            0.19420468820503340806e-07,
            -0.17064703711658711163e-07,
            0.14994700450203629605e-07,
            -0.13175795208633650552e-07,
            0.11577529007302610104e-07,
            -0.10173137628831489796e-07,
            0.89391034277419880612e-08,
            -0.78547615254943383801e-08,
            0.69019537690386798258e-08,
            -0.60647246504485460851e-08,
            0.53290541108617648523e-08,
            -0.46826227654829832319e-08,
            0.41146056122897838021e-08,
            -0.36154907601472161940e-08,
            0.31769201408808051073e-08,
            -0.27915495445014601903e-08,
            0.24529256362912200658e-08,
            -0.21553778928767579448e-08,
            0.18939236446342600392e-08,
            -0.16641846353897590312e-08,
            0.14623137046133780063e-08,
            -0.12849303645740829450e-08,
            0.11290641920775340868e-08,
            -0.99210508606754888054e-09,
            0.87175955886335064848e-09,
            -0.76601232990455893213e-09,
            0.67309257867770258852e-09,
            -0.59144429108839117984e-09,
            0.51970020258742657674e-09,
            -0.45665890207353687588e-09,
            0.40126471338250552359e-09,
            -0.35259001730495680327e-09,
            0.30981971788986289322e-09,
            -0.27223759293244408980e-09,
            0.23921430022514762304e-09,
            -0.21019683877088560052e-09,
            0.18469928839465210504e-09,
            -0.16229467261876449390e-09,
            0.14260780857078909327e-09,
            -0.12530902424439949164e-09,
            0.11010863790384879428e-09,
            -0.96752107149079824750e-10,
            0.85015766394465004630e-10,
            -0.74703081400610626905e-10,
            0.65641358155343214620e-10,
            -0.57678850985617859679e-10,
            0.50682221459754282741e-10,
            -0.44534305525909120100e-10,
            0.39132151504364007283e-10,
            -0.34385296082594761887e-10,
            0.30214249439239317785e-10,
            -0.26549164124680559140e-10,
            0.23328665408564599349e-10,
            -0.20498823510896081544e-10,
            0.18012250506600650469e-10,
            -0.15827306778509629286e-10,
            0.13907403728951850551e-10,
            -0.12220391074163700618e-10,
            0.10738018462452050425e-10,
            -0.94354624003716234485e-11,
            0.82909105640953193728e-11,
            -0.72851965346947523766e-11,
            0.64014788411436639934e-11,
            -0.56249589367146987257e-11,
            0.49426333858928202302e-11,
            -0.43430761116809759680e-11,
            0.38162470568545827330e-11,
            -0.33533240552467311054e-11,
            0.29465550976975220819e-11,
            -0.25891285186428910646e-11,
            0.22750589294212621855e-11,
            -0.19990869881930792011e-11,
            0.17565913281172699705e-11,
            -0.15435111689614020217e-11,
            0.13562783161689700960e-11,
            -0.11917574086305238998e-11,
            0.10471934145827610682e-11,
            -0.92016549643814607864e-12,
            0.80854647197038528016e-12,
            -0.71046719298279401734e-12,
            0.62428524494839663246e-12,
            -0.54855744348363701401e-12,
            0.48201566709453555482e-12,
            -0.42354562149103289211e-12,
            0.37216817985480538490e-12,
            -0.32702298658923321260e-12,
            0.28735404998800310208e-12,
            -0.25249708256169828971e-12,
            0.22186837702370389530e-12,
            -0.19495503165314128815e-12,
            0.17130636135177430412e-12,
            -0.15052635056794931119e-12,
            0.13226702170589389284e-12,
            -0.11622260796808300942e-12,
            0.10212443304988860003e-12,
            -0.89736411943535084833e-13,
            0.78851097509314687508e-13,
            -0.69286206610625636538e-13,
            0.60881567639821682927e-13,
            -0.53496438318744010298e-13,
            0.47007148858659282416e-13,
            -0.41305031012602458038e-13,
            0.36294598340360191951e-13,
            -0.31891947213064369610e-13,
            0.28023351780988580919e-13,
            -0.24624029376283279094e-13,
            0.21637055676372050574e-13,
            -0.19012411461508211104e-13,
            0.16706145003659701281e-13,
            -0.14679636060291389006e-13,
            0.12898949148078830678e-13,
            -0.11334265266611379683e-13,
            0.99593825558268706535e-14,
            -0.87512775252863140427e-14,
            0.76897195077421143642e-14,
            -0.67569318807328292835e-14,
            0.59372943830891981248e-14,
            -0.52170815413984568005e-14,
            0.45842328261723863703e-14,
            -0.40281506888097158469e-14,
            0.35395230973258291850e-14,
            -0.31101675990681859105e-14,
            0.27328942991223000800e-14,
            -0.24013854598749791415e-14,
            0.21100897055373030150e-14,
            -0.18541290599993040151e-14,
            0.16292172612908320081e-14,
            -0.14315879847593459752e-14,
            0.12579317730060459754e-14,
            -0.11053406164233719234e-14,
            0.97125925629132513426e-15,
            -0.85344239496416429968e-15,
            0.74991709659831521783e-15,
            -0.65894974879250159170e-15,
            0.57901703188703568188e-15,
            -0.50878040978025863482e-15,
            0.44706371509062259811e-15,
            -0.39283345331036878858e-15,
            0.34518149612852799007e-15,
            -0.30330987410939582363e-15,
            0.26651741406787180955e-15,
            -0.23418799737396022297e-15,
            0.20578024256254929268e-15,
            -0.18081843947571900690e-15,
            0.15888458312267581200e-15,
            -0.13961137385800899984e-15,
            0.12267606666074510376e-15,
            -0.10779506651556279801e-15,
            0.94719179391603797610e-16,
            -0.83229439292786975771e-16,
            0.73133441500293154255e-16,
            -0.64262120604484965377e-16,
            0.56466919371882109638e-16,
            -0.49617301037651720597e-16,
            0.43598563364994760812e-16,
            -0.38309917866133909996e-16,
            0.33662802019951652661e-16,
            -0.29579396223041629810e-16,
            0.25991320639354369445e-16,
            -0.22838490126158601174e-16,
            0.20068108061152711033e-16,
            -0.17633782221571330287e-16,
            0.15494747909989438574e-16,
            -0.13615185317456429361e-16,
            0.11963619692670830272e-16,
            -0.10512394272558709913e-16,
            0.92372071480646483496e-17,
            -0.81167043096073984687e-17,
            0.71321220574247113037e-17,
            -0.62669727886873747397e-17,
            0.55067688996430172615e-17,
            -0.48387801793162163364e-17,
            0.42518206321314128635e-17,
            -0.37360611596341719162e-17,
            0.32828649645187700314e-17,
            -0.28846429206752079096e-17,
            0.25347264873153391346e-17,
            -0.22272560390247178611e-17,
            0.19570827417369101585e-17,
            -0.17196823314652230881e-17,
            0.15110793519499210062e-17,
            -0.13277805825221979748e-17,
            0.11667165413822330724e-17,
            -0.10251900847168659743e-17,
            0.90083124090830315999e-18,
            -0.79155752349720990603e-18,
            0.69553905830937584557e-18,
            -0.61116794077226843712e-18,
            0.53703131028706945667e-18,
            -0.47188769076765341320e-18,
            0.41464620115119749872e-18,
            -0.36434828774274431505e-18,
            0.32015167248500091253e-18,
            -0.28131624835279290921e-18,
            0.24719168567911820595e-18,
            -0.21720654187174320654e-18,
            0.19085869215550189721e-18,
            -0.16770692109721899111e-18,
            0.14736353410746910352e-18,
            -0.12948786519216900428e-18,
            0.11378057223455149305e-18,
            -0.99978624276428549682e-19,
            0.87850896856935804010e-19,
            -0.77194301650481244444e-19,
            0.67830385593241861930e-19,
            -0.59602342548611573044e-19,
            0.52372387468676202365e-19,
            -0.46019449077998791288e-19,
            0.40437142440414807125e-19,
            -0.35531987457892000724e-19,
            0.31221843469976618630e-19,
            -0.27434533741424270108e-19,
            0.24106636804928459342e-19,
            -0.21182424419102399996e-19,
            0.18612928356413968765e-19,
            -0.16355120393022199627e-19,
            0.14371191768318589670e-19,
            -0.12627920047644379558e-19,
            0.11096112785317059883e-19,
            -0.97501186719036165739e-20,
            0.85673979808058807363e-20,
            -0.75281451224231245595e-20,
            0.66149569855872109159e-20,
            -0.58125415113657893624e-20,
            0.51074616176521397340e-20,
            -0.44879101854824920831e-20,
            0.39435123390814399527e-20,
            -0.34651517093082292638e-20,
            0.30448177717368800233e-20,
            -0.26754717030230440586e-20,
            0.23509285093699011334e-20,
            -0.20657534540814548706e-20,
            0.18151710504142631532e-20,
            -0.15949850948474660527e-20,
            0.14015083996708770500e-20,
            -0.12315010469494319376e-20,
            0.10821161305372590522e-20,
            -0.95085207960662486644e-21,
            0.83551076750596628203e-21,
            -0.73416070535124345550e-21,
            0.64510470249568779156e-21,
            -0.56685144894552668532e-21,
            0.49809054123125183129e-21,
            -0.43767053435643358195e-21,
            0.38457965596782228367e-21,
            -0.33792886257541818238e-21,
            0.29693695347390241880e-21,
            -0.26091749061412948769e-21,
            0.22926730418560010001e-21,
            -0.20145639198759520765e-21,
            0.17701904387406560389e-21,
            -0.15554604225258168932e-21,
            0.13667780796729550641e-21,
            -0.12009837779858251158e-21,
            0.10553011379060780116e-21,
            -0.92729055585803990884e-22,
            0.81480836646395174933e-22,
            -0.71597094891875700200e-22,
            0.62912317547278588468e-22,
            -0.55281068025898321612e-22,
            0.48575549320856098216e-22,
            -0.42683463880645999618e-22,
            0.37506134255287311446e-22,
            -0.32956852260392350507e-22,
            0.28959427878761049247e-22,
            -0.25446913153691919200e-22,
            0.22360480394754259887e-22,
            -0.19648436888446069881e-22,
            0.17265359799157371192e-22,
            -0.15171336029025659557e-22,
            0.13331293566397819119e-22,
            -0.11714413365284629986e-22,
            0.10293612991331600087e-22,
            -0.90450942661818855010e-23,
            0.79479472072663889120e-23,
            -0.69838025973983943288e-23,
            0.61365260759698707238e-23,
            -0.53919477932897098293e-23,
            0.47376234574340661915e-23,
            -0.41626240529454467843e-23,
            0.36573513415638608941e-23,
            -0.32133750545472298724e-23,
            0.28232876298439711191e-23,
            -0.24805746766402519629e-23,
            0.21795021088492381699e-23,
            -0.19150207966391389939e-23,
            0.16826867531148651179e-23,
            -0.14785926095987099109e-23,
            0.12993065708576839773e-23,
            -0.11418169017687040810e-23,
            0.10034812450227850599e-23,
            -0.88198046609572075136e-24,
            0.77527684778131062791e-24,
            -0.68157678580025223715e-24,
            0.59929859006652480651e-24,
            -0.52704590637343920302e-24,
            0.46358624111797927966e-24,
            -0.40783289235105509668e-24,
            0.35882870194420578191e-24,
            -0.31573129057568951175e-24,
            0.27780031811537501369e-24,
            -0.24438721059930989498e-24,
            0.21492709123302868776e-24,
            -0.18893185489531158875e-24,
            0.16598283447655300523e-24,
            -0.14572197983278360944e-24,
            0.12784189795680629463e-24,
            -0.11207611655384429414e-24,
            0.98190635833986939235e-25,
            -0.85977069575473510383e-25,
            0.75247618025397713641e-25,
            -0.65832296928724367365e-25,
            0.57578153084183532562e-25,
            -0.50349088175171190647e-25,
            0.44024756065989681357e-25,
            -0.38498191971011007846e-25,
            0.33673291115583042791e-25,
            -0.29463411791224339510e-25,
            0.25791217847239958719e-25,
            -0.22588731181811621204e-25,
            0.19796670767579898895e-25,
            -0.17363295672276158774e-25,
            0.15243569288426309447e-25,
            -0.13398787953334260281e-25,
            0.11796045820480120133e-25,
            -0.10407147490908270710e-25,
            0.92071944889040744035e-26,
            -0.81731249295591442672e-26,
            0.72825159703487444195e-26,
            -0.65131006447129183541e-26,
            0.58430301262485386924e-26,
            -0.52513930081011199285e-26,
            0.47187919679302219783e-26,
            -0.42284069142292412393e-26,
            0.37676954747502728537e-26,
            -0.33299134001135826479e-26,
            0.29143004338630110718e-26,
            -0.25243668234065648260e-26,
            0.21646903157623978348e-26,
            -0.18376734028052161580e-26,
            0.15423017366879009263e-26,
            -0.12758944190588119779e-26,
            0.10371662830395429178e-26,
            -0.82748173500706471962e-27,
            0.64909427840638429032e-27,
            -0.50234666376562531025e-27,
            0.38452110973130778698e-27,
            -0.29091636205058451455e-27,
            0.21678009467166822096e-27,
            -0.15874578619111640200e-27,
            0.11537394235020309101e-27,
            -0.86900364189788941857e-28,
            0.74355723662636970071e-28,
            -0.78471428315427385396e-28,
            0.98949852128730596091e-28,
            -0.13427202805922251023e-27,
            0.18174242734021449278e-27,
            -0.23763619609508699858e-27,
            0.29778198107110528091e-27,
            -0.35868369680655068529e-27,
            0.41844766665021181973e-27,
            -0.47660851066880372752e-27,
            0.53287774452581460041e-27,
            -0.58588249858342115784e-27,
            0.63294098597067369917e-27,
            -0.67092689718604759703e-27,
            0.69754161721794173542e-27,
            -0.71229292612293631973e-27,
            0.71678213553979546309e-27,
            -0.71422093211775989035e-27,
            0.70847857438160299782e-27,
            -0.70324864718393631473e-27,
            0.70176504791510964044e-27,
            -0.70696084721646144402e-27,
            0.72149205056137611593e-27,
        ]
    )

    return WT0, WT1, YBASE
