from enum import Enum
from typing import Callable, Generic, TypeVar, get_type_hints, Optional, Dict, Any, get_args, Union, Type, Tuple
from functools import wraps

from drf_yasg.openapi import Schema, SchemaRef
from rest_framework.request import Request
from rest_framework.response import Response as _Response
from rest_framework.serializers import Serializer
from drf_yasg.utils import unset, swagger_auto_schema
from drf_yasg.openapi import Response as YasgResponse

__version__ = '0.1.0'


class Location(Enum):
    QUERY = 'query'
    BODY = 'body'


def get_location(serializer: Union[Type[Serializer], Serializer]) -> Location:
    metaclass = getattr(serializer, 'Meta', None)
    if metaclass is None:
        return Location.BODY
    return getattr(metaclass, 'location', Location.BODY)


def restapi(method=None, methods=None, auto_schema=unset, request_body=None, query_serializer=None,
            manual_parameters=None, operation_id=None, operation_description=None, operation_summary=None,
            security=None, deprecated=None, responses=None, field_inspectors=None, filter_inspectors=None,
            paginator_inspectors=None, tags=None, **extra_overrides):
    """Decorate a view method to customize the :class:`.Operation` object generated from it.

        `method` and `methods` are mutually exclusive and must only be present when decorating a view method that accepts
        more than one HTTP request method.

        The `auto_schema` and `operation_description` arguments take precedence over view- or method-level values.

        :param str method: for multi-method views, the http method the options should apply to
        :param list[str] methods: for multi-method views, the http methods the options should apply to
        :param drf_yasg.inspectors.SwaggerAutoSchema auto_schema: custom class to use for generating the Operation object;
            this overrides both the class-level ``swagger_schema`` attribute and the ``DEFAULT_AUTO_SCHEMA_CLASS``
            setting, and can be set to ``None`` to prevent this operation from being generated
        :param request_body: custom request body which will be used as the ``schema`` property of a
            :class:`.Parameter` with ``in: 'body'``.

            A Schema or SchemaRef is not valid if this request consumes form-data, because ``form`` and ``body`` parameters
            are mutually exclusive in an :class:`.Operation`. If you need to set custom ``form`` parameters, you can use
            the `manual_parameters` argument.

            If a ``Serializer`` class or instance is given, it will be automatically converted into a :class:`.Schema`
            used as a ``body`` :class:`.Parameter`, or into a list of ``form`` :class:`.Parameter`\\ s, as appropriate.
        :type request_body: drf_yasg.openapi.Schema or drf_yasg.openapi.SchemaRef  or rest_framework.serializers.Serializer
            or type[no_body]

        :param rest_framework.serializers.Serializer query_serializer: if you use a ``Serializer`` to parse query
            parameters, you can pass it here and have :class:`.Parameter` objects be generated automatically from it.

            If any ``Field`` on the serializer cannot be represented as a ``query`` :class:`.Parameter`
            (e.g. nested Serializers, file fields, ...), the schema generation will fail with an error.

            Schema generation will also fail if the name of any Field on the `query_serializer` conflicts with parameters
            generated by ``filter_backends`` or ``paginator``.

        :param list[drf_yasg.openapi.Parameter] manual_parameters: a list of manual parameters to override the
            automatically generated ones

            :class:`.Parameter`\\ s are identified by their (``name``, ``in``) combination, and any parameters given
            here will fully override automatically generated parameters if they collide.

            It is an error to supply ``form`` parameters when the request does not consume form-data.

        :param str operation_id: operation ID override; the operation ID must be unique across the whole API
        :param str operation_description: operation description override
        :param str operation_summary: operation summary string
        :param list[dict] security: security requirements override; used to specify which authentication mechanism
            is required to call this API; an empty list marks the endpoint as unauthenticated (i.e. removes all accepted
            authentication schemes), and ``None`` will inherit the top-level security requirements
        :param bool deprecated: deprecation status for operation
        :param responses: a dict of documented manual responses
            keyed on response status code. If no success (``2xx``) response is given, one will automatically be
            generated from the request body and http method. If any ``2xx`` response is given the automatic response is
            suppressed.

            * if a plain string is given as value, a :class:`.Response` with no body and that string as its description
              will be generated
            * if ``None`` is given as a value, the response is ignored; this is mainly useful for disabling default
              2xx responses, i.e. ``responses={200: None, 302: 'something'}``
            * if a :class:`.Schema`, :class:`.SchemaRef` is given, a :class:`.Response` with the schema as its body and
              an empty description will be generated
            * a ``Serializer`` class or instance will be converted into a :class:`.Schema` and treated as above
            * a :class:`.Response` object will be used as-is; however if its ``schema`` attribute is a ``Serializer``,
              it will automatically be converted into a :class:`.Schema`
        :type responses: dict[int or str, (drf_yasg.openapi.Schema or drf_yasg.openapi.SchemaRef or
            drf_yasg.openapi.Response or str or rest_framework.serializers.Serializer)]

        :param list[type[drf_yasg.inspectors.FieldInspector]] field_inspectors: extra serializer and field inspectors; these
            will be tried before :attr:`.ViewInspector.field_inspectors` on the :class:`.inspectors.SwaggerAutoSchema`
        :param list[type[drf_yasg.inspectors.FilterInspector]] filter_inspectors: extra filter inspectors; these will be
            tried before :attr:`.ViewInspector.filter_inspectors` on the :class:`.inspectors.SwaggerAutoSchema`
        :param list[type[drf_yasg.inspectors.PaginatorInspector]] paginator_inspectors: extra paginator inspectors; these
            will be tried before :attr:`.ViewInspector.paginator_inspectors` on the :class:`.inspectors.SwaggerAutoSchema`
        :param list[str] tags: tags override
        :param extra_overrides: extra values that will be saved into the ``overrides`` dict; these values will be available
            in the handling :class:`.inspectors.SwaggerAutoSchema` instance via ``self.overrides``
        """

    def decorator(func: Callable):
        hints = get_type_hints(func)

        _query_serializer = query_serializer
        _request_body = request_body
        _responses = responses or {}
        serializer_cls_dict: Dict[str, Tuple[Location, Type[Serializer]]] = {}

        if not _responses and (200 not in _responses):
            return_hint = hints.pop('return', None)
            return_hint_args = get_args(return_hint)
            if return_hint_args:
                serializer_cls = return_hint_args[0]
                if issubclass(serializer_cls, Serializer):
                    _responses[200] = serializer_cls()

        for key, value in hints.items():
            if issubclass(value, Serializer):
                location = get_location(value)
                serializer = value()
                assert isinstance(location, Location), 'invalid location'
                if location == Location.QUERY:
                    _query_serializer = serializer
                    serializer_cls_dict[key] = (Location.QUERY, value)
                else:
                    _request_body = serializer
                    serializer_cls_dict[key] = (Location.BODY, value)

        @wraps(func)
        def inner(self, request: Request, *args, **kwargs):
            for k, val in serializer_cls_dict.items():
                loc, ser_lcs = val
                data = request.data if loc == Location.BODY else request.query_params
                ser = ser_lcs(data=data)
                ser.is_valid(raise_exception=True)
                kwargs[k] = ser

            return func(self, request, *args, **kwargs)

        return swagger_auto_schema(method, methods, auto_schema, _request_body, _query_serializer,
                                   manual_parameters, operation_id, operation_description, operation_summary,
                                   security, deprecated, _responses, field_inspectors, filter_inspectors,
                                   paginator_inspectors, tags, **extra_overrides)(inner)

    return decorator


T = TypeVar('T', bound=Union[Serializer, Schema, SchemaRef, YasgResponse])


class Response(_Response, Generic[T]):
    def __init__(
            self,
            data: Optional[Union[dict, list]] = None,
            status: Optional[int] = None,
            template_name: Optional[str] = None,
            headers: Optional[Dict[str, str]] = None,
            exception: bool = False,
            content_type: Optional[str] = None
    ):
        super().__init__(data=data, status=status, template_name=template_name, headers=headers, exception=exception,
                         content_type=content_type)
