"""
Test step definitions for behave tests generated by the Omnata Plugin Devkit
"""
import datetime
import importlib
import json
import logging
import os
from pathlib import Path
import sys
import unittest
from types import ModuleType
from typing import List, Optional
from omnata_plugin_devkit.native_app_packaging import NativeAppPackaging
from omnata_plugin_devkit.plugin_uploader import PluginUploader
from omnata_plugin_devkit.utils import get_snowpark_session
from snowflake.connector.pandas_tools import write_pandas
from snowflake.snowpark import Session
from jinja2 import Environment
import pandas
import vcr

# monkey-patch time.sleep to avoid waiting in tests
import time
def dont_sleep(seconds):
    pass
time.sleep = dont_sleep

# see (https://github.com/behave/behave/issues/641)
from behave import given, step, then, when  # pylint: disable=no-name-in-module
from behave_pandas import table_to_dataframe
from omnata_plugin_runtime.configuration import (
    InboundSyncConfigurationParameters,
    OutboundSyncConfigurationParameters,
    StoredStreamConfiguration,
)
from omnata_plugin_runtime.logging import OmnataPluginLogHandler
from omnata_plugin_runtime.omnata_plugin import InboundSyncRequest, OutboundSyncRequest
from pandas.testing import assert_frame_equal
from pydantic import TypeAdapter

case = unittest.TestCase()
from logging import getLogger

logger = getLogger(__name__)

@given("we use the outbound source records from {filename}")
def step_impl_use_source_records_from_file(context, filename):
    # the filename is a pipe-separated file, so we'll read it as such
    file_path = os.path.join(
        context._root["config"].base_dir,
        "outbound_source_records",
        filename,  # pylint: disable=protected-access
    )
    context.source_records = pandas.read_csv(file_path, sep="|")
    context.source_records = context.source_records.rename(str.strip, axis = 'columns')
    # we need to assert that the frame contains, at a minimum, the expected columns:
    # IDENTIFIER, SYNC_ACTION, and TRANSFORMED_RECORD
    expected_columns = ["IDENTIFIER", "SYNC_ACTION", "TRANSFORMED_RECORD"]
    for column in expected_columns:
        case.assertIn(column, context.source_records.columns, f"Column {column} not found in source records")
    # add missing columns if they don't exist
    if "TRANSFORMED_RECORD_PREVIOUS" not in context.source_records.columns:
        context.source_records["TRANSFORMED_RECORD_PREVIOUS"] = None
    if "APPLY_STATE" not in context.source_records.columns:
        context.source_records["APPLY_STATE"] = None
    if "APPLY_STATE_DATETIME" not in context.source_records.columns:
        context.source_records["APPLY_STATE_DATETIME"] = None
    if "LOADBATCH_ID" not in context.source_records.columns:
        context.source_records["LOADBATCH_ID"] = None
    if "APP_IDENTIFIER" not in context.source_records.columns:
        context.source_records["APP_IDENTIFIER"] = None
    if "RESULT" not in context.source_records.columns:
        context.source_records["RESULT"] = None
    # the IDENTIFIER column should be cast to string
    context.source_records["IDENTIFIER"] = context.source_records["IDENTIFIER"].astype(str)
    # strip whitespace from the IDENTIFIER column
    context.source_records["IDENTIFIER"] = context.source_records["IDENTIFIER"].str.strip()
    # the APP_IDENTIFIER column should be cast to string
    context.source_records["APP_IDENTIFIER"] = context.source_records["APP_IDENTIFIER"].astype(str)
    # strip whitespace from the APP_IDENTIFIER column
    context.source_records["APP_IDENTIFIER"] = context.source_records["APP_IDENTIFIER"].str.strip()
    # the RESULT column should be cast to string
    context.source_records["RESULT"] = context.source_records["RESULT"].astype(str)
    # strip whitespace from the RESULT column
    context.source_records["RESULT"] = context.source_records["RESULT"].str.strip()
    # the SYNC_ACTION column should be cast to string
    context.source_records["SYNC_ACTION"] = context.source_records["SYNC_ACTION"].astype(str)
    # strip whitespace from the SYNC_ACTION column
    context.source_records["SYNC_ACTION"] = context.source_records["SYNC_ACTION"].str.strip()
    # the TRANSFORMED_RECORD column should be cast to string
    context.source_records["TRANSFORMED_RECORD"] = context.source_records["TRANSFORMED_RECORD"].astype(str)
    # strip whitespace from the TRANSFORMED_RECORD column
    context.source_records["TRANSFORMED_RECORD"] = context.source_records["TRANSFORMED_RECORD"].str.strip()
    
    # rename TRANSFORMED_RECORD to TRANSFORMED_RECORD_VARCHAR
    context.source_records = context.source_records.rename(columns={"TRANSFORMED_RECORD":"TRANSFORMED_RECORD_VARCHAR"})
    # rename TRANSFORMED_RECORD_PREVIOUS to TRANSFORMED_RECORD_PREVIOUS_VARCHAR
    context.source_records = context.source_records.rename(columns={"TRANSFORMED_RECORD_PREVIOUS":"TRANSFORMED_RECORD_PREVIOUS_VARCHAR"})
    # rename RESULT to RESULT_VARCHAR
    context.source_records = context.source_records.rename(columns={"RESULT":"RESULT_VARCHAR"})
    context.source_records['TRANSFORMED_RECORD'] = None
    context.source_records['TRANSFORMED_RECORD_PREVIOUS'] = None
    context.source_records['RESULT'] = None


@step("the plugin sources outbound records directly from a real table")
def step_impl_source_outbound_records_from_table(context):
    # as of late 2024, we've started leaning more and more on Snowflake UDTFs piping 
    # data directly to and from tables provided by the sync engine
    # so this step takes the pre-baked source records and loads them into a real table
    case.assertIn('PLUGIN_APP_NAME',os.environ, "PLUGIN_APP_NAME not found in environment, this is required when sourcing records from a table")
    context.source_app_name = os.environ['PLUGIN_APP_NAME']
    case.assert_(hasattr(context,'source_records'), "source_records not found in context, this is required when sourcing records from a table")
    # calling this step implies that a snowpark session can be created in the environment,
    # and that you it attached to the sync request
    connection_name = os.environ.get('SNOWFLAKE_CONNECTION_NAME','dev')
    session:Session = get_snowpark_session(connection_name=connection_name)
    context.session = session
    context.results_schema_name = 'TEMP_DATA'
    context.records_schema_name = 'TEMP_DATA'
    context.records_table_name = f"RECORDS_{datetime.datetime.now().strftime('%Y%m%d%H%M%S')}"
    context.results_table_name = f"RESULTS_{datetime.datetime.now().strftime('%Y%m%d%H%M%S')}"
    session.sql(f"CREATE SCHEMA IF NOT EXISTS {context.source_app_name}.{context.results_schema_name}").collect()
    session.sql(
        f"""CREATE OR REPLACE TRANSIENT TABLE {context.source_app_name}.{context.records_schema_name}.{context.records_table_name}(
                                IDENTIFIER VARCHAR(255),
                                APP_IDENTIFIER VARCHAR(255),
                                APPLY_STATE VARCHAR(255),
                                SYNC_ACTION VARCHAR(255),
                                TRANSFORMED_RECORD_VARCHAR VARCHAR,
                                TRANSFORMED_RECORD OBJECT,
                                TRANSFORMED_RECORD_PREVIOUS_VARCHAR VARCHAR,
                                TRANSFORMED_RECORD_PREVIOUS OBJECT,
                                APPLY_STATE_DATETIME TIMESTAMP_LTZ(9),
                                LOADBATCH_ID NUMBER(18,0),
                                RESULT_VARCHAR VARCHAR,
                                RESULT OBJECT)"""
    ).collect()
    session.sql(
        f"""CREATE OR REPLACE TRANSIENT TABLE {context.source_app_name}.{context.results_schema_name}.{context.results_table_name}(
                                IDENTIFIER VARCHAR(255),
                                APP_IDENTIFIER VARCHAR(255),
                                APPLY_STATE VARCHAR(255),
                                SYNC_ACTION VARCHAR(255),
                                TRANSFORMED_RECORD OBJECT,
                                APPLY_STATE_DATETIME TIMESTAMP_LTZ(9),
                                LOADBATCH_ID NUMBER(18,0),
                                RESULT OBJECT,
                                RESULT_OBJECT OBJECT)"""
    ).collect()
    source_records:pandas.DataFrame = context.source_records
    # upload the source records dataframe to the records table
    success, nchunks, nrows, _ = write_pandas(
        conn=session._conn._cursor.connection,  # pylint: disable=protected-access
        df=source_records,
        table_name=f"{context.source_app_name}.{context.records_schema_name}.{context.records_table_name}",
        quote_identifiers=False,  # already done in get_temp_table_name
        table_type="transient"
    )
    # we run an update statement to convert the TRANSFORMED_RECORD_VARCHAR and TRANSFORMED_RECORD_PREVIOUS_VARCHAR columns to OBJECT
    session.sql(
        f"""UPDATE {context.source_app_name}.{context.records_schema_name}.{context.records_table_name}
            SET TRANSFORMED_RECORD = PARSE_JSON(TRANSFORMED_RECORD_VARCHAR),
                TRANSFORMED_RECORD_PREVIOUS = PARSE_JSON(TRANSFORMED_RECORD_PREVIOUS_VARCHAR),
                RESULT = PARSE_JSON(RESULT_VARCHAR)
            """
    ).collect()
    # then drop the TRANSFORMED_RECORD_VARCHAR and TRANSFORMED_RECORD_PREVIOUS_VARCHAR columns
    session.sql(
        f"""ALTER TABLE {context.source_app_name}.{context.records_schema_name}.{context.records_table_name}
            DROP COLUMN TRANSFORMED_RECORD_VARCHAR
            """
    ).collect()
    session.sql(
        f"""ALTER TABLE {context.source_app_name}.{context.records_schema_name}.{context.records_table_name}
            DROP COLUMN TRANSFORMED_RECORD_PREVIOUS_VARCHAR
            """
    ).collect()
    session.sql(
        f"""ALTER TABLE {context.source_app_name}.{context.records_schema_name}.{context.records_table_name}
            DROP COLUMN RESULT_VARCHAR
            """
    ).collect()
    case.assertTrue(success, "Failed to write source records to table")


@step("we use the recorded results from {filename}")
def step_impl_response_inbound_from_file(context, filename):
    # the filename is a pipe-separated file, so we'll read it as such
    file_path = os.path.join(
        context._root["config"].base_dir,
        "inbound_results",
        filename,  # pylint: disable=protected-access
    )
    expected_result = pandas.read_csv(file_path, sep="|")
    expected_result = expected_result.rename(str.strip, axis = 'columns')
    # the results file contains the following columns:
    # 1. RECORD_OBJECT, needs to be cast to string and stripped of whitespace
    # 2. APP_IDENTIFIER, needs to be cast to string and stripped of whitespace
    # 3. RETRIEVE_DATE, looks like 2024-08-26 22:01:24.425258+10:00 and needs to be parsed as a datetime
    # 4. STREAM_NAME, needs to be cast to string and stripped of whitespace
    # we have to convert the RECORD_OBJECT column to a json object, named as RECORD_OBJECT
    expected_result["RECORD_OBJECT"] = expected_result["RECORD_OBJECT"].apply(json.loads)
    expected_result["APP_IDENTIFIER"] = expected_result["APP_IDENTIFIER"].astype(str)
    expected_result["APP_IDENTIFIER"] = expected_result["APP_IDENTIFIER"].str.strip()
    #expected_result["RETRIEVE_DATE"] = pandas.to_datetime(expected_result["RETRIEVE_DATE"])
    expected_result["STREAM_NAME"] = expected_result["STREAM_NAME"].astype(str)
    expected_result["STREAM_NAME"] = expected_result["STREAM_NAME"].str.strip()
    # we'll group the results by STREAM_NAME and store them in a dictionary keyed by the stream name
    context.sync_inbound_expected_results = {}
    for stream_name, group in expected_result.groupby("STREAM_NAME"):
        context.sync_inbound_expected_results[stream_name] = (group
            .sort_values(by=["APP_IDENTIFIER"])
            .sort_index(axis=1)
            .reset_index(drop=True)
        )

@step("we assert that the recorded results for the {stream_name} stream are correct")
def step_impl_assert_inbound_results(context, stream_name):
    if stream_name not in context.sync_inbound_expected_results:
        #logger.info(context.sync_inbound_result)
        raise ValueError(
            f"Stream {stream_name} not found in sync_inbound_expected_results. Keys: {list(context.sync_inbound_result.keys())}"
        )
    inbound_results_match_expected(context, stream_name, context.sync_inbound_expected_results[stream_name])

def inbound_results_match_expected(context, stream_name, expected_result:pandas.DataFrame):
    if stream_name not in context.sync_inbound_result:
        #logger.info(context.sync_inbound_result)
        raise ValueError(
            f"Stream {stream_name} not found in sync_inbound_result. Keys: {list(context.sync_inbound_result.keys())}"
        )
    
    sync_inbound_result_for_stream:pandas.DataFrame = context.sync_inbound_result[stream_name]
    # convert APP_IDENTIFIER to string (it appears as object for some reason)
    sync_inbound_result_for_stream["APP_IDENTIFIER"] = sync_inbound_result_for_stream[
        "APP_IDENTIFIER"
    ].astype(str)
    # re-sort by APP_IDENTIFIER
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_values(by=["APP_IDENTIFIER"]).sort_index(axis=1).reset_index(drop=True)
    # because behave tables work best with a string representation of json, we'll convert the results objects to string
    # expected_result['RESULT'] = expected_result['RESULT'].apply(json.dumps)
    #sync_inbound_result_for_stream["RECORD_DATA"] = sync_inbound_result_for_stream[
    #    "RECORD_DATA"
    #].apply(json.dumps)
    
    # exclude RETRIEVE_DATE column from both dataframes, naturally they will differ
    expected_result = expected_result.drop(columns=["RETRIEVE_DATE"])
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.drop(
        columns=["RETRIEVE_DATE"]
    )
    # also convert the expected_result APP_IDENTIFIER to string
    expected_result["APP_IDENTIFIER"] = expected_result["APP_IDENTIFIER"].astype(str)
    
    # drop any of the following result columns if they are not present in the expected result
    droppable_columns = ["IS_DELETED"]
    for column in droppable_columns:
        # in order to not break existing tests, if the actual result contains the IS_DELETED column but the expected result doesn't, we'll remove it from the actual result
        if column in sync_inbound_result_for_stream.columns and column not in expected_result.columns:
            sync_inbound_result_for_stream = sync_inbound_result_for_stream.drop(columns=[column])

    #expected_result = expected_result.sort_index(axis=1)
    #sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_index(axis=1)
    logger.info("Setting index of expected_result to APP_IDENTIFIER")
    expected_result.set_index("APP_IDENTIFIER", inplace=True, drop=False)
    logger.info("Sorting index of expected result")
    expected_result = expected_result.sort_index()
    logger.info("Setting index of sync_inbound_result to APP_IDENTIFIER")
    sync_inbound_result_for_stream.set_index("APP_IDENTIFIER", inplace=True, drop=False)
    logger.info("Sorting index of sync_inbound_result")
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_index()
    logger.info(f"expected_result dtypes:\n{expected_result.dtypes}")
    logger.info(f"sync_inbound_result dtypes:\n{sync_inbound_result_for_stream.dtypes}")    
    #logger.info("Checking for duplicates in expected_result")
    #duplicates = expected_result[expected_result.duplicated(['APP_IDENTIFIER'], keep="first")]['APP_IDENTIFIER'].unique().tolist()
    #case.assertEqual(
    #    len(duplicates),0, f"Duplicate APP_IDENTIFIER in expected_result:\n{duplicates.tolist()}"
    #)
    #logger.info("Checking for duplicates in sync_inbound_result")
    #duplicates = sync_inbound_result_for_stream[sync_inbound_result_for_stream.duplicated(['APP_IDENTIFIER'], keep="first")]['APP_IDENTIFIER'].unique().tolist()
    #case.assertFalse(
    #    len(duplicates),0, f"Duplicate APP_IDENTIFIER in sync_inbound_result:\n{duplicates.tolist()}"
    #)
    # remove duplicates from expected_result and sync_inbound_result
    expected_result.drop_duplicates(subset=['APP_IDENTIFIER'], keep="first", inplace=True)
    sync_inbound_result_for_stream.drop_duplicates(subset=['APP_IDENTIFIER'], keep="first", inplace=True)
    logger.info("Resetting index of expected_result to match sync_inbound_result")
    expected_result = expected_result.reindex_like(sync_inbound_result_for_stream)
    assert list(expected_result.columns) == list(
        sync_inbound_result_for_stream.columns
    ), f"Column headings didn't match. Expected: {expected_result.columns}, actual: {sync_inbound_result_for_stream.columns}"
    pandas.set_option("display.max_columns", 10)
    pandas.set_option("display.width", 150)
    logger.info(f"expected_result:\n{expected_result}")
    logger.info(f"sync_inbound_result:\n{sync_inbound_result_for_stream}")
    case.assertCountEqual(
        expected_result.columns.to_list(), sync_inbound_result_for_stream.columns.to_list()
    )
    logger.info(f"differences:\n{expected_result.compare(sync_inbound_result_for_stream)}")
    assert_frame_equal(expected_result, sync_inbound_result_for_stream)


@step("the outbound response will match the records from {filename}")
def step_impl_response_outbound_from_file(context, filename):
    # the filename is a pipe-separated file, so we'll read it as such
    file_path = os.path.join(
        context._root["config"].base_dir,
        "outbound_results",
        filename,  # pylint: disable=protected-access
    )
    expected_result = pandas.read_csv(file_path, sep="|")
    expected_result = expected_result.rename(str.strip, axis = 'columns')
    # the results file contains the following columns:
    # 1. APP_IDENTIFIER, needs to be cast to string and stripped of whitespace
    # 2. IDENTIFIER, needs to be cast to string and stripped of whitespace
    # 3. RESULT, needs to be cast to string and stripped of whitespace
    # 4. SUCCESS, needs to be cast to boolean
    if 'APP_IDENTIFIER' in expected_result.columns:
        expected_result["APP_IDENTIFIER"] = expected_result["APP_IDENTIFIER"].astype(str)
        expected_result["APP_IDENTIFIER"] = expected_result["APP_IDENTIFIER"].str.strip()
    expected_result["IDENTIFIER"] = expected_result["IDENTIFIER"].astype(str)
    expected_result["IDENTIFIER"] = expected_result["IDENTIFIER"].str.strip()
    expected_result["RESULT"] = expected_result["RESULT"].astype(str)
    expected_result["RESULT"] = expected_result["RESULT"].str.strip()
    if "SUCCESS" in expected_result.columns:
        expected_result["SUCCESS"] = expected_result["SUCCESS"].astype(bool)
    if "APPLY_STATE" in expected_result.columns:
        expected_result["APPLY_STATE"] = expected_result["APPLY_STATE"].astype(str)
        expected_result["APPLY_STATE"] = expected_result["APPLY_STATE"].str.strip()
    
    outbound_results_match_expected(context,expected_result)

def outbound_results_match_expected(context,expected_result:pandas.DataFrame):
    expected_result = (
        expected_result.sort_values(by=["IDENTIFIER"])
        .sort_index(axis=1)
        .reset_index(drop=True)
    )
    context.sync_outbound_result = (
        context.sync_outbound_result.sort_values(by=["IDENTIFIER"])
        .sort_index(axis=1)
        .reset_index(drop=True)
    )
    # drop any of the following result columns if they are not present in the expected result
    droppable_columns = ["IS_DELETED",'APPLY_STATE_DATETIME','LOADBATCH_ID','SYNC_ACTION','TRANSFORMED_RECORD','APPLY_STATE','SUCCESS','APP_IDENTIFIER','RESULT_OBJECT']
    for column in droppable_columns:
        # in order to not break existing tests, if the actual result contains the IS_DELETED column but the expected result doesn't, we'll remove it from the actual result
        if column in context.sync_outbound_result.columns and column not in expected_result.columns:
            context.sync_outbound_result = context.sync_outbound_result.drop(columns=[column])

    # because behave tables work best with a string representation of json, we'll convert the results objects to string
    # expected_result['RESULT'] = expected_result['RESULT'].apply(json.dumps)
    context.sync_outbound_result["RESULT"] = context.sync_outbound_result[
        "RESULT"
    ].apply(json.dumps)
    assert list(expected_result.columns) == list(
        context.sync_outbound_result.columns
    ), f"Column headings didn't match. Expected: {expected_result.columns}, actual: {context.sync_outbound_result.columns}"
    pandas.set_option("display.max_columns", 10)
    pandas.set_option("display.width", 150)
    logger.info(f"expected_result: {expected_result}")
    logger.info(f"expected_result single: {expected_result['RESULT'][0]}")
    logger.info(f"expected_result dtypes: {expected_result.dtypes}")
    logger.info(f"apply_result: {context.sync_outbound_result}")
    logger.info(f"apply_result single: {context.sync_outbound_result['RESULT'][0]}")
    logger.info(f"apply_result dtypes: {context.sync_outbound_result.dtypes}")
    logger.info(f"differences: {expected_result.compare(context.sync_outbound_result)}")
    case.assertCountEqual(
        expected_result.columns.to_list(),
        context.sync_outbound_result.columns.to_list(),
    )
    assert_frame_equal(expected_result, context.sync_outbound_result)


@step("the plugin returns inbound sync results directly to a real table")
def step_impl_return_inbound_records_to_table(context):
    # as of late 2024, we've started leaning more and more on Snowflake UDTFs piping 
    # data directly to and from tables provided by the sync engine
    # so this step takes the pre-baked source records and loads them into a real table
    case.assertIn('PLUGIN_APP_NAME',os.environ, "PLUGIN_APP_NAME not found in environment, this is required when reading results from a table")
    context.source_app_name = os.environ['PLUGIN_APP_NAME']
    # calling this step implies that a snowpark session can be created in the environment,
    # and that you it attached to the sync request
    connection_name = os.environ.get('SNOWFLAKE_CONNECTION_NAME','dev')
    session:Session = get_snowpark_session(connection_name=connection_name)
    context.session = session
    # we need to set the current database to the source app name
    session.sql(f"USE DATABASE {context.source_app_name}").collect()
    context.results_schema_name = 'TEMP_DATA'
    context.results_table_name = f"RESULTS_{datetime.datetime.now().strftime('%Y%m%d%H%M%S')}"
    context.full_results_table_name = f"{context.source_app_name}.{context.results_schema_name}.{context.results_table_name}"
    session.sql(f"CREATE SCHEMA IF NOT EXISTS {context.source_app_name}.{context.results_schema_name}").collect()
    session.sql(
        f"""CREATE OR REPLACE TRANSIENT TABLE {context.full_results_table_name}(
                                        APP_IDENTIFIER VARCHAR(255),
                                        STREAM_NAME VARCHAR(255),
                                        RETRIEVE_DATE TIMESTAMP_LTZ(9),
                                        RECORD_DATA VARCHAR,
                                        RECORD_OBJECT OBJECT,
                                        IS_DELETED BOOLEAN default false
                                        )"""
    ).collect()
    

@given("the following records")
def step_impl_records(context):
    logger.info(f"context.table: {context.table}")
    logger.info(f"context.table.headings: {context.table.headings}")
    context.source_records = table_to_dataframe(context.table)
    logger.info(f"source dataframe: {context.source_records}")


@given("the following streams state")
def step_impl_streams_state(context):
    context.streams_state = {}
    for row in context.table:
        if "Stream" not in row.headings:
            raise ValueError("Streams state must have a Stream column")
        if "Value" not in row.headings:
            raise ValueError("Streams state must have a Value column")
        context.streams_state[row["Stream"]] = json.loads(row["Value"])
    logger.info(f"streams state: {context.streams_state}")


@when("we perform an outbound sync with configuration parameters")
def step_impl_outbound_sync(context):
    if context.plugin_class is None:
        raise ValueError(
            'You must define which plugin class and module is to be used ("we use the x class from the y module")'
        )
    strategy = None
    connection_parameters = {}
    connection_secrets = {}
    sync_parameters = {}
    field_mappings = []
    api_limits = []
    connection_method = None

    for row in context.table:
        if row["Property"] == "strategy":
            strategy = json.loads(row["Value"])
        elif row["Property"] == "connection_method":
            connection_method = row["Value"]
        elif row["Property"] == "connection_parameters":
            connection_parameters = json.loads(jinja_render(row["Value"],context))
        elif row["Property"] == "connection_secrets":
            connection_secrets = json.loads(jinja_render(row["Value"],context))
        elif row["Property"] == "sync_parameters":
            sync_parameters = json.loads(jinja_render(row["Value"],context))
        elif row["Property"] == "field_mappings":
            field_mappings = json.loads(jinja_render(row["Value"],context))
        elif row["Property"] == "api_limits":
            api_limits = json.loads(row["Value"])
        else:
            raise ValueError(f"Unknown apply parameter {row['Property']}")

    parameters = OutboundSyncConfigurationParameters.model_validate(
        {
            "sync_strategy": strategy,
            "connection_method": connection_method,
            "connection_parameters": connection_parameters,
            "connection_secrets": connection_secrets,
            "sync_parameters": sync_parameters,
            "field_mappings": field_mappings,
        }
    )
    # With API Limits, we remove rate limits to remove needless waiting
    # TODO: override concurrency setting due to pyvcr's thread non-safety
    context.plugin_instance = context.plugin_class()
    # as of later 2024, we've started leaning more and more on Snowflake UDTFs piping data directly
    # back into the results tables.
    # But we still test the plugins in isolation, without a sync engine instance
    # so to accomodate this, we set the source_app_name to the temporary plugin instance,
    # and create a results table inside the plugin instance
    source_app_name = None
    results_schema_name = None
    records_schema_name = None
    results_table_name = None
    records_table_name = None
    if hasattr(context, "records_table_name"):
        source_app_name = context.source_app_name
        results_schema_name = context.results_schema_name
        records_schema_name = context.records_schema_name
        results_table_name = context.results_table_name
        records_table_name = context.records_table_name
    session:Optional[Session] = None
    if hasattr(context, "session"):
        session = context.session

    outbound_sync_request = OutboundSyncRequest(
        run_id=None,
        session=session,
        source_app_name=source_app_name,
        records_schema_name=records_schema_name,
        records_table_name=records_table_name,
        results_schema_name=results_schema_name,
        results_table_name=results_table_name,
        plugin_instance=context.plugin_instance,
        api_limits=api_limits,
        rate_limit_state_all={},
        rate_limit_state_this_sync_and_branch={},
        # make it 5 mins since we should be using vcr
        run_deadline=datetime.datetime.now(datetime.timezone.utc) + datetime.timedelta(minutes=5),
        development_mode=True,
        test_replay_mode=True,
    )
    if records_table_name is None:
        outbound_sync_request._prebaked_record_state = (
            context.source_records
        )  # pylint: disable=protected-access
    context.error = None
    try:
        context.sync_direction = "outbound"
        context.sync_outbound_result = context.plugin_instance.sync_outbound(
            parameters, outbound_sync_request
        )
        # when using a managed_outbound_processing decorator, the results aren't returned from sync_outbound
        if context.sync_outbound_result is None:
            context.sync_outbound_result = outbound_sync_request.get_queued_results()
        if hasattr(context,'full_results_table_name'):
            full_results_table = session.table(context.full_results_table_name)
            if full_results_table.count() > 0:
                context.sync_outbound_result = full_results_table.to_pandas()
    except Exception as exception:  # pylint: disable=broad-except
        context.error = exception
        logging.exception(exception)

@when("we perform an inbound sync with configuration parameters")
def step_impl_inbound_sync(context):
    if context.plugin_class is None:
        raise ValueError(
            'You must define which plugin class and module is to be used ("we use the x class from the y module")'
        )
    connection_parameters = {}
    connection_secrets = {}
    sync_parameters = {}
    streams = []
    api_limits = []
    connection_method = None

    for row in context.table:
        if row["Property"] == "connection_method":
            connection_method = row["Value"]
        elif row["Property"] == "connection_parameters":
            connection_parameters = json.loads(jinja_render(row["Value"],context))
        elif row["Property"] == "connection_secrets":
            connection_secrets = json.loads(jinja_render(row["Value"],context))
        elif row["Property"] == "sync_parameters":
            sync_parameters = json.loads(jinja_render(row["Value"],context))
        elif row["Property"] == "streams":
            streams = json.loads(row["Value"])
        elif row["Property"] == "api_limits":
            api_limits = json.loads(row["Value"])
        else:
            raise ValueError(f"Unknown apply parameter {row['Property']}")

    parameters = InboundSyncConfigurationParameters.model_validate(
        {
            "connection_method": connection_method,
            "connection_parameters": connection_parameters,
            "connection_secrets": connection_secrets,
            "sync_parameters": sync_parameters,
        }
    )
    source_app_name = None
    results_schema_name = None
    results_table_name = None
    if hasattr(context, "results_table_name"):
        source_app_name = context.source_app_name
        results_schema_name = context.results_schema_name
        results_table_name = context.results_table_name
    session:Optional[Session] = None
    if hasattr(context, "session"):
        session = context.session

    # With API Limits, we remove rate limits to remove needless waiting
    # TODO: override concurrency setting due to pyvcr's thread non-safety
    context.plugin_instance = context.plugin_class()
    streams_list = TypeAdapter(List[StoredStreamConfiguration]).validate_python(streams)
    for stream in streams_list:
        if stream.stream_name in context.streams_state:
            stream.latest_state = context.streams_state[stream.stream_name]
    omnata_log_handler = OmnataPluginLogHandler(
        session=session,
        sync_id=None,
        sync_branch_id=None,
        connection_id=None,
        sync_run_id=None,
    )
    # make it 5 mins since we should be using vcr
    run_deadline = datetime.datetime.now(datetime.timezone.utc) + datetime.timedelta(minutes=5)
    development_mode = True
    test_replay_mode = True
    if not hasattr(context, "cassette"):
        run_deadline = datetime.datetime.now(datetime.timezone.utc) + datetime.timedelta(minutes=30)
        test_replay_mode = False
    inbound_sync_request = InboundSyncRequest(
        run_id=1,
        session=session,
        source_app_name=source_app_name,
        results_schema_name=results_schema_name,
        results_table_name=results_table_name,
        plugin_instance=context.plugin_instance,
        api_limits=api_limits,
        rate_limit_state_all={},
        rate_limit_state_this_sync_and_branch={},
        run_deadline=run_deadline,
        development_mode=development_mode,
        streams=streams_list,
        test_replay_mode=test_replay_mode,
        omnata_log_handler=omnata_log_handler
    )

    # outbound_sync_request._prebaked_record_state = context.source_records
    context.error = None
    try:
        context.sync_direction = "inbound"
        context.plugin_instance.sync_inbound(parameters, inbound_sync_request)
        context.sync_inbound_result = (
            inbound_sync_request._apply_results
        )  # pylint: disable=protected-access
        for stream_name in context.sync_inbound_result:
            if isinstance(context.sync_inbound_result[stream_name],list):
                context.sync_inbound_result[stream_name] = pandas.concat(context.sync_inbound_result[stream_name])
                # sometimes we have a RECORD_DATA object instead of a RECORD_OBJECT object
                # if so, rename it
                if "RECORD_DATA" in context.sync_inbound_result[stream_name].columns:
                    context.sync_inbound_result[stream_name] = context.sync_inbound_result[stream_name].rename(columns={"RECORD_DATA":"RECORD_OBJECT"})
                    # we also need to parse the RECORD_OBJECT column as json
                    context.sync_inbound_result[stream_name]["RECORD_OBJECT"] = context.sync_inbound_result[stream_name]["RECORD_OBJECT"].apply(json.loads)

        if (
            len(context.sync_inbound_result) == 0
        ):  # pylint: disable=protected-access
            # have a look in the table instead
            # when using a managed_outbound_processing decorator, the results aren't returned from sync_outbound
            if hasattr(context,'full_results_table_name'):
                full_results_table = session.table(context.full_results_table_name)
                if full_results_table.count() > 0:
                    # context.sync_outbound_result is a dict keyed by stream name
                    # so group the results by the STREAM_NAME column
                    full_results = full_results_table.to_pandas()
                    # convert RECORD_OBJECT to a json object
                    full_results["RECORD_OBJECT"] = full_results["RECORD_OBJECT"].apply(json.loads)
                    context.sync_inbound_result = {}
                    for stream_name, group in full_results.groupby("STREAM_NAME"):
                        logger.info(f"Stream {stream_name} has {len(group)} records")
                        context.sync_inbound_result[stream_name] = (group
                            .sort_values(by=["APP_IDENTIFIER"])
                            .sort_index(axis=1)
                            .reset_index(drop=True)
                        )
                else:
                    logger.warning("No apply results found in inbound_sync_request._apply_results, result table bound to the sync request is empty")
            else:
                logger.warning("No apply results found in inbound_sync_request._apply_results, and no result table bound to the sync request")
                    
    except Exception as exception:  # pylint: disable=broad-except
        context.error = exception
        logging.exception(exception)


def load_script_from_file(app_name, action_name):
    f = open(os.path.join("scripts", app_name, f"{action_name}.py"), encoding="utf-8")
    script_contents = f.read()
    mod = ModuleType("whatever.py", "")
    exec(script_contents, mod.__dict__)  # pylint: disable=exec-used
    return mod

@step("we attach a Snowpark session to the sync request")
def step_impl_attach_snowpark_session(context: any):
    connection_name = os.environ.get('SNOWFLAKE_CONNECTION_NAME','dev')
    session:Session = get_snowpark_session(connection_name=connection_name)
    context.session = session
    if 'PLUGIN_APP_NAME' not in os.environ:
        raise ValueError("PLUGIN_APP_NAME not found in environment, this is required when attaching a Snowflake session to the sync request")
    context.source_app_name = os.environ['PLUGIN_APP_NAME']
    session.sql(f"USE DATABASE {context.source_app_name}").collect()

@step("we use the {plugin_class} class from the {plugin_module} module")
def step_impl_use_plugin_class(context, plugin_class, plugin_module):
    # Assuming the module is defined in the parent of the tests directory
    sys.path.insert(
        0,
        os.path.abspath(
            os.path.join(context._root["config"].base_dir, "..")
        ),  # pylint: disable=protected-access
    )
    module = importlib.import_module(plugin_module)
    context.plugin_class = getattr(module, plugin_class)


@step("when matching requests, we ignore query parameters {params}")
def step_impl_ignore_params(context, params):
    context.ignore_query_params = params.split(",")

@step("we use the HTTP recordings from {filename}")
def step_impl_use_http_recordings(context, filename):
    if hasattr(context,"cassette"):
        context.cassette.__exit__(None, None, None)
    file_name = os.path.join(
        context._root["config"].base_dir,
        "vcr_cassettes",
        filename,  # pylint: disable=protected-access
    )
    logger.info(f"using cassette at {file_name}")
    my_vcr = vcr.VCR(record_mode="none")
    if "VCRPY_LOG_LEVEL" in os.environ:
        logging.basicConfig(
            format="%(asctime)s %(levelname)-8s %(message)s",
            level=logging.INFO,
            datefmt="%Y-%m-%d %H:%M:%S",
        )
        vcr_log = logging.getLogger("vcr")
        vcr_log.setLevel(
            logging._nameToLevel[os.environ["VCRPY_LOG_LEVEL"]]
        )  # pylint: disable=protected-access
    filter_query_params = None
    if hasattr(context, "ignore_query_params"):
        filter_query_params = context.ignore_query_params
    context.cassette = my_vcr.use_cassette(file_name, filter_query_parameters=filter_query_params)
    context.cassette.__enter__()


@step("the response will be")
def step_impl_response_outbound(context):
    expected_result = table_to_dataframe(context.table)
    # cast IDENTIFIER to string (it appears as object for some reason)
    expected_result["IDENTIFIER"] = expected_result["IDENTIFIER"].astype(str)
    expected_result = (
        expected_result.sort_values(by=["IDENTIFIER"])
        .sort_index(axis=1)
        .reset_index(drop=True)
    )
    context.sync_outbound_result = (
        context.sync_outbound_result.sort_values(by=["IDENTIFIER"])
        .sort_index(axis=1)
        .reset_index(drop=True)
    )
    # because behave tables work best with a string representation of json, we'll convert the results objects to string
    # expected_result['RESULT'] = expected_result['RESULT'].apply(json.dumps)
    context.sync_outbound_result["RESULT"] = context.sync_outbound_result[
        "RESULT"
    ].apply(json.dumps)
    assert list(expected_result.columns) == list(
        context.sync_outbound_result.columns
    ), f"Column headings didn't match. Expected: {expected_result.columns}, actual: {context.sync_outbound_result.columns}"
    pandas.set_option("display.max_columns", 10)
    pandas.set_option("display.width", 150)
    logger.info(f"expected_result: {expected_result}")
    logger.info(f"expected_result single: {expected_result['RESULT'][0]}")
    logger.info(f"expected_result dtypes: {expected_result.dtypes}")
    logger.info(f"apply_result: {context.sync_outbound_result}")
    logger.info(f"apply_result single: {context.sync_outbound_result['RESULT'][0]}")
    logger.info(f"apply_result dtypes: {context.sync_outbound_result.dtypes}")
    logger.info(f"differences: {expected_result.compare(context.sync_outbound_result)}")
    case.assertCountEqual(
        expected_result.columns.to_list(),
        context.sync_outbound_result.columns.to_list(),
    )
    assert_frame_equal(expected_result, context.sync_outbound_result)


@step("the response for the {stream_name} stream will be")
def step_impl_response_inbound(context, stream_name):
    expected_result = table_to_dataframe(context.table)
    expected_result = (
        expected_result.sort_values(by=["APP_IDENTIFIER"])
        .sort_index(axis=1)
        .reset_index(drop=True)
    )
    if stream_name not in context.sync_inbound_result:
        #logger.info(context.sync_inbound_result)
        raise ValueError(
            f"Stream {stream_name} not found in sync_inbound_result. Keys: {list(context.sync_inbound_result.keys())}"
        )
    
    sync_inbound_result_for_stream:pandas.DataFrame = context.sync_inbound_result[stream_name]
    
    # convert APP_IDENTIFIER to string (it appears as object for some reason)
    sync_inbound_result_for_stream["APP_IDENTIFIER"] = sync_inbound_result_for_stream[
        "APP_IDENTIFIER"
    ].astype(str)
    # re-sort by APP_IDENTIFIER
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_values(by=["APP_IDENTIFIER"]).sort_index(axis=1).reset_index(drop=True)
    # because behave tables work best with a string representation of json, we'll convert the results objects to string
    # expected_result['RESULT'] = expected_result['RESULT'].apply(json.dumps)
    #sync_inbound_result_for_stream["RECORD_DATA"] = sync_inbound_result_for_stream[
    #    "RECORD_DATA"
    #].apply(json.dumps)
    
    # exclude RETRIEVE_DATE column from both dataframes, naturally they will differ
    expected_result = expected_result.drop(columns=["RETRIEVE_DATE"])
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.drop(
        columns=["RETRIEVE_DATE"]
    )
    # also convert the expected_result APP_IDENTIFIER to string
    expected_result["APP_IDENTIFIER"] = expected_result["APP_IDENTIFIER"].astype(str)
    # in order to not break existing tests, if the actual result contains the IS_DELETED column but the expected result doesn't, we'll remove it from the actual result
    if "IS_DELETED" in sync_inbound_result_for_stream.columns and "IS_DELETED" not in expected_result.columns:
        sync_inbound_result_for_stream = sync_inbound_result_for_stream.drop(columns=["IS_DELETED"])

    #expected_result = expected_result.sort_index(axis=1)
    #sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_index(axis=1)
    logger.info("Setting index of expected_result to APP_IDENTIFIER")
    expected_result.set_index("APP_IDENTIFIER", inplace=True, drop=False)
    logger.info("Sorting index of expected result")
    expected_result = expected_result.sort_index()
    logger.info("Setting index of sync_inbound_result to APP_IDENTIFIER")
    sync_inbound_result_for_stream.set_index("APP_IDENTIFIER", inplace=True, drop=False)
    logger.info("Sorting index of sync_inbound_result")
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_index()
    logger.info(f"expected_result dtypes:\n{expected_result.dtypes}")
    logger.info(f"sync_inbound_result dtypes:\n{sync_inbound_result_for_stream.dtypes}")    
    #logger.info("Checking for duplicates in expected_result")
    #duplicates = expected_result[expected_result.duplicated(['APP_IDENTIFIER'], keep="first")]['APP_IDENTIFIER'].unique().tolist()
    #case.assertEqual(
    #    len(duplicates),0, f"Duplicate APP_IDENTIFIER in expected_result:\n{duplicates.tolist()}"
    #)
    #logger.info("Checking for duplicates in sync_inbound_result")
    #duplicates = sync_inbound_result_for_stream[sync_inbound_result_for_stream.duplicated(['APP_IDENTIFIER'], keep="first")]['APP_IDENTIFIER'].unique().tolist()
    #case.assertFalse(
    #    len(duplicates),0, f"Duplicate APP_IDENTIFIER in sync_inbound_result:\n{duplicates.tolist()}"
    #)
    # remove duplicates from expected_result and sync_inbound_result
    expected_result.drop_duplicates(subset=['APP_IDENTIFIER'], keep="first", inplace=True)
    sync_inbound_result_for_stream.drop_duplicates(subset=['APP_IDENTIFIER'], keep="first", inplace=True)
    logger.info("Resetting index of expected_result to match sync_inbound_result")
    expected_result = expected_result.reindex_like(sync_inbound_result_for_stream)
    assert list(expected_result.columns) == list(
        sync_inbound_result_for_stream.columns
    ), f"Column headings didn't match. Expected: {expected_result.columns}, actual: {sync_inbound_result_for_stream.columns}"
    pandas.set_option("display.max_columns", 10)
    pandas.set_option("display.width", 150)
    logger.info(f"expected_result:\n{expected_result}")
    logger.info(f"sync_inbound_result:\n{sync_inbound_result_for_stream}")
    case.assertCountEqual(
        expected_result.columns.to_list(), sync_inbound_result_for_stream.columns.to_list()
    )
    logger.info(f"differences:\n{expected_result.compare(sync_inbound_result_for_stream)}")
    assert_frame_equal(expected_result, sync_inbound_result_for_stream)

@step("we exclude the following fields from the inbound results for the {stream_name} stream: {fields}")
def step_impl_exclude_fields(context, stream_name, fields):
    fields: List[str] = fields.split(",")
    if stream_name in context.sync_inbound_result:
        stream_frame = context.sync_inbound_result[stream_name]
        stream_frame.loc[
            stream_frame["STREAM_NAME"] == stream_name,
            "RECORD_OBJECT",
        ] = stream_frame.loc[
            stream_frame["STREAM_NAME"] == stream_name,
            "RECORD_OBJECT",
        ].apply(
            lambda x: {
                k: v
                for k, v in x.items()
                if k not in fields
            }
        )
        context.sync_inbound_result[stream_name] = stream_frame

@step("we exclude all inbound results where the field {field_name} has the value {value}")
def step_impl_delete_nulls(context, field_name, value):
    for stream_name in context.sync_inbound_result:
        total_matches = 0
        df:pandas.DataFrame = context.sync_inbound_result[stream_name]
        df["value_test"] = df["RECORD_OBJECT"].apply(
            lambda x: x[field_name] if x is not None and field_name in x else None
        )
        if value == 'true':
            value = True
        elif value == 'false':
            value = False
        total_matches += len(df[df["value_test"] == value])
        context.sync_inbound_result[stream_name] = df[df["value_test"] != value]
        logger.info(f"Deleted {total_matches} records from {stream_name} where {field_name} is {value}")


@step("when indexed on the {field_name} RECORD_OBJECT field, we assert that the recorded results for the {stream_name} stream are correct")
def step_impl_assert_inbound_results(context, field_name, stream_name):
    if stream_name not in context.sync_inbound_expected_results:
        raise ValueError(
            f"Stream {stream_name} not found in sync_inbound_expected_results. Keys: {list(context.sync_inbound_result.keys())}"
        )
    expected_result:pandas.DataFrame = context.sync_inbound_expected_results[stream_name]
    if stream_name not in context.sync_inbound_result:
        raise ValueError(
            f"Stream {stream_name} not found in sync_inbound_result. Keys: {list(context.sync_inbound_result.keys())}"
        )
    sync_inbound_result_for_stream:pandas.DataFrame = context.sync_inbound_result[stream_name]
    
    # determine the names of all columns that aren't named RECORD_OBJECT
    non_record_object_columns = [column for column in sync_inbound_result_for_stream.columns if column != "RECORD_OBJECT"]
    # drop all columns that aren't named RECORD_OBJECT
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.drop(columns=non_record_object_columns)
    # output a little preview
    logger.info(f"Preview of sync_inbound_result_for_stream:\n{sync_inbound_result_for_stream.head()}")
    # same for expected_result
    non_record_object_columns = [column for column in expected_result.columns if column != "RECORD_OBJECT"]
    expected_result = expected_result.drop(columns=non_record_object_columns)

    # now for both dataframes, add a column named test_index that contains the value of the {field_name} field in RECORD_OBJECT
    # sync_inbound_result_for_stream will be in JSON as the Snowflake connector returns it as such
    sync_inbound_result_for_stream["test_index"] = sync_inbound_result_for_stream["RECORD_OBJECT"].apply(
        lambda x: x[field_name] if x is not None and field_name in x else None
    )
    expected_result["test_index"] = expected_result["RECORD_OBJECT"].apply(
        lambda x: x[field_name] if x is not None and field_name in x else None
    )
    # now we'll set the index of both dataframes to the test_index column
    sync_inbound_result_for_stream.set_index("test_index", inplace=True, drop=False)
    expected_result.set_index("test_index", inplace=True, drop=False)
    # count how many null values are in the test_index column
    nulls_in_sync_inbound = sync_inbound_result_for_stream["test_index"].isnull().sum()
    nulls_in_expected = expected_result["test_index"].isnull().sum()
    # log all rows with null values in the test_index column
    rows_with_nulls = sync_inbound_result_for_stream[sync_inbound_result_for_stream['test_index'].isnull()]
    logger.info(f"Rows with null values in test_index column in sync_inbound_result:\n{rows_with_nulls}")
    # log the first row, column RECORD_OBJECT
    #logger.info(f"First row, column RECORD_OBJECT in sync_inbound_result:\n{rows_with_nulls.iloc[0]['RECORD_OBJECT']}")
    # raise an error if there are any null values in the test_index column
    case.assertEqual(
        nulls_in_sync_inbound, 0, f"Null values in test_index column in sync_inbound_result: {nulls_in_sync_inbound}. Total rows: {len(sync_inbound_result_for_stream)}"
    )
    case.assertEqual(
        nulls_in_expected, 0, f"Null values in test_index column in expected_result: {nulls_in_expected}. Total rows: {len(expected_result)}"
    )
    logger.info("Sorting index of expected result")
    expected_result = expected_result.sort_index()
    logger.info("Sorting index of sync_inbound_result")
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_index()
    assert list(expected_result.columns) == list(
        sync_inbound_result_for_stream.columns
    ), f"Column headings didn't match. Expected: {expected_result.columns}, actual: {sync_inbound_result_for_stream.columns}"
    pandas.set_option("display.max_columns", 10)
    pandas.set_option("display.width", 150)
    logger.info(f"expected_result:\n{expected_result}")
    logger.info(f"sync_inbound_result:\n{sync_inbound_result_for_stream}")
    logger.info(f"expected_result dtypes:\n{expected_result.dtypes}")
    logger.info(f"sync_inbound_result dtypes:\n{sync_inbound_result_for_stream.dtypes}")
    # check for duplicates in the indexes of both dataframes, raise an error if there are any
    duplicates = expected_result[expected_result.duplicated(['test_index'], keep="first")]['test_index'].unique().tolist()
    case.assertEqual(
        len(duplicates),0, f"Duplicate test_index in expected_result:\n{duplicates}"
    )
    duplicates = sync_inbound_result_for_stream[sync_inbound_result_for_stream.duplicated(['test_index'], keep="first")]['test_index'].unique().tolist()
    case.assertEqual(
        len(duplicates),0, f"Duplicate test_index in sync_inbound_result:\n{duplicates}"
    )
    logger.info("Resetting index of expected_result to match sync_inbound_result")
    expected_result = expected_result.reindex_like(sync_inbound_result_for_stream)
    case.assertCountEqual(
        expected_result.columns.to_list(), sync_inbound_result_for_stream.columns.to_list()
    )
    # it's best to compare the RECORD_OBJECT columns as json strings
    expected_result["RECORD_OBJECT"] = expected_result["RECORD_OBJECT"].apply(json.dumps)
    sync_inbound_result_for_stream["RECORD_OBJECT"] = sync_inbound_result_for_stream["RECORD_OBJECT"].apply(json.dumps)
    differences = expected_result.compare(sync_inbound_result_for_stream)
    if len(differences) > 0:
        first_difference = differences.iloc[0]
        logger.info(f"First difference self:\n{first_difference['RECORD_OBJECT','self']}")
        logger.info(f"First difference other:\n{first_difference['RECORD_OBJECT','other']}")
    logger.info(f"differences:\n{differences}")
    assert_frame_equal(expected_result, sync_inbound_result_for_stream)

@step("no error will be raised")
def step_impl_no_error(context):
    assert (
        context.error is None
    ), f"Expected no error from action, instead got {context.error}"

def jinja_environment():
    env = Environment()
    return env

def jinja_arguments(context):
    return {
        "environ": os.environ,
    }

def jinja_render(template:str, context):
    variables = {}
    if hasattr(context, "variables"):
        variables = context.variables
    return jinja_environment().from_string(template).render(
        {**jinja_arguments(context), **variables}
    )