# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/10_staking.ipynb.

# %% auto 0
__all__ = ['BaseStaker', 'NumeraiClassicStaker', 'NumeraiSignalsStaker']

# %% ../nbs/10_staking.ipynb 4
import numpy as np
from typing import Union
from tqdm.auto import tqdm
from numerapi import NumerAPI
from typeguard import typechecked
from rich import print as rich_print

from .key import Key, load_key_from_json

# %% ../nbs/10_staking.ipynb 7
@typechecked
class BaseStaker:
    """
    Base class for staking functionality which holds in both Numerai Classic and Signals.

    :param key: a `Key` object \n
    :param tournament_number: ID of the tournament (8 for Numerai Classic and 11 for Numerai Signals).
    """
    def __init__(self, key: Key, tournament_number: int, *args, **kwargs):
        self.tournament_number = tournament_number
        self.api = NumerAPI(public_id=key.pub_id, secret_key=key.secret_key, *args, **kwargs)

    def stake_change(self, model_name: str, amount: Union[float, str]):
        """
        Change arbitrary stake amount.
        :param model_name: Lowercase raw model name (For example, 'integration_test').
        :param amount: NMR amount to increase or decrease.
        negative number = Stake decrease.
        positive number = Stake increase.
        """
        action = 'decrease' if amount < 0 else 'increase'
        color = 'red' if amount < 0 else 'green'
        model_id = self._get_model_id(model_name=model_name)
        rich_print(f"[bold {color}]{action[:-1]}ing[/bold {color}] stake by amount '[bold {color}]{amount}[/bold {color}]' for model: [bold blue]'{model_name}'[/bold blue] (id='{model_id}')!")
        self.api.stake_change(nmr=amount, action=action,
                              model_id=model_id, tournament=self.tournament_number)

    def stake_drain_all(self):
        """
        -WARNING- This function will remove all stakes for all models in your account!!! -WARNING-
        User must answer prompt with 'Y' to confirm stake drain.
        """
        prompt = input(f"WARNING: You are about to remove all stakes for all your models! Are you sure? [Y/n]")
        if prompt == "Y":
            model_names = list(self.get_model_mapping.keys())
            for name in tqdm(model_names, desc="Full stake drain!"):
                self.stake_drain_single(model_name=name)
        else:
            rich_print("Aborting full stake draining.")

    def stake_drain_single(self, model_name: str):
        """
        -WARNING- This function removes your full stake for a given model name!!! -WARNING-
        :param model_name: Lowercase raw model name (For example, 'integration_test').
        """
        model_id = self._get_model_id(model_name=model_name)
        rich_print(f":warning: Draining stake for model: '{model_name}' (id: '{model_id}' :warning:")
        self.api.stake_drain(model_id=model_id, tournament=self.tournament_number)

    def _get_model_id(self, model_name: str) -> str:
        """
        Get ID needed for staking.
        :param model_name: Lowercase raw model name (For example, 'integration_test').
        """
        return self.get_model_mapping[model_name]

    @property
    def get_model_mapping(self) -> dict:
        """ Mapping between raw model names and model IDs. """
        return self.api.get_models()

    @property
    def available_nmr(self):
        """ Get amount of NMR that is available in your local wallet. """
        return np.float64(self.api.get_account()['availableNmr'])

# %% ../nbs/10_staking.ipynb 22
class NumeraiClassicStaker(BaseStaker):
    """
    Staking functionality specific to Numerai Classic (tournament number 8). \n
    :param key: a `Key` object.
    """
    CLASSIC_TOURNAMENT_NUMBER = 8
    def __init__(self, key: Key, *args, **kwargs):
        super().__init__(key=key, tournament_number=self.CLASSIC_TOURNAMENT_NUMBER,
                         *args, **kwargs
                         )

    @property
    def get_stakes(self) -> dict:
        """
        Get mapping of stakes for all models.
        Example output:
            {
            "my_model_1": 20.2,
            "my_model_2": 100.5,
            "my_model_3: 0.01
            }
        """
        stakes = [self._get_single_stake(model_name=model_name) for model_name in self.get_model_mapping.keys()]
        return {name: stake for name, stake in zip(self.get_model_mapping, stakes)}

    def _get_single_stake(self, model_name: str) -> np.float64:
        return np.float64(self.api.stake_get(username=model_name))

# %% ../nbs/10_staking.ipynb 28
class NumeraiSignalsStaker(BaseStaker):
    """
    Staking functionality specific to Numerai Signals (tournament number 11). \n
    :param key: a `Key` object.
    """
    SIGNALS_TOURNAMENT_NUMBER = 11
    def __init__(self, key: Key, *args, **kwargs):
        super().__init__(key=key,
                         tournament_number=self.SIGNALS_TOURNAMENT_NUMBER,
                         *args, **kwargs
                         )
