from airflow import DAG
from kubernetes.client import models as k8s
from datetime import datetime
from typing import Callable

from flowui.client.flowui_backend_client import FlowuiBackendRestClient
from flowui.operators.python_operator import PythonOperator
from flowui.operators.bash_operator import BashOperator
from flowui.operators.k8s_operator import FlowuiKubernetesPodOperator
from flowui.schemas.shared_storage import shared_storage_map
from flowui.utils import dict_deep_update
from flowui.logger import get_configured_logger
from flowui.schemas.shared_storage import StorageSource
from flowui.schemas.container_resources import ContainerResourcesModel
from kubernetes import client, config
import os

class Task(object):
    """
    The Task object represents a task in a workflow. 
    It is only instantiated by processes parsing dag files in Airflow.
    """

    def __init__(
        self, 
        dag: DAG, 
        task_id: str,
        operator: dict,
        operator_input_kwargs: dict,
        workflow_shared_storage: dict = None,
        container_resources: dict = None,
        deploy_mode: str = 'k8s',
    ) -> None:
        # Task configuration and attributes
        self.task_id = task_id
        # Logger
        self.logger = get_configured_logger(f"{self.__class__.__name__ }-{self.task_id}")
        self.logger.info('### Configuring task object ###')
        self.dag = dag
        self.dag_id = self.dag.dag_id
        self.repository_id = operator["repository_id"]
        self.operator = operator
        if not workflow_shared_storage:
            workflow_shared_storage = {}
        # Shared storage
        workflow_shared_storage_source = StorageSource(workflow_shared_storage.pop("source", "None")).name
        provider_options = workflow_shared_storage.pop("provider_options", {})
        self.workflow_shared_storage = shared_storage_map[workflow_shared_storage_source](**workflow_shared_storage, **provider_options) if shared_storage_map[workflow_shared_storage_source] else shared_storage_map[workflow_shared_storage_source]
        # Container resources
        self.container_resources = container_resources if container_resources else {}
        self.provide_gpu = self.container_resources.pop("use_gpu", False)
        
        self.deploy_mode = deploy_mode

        config.load_incluster_config()
        self.k8s_client = client.CoreV1Api()

        # Clients
        self.backend_client = FlowuiBackendRestClient(base_url="http://flowui-backend-service:8000/")

        # Set up Airflow operator using custom function
        self._task_operator = self._set_operator(operator_input_kwargs)

    def _set_operator(self, operator_input_kwargs) -> None:
        """
        Set airflow operator based on task configuration
        """

        if self.deploy_mode == "local-python":
            return PythonOperator(
                dag=self.dag,
                task_id=self.task_id,
                start_date=datetime(2021, 1, 1), # TODO - get correct start_date
                provide_context=True,
                op_kwargs=operator_input_kwargs,
                # queue=dependencies_group,
                make_python_callable_kwargs=dict(
                    operator_name=self.operator_name,
                    deploy_mode=self.deploy_mode,
                    task_id=self.task_id,
                    dag_id=self.dag_id,
                )
            )
        
        elif self.deploy_mode == "local-bash":
            cmds = 'source /opt/airflow/flowui_env/bin/activate && pip install -e /opt/flowui && flowui run-operator-bash'
            queue_name = self.operator.get("repository") + '-' + self.operator.get("dependency")
            return BashOperator(
                dag=self.dag,
                task_id=self.task_id,
                queue=queue_name,
                bash_command=cmds,
                env={
                    "FLOWUI_BASHOPERATOR_OPERATOR": str(self.operator),
                    "FLOWUI_BASHOPERATOR_INSTANTIATE_OP_KWARGS": str({
                        "deploy_mode": self.deploy_mode,
                        "task_id": self.task_id,
                        "dag_id": self.dag_id,
                    }),
                    "FLOWUI_BASHOPERATOR_RUN_OP_KWARGS": str(operator_input_kwargs),
                },
                append_env=True
            )

        # Ref: https://airflow.apache.org/docs/apache-airflow/1.10.14/_api/airflow/contrib/operators/kubernetes_pod_operator/index.html
        # Good example: https://github.com/apache/airflow/blob/main/tests/system/providers/cncf/kubernetes/example_kubernetes.py
        # Commands HAVE to go in a list object, ref: https://stackoverflow.com/a/55149915/11483674
        elif self.deploy_mode == "k8s":
            # https://airflow.apache.org/docs/apache-airflow/stable/templates-ref.html
            # https://www.astronomer.io/guides/templating/
            container_env_vars = {
                "FLOWUI_K8S_OPERATOR": self.operator["name"],
                "FLOWUI_K8S_INSTANTIATE_OP_KWARGS": str({
                    "deploy_mode": self.deploy_mode,
                    "task_id": self.task_id,
                    "dag_id": self.dag_id,
                }),
                "FLOWUI_K8S_RUN_OP_KWARGS": str(operator_input_kwargs),
                "FLOWUI_WORKFLOW_SHARED_STORAGE": self.workflow_shared_storage.json() if self.workflow_shared_storage else "",
                "AIRFLOW_CONTEXT_EXECUTION_DATETIME": "{{ dag_run.logical_date | ts_nodash }}",
                "AIRFLOW_CONTEXT_DAG_RUN_ID": "{{ run_id }}",
            }
            base_container_resources_model = ContainerResourcesModel(
                requests={
                    "cpu": "100m",
                    "memory": "128Mi",
                },
                limits={
                    "cpu": "100m",
                    "memory": "128Mi",
                }
            )
            
            # Container resources
            basic_container_resources = base_container_resources_model.dict()
            basic_container_resources = dict_deep_update(basic_container_resources, self.container_resources)
            if self.provide_gpu:
                basic_container_resources["limits"]["nvidia.com/gpu"] = "1"
            #self.container_resources = ContainerResourcesModel(**self.container_resources)
            container_resources_obj = k8s.V1ResourceRequirements(**basic_container_resources)

            # Volumes
            all_volumes = []
            all_volume_mounts = []

            ######################## For local Operators dev ###########################################
            if os.environ.get('FLOWUI_DEV_MODE') == 'local':
                source_image = self.operator.get('source_image')
                repository_raw_project_name = str(source_image).split('/')[1].split(':')[0]
                persistent_volume_claim_name = 'pvc-{}'.format(str(repository_raw_project_name.lower().replace('_', '-')))

                persistent_volume_name = 'pv-{}'.format(str(repository_raw_project_name.lower().replace('_', '-')))
                persistent_volume_claim_name = 'pvc-{}'.format(str(repository_raw_project_name.lower().replace('_', '-')))

                pvc_exists = False
                try:
                    print('Checking for PVC')
                    self.k8s_client.read_namespaced_persistent_volume_claim(name=persistent_volume_claim_name, namespace='default')
                    pvc_exists = True
                except client.rest.ApiException as e:
                    if e.status != 404:
                        raise e

                pv_exists = False
                try:
                    self.k8s_client.read_persistent_volume(name=persistent_volume_name)
                    pv_exists = True
                except client.rest.ApiException as e:
                    if e.status != 404:
                        raise e

                if pv_exists and pvc_exists:
                    volume_dev_operators = k8s.V1Volume(
                        name='dev-op-{path_name}'.format(path_name=str(repository_raw_project_name.lower().replace('_', '-'))),
                        persistent_volume_claim=k8s.V1PersistentVolumeClaimVolumeSource(
                            claim_name=persistent_volume_claim_name
                        ),
                    )
                    volume_mount_dev_operators = k8s.V1VolumeMount(
                        name='dev-op-{path_name}'.format(path_name=str(repository_raw_project_name.lower().replace('_', '-'))), 
                        mount_path=f'/home/flowui/operators_repository',
                        sub_path=None, 
                        read_only=True
                    )
                    all_volumes.append(volume_dev_operators)
                    all_volume_mounts.append(volume_mount_dev_operators)
                
                ######################## For local FlowUI dev ###############################################

                flowui_package_local_claim_name = 'flowui-dev-volume-claim'
                pvc_exists = False
                try:
                    self.k8s_client.read_namespaced_persistent_volume_claim(name=flowui_package_local_claim_name, namespace='default')
                    pvc_exists = True
                except client.rest.ApiException as e:
                    if e.status != 404:
                        raise e

                if pvc_exists:
                    volume_dev = k8s.V1Volume(
                        name='jobs-persistent-storage-dev',
                        persistent_volume_claim=k8s.V1PersistentVolumeClaimVolumeSource(claim_name=flowui_package_local_claim_name),
                    )
                    volume_mount_dev = k8s.V1VolumeMount(
                        name='jobs-persistent-storage-dev', 
                        mount_path='/home/flowui_dev', 
                        sub_path=None, 
                        read_only=True
                    )
                    all_volumes.append(volume_dev)
                    all_volume_mounts.append(volume_mount_dev)
            ############################################################################################

            pod_startup_timeout_in_seconds = 600
            return FlowuiKubernetesPodOperator(
                operator_name=self.operator.get('name'),
                repository_id=self.repository_id,
                workflow_shared_storage=self.workflow_shared_storage,
                namespace='default',  # TODO - separate namespace by User or Workspace?
                image=self.operator["source_image"],
                image_pull_policy='IfNotPresent',
                task_id=self.task_id,
                name=f"airflow-worker-pod-{self.task_id}",
                startup_timeout_seconds=pod_startup_timeout_in_seconds,
                # cmds=["python"],
                # arguments=["-c", "'import time; time.sleep(10000)'"],
                cmds=["flowui"],
                arguments=["run-operator-k8s"],
                env_vars=container_env_vars,
                do_xcom_push=True,
                in_cluster=True,
                # labels={"foo": "bar"},
                # secrets=[secret_file, secret_env, secret_all_keys],
                # ports=[port],
                volumes=all_volumes,
                volume_mounts=all_volume_mounts,
                container_resources=container_resources_obj,
                # env_from=configmaps,
                # affinity=affinity,
                # is_delete_operator_pod=True,
                # hostnetwork=False,
                # tolerations=tolerations,
                # init_containers=[init_container],
                # priority_class_name="medium",
            )

    def __call__(self) -> Callable:
        return self._task_operator