"""
Example usage of dry run:

from flowui.scripts.run_operator_dry import run_operator_dry

run_operator_dry(
    repository_folder_path=".", 
    operator_name="DownloadPicsumImageOperator", 
    operator_input={
        "width_pixels": 200,
        "height_pixels": 200,
        "save_as_file": False
    }
)
"""
from pathlib import Path
import importlib
import json
import sys




def run_operator_dry(
    repository_folder_path: str, 
    operator_name: str, 
    operator_input: dict,
    secrets_input: dict = None,
    results_path: str = None
):
    if not results_path:
        results_path = Path('./dry_run_results')
        results_path.mkdir(parents=True, exist_ok=True)

    operators_folder_path = str(Path(repository_folder_path).resolve() / "operators")
    if operators_folder_path not in sys.path:
        sys.path.append(operators_folder_path)

    # Load Operator class
    importlib.invalidate_caches()
    operator_module = importlib.import_module(f"{operator_name}.operator")
    operator_class = getattr(operator_module, operator_name)

    # Load metadata
    metadata_path = Path(operators_folder_path) / f"{operator_name}/metadata.json"
    with open(metadata_path, "r") as f:
        metadata = json.load(f)
    
    # Set Operator class metadata
    operator_class.set_metadata(metadata=metadata)

    # Load Operator Models
    importlib.invalidate_caches()
    operator_model_module = importlib.import_module(f"{operator_name}.models")
    operator_input_model_class = getattr(operator_model_module, "InputModel")
    operator_output_model_class = getattr(operator_model_module, "OutputModel")
    operator_secrets_model_class = getattr(operator_model_module, "SecretsModel", None)

    # Dry run Operator
    return operator_class.dry_run(
        operator_input=operator_input,
        operator_input_model=operator_input_model_class,
        operator_output_model=operator_output_model_class,
        operator_secrets_model=operator_secrets_model_class,
        secrets_input=secrets_input,
        results_path=str(results_path)
    )
