"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const ssm = require("../lib");
module.exports = {
    'creating a String SSM Parameter'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ssm.StringParameter(stack, 'Parameter', {
            allowedPattern: '.*',
            description: 'The value Foo',
            parameterName: 'FooParameter',
            stringValue: 'Foo',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            AllowedPattern: '.*',
            Description: 'The value Foo',
            Name: 'FooParameter',
            Type: 'String',
            Value: 'Foo',
        }));
        test.done();
    },
    'String SSM Parameter rejects invalid values'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new ssm.StringParameter(stack, 'Parameter', { allowedPattern: '^Bar$', stringValue: 'FooBar' }), /does not match the specified allowedPattern/);
        test.done();
    },
    'String SSM Parameter allows unresolved tokens'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.doesNotThrow(() => {
            new ssm.StringParameter(stack, 'Parameter', {
                allowedPattern: '^Bar$',
                stringValue: cdk.Lazy.stringValue({ produce: () => 'Foo!' }),
            });
        });
        test.done();
    },
    'creating a StringList SSM Parameter'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ssm.StringListParameter(stack, 'Parameter', {
            allowedPattern: '(Foo|Bar)',
            description: 'The values Foo and Bar',
            parameterName: 'FooParameter',
            stringListValue: ['Foo', 'Bar'],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            AllowedPattern: '(Foo|Bar)',
            Description: 'The values Foo and Bar',
            Name: 'FooParameter',
            Type: 'StringList',
            Value: 'Foo,Bar',
        }));
        test.done();
    },
    'StringList SSM Parameter values cannot contain commas'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new ssm.StringListParameter(stack, 'Parameter', { stringListValue: ['Foo,Bar'] }), /cannot contain the ',' character/);
        test.done();
    },
    'StringList SSM Parameter rejects invalid values'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new ssm.StringListParameter(stack, 'Parameter', { allowedPattern: '^(Foo|Bar)$', stringListValue: ['Foo', 'FooBar'] }), /does not match the specified allowedPattern/);
        test.done();
    },
    'StringList SSM Parameter allows unresolved tokens'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.doesNotThrow(() => new ssm.StringListParameter(stack, 'Parameter', {
            allowedPattern: '^(Foo|Bar)$',
            stringListValue: ['Foo', cdk.Lazy.stringValue({ produce: () => 'Baz!' })],
        }));
        test.done();
    },
    'parameterArn is crafted correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const param = new ssm.StringParameter(stack, 'Parameter', { stringValue: 'Foo' });
        // THEN
        test.deepEqual(stack.resolve(param.parameterArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ssm:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':parameter',
                    { Ref: 'Parameter9E1B4FBA' }
                ]]
        });
        test.done();
    },
    'StringParameter.fromStringParameterName'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const param = ssm.StringParameter.fromStringParameterName(stack, 'MyParamName', 'MyParamName');
        // THEN
        test.deepEqual(stack.resolve(param.parameterArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ssm:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':parameterMyParamName'
                ]]
        });
        test.deepEqual(stack.resolve(param.parameterName), 'MyParamName');
        test.deepEqual(stack.resolve(param.parameterType), 'String');
        test.deepEqual(stack.resolve(param.stringValue), { Ref: 'MyParamNameParameter' });
        assert_1.expect(stack).toMatch({
            Parameters: {
                MyParamNameParameter: {
                    Type: "AWS::SSM::Parameter::Value<String>",
                    Default: "MyParamName"
                }
            }
        });
        test.done();
    },
    'StringParameter.fromStringParameterAttributes'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const param = ssm.StringParameter.fromStringParameterAttributes(stack, 'MyParamName', {
            parameterName: 'MyParamName',
            version: 2
        });
        // THEN
        test.deepEqual(stack.resolve(param.parameterArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ssm:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':parameterMyParamName'
                ]]
        });
        test.deepEqual(stack.resolve(param.parameterName), 'MyParamName');
        test.deepEqual(stack.resolve(param.parameterType), 'String');
        test.deepEqual(stack.resolve(param.stringValue), '{{resolve:ssm:MyParamName:2}}');
        test.done();
    },
    'StringListParameter.fromName'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const param = ssm.StringListParameter.fromStringListParameterName(stack, 'MyParamName', 'MyParamName');
        // THEN
        test.deepEqual(stack.resolve(param.parameterArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ssm:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':parameterMyParamName'
                ]]
        });
        test.deepEqual(stack.resolve(param.parameterName), 'MyParamName');
        test.deepEqual(stack.resolve(param.parameterType), 'StringList');
        test.deepEqual(stack.resolve(param.stringListValue), { 'Fn::Split': [',', '{{resolve:ssm:MyParamName}}'] });
        test.done();
    },
    'fromLookup will use the SSM context provider to read value during synthesis'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'my-staq', { env: { region: 'us-east-1', account: '12344' } });
        // WHEN
        const value = ssm.StringParameter.valueFromLookup(stack, 'my-param-name');
        // THEN
        test.deepEqual(value, 'dummy-value-for-my-param-name');
        test.deepEqual(app.synth().manifest.missing, [
            {
                key: 'ssm:account=12344:parameterName=my-param-name:region=us-east-1',
                props: {
                    account: '12344',
                    region: 'us-east-1',
                    parameterName: 'my-param-name'
                },
                provider: 'ssm'
            }
        ]);
        test.done();
    },
    'valueForStringParameter': {
        'returns a token that represents the SSM parameter value'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const value = ssm.StringParameter.valueForStringParameter(stack, 'my-param-name');
            // THEN
            assert_1.expect(stack).toMatch({
                Parameters: {
                    SsmParameterValuemyparamnameC96584B6F00A464EAD1953AFF4B05118Parameter: {
                        Type: "AWS::SSM::Parameter::Value<String>",
                        Default: "my-param-name"
                    }
                }
            });
            test.deepEqual(stack.resolve(value), { Ref: 'SsmParameterValuemyparamnameC96584B6F00A464EAD1953AFF4B05118Parameter' });
            test.done();
        },
        'de-dup based on parameter name'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            ssm.StringParameter.valueForStringParameter(stack, 'my-param-name');
            ssm.StringParameter.valueForStringParameter(stack, 'my-param-name');
            ssm.StringParameter.valueForStringParameter(stack, 'my-param-name-2');
            ssm.StringParameter.valueForStringParameter(stack, 'my-param-name');
            // THEN
            assert_1.expect(stack).toMatch({
                Parameters: {
                    SsmParameterValuemyparamnameC96584B6F00A464EAD1953AFF4B05118Parameter: {
                        Type: "AWS::SSM::Parameter::Value<String>",
                        Default: "my-param-name"
                    },
                    SsmParameterValuemyparamname2C96584B6F00A464EAD1953AFF4B05118Parameter: {
                        Type: "AWS::SSM::Parameter::Value<String>",
                        Default: "my-param-name-2"
                    }
                }
            });
            test.done();
        },
        'can query actual SSM Parameter Names, multiple times'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            ssm.StringParameter.valueForStringParameter(stack, '/my/param/name');
            ssm.StringParameter.valueForStringParameter(stack, '/my/param/name');
            test.done();
        },
    }
};
//# sourceMappingURL=data:application/json;base64,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