"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const ssm = require("./ssm.generated");
/**
 * Basic features shared across all types of SSM Parameters.
 */
class ParameterBase extends core_1.Resource {
    grantRead(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'ssm:DescribeParameters',
                'ssm:GetParameters',
                'ssm:GetParameter',
                'ssm:GetParameterHistory'
            ],
            resourceArns: [this.parameterArn],
        });
    }
    grantWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['ssm:PutParameter'],
            resourceArns: [this.parameterArn],
        });
    }
}
const STRING_PARAM_TYPE = 'String';
const SECURE_STRING_PARAM_TYPE = 'SecureString';
const STRINGLIST_PARAM_TYPE = 'StringList';
/**
 * Creates a new String SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringParameter extends ParameterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.allowedPattern) {
            _assertValidValue(props.stringValue, props.allowedPattern);
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            type: STRING_PARAM_TYPE,
            value: props.stringValue,
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = arnForParameterName(this, this.parameterName);
        this.parameterType = resource.attrType;
        this.stringValue = resource.attrValue;
    }
    /**
     * Imports an external string parameter by name.
     */
    static fromStringParameterName(scope, id, stringParameterName) {
        return this.fromStringParameterAttributes(scope, id, { parameterName: stringParameterName });
    }
    /**
     * Imports an external string parameter with name and optional version.
     */
    static fromStringParameterAttributes(scope, id, attrs) {
        if (!attrs.parameterName) {
            throw new Error(`parameterName cannot be an empty string`);
        }
        const stringValue = attrs.version
            ? new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, `${attrs.parameterName}:${attrs.version}`).toString()
            : new core_1.CfnParameter(scope, `${id}.Parameter`, { type: 'AWS::SSM::Parameter::Value<String>', default: attrs.parameterName }).valueAsString;
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = arnForParameterName(this, this.parameterName);
                this.parameterType = STRING_PARAM_TYPE;
                this.stringValue = stringValue;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a secure string parameter from the SSM parameter store.
     */
    static fromSecureStringParameterAttributes(scope, id, attrs) {
        const stringValue = new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM_SECURE, `${attrs.parameterName}:${attrs.version}`).toString();
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = arnForParameterName(this, this.parameterName);
                this.parameterType = SECURE_STRING_PARAM_TYPE;
                this.stringValue = stringValue;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Reads the value of an SSM parameter during synthesis through an
     * environmental context provider.
     *
     * Requires that the stack this scope is defined in will have explicit
     * account/region information. Otherwise, it will fail during synthesis.
     */
    static valueFromLookup(scope, parameterName) {
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxapi.SSM_PARAMETER_PROVIDER,
            props: { parameterName },
            dummyValue: `dummy-value-for-${parameterName}`
        });
        return value;
    }
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForStringParameter(scope, parameterName, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromStringParameterAttributes(stack, id, { parameterName, version }).stringValue;
    }
    /**
     * Returns a token that will resolve (during deployment)
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter
     * @param version The parameter version (required for secure strings)
     */
    static valueForSecureStringParameter(scope, parameterName, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromSecureStringParameterAttributes(stack, id, { parameterName, version }).stringValue;
    }
}
exports.StringParameter = StringParameter;
/**
 * Creates a new StringList SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringListParameter extends ParameterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.stringListValue.find(str => !core_1.Token.isUnresolved(str) && str.indexOf(',') !== -1)) {
            throw new Error('Values of a StringList SSM Parameter cannot contain the \',\' character. Use a string parameter instead.');
        }
        if (props.allowedPattern && !core_1.Token.isUnresolved(props.stringListValue)) {
            props.stringListValue.forEach(str => _assertValidValue(str, props.allowedPattern));
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            type: STRINGLIST_PARAM_TYPE,
            value: props.stringListValue.join(','),
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = arnForParameterName(this, this.parameterName);
        this.parameterType = resource.attrType;
        this.stringListValue = core_1.Fn.split(',', resource.attrValue);
    }
    /**
     * Imports an external parameter of type string list.
     */
    static fromStringListParameterName(scope, id, stringListParameterName) {
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = stringListParameterName;
                this.parameterArn = arnForParameterName(this, this.parameterName);
                this.parameterType = STRINGLIST_PARAM_TYPE;
                this.stringListValue = core_1.Fn.split(',', new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, stringListParameterName).toString());
            }
        }
        return new Import(scope, id);
    }
}
exports.StringListParameter = StringListParameter;
/**
 * Validates whether a supplied value conforms to the allowedPattern, granted neither is an unresolved token.
 *
 * @param value          the value to be validated.
 * @param allowedPattern the regular expression to use for validation.
 *
 * @throws if the ``value`` does not conform to the ``allowedPattern`` and neither is an unresolved token (per
 *         ``cdk.unresolved``).
 */
function _assertValidValue(value, allowedPattern) {
    if (core_1.Token.isUnresolved(value) || core_1.Token.isUnresolved(allowedPattern)) {
        // Unable to perform validations against unresolved tokens
        return;
    }
    if (!new RegExp(allowedPattern).test(value)) {
        throw new Error(`The supplied value (${value}) does not match the specified allowedPattern (${allowedPattern})`);
    }
}
function makeIdentityForImportedValue(parameterName) {
    return `SsmParameterValue:${parameterName}:C96584B6-F00A-464E-AD19-53AFF4B05118`;
}
function arnForParameterName(scope, parameterName) {
    return core_1.Stack.of(scope).formatArn({
        service: 'ssm',
        resource: 'parameter',
        sep: '',
        resourceName: parameterName,
    });
}
//# sourceMappingURL=data:application/json;base64,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