import iam = require('@aws-cdk/aws-iam');
import { Construct, IResource, Resource } from '@aws-cdk/core';
/**
 * An SSM Parameter reference.
 */
export interface IParameter extends IResource {
    /**
     * The ARN of the SSM Parameter resource.
     * @attribute
     */
    readonly parameterArn: string;
    /**
     * The name of the SSM Parameter resource.
     * @attribute
     */
    readonly parameterName: string;
    /**
     * The type of the SSM Parameter resource.
     * @attribute
     */
    readonly parameterType: string;
    /**
     * Grants read (DescribeParameter, GetParameter, GetParameterHistory) permissions on the SSM Parameter.
     *
     * @param grantee the role to be granted read-only access to the parameter.
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grants write (PutParameter) permissions on the SSM Parameter.
     *
     * @param grantee the role to be granted write access to the parameter.
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
}
/**
 * A String SSM Parameter.
 */
export interface IStringParameter extends IParameter {
    /**
     * The parameter value. Value must not nest another parameter. Do not use {{}} in the value.
     *
     * @attribute Value
     */
    readonly stringValue: string;
}
/**
 * A StringList SSM Parameter.
 */
export interface IStringListParameter extends IParameter {
    /**
     * The parameter value. Value must not nest another parameter. Do not use {{}} in the value. Values in the array
     * cannot contain commas (``,``).
     *
     * @attribute Value
     */
    readonly stringListValue: string[];
}
/**
 * Properties needed to create a new SSM Parameter.
 */
export interface ParameterOptions {
    /**
     * A regular expression used to validate the parameter value. For example, for String types with values restricted to
     * numbers, you can specify the following: ``^\d+$``
     *
     * @default no validation is performed
     */
    readonly allowedPattern?: string;
    /**
     * Information about the parameter that you want to add to the system.
     *
     * @default none
     */
    readonly description?: string;
    /**
     * The name of the parameter.
     *
     * @default - a name will be generated by CloudFormation
     */
    readonly parameterName?: string;
}
/**
 * Properties needed to create a String SSM parameter.
 */
export interface StringParameterProps extends ParameterOptions {
    /**
     * The value of the parameter. It may not reference another parameter and ``{{}}`` cannot be used in the value.
     */
    readonly stringValue: string;
}
/**
 * Properties needed to create a StringList SSM Parameter
 */
export interface StringListParameterProps extends ParameterOptions {
    /**
     * The values of the parameter. It may not reference another parameter and ``{{}}`` cannot be used in the value.
     */
    readonly stringListValue: string[];
}
/**
 * Basic features shared across all types of SSM Parameters.
 */
declare abstract class ParameterBase extends Resource implements IParameter {
    abstract readonly parameterArn: string;
    abstract readonly parameterName: string;
    abstract readonly parameterType: string;
    grantRead(grantee: iam.IGrantable): iam.Grant;
    grantWrite(grantee: iam.IGrantable): iam.Grant;
}
export interface StringParameterAttributes {
    /**
     * The name of the parameter store value
     */
    readonly parameterName: string;
    /**
     * The version number of the value you wish to retrieve.
     *
     * @default The latest version will be retrieved.
     */
    readonly version?: number;
}
export interface SecureStringParameterAttributes {
    /**
     * The name of the parameter store value
     */
    readonly parameterName: string;
    /**
     * The version number of the value you wish to retrieve. This is required for secure strings.
     */
    readonly version: number;
}
/**
 * Creates a new String SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
export declare class StringParameter extends ParameterBase implements IStringParameter {
    /**
     * Imports an external string parameter by name.
     */
    static fromStringParameterName(scope: Construct, id: string, stringParameterName: string): IStringParameter;
    /**
     * Imports an external string parameter with name and optional version.
     */
    static fromStringParameterAttributes(scope: Construct, id: string, attrs: StringParameterAttributes): IStringParameter;
    /**
     * Imports a secure string parameter from the SSM parameter store.
     */
    static fromSecureStringParameterAttributes(scope: Construct, id: string, attrs: SecureStringParameterAttributes): IStringParameter;
    /**
     * Reads the value of an SSM parameter during synthesis through an
     * environmental context provider.
     *
     * Requires that the stack this scope is defined in will have explicit
     * account/region information. Otherwise, it will fail during synthesis.
     */
    static valueFromLookup(scope: Construct, parameterName: string): string;
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForStringParameter(scope: Construct, parameterName: string, version?: number): string;
    /**
     * Returns a token that will resolve (during deployment)
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter
     * @param version The parameter version (required for secure strings)
     */
    static valueForSecureStringParameter(scope: Construct, parameterName: string, version: number): string;
    readonly parameterArn: string;
    readonly parameterName: string;
    readonly parameterType: string;
    readonly stringValue: string;
    constructor(scope: Construct, id: string, props: StringParameterProps);
}
/**
 * Creates a new StringList SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
export declare class StringListParameter extends ParameterBase implements IStringListParameter {
    /**
     * Imports an external parameter of type string list.
     */
    static fromStringListParameterName(scope: Construct, id: string, stringListParameterName: string): IStringListParameter;
    readonly parameterArn: string;
    readonly parameterName: string;
    readonly parameterType: string;
    readonly stringListValue: string[];
    constructor(scope: Construct, id: string, props: StringListParameterProps);
}
export {};
