"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const kinesisfirehose = require("@aws-cdk/aws-kinesisfirehose");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-konstruk/core");
const iam = require("@aws-cdk/aws-iam");
const core_2 = require("@aws-solutions-konstruk/core");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
class KinesisFirehoseToS3 extends core_1.Construct {
    /**
     * Constructs a new instance of the IotToLambda class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup S3 Bucket
        this.s3Bucket = defaults.buildS3Bucket(this, {
            deployBucket: props.deployBucket,
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        // Extract the CfnBucket from the s3Bucket
        const s3BucketResource = this.s3Bucket.node.findChild('Resource');
        s3BucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W51',
                        reason: `This S3 bucket Bucket does not need a bucket policy`
                    }]
            }
        };
        // Setup Cloudwatch Log group & stream for Kinesis Firehose
        const cwLogGroup = new logs.LogGroup(this, 'firehose-log-group', defaults.DefaultLogGroupProps());
        const cwLogStream = cwLogGroup.addStream('firehose-log-stream');
        // Setup the IAM Role for Kinesis Firehose
        const firehoseRole = new iam.Role(this, 'KinesisFirehoseRole', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
        });
        // Setup the IAM policy for Kinesis Firehose
        const firehosePolicy = new iam.Policy(this, 'KinesisFirehosePolicy', {
            statements: [new iam.PolicyStatement({
                    actions: [
                        's3:AbortMultipartUpload',
                        's3:GetBucketLocation',
                        's3:GetObject',
                        's3:ListBucket',
                        's3:ListBucketMultipartUploads',
                        's3:PutObject'
                    ],
                    resources: [`${this.s3Bucket.bucketArn}`, `${this.s3Bucket.bucketArn}/*`]
                }),
                new iam.PolicyStatement({
                    actions: [
                        'logs:PutLogEvents'
                    ],
                    resources: [`arn:aws:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:${cwLogGroup.logGroupName}:log-stream:${cwLogStream.logStreamName}`]
                })
            ]
        });
        // Attach policy to role
        firehosePolicy.attachToRole(firehoseRole);
        // Setup the default Kinesis Firehose props
        const defaultKinesisFirehoseProps = defaults.DefaultCfnDeliveryStreamProps(this.s3Bucket.bucketArn, firehoseRole.roleArn, cwLogGroup.logGroupName, cwLogStream.logStreamName);
        // Override with the input props
        if (props.kinesisFirehoseProps) {
            const kinesisFirehoseProps = core_2.overrideProps(defaultKinesisFirehoseProps, props.kinesisFirehoseProps);
            this.firehose = new kinesisfirehose.CfnDeliveryStream(this, 'KinesisFirehose', kinesisFirehoseProps);
        }
        else {
            this.firehose = new kinesisfirehose.CfnDeliveryStream(this, 'KinesisFirehose', defaultKinesisFirehoseProps);
        }
    }
    /**
     * Returns an instance of kinesisfirehose.CfnDeliveryStream created by the construct
     */
    kinesisFirehose() {
        return this.firehose;
    }
    /**
     * Returns an instance of s3.Bucket created by the construct
     */
    bucket() {
        return this.s3Bucket;
    }
}
exports.KinesisFirehoseToS3 = KinesisFirehoseToS3;
//# sourceMappingURL=data:application/json;base64,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