"""
Produces a maximum cardinality matching of a bipartite graph

Example:

 0---0
  \   
   \
    \
 1---1
  \ /
   /
  / \ 
 2   2
    /
   /
  /
 3

>>>> n = 4
>>>> m = 3
>>>> graph = [[0, 1], [1, 2], [1], [2]]
>>>> match1, match2 = hopcroft_karp(graph, n, m)
>>>> match1
[0, 1, -1, 2]
>>>> match2
[0, 1, 3]

Meaning
 0---0
    
 1---1
  
 2   2
    /
   /
  /
 3

"""


def hopcroft_karp(graph, n, m):
    """
    Maximum bipartite matching using Hopcroft-Karp algorithm, running in O(|E| sqrt(|V|))
    """
    assert(n == len(graph))
    match1 = [-1]*n
    match2 = [-1]*m
     
    # Find a greedy match for possible speed up
    for node in range(n):
        for nei in graph[node]:
            if match2[nei] == -1:
                match1[node] = nei
                match2[nei] = node
                break
    while 1:
        bfs = [node for node in range(n) if match1[node] == -1]
        depth = [-1] * n
        for node in bfs:
            depth[node] = 0
     
        for node in bfs:
            for nei in graph[node]:
                next_node = match2[nei]
                if next_node == -1:
                    break
                if depth[next_node] == -1:
                    depth[next_node] = depth[node] + 1
                    bfs.append(next_node)
            else:
                continue
            break
        else:
            break
     
        pointer = [len(c) for c in graph]
        dfs = [node for node in range(n) if depth[node] == 0]
        while dfs:
            node = dfs[-1]
            while pointer[node]:
                pointer[node] -= 1
                nei = graph[node][pointer[node]]
                next_node = match2[nei]
                if next_node == -1:
                    # Augmenting path found
                    while nei != -1:
                        node = dfs.pop()
                        match2[nei], match1[node], nei = node, nei, match1[node]
                    break
                elif depth[node] + 1 == depth[next_node]:
                    dfs.append(next_node)
                    break
            else:
                dfs.pop()
    return match1, match2
