# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['starlite_jwt']

package_data = \
{'': ['*']}

install_requires = \
['cryptography', 'python-jose', 'starlite>=1.24.0']

setup_kwargs = {
    'name': 'starlite-jwt',
    'version': '1.5.0',
    'description': 'A JWT auth toolkit for Starlite',
    'long_description': '# Starlite JWT\n\n<!-- markdownlint-disable -->\n<img alt="Starlite logo" src="./starlite-banner.svg" width="100%" height="auto">\n<!-- markdownlint-restore -->\n\n<div align="center">\n\n![PyPI - License](https://img.shields.io/pypi/l/starlite-jwt?color=blue)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/starlite-jwt)\n\n[![Quality Gate Status](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_jwt-auth&metric=alert_status)](https://sonarcloud.io/summary/new_code?id=starlite-api_jwt-auth)\n[![Coverage](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_jwt-auth&metric=coverage)](https://sonarcloud.io/summary/new_code?id=starlite-api_jwt-auth)\n\n[![Maintainability Rating](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_jwt-auth&metric=sqale_rating)](https://sonarcloud.io/summary/new_code?id=starlite-api_jwt-auth)\n[![Security Rating](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_jwt-auth&metric=security_rating)](https://sonarcloud.io/summary/new_code?id=starlite-api_jwt-auth)\n[![Reliability Rating](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_jwt-auth&metric=reliability_rating)](https://sonarcloud.io/summary/new_code?id=starlite-api_jwt-auth)\n[![Code Smells](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_jwt-auth&metric=code_smells)](https://sonarcloud.io/summary/new_code?id=starlite-api_jwt-auth)\n\n[![Discord](https://img.shields.io/discord/919193495116337154?color=blue&label=chat%20on%20discord&logo=discord)](https://discord.gg/X3FJqy8d2j)\n[![Matrix](https://img.shields.io/badge/%5Bm%5D%20chat%20on%20Matrix-bridged-blue)](https://matrix.to/#/#starlitespace:matrix.org)\n\n</div>\n\nThis library offers simple JWT authentication for [Starlite](https://github.com/starlite-api/starlite).\n\nCheckout [the docs 📚](https://starlite-api.github.io/starlite-jwt/).\n\n## Installation\n\n```shell\npip install starlite-jwt\n```\n\nThis library uses the excellent [python-jose](https://github.com/mpdavis/python-jose) library, which supports multiple\ncryptographic backends. You can install either [pyca/cryptography](http://cryptography.io/)\nor [pycryptodome](https://pycryptodome.readthedocs.io/en/latest/), and it will be used as the backend automatically. Note\nthat if you want to use a certificate based encryption scheme, you must install one of these backends - please refer to\nthe [python-jose](https://github.com/mpdavis/python-jose) readme for more details.\n\n## Example\n\n```python\nimport os\nfrom typing import Any, Optional\nfrom uuid import UUID, uuid4\n\nfrom pydantic import BaseModel, EmailStr\nfrom starlite import OpenAPIConfig, Request, Response, ASGIConnection, Starlite, get\n\nfrom starlite_jwt import JWTAuth, Token\n\n\n# Let\'s assume we have a User model that is a pydantic model.\n# This though is not required - we need some sort of user class -\n# but it can be any arbitrary value, e.g. an SQLAlchemy model, a representation of a MongoDB  etc.\nclass User(BaseModel):\n    id: UUID\n    name: str\n    email: EmailStr\n\n\n# The JWTAuth package requires a handler callable that takes a unique identifier, and returns the \'User\'\n# instance correlating to it.\n#\n# The identifier is the \'sub\' key of the JWT, and it usually correlates to a user ID.\n# It can be though any arbitrary value you decide upon - as long as the handler function provided\n# can receive this value and return the model instance for it.\n#\n# Note: The callable can be either sync or async - both will work.\nasync def retrieve_user_handler(\n    unique_identifier: str, connection: ASGIConnection[Any, Any, Any]\n) -> Optional[User]:\n    # logic here to retrieve the user instance\n    ...\n\n\n# The minimal configuration required for the library is the callable for the \'retrieve_user_handler\' key, and a string\n# value for the token secret.\n#\n# Important: secrets should never be hardcoded. Its best practice to pass the secret using ENV.\n#\n# Tip: It\'s also a good idea to use the pydantic settings management functionality\njwt_auth = JWTAuth(\n    retrieve_user_handler=retrieve_user_handler,\n    token_secret=os.environ.get("JWT_SECRET", "abcd123"),\n    # we are specifying which endpoints should be excluded from authentication. In this case the login endpoint\n    # and our openAPI docs.\n    exclude=["/login", "/schema"],\n)\n\n\n# Given an instance of \'JWTAuth\' we can create a login handler function:\n@get("/login")\ndef login_handler() -> Response[User]:\n    # we have a user instance - probably by retrieving it from persistence using another lib.\n    # what\'s important for our purposes is to have an identifier:\n    user = User(name="Moishe Zuchmir", email="zuchmir@moishe.com", id=uuid4())\n\n    response = jwt_auth.login(identifier=str(user.id), response_body=user)\n\n    # you can do whatever you want to update the response instance here\n    # e.g. response.set_cookie(...)\n\n    return response\n\n\n# We also have some other routes, for example:\n@get("/some-path")\ndef some_route_handler(request: Request[User, Token]) -> Any:\n    # request.user is set to the instance of user returned by the middleware\n    assert isinstance(request.user, User)\n    # request.auth is the instance of \'starlite_jwt.Token\' created from the data encoded in the auth header\n    assert isinstance(request.auth, Token)\n    # do stuff ...\n\n\n# We add the jwt security schema to the OpenAPI config.\nopenapi_config = OpenAPIConfig(\n    title="My API",\n    version="1.0.0",\n    components=[jwt_auth.openapi_components],\n    security=[jwt_auth.security_requirement],\n)\n\n# We initialize the app instance, passing to it the \'jwt_auth.middleware\' and the \'openapi_config\'.\napp = Starlite(\n    route_handlers=[login_handler, some_route_handler],\n    middleware=[jwt_auth.middleware],\n    openapi_config=openapi_config,\n)\n```\n\n## Customization\n\nThis integrates with the OpenAPI configuration of Starlite, and it uses the `SecurityScheme` configuration to format the header and/or cookie value.\n\nThe default implementation follows the `Bearer {encoded_token}` format, but you may optionally override this configuration by modifying the openapi_component attribute of your `JWTAuth` instance.\n\nIf you wanted your authentication header to be `Token {encoded_token}`, you could use the following as your security scheme configuration:\n\n```python\nfrom pydantic_openapi_schema.v3_1_0 import Components, SecurityScheme\nfrom starlite_jwt import JWTAuth\n\n\nclass CustomJWTAuth(JWTAuth):\n    @property\n    def openapi_components(self) -> Components:\n        """Creates OpenAPI documentation for the JWT auth schema used.\n\n        Returns:\n            An [Components][pydantic_schema_pydantic.v3_1_0.components.Components] instance.\n        """\n        return Components(\n            securitySchemes={\n                self.openapi_security_scheme_name: SecurityScheme(\n                    type="http",\n                    scheme="Token",\n                    name=self.auth_header,\n                    bearerFormat="JWT",\n                    description="JWT api-key authentication and authorization.",\n                )\n            }\n        )\n```\n\n## Contributing\n\nStarlite and all its official libraries is open to contributions big and small.\n\nYou can always [join our discord](https://discord.gg/X3FJqy8d2j) server\nor [join our Matrix](https://matrix.to/#/#starlitespace:matrix.org) space to discuss contributions and project\nmaintenance. For guidelines on how to contribute to this library, please see [the contribution guide](CONTRIBUTING.md).\n',
    'author': "Na'aman Hirschfeld",
    'author_email': 'nhirschfeld@gmail.com',
    'maintainer': "Na'aman Hirschfeld",
    'maintainer_email': 'nhirschfeld@gmail.com',
    'url': 'https://github.com/starlite-api/jwt-auth',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
