#include <cmath>
#include <numeric>
#include <algorithm>
#include <iterator>
#include <sstream>
#include <utility>
#include <array>

#include "reverb_rir.h"

using namespace std;


static const array<const float, 800> DEFAULT_RIR = {-3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 6.1035156e-05, -9.1552734e-05, 6.1035156e-05, -9.1552734e-05, 6.1035156e-05, -0.00012207031, 9.1552734e-05, -0.00012207031, 0.00012207031, -0.00015258789, 0.00012207031, -0.00018310547, 0.00018310547, -0.00024414062, 0.00024414062, -0.00033569336, 0.0004272461, -0.0007019043, 0.001739502, 0.046875, 0.0008544922, -0.0004272461, 0.00048828125, -0.0008239746, 0.002166748, 0.038726807, -0.000579834, 0.00045776367, -0.00039672852, 0.0002746582, -0.0002746582, 0.00018310547, -0.00021362305, 0.00015258789, -0.00015258789, 0.00012207031, -0.00012207031, 9.1552734e-05, -0.00012207031, 6.1035156e-05, -9.1552734e-05, 6.1035156e-05, -9.1552734e-05, 6.1035156e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 0.0, -3.0517578e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, -9.1552734e-05, 6.1035156e-05, -9.1552734e-05, 6.1035156e-05, -9.1552734e-05, 6.1035156e-05, -0.00012207031, 9.1552734e-05, -0.00012207031, 0.00012207031, -0.00015258789, 0.00015258789, -0.00021362305, 0.00021362305, -0.0002746582, 0.00030517578, -0.00039672852, 0.0005187988, -0.0008239746, 0.0017700195, 0.024841309, -0.0011901855, 0.00061035156, -0.0004272461, 0.00030517578, -0.00024414062, 0.00018310547, -0.00018310547, 0.00012207031, -0.00012207031, 6.1035156e-05, -9.1552734e-05, 3.0517578e-05, -3.0517578e-05, 0.0, 0.0, -3.0517578e-05, 3.0517578e-05, -9.1552734e-05, 9.1552734e-05, -0.00021362305, 0.00045776367, 0.01953125, -0.00018310547, 6.1035156e-05, -6.1035156e-05, 0.0, 0.0, -3.0517578e-05, 3.0517578e-05, -9.1552734e-05, 6.1035156e-05, -0.00012207031, 9.1552734e-05, -0.00015258789, 0.00012207031, -0.00018310547, 0.00015258789, -0.00021362305, 0.00021362305, -0.0002746582, 0.0002746582, -0.00033569336, 0.00033569336, -0.0004272461, 0.00045776367, -0.0005493164, 0.00061035156, -0.00079345703, 0.0009460449, -0.0013122559, 0.0018615723, -0.0032043457, 0.0093688965, 0.023834229, -0.0018920898, 0.016540527, 0.00061035156, 0.004638672, 0.0052490234, -0.0011901855, 0.00048828125, -0.0002746582, 0.00015258789, -0.00012207031, 6.1035156e-05, -9.1552734e-05, 6.1035156e-05, -0.00012207031, 0.00012207031, -0.00018310547, 0.00018310547, -0.00030517578, 0.00039672852, -0.00061035156, 0.00091552734, -0.0017089844, 0.0049438477, 0.012176514, -0.003479004, 0.002380371, -0.002166748, 0.0022277832, -0.0029296875, 0.009185791, 0.0005187988, 0.0046081543, 0.011871338, -0.0024719238, 0.0069885254, 0.00021362305, 3.0517578e-05, -6.1035156e-05, 3.0517578e-05, 0.0, -6.1035156e-05, 6.1035156e-05, -0.00012207031, 0.00012207031, -0.00018310547, 0.00021362305, -0.00030517578, 0.00033569336, -0.00045776367, 0.0005493164, -0.00076293945, 0.0010681152, -0.0018920898, 0.006652832, 0.00289917, 0.005004883, 0.002105713, -0.0013122559, 0.0009460449, -0.0008239746, 0.0007019043, -0.0006713867, 0.00061035156, -0.00064086914, 0.00064086914, -0.0007324219, 0.0008544922, -0.0012207031, 0.0022583008, 0.0049743652, 0.0023498535, 0.004272461, -0.0018615723, 0.0013427734, -0.0012207031, 0.001159668, -0.0012512207, 0.0014343262, -0.0020141602, 0.004272461, 0.0032653809, 0.0024719238, 0.0037841797, -0.0015258789, 0.00091552734, -0.0007019043, 0.0005187988, -0.00045776367, 0.00036621094, -0.00039672852, 0.0004272461, -0.0008239746, 0.0048217773, 0.0032348633, 0.008911133, 0.00091552734, 0.004058838, -0.0007019043, 0.00024414062, -9.1552734e-05, -0.00012207031, 0.00033569336, -0.00076293945, 0.002166748, 0.015014648, -0.0027160645, 0.0140686035, -0.00048828125, 0.00079345703, -0.0012512207, 0.003112793, 0.0031433105, 0.0030517578, 0.0010681152, -0.00088500977, 0.00088500977, 0.0039367676, 0.001739502, 0.0033569336, -0.00064086914, 0.00024414062, -0.00012207031, 0.0, 3.0517578e-05, -9.1552734e-05, 0.00012207031, -0.00018310547, 0.00018310547, -0.0002746582, 0.00030517578, -0.00039672852, 0.00048828125, -0.0008239746, 0.0037841797, 0.0011901855, 0.0032653809, -0.0005493164, 0.00048828125, -0.0005493164, 0.00061035156, -0.00076293945, 0.00088500977, -0.0011291504, 0.0014953613, -0.002380371, 0.0058288574, 0.00869751, -0.0020751953, 0.00091552734, -0.00039672852, -6.1035156e-05, 0.0005493164, -0.0015563965, 0.007843018, 0.004058838, -0.0016174316, 0.00076293945, 0.00024414062, 0.010345459, -0.0018920898, 0.0013122559, -0.0010986328, 0.0009460449, -0.00088500977, 0.00079345703, -0.00079345703, 0.0007324219, -0.00076293945, 0.0007324219, -0.00079345703, 0.00079345703, -0.00088500977, 0.0009460449, -0.001159668, 0.0014343262, -0.002105713, 0.004486084, 0.0128479, 0.006134033, 0.012786865, 0.0019836426, 0.000579834, -0.000579834, 0.00045776367, -0.0004272461, 0.00036621094, -0.00036621094, 0.00030517578, -0.00033569336, 0.00030517578, -0.00039672852, 0.000579834, 0.0074768066, -0.0002746582, 0.00015258789, -0.00033569336, 0.003540039, 0.0032653809, 0.0062561035, 0.0018920898, 0.0022888184, -0.0010070801, 0.0006713867, -0.000579834, 0.0005187988, -0.0005187988, 0.00048828125, -0.0005493164, 0.000579834, -0.0007019043, 0.0008544922, -0.0013122559, 0.0038146973, 0.0028076172, 0.0053710938, 0.005493164, -0.0010986328, 0.00048828125, -0.0002746582, 0.003479004, 0.003112793, 0.0009460449, -0.00030517578, 0.0036010742, 0.0029907227, 0.00091552734, -0.00036621094, 0.00012207031, 6.1035156e-05, -0.00039672852, 0.0014648438, 0.003326416, 0.0022888184, 0.00048828125, -0.00039672852, 0.00024414062, -0.00024414062, 0.0028381348, 0.0021362305, 0.001739502, 0.0058288574, 0.011962891, 0.0034484863, 0.003967285, -0.0018615723, 0.0015258789, -0.0015258789, 0.0016174316, -0.0020141602, 0.0038757324, 0.0013122559, 0.004211426, 0.0033569336, 0.004547119, 0.004486084, 0.0, -0.00024414062, 0.0032348633, 0.001373291, 0.006713867, 0.009765625, 0.006011963, 0.0049743652, -0.0020751953, 0.0012817383, -0.0009460449, 0.0007019043, -0.000579834, 0.00039672852, -0.00021362305, 0.0026550293, 0.0016174316, 0.0007019043, 0.0014038086, 0.0036315918, 0.0002746582, 0.0002746582, -0.00039672852, 0.00045776367, -0.00064086914, 0.0012817383, 0.0030517578, -0.00091552734, 0.0067749023, 0.0, 0.009307861, 0.0042419434, 0.00048828125, 0.0017700195, -0.0018615723, 0.012023926, 0.0060424805, 0.0004272461, 0.0018920898, -0.0011291504, 0.00076293945, -0.00018310547, 0.0038757324, -0.00021362305, 0.0029296875, 0.0059814453, -0.0012207031, 0.0005187988, -0.00021362305, -0.00018310547, 0.0071105957, 0.0018005371, 0.001373291, 0.0009460449, -0.0009460449, 0.0010375977, -0.0014648438, 0.0036315918, 0.0026245117, 0.011047363, 0.010314941, 0.003753662, 0.004425049, -0.0012817383, 0.0032043457, -0.0015258789, 0.002319336, 0.0006713867, 0.0007324219, 0.0014953613, 9.1552734e-05, 0.00592041, -6.1035156e-05, -0.00030517578, 0.0024108887, 0.0042419434, 0.0050354004, 0.00088500977, 0.0018005371, 0.0039367676, 0.00076293945, 0.0015563965, 0.0015563965, 0.005432129, -0.0004272461, 0.0005493164, -0.00061035156, 0.003967285, 0.0030212402, 0.00033569336, -0.00039672852, 0.00048828125, 0.0016174316, 9.1552734e-05, 0.0013122559, 0.0012512207, 0.003692627, 0.0032958984, 0.0015563965, 0.0006713867, 0.004180908, 0.0036010742, 0.0044555664, 0.0015869141, -0.00015258789, 0.00012207031, -0.00033569336, 0.0022583008, 0.0020751953, 0.0068359375, 0.0067749023, 0.0032958984, -0.00012207031, 0.0, 3.0517578e-05, 0.0017700195, 0.0016174316, 3.0517578e-05, -3.0517578e-05, 0.00012207031, 0.0032653809, 0.0012817383, 0.0005493164, 0.0069274902, 0.0014953613, 0.0008239746, -0.0006713867, 0.0025634766, 0.0016174316, 0.0002746582, -0.00039672852, 0.00048828125, -0.0008544922, 0.002960205, 0.0027770996, 0.0045166016, 0.0025939941, 0.001953125, 0.003753662, 0.0014953613, 0.0013427734, 0.004272461, 0.0016479492, 0.0013122559, 0.00064086914, -0.0010375977, 0.00390625, 0.005554199, -0.00088500977, 0.004333496, 0.00021362305, 0.00024414062, -0.0005187988, 0.00091552734, 0.0012817383, 0.002319336, 0.0018310547, 0.0008544922, 0.00039672852, -0.000579834, 0.0030212402, 0.0023498535, 0.00061035156, 0.0055236816, 0.0014343262, 0.0021972656, 0.0014343262, 0.0010986328, 0.0037231445, 0.0022277832, 0.0012207031, 3.0517578e-05, -0.00018310547, 0.00021362305, -0.00030517578, 0.00036621094, -0.00048828125, 0.000579834, -0.0008544922, 0.0016174316, 0.0017700195, 0.004272461, 0.0038757324, 0.00091552734, 6.1035156e-05, 0.00091552734, 0.002960205, 0.0030517578, 0.001739502, -0.00088500977, 0.0036621094, 0.002532959, 0.0014343262, 0.0015869141, 0.0014343262, -0.00045776367, 0.00018310547, 3.0517578e-05, 0.0014038086, 0.00024414062, 0.001159668, -0.0005493164, 0.0004272461, 0.0021362305, 0.0011291504, 0.0043029785, 0.0062561035, 0.0019226074, 0.00015258789, -0.00033569336, 0.0002746582, 6.1035156e-05, 0.005584717, 0.0034484863, 0.0022888184, 0.00036621094, 0.00079345703, 0.0030517578, 0.0006713867, 0.0020446777, 0.00012207031, -0.00018310547, 0.000579834, 0.00076293945, 0.0004272461, 0.0007019043, -0.00048828125, 0.0010681152, 0.0022583008, 0.0024108887, 0.0020446777, 0.00061035156, 0.0011291504, 3.0517578e-05, 0.0016174316, 0.0011901855, 0.0032653809, 0.0022583008, 0.0020446777, 0.0017700195, 0.0048217773, 0.0037841797, 0.0015258789, 0.0011291504, -0.0002746582, 0.0010375977, 0.0026550293, 0.0019226074, -0.00015258789, -3.0517578e-05, 6.1035156e-05, -0.00021362305, 0.0022888184, 0.0024719238, 0.002532959, 0.00012207031, 0.0011901855, 0.009918213, 0.011657715, 0.0018615723, 0.003326416, 0.0017700195, 0.0014343262, -3.0517578e-05, 0.002319336, 0.0015563965, 0.0013427734, 0.0010375977, 0.0019836426, 0.0030212402, 0.00061035156, 0.00012207031, 0.0012207031, 0.0022277832, -0.00088500977, 0.0015869141, 0.00061035156, 0.00045776367, 0.0047912598, 0.005340576, -0.00076293945, 0.0024414062, -0.00088500977, 0.0014953613, 0.0017700195, 0.0074768066, -0.00033569336, 0.0011901855, 0.00030517578, 0.0005187988, 0.0014038086, 0.0024719238, 0.0019226074, 0.0013122559, 0.00030517578, 0.008850098, 0.009246826, 0.0019226074, 0.0010681152, 6.1035156e-05, -0.00024414062, 0.0011901855, 0.00024414062, 0.0021972656, 0.004272461, -0.00064086914, 0.004211426, 0.0045166016, 0.005126953, -0.00088500977, 0.0022277832, 0.00024414062, 0.0016174316, -0.00015258789, 0.0006713867, 0.001953125, 0.0032043457, -0.0005187988, 0.0011901855, 0.00030517578, 0.0016174316, 0.003326416, 0.005065918, 0.0012817383, 0.0016174316, -0.00018310547, 0.002319336, 0.0012817383, 0.0019226074, 0.0027160645, -0.00048828125, 0.0016784668, 0.003540039, 0.0050964355, 0.0007019043, 0.00088500977};


void validate_mix(const float mix) {
    if (mix < 0.0f || mix > 1.0f) {
        throw runtime_error{"Mix argument must have value between 0 and 1."};
    }
}


ReverbRIR::ReverbRIR(const float mix) : mix_{mix} {
    std::copy(DEFAULT_RIR.crbegin(), DEFAULT_RIR.crend(), back_inserter(rir_));
    reset();
    validate_mix(mix_);
}

ReverbRIR::ReverbRIR(const float * const rir, const size_t num_samples, const float mix) : mix_{mix} {
    for (unsigned int i = num_samples; i > 0; --i) {
        rir_.push_back(rir[i-1]);
    }
    reset();
    validate_mix(mix_);
}


float inner_product_clang(const vector<float> &buffer_, const vector<float> &rir_, int offset) {
    const auto SIZE = rir_.size();
    float sum = 0.0f;
    #pragma clang loop vectorize(assume_safety)
    for (unsigned int i = 0; i < SIZE; ++i) {
        sum += buffer_[offset + i] * rir_[i];
    }
    return sum;
}


float inner_product_gcc(const vector<float> &buffer_, const vector<float> &rir_, int offset) {
    const auto SIZE = rir_.size();
    float sum = 0.0f;
    #pragma omp simd reduction(+:sum)
    for (unsigned int i = 0; i < SIZE; ++i) {
        sum += buffer_[offset + i] * rir_[i];
    }
    return sum;
}

void ReverbRIR::apply_inplace(float * const audio, const size_t num_samples) {
    const auto dry = 1.0f - mix_;
    copy(audio, audio + num_samples, back_inserter(buffer_));
    for (unsigned int idx = 0; idx < num_samples; ++idx) {

        // Apparently std::inner_product does not get auto-vectorized,
        // but we can help the compiler with a few pragmas, so we write
        // it out explicitly.
        #if defined(__clang__)
        const auto sum = inner_product_clang(buffer_, rir_, idx);
        #elif defined(__GNUC__)
        const auto sum = inner_product_gcc(buffer_, rir_, idx);
        #else
        const auto sum = inner_product(buffer_.cbegin() + idx, buffer_.cbegin() + idx + rir_.size(), rir_.cbegin(), 0.0f);
        #endif

        audio[idx] = mix_ * sum + dry * audio[idx];
    }
    buffer_.erase(buffer_.begin(), buffer_.begin() + num_samples);
}

vector<float> ReverbRIR::apply(float const * const audio, const size_t num_samples) {
    vector<float> out;
    copy(audio, audio + num_samples, back_inserter(out));
    apply_inplace(out.data(), out.size());
    return out;
}

void ReverbRIR::reset() {
    buffer_.resize(rir_.size() - 1);
    for (unsigned long i = 0; i < buffer_.size(); ++i) {
        buffer_[i] = 0.0f;
    }
}

void ReverbRIR::read_from_string(const string &data) {
    istringstream str{data};
    unsigned int size;
    float sample;

    str >> size;
    rir_.clear();
    for (unsigned int i = 0; i < size; ++i) {
        str >> sample;
        rir_.push_back(sample);
    }

    str >> size;
    buffer_.clear();
    for (unsigned int i = 0; i < size; ++i) {
        str >> sample;
        buffer_.push_back(sample);
    }
}

string ReverbRIR::write_to_string() const {
    ostringstream str;
    const auto s = " ";
    str << rir_.size() << s;
    for (const auto item : rir_) {
        str << item << s;
    }
    str << buffer_.size() << s;
    for (const auto item : buffer_) {
        str << item << s;
    }
    return str.str();
}

