# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dotmatrix', 'dotmatrix.displays']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'dotmatrix',
    'version': '0.2.0',
    'description': 'A pixel matrix rendered using braille characters.',
    'long_description': '# ⣿ dotmatrix\n_A dot matrix rendered using braille characters._\n\n[![PyPI](https://img.shields.io/pypi/v/dotmatrix)](https://pypi.org/project/dotmatrix/)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/dotmatrix)](https://pypi.org/project/dotmatrix/)\n[![PyPI - License](https://img.shields.io/pypi/l/dotmatrix)](https://pypi.org/project/dotmatrix/)\n[![Checked with mypy](http://www.mypy-lang.org/static/mypy_badge.svg)](http://mypy-lang.org/)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n## Description\n\nThis library provides class called `Matrix` which represents a dot matrix that can be rendered to a string of [Braille](https://en.wikipedia.org/wiki/Braille) [characters](https://en.wikipedia.org/wiki/Braille_Patterns). In addition the class also provides some usefull functions for drawing all kinds of things onto said matrix.\n\n### A word on fonts...\n\nThis heavily relies on the font you want display the resulting characters with. Some "monospace" fonts/systems **dot not** treat **all** characters as having the same width! In particular this affects the [blank braille character](https://en.wikipedia.org/wiki/Braille_pattern_dots-0) (this:\xa0`⠀`). The system that causes the most problems seems to be Windows while both mac OS and your average linux distribution don\'t screw it up. If you are having problems with the images in this readme you can have a look at the images included in the spoilers.\n\n## Install\n\nUse can install this library from [PyPI](https://pypi.org/project/dotmatrix/):\n\n```sh\npip install dotmatrix\n```\n\n### Example\n\n**Code**\n\n```python\nfrom dotmatrix import Matrix\n\nm = Matrix(64, 64)\n\nm.rectangle((0, 0), (63, 63))\nm.circle((31, 31), 31)\n\nprint(m.render())\n```\n\n**Output**\n\n```\n⡏⠉⠉⠉⠉⠉⠉⠉⢉⡩⠭⠛⠛⠉⠉⠉⠉⠉⠙⠛⠫⠭⣉⠉⠉⠉⠉⠉⠉⠉⠉⢹\n⡇⠀⠀⠀⠀⢀⡠⠊⠁⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠉⠢⣀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⢀⠔⠁⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠑⢄⠀⠀⠀⢸\n⡇⠀⡠⠊⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠢⡀⠀⢸\n⡇⡰⠁⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠱⡀⢸\n⣧⠃⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢣⢸\n⡟⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠘⣼\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⣿\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⣿\n⣷⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢰⢹\n⡏⡆⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⡎⢸\n⡇⠘⡄⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⡜⠀⢸\n⡇⠀⠈⢢⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢠⠊⠀⠀⢸\n⡇⠀⠀⠀⠑⢄⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⣀⠔⠁⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠈⠢⢄⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⣀⠤⠊⠀⠀⠀⠀⠀⠀⢸\n⣇⣀⣀⣀⣀⣀⣀⣀⣀⣈⣉⣒⣒⣤⣤⣤⣤⣤⣔⣒⣊⣉⣀⣀⣀⣀⣀⣀⣀⣀⣀⣸\n```\n\n<details><summary>image</summary>\n\nThis is what it should look like:\n\n![](https://github.com/timfi/dotmatrix/blob/root/.resources/img/basic_example.png)\n</details>\n\n## Drawing functions\n\nAs of now this library contains the following drawing functions:\n- `scatter` – Draws some points.\n- `iscatter` – Draws some points (from an iterator).\n- `show` – Draws an object implementing the `Dotted` protocol.\n- `line` – Draws a line.\n- `chain` – Draws a chain of segments.\n- `polygon` – Draws a polygon.\n- `rectangle` – Draws an axis aligned rectangle. (from two opposing corners)\n- `cricle` – Draws a circle.\n- `ellipse` – Draws an axis aligned ellipse.\n- `curve` – Draws a [Bézier curve](https://en.wikipedia.org/wiki/B%C3%A9zier_curve).\n- `plot` – Plots a series of XY-coordinates. (matplotlib.pyplot style)\n- `plotf` – Plots a function.\n\n<details><summary>Dotted protocol</summary>\n\n---\n```python\nclass Dotted(Protocol):\n    """An object that can be drawn on a Matrix."""\n\n    def __dots__(self) -> Iterable[Point]:\n        """Generate the pixel positions representing this object.\n\n        :return: pixels to draw\n        :rtype: Iterable[Point]\n        """\n```\n---\n</details>\n\n⚠️\xa0\xa0_The origin of the coordinate system, i.e. `(0, 0)`, is at the top left corner!_\n\n\n## Does it need to be Braille characters?\n\nNo, no it does not. It\'s just the default; you are free to choose how you want to render things. To facilitate this any given `Matrix` object internally makes use of an object implementing the `Display` protocol. For example this library implements, next to the `Braille` displays, some more display like `Block` or `Unit`.\n\n<details><summary>Display protocol</summary>\n\n---\n```python\nclass Display(Protocol[V, O]):\n    """An object that can be used as a matrix display."""\n\n    width: int\n    height: int\n    default_brush: V\n\n    def __init__(\n        self, width: int, height: int, *, default_brush: Union[V, UseDefault]\n    ) -> None:\n        """Initialize a matrix object.\n\n        :param width: width of the matrix\n        :type width: int\n        :param height: height of the matrix\n        :type height: int\n        """\n\n    def render(self) -> O:\n        """Render the current matrix state.\n\n        :return: render result\n        :rtype: O\n        """\n\n    def __getitem__(self, pos: Point) -> V:\n        """Get the value of a pixel.\n\n        :param pos: position of pixel to get\n        :type pos: Point\n        :raises IndexError: requested pixel is out of the bounds of the matrix\n        :return: state of the pixel\n        :rtype: bool\n        """\n\n    def __setitem__(self, pos: Point, val: V):\n        """Set the value of a pixel.\n\n        :param pos: position of the pixel to set\n        :type pos: Point\n        :param val: the value to set the pixel to\n        :type val: bool\n        :raises IndexError: requested pixel is out of the bounds of the matrix\n        """\n```\n---\n</details>\n<details><summary>Block display</summary>\n\n---\n**Code**\n\n```python\nfrom dotmatrix import Matrix\nfrom dotmatrix.displays import Block\n\n# Using a different display is as simple as passing it\n# into the display-argument of the initializer.\nm = Matrix(16, 16, display=Block)\n\nm.rectangle((0, 0), (15, 15))\nm.circle((7, 7), 7)\n\nprint(m.render())\n```\n\n**Output**\n\n```\n█▀▀██▀▀▀▀▀██▀▀▀█\n█▄▀         ▀▄ █\n█▀           ▀▄█\n█             ██\n█             ██\n██           █ █\n█ ▀▄▄     ▄▄▀  █\n█▄▄▄▄█████▄▄▄▄▄█\n```\n\n---\n</details>\n<details><summary>Unit display</summary>\n\n---\n**Code**\n\n```python\nfrom dotmatrix import Matrix\nfrom dotmatrix.displays import Block\n\n# The following isn\'t required for using the Unit display.\n# It\'s just here to demonstrate that you "pre-instantiate"\n# a display and construct a Matrix object from it using\n# Matrix.from_display.\nd = Unit(16, 16, chars=["  ", "##"])\nm = Matrix.from_display(d)\n\nm.curve((0, 0), (15, 0), (0, 15), (15, 15))\n\nprint(m.render())\n```\n\n**Output**\n\n```\n########\n        ####\n            ##\n              ##\n              ##\n              ##\n              ##\n              ##\n                ##\n                ##\n                ##\n                ##\n                ##\n                  ##\n                    ##\n                      ##########\n```\n\n---\n</details>\n\n## More examples\n\n<details><summary>Bézier flower</summary>\n\n---\n**Code**\n\n```python\nfrom dotmatrix import Matrix\n\nm = Matrix(64, 64)\n\nm.curve((0, 0), (63, 0), (0, 63), (63, 63))\nm.curve((0, 0), (0, 63), (63, 0), (63, 63))\nm.curve((63, 0), (0, 0), (63, 63), (0, 63))\nm.curve((63, 0), (63, 63), (0, 0), (0, 63))\n\nprint(m.render())\n```\n\n**Output**\n\n```\n⡏⠉⠉⠉⠉⠒⠒⠤⠤⣀⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢀⣀⠤⠤⠒⠒⠉⠉⠉⠉⢹\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠒⢄⠀⠀⠀⠀⠀⠀⡠⠒⠁⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⢣⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠑⡄⠀⠀⢠⠊⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⡜\n⠘⡄⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⡆⢰⠁⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢠⠃\n⠀⢣⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢱⡎⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⡜⠀\n⠀⠈⢢⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⡔⠁⠀\n⠀⠀⠀⠑⢄⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⡠⠊⠀⠀⠀\n⠀⠀⠀⠀⠀⠉⠢⠤⢄⣀⣀⣀⣀⣀⣀⣸⣇⣀⣀⣀⣀⣀⣀⡠⠤⠔⠉⠀⠀⠀⠀⠀\n⠀⠀⠀⠀⠀⣀⠤⠒⠒⠉⠉⠉⠉⠉⠉⢹⡏⠉⠉⠉⠉⠉⠉⠒⠒⠤⣀⠀⠀⠀⠀⠀\n⠀⠀⠀⡠⠊⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠑⢄⠀⠀⠀\n⠀⢀⠎⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠱⡀⠀\n⠀⡜⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⡜⢣⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢣⠀\n⢰⠁⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢠⠃⠘⡄⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⡆\n⡜⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⡠⠃⠀⠀⠘⢄⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢣\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⢀⡠⠊⠀⠀⠀⠀⠀⠀⠑⢄⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⣇⣀⣀⣀⣀⠤⠤⠔⠒⠉⠁⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠉⠒⠢⠤⠤⣀⣀⣀⣀⣸\n```\n\n<details><summary>image</summary>\n\nThis is what it should look like:\n\n![](https://github.com/timfi/dotmatrix/blob/root/.resources/img/bezier_flower.png)\n</details>\n\n---\n</details>\n\n<details><summary>Function plotting</summary>\n\n---\n**Code**\n\n```python\nfrom dotmatrix import Matrix\n\nm = Matrix(64, 64)\n\nm.rectangle((0, 0), (63, 63))\nm.plotf(\n    lambda x: 0.005 * x ** 3,\n    range(-31, 31),\n    origin=(31,31),\n)\n\nprint(m.render())\n```\n\n**Output**\n\n```\n⡏⠉⠉⠉⠉⠉⢹⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⠉⢹\n⡇⠀⠀⠀⠀⠀⠀⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⢱⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠈⡆⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠸⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠱⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠱⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠢⢄⣀⣀⣀⣀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠉⠢⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⢆⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⡆⠀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠸⡀⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢣⠀⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠘⡄⠀⠀⠀⠀⠀⠀⢸\n⡇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢣⠀⠀⠀⠀⠀⠀⢸\n⣇⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣀⣸⣀⣀⣀⣀⣀⣀⣸\n```\n\n<details><summary>image</summary>\n\nThis is what it should look like:\n\n![](https://github.com/timfi/dotmatrix/blob/root/.resources/img/plotting.png)\n</details>\n\n---\n</details>\n\n## Development\n\nIn case you want to add some code to this project your need to first make sure you have [poetry](https://python-poetry.org/) installed. Afterwards you can run the following commands to get your setup up and running:\n\n```sh\npoetry install\npoetry shell\npre-commit install\n```\n\nDue note that you will have to commit from _inside the virtual environment_ or you need to have the dev-tools installed in your local python installation.\n\nAll PRs will be style checked with [isort](https://github.com/PyCQA/isort/), [pydocstyle](https://github.com/PyCQA/pydocstyle/) and [black](https://github.com/psf/black) as well as type checked with [mypy](http://www.mypy-lang.org/). In addition to this all PRs should target the `dev`-branch and contain as many signed commits as possible (better yet _only_ signed commits 😉 ). If you have no clue how or why to sign your commits have a look at the [GitHub docs](https://docs.github.com/en/github/authenticating-to-github/managing-commit-signature-verification) on this topic.\n',
    'author': 'Tim Fischer',
    'author_email': 'me@timfi.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/timfi/dotmatrix',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
