"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const handlerCodeBundle = path.join(__dirname, '..', 'lambda', 'bundle.zip');
const handlerSourceDirectory = path.join(__dirname, '..', 'lambda', 'src');
class BucketDeployment extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.distributionPaths && !props.distribution) {
            throw new Error('Distribution must be specified if distribution paths are specified');
        }
        const assetHash = calcSourceHash(handlerSourceDirectory);
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit),
            code: lambda.Code.fromAsset(handlerCodeBundle, { assetHash }),
            runtime: lambda.Runtime.PYTHON_3_6,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_a = props.prune) !== null && _a !== void 0 ? _a : true,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: props.distribution ? props.distribution.distributionId : undefined,
                DistributionPaths: props.distributionPaths,
            },
        });
    }
    renderSingletonUuid(memoryLimit) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        return uuid;
    }
}
exports.BucketDeployment = BucketDeployment;
/**
 * We need a custom source hash calculation since the bundle.zip file
 * contains python dependencies installed during build and results in a
 * non-deterministic behavior.
 *
 * So we just take the `src/` directory of our custom resoruce code.
 */
function calcSourceHash(srcDir) {
    const sha = crypto.createHash('sha256');
    for (const file of fs.readdirSync(srcDir)) {
        const data = fs.readFileSync(path.join(srcDir, file));
        sha.update(`<file name=${file}>`);
        sha.update(data);
        sha.update('</file>');
    }
    return sha.digest('hex');
}
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase().startsWith('x-amzn-meta-')
        ? key.toLowerCase()
        : `x-amzn-meta-${key.toLowerCase()}`;
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    constructor(value) {
        this.value = value;
    }
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    static noCache() { return new CacheControl('no-cache'); }
    static noTransform() { return new CacheControl('no-transform'); }
    static setPublic() { return new CacheControl('public'); }
    static setPrivate() { return new CacheControl('private'); }
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    ServerSideEncryption["AES_256"] = "AES256";
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    StorageClass["STANDARD"] = "STANDARD";
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    StorageClass["GLACIER"] = "GLACIER";
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 *
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(value) {
        this.value = value;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    static fromString(s) { return new Expires(s); }
}
exports.Expires = Expires;
//# sourceMappingURL=data:application/json;base64,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