"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Source = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *      Source.bucket(bucket, key)
 *      Source.asset('/local/path/to/directory')
 *      Source.asset('/local/path/to/a/file.zip')
 *
 * @stability stable
 */
class Source {
    constructor() { }
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     *
     * Make sure you trust the producer of the archive.
     *
     * @param bucket The S3 Bucket.
     * @param zipObjectKey The S3 object key of the zip file with contents.
     * @stability stable
     */
    static bucket(bucket, zipObjectKey) {
        return {
            bind: (_, context) => {
                if (!context) {
                    throw new Error('To use a Source.bucket(), context must be provided');
                }
                bucket.grantRead(context.handlerRole);
                return { bucket, zipObjectKey };
            },
        };
    }
    /**
     * Uses a local asset as the deployment source.
     *
     * If the local asset is a .zip archive, make sure you trust the
     * producer of the archive.
     *
     * @param path The path to a local .zip file or a directory.
     * @stability stable
     */
    static asset(path, options) {
        return {
            bind(scope, context) {
                if (!context) {
                    throw new Error('To use a Source.asset(), context must be provided');
                }
                let id = 1;
                while (scope.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(scope, `Asset${id}`, {
                    path,
                    ...options,
                });
                if (!asset.isZipArchive) {
                    throw new Error('Asset path must be either a .zip file or a directory');
                }
                asset.grantRead(context.handlerRole);
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey,
                };
            },
        };
    }
}
exports.Source = Source;
_a = JSII_RTTI_SYMBOL_1;
Source[_a] = { fqn: "@aws-cdk/aws-s3-deployment.Source", version: "1.115.0" };
//# sourceMappingURL=data:application/json;base64,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