"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubActionsRole = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const owner_regexp_1 = require("./owner-regexp");
const provider_1 = require("./provider");
/**
 * Define an IAM Role that can be assumed by Github Actions workflow
 * via Github OpenID Connect Identity Provider.
 *
 * Besides `GithubConfiguration`, you may pass in any `iam.RoleProps` except `assumedBy`
 * which will be defined by this construct (CDK will fail if you do).
 *
 * @example
 * const uploadRole = new GithubActionsRole(scope, "UploadRole", {
 *   provider: GithubActionsIdentityProvider.fromAccount(scope, "GithubProvider"),
 *   owner: 'octo-org',
 *   repo: 'octo-repo',
 *   filter: 'ref:refs/tags/v*',
 *   roleName: 'MyUploadRole',
 * });
 *
 * myBucket.grantWrite(uploadRole);
 */
class GithubActionsRole extends iam.Role {
    /**
     * Define an IAM Role that can be assumed by Github Actions workflow
     * via Github OpenID Connect Identity Provider.
     *
     * Besides `GithubConfiguration`, you may pass in any `iam.RoleProps` except `assumedBy`
     * which will be defined by this construct (CDK will fail if you do).
     *
     * @example
     * const uploadRole = new GithubActionsRole(scope, "UploadRole", {
     *   provider: GithubActionsIdentityProvider.fromAccount(scope, "GithubProvider"),
     *   owner: 'octo-org',
     *   repo: 'octo-repo',
     *   filter: 'ref:refs/tags/v*',
     *   roleName: 'MyUploadRole',
     * });
     *
     * myBucket.grantWrite(uploadRole);
     */
    constructor(scope, id, props) {
        const { provider, owner, repo } = props;
        // Perform validations
        GithubActionsRole.validateOwner(scope, owner);
        GithubActionsRole.validateRepo(scope, repo);
        // Prepare values
        const subject = GithubActionsRole.formatSubject(props);
        const roleProps = GithubActionsRole.extractRoleProps(props);
        // The actual IAM Role creation
        super(scope, id, {
            ...roleProps,
            assumedBy: new iam.WebIdentityPrincipal(provider.openIdConnectProviderArn, {
                StringLike: {
                    // Only allow specified subjects to assume this role
                    [`${provider_1.GithubActionsIdentityProvider.issuer}:sub`]: subject,
                },
                StringEquals: {
                    // Audience is always sts.amazonaws.com with AWS official Github Action
                    // https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services#adding-the-identity-provider-to-aws
                    [`${provider_1.GithubActionsIdentityProvider.issuer}:aud`]: 'sts.amazonaws.com',
                },
            }),
        });
    }
    /**
     * Extracts props given for the created IAM Role Construct.
     * @param props for the GithubActionsRole
     * @returns for the IAM Role
     */
    static extractRoleProps(props) {
        const extractProps = props;
        delete extractProps.provider;
        delete extractProps.owner;
        delete extractProps.repo;
        delete extractProps.filter;
        return extractProps;
    }
    /** Validates the Github owner (organization or user) name. */
    static validateOwner(scope, owner) {
        if (owner_regexp_1.default.test(owner) !== true) {
            cdk.Annotations.of(scope).addError(`Invalid Github Repository Owner "${owner}". Must only contain alphanumeric characters or hyphens, cannot have multiple consecutive hyphens, cannot begin or end with a hypen and maximum lenght is 39 characters.`);
        }
    }
    /** Validates the Github repository name (without owner). */
    static validateRepo(scope, repo) {
        if (repo === '') {
            cdk.Annotations.of(scope).addError(`Invalid Github Repository Name "${repo}". May not be empty string.`);
        }
    }
    /** Formats the `sub` value used in trust policy. */
    static formatSubject(props) {
        const { owner, repo, filter = '*' } = props;
        return `repo:${owner}/${repo}:${filter}`;
    }
}
exports.GithubActionsRole = GithubActionsRole;
_a = JSII_RTTI_SYMBOL_1;
GithubActionsRole[_a] = { fqn: "aws-cdk-github-oidc.GithubActionsRole", version: "2.0.0" };
//# sourceMappingURL=data:application/json;base64,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