# Using IAM Policy Validator as a Python Library

This guide shows you how to use IAM Policy Validator as a Python library in your own applications, scripts, and automation tools.

## Table of Contents

- [Installation](#installation)
- [Quick Start](#quick-start)
- [Configuration Methods](#configuration-methods)
- [Common Use Cases](#common-use-cases)
- [Advanced Usage](#advanced-usage)
- [API Reference](#api-reference)
- [Examples](#examples)

## Installation

Install the package using pip or uv:

```bash
# Using pip
pip install iam-policy-validator

# Using uv (recommended)
uv add iam-policy-validator

# For development
uv add --dev iam-policy-validator
```

## Quick Start

The simplest way to validate IAM policies programmatically:

```python
import asyncio
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies
from iam_validator.core.report import ReportGenerator

async def validate_policies_example():
    # Load policies from file or directory
    loader = PolicyLoader()
    policies = loader.load_from_path("./policies/my-policy.json")

    # Validate policies (uses default configuration)
    results = await validate_policies(policies)

    # Generate and print report
    generator = ReportGenerator()
    report = generator.generate_report(results)
    generator.print_console_report(report)

    # Check if all policies are valid
    all_valid = all(r.is_valid for r in results)
    return 0 if all_valid else 1

# Run the validation
exit_code = asyncio.run(validate_policies_example())
```

**Key Points:**
- All validation is **asynchronous** (uses `async`/`await`)
- Returns `PolicyValidationResult` objects with issues and metadata
- Default configuration includes all built-in security checks

## Configuration Methods

There are multiple ways to configure the validator based on your needs.

### 1. Default Configuration (No Config)

Use built-in defaults - suitable for testing and simple validation:

```python
import asyncio
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies

async def validate_with_defaults():
    loader = PolicyLoader()
    policies = loader.load_from_path("./policies/")

    # Uses all built-in checks with default settings
    results = await validate_policies(policies)
    return results

asyncio.run(validate_with_defaults())
```

### 2. YAML Configuration File

Use a YAML file for persistent, shareable configuration:

**Create `iam-validator.yaml`:**
```yaml
settings:
  fail_on_severity: ["error", "critical"]
  cache_enabled: true
  cache_ttl_hours: 168  # 7 days
  parallel_execution: true
  enable_builtin_checks: true

# Configure built-in checks
security_best_practices:
  enabled: true
  severity: high

action_validation:
  enabled: true
  severity: error

action_condition_enforcement:
  enabled: true
  severity: critical
  action_condition_requirements:
    - actions:
        - "iam:PassRole"
      required_conditions:
        - condition_key: "iam:PassedToService"
```

**Use in Python:**
```python
import asyncio
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies

async def validate_with_config_file():
    loader = PolicyLoader()
    policies = loader.load_from_path("./policies/")

    # Load and apply configuration from file
    results = await validate_policies(
        policies,
        config_path="./iam-validator.yaml",
    )
    return results

asyncio.run(validate_with_config_file())
```

**Config File Auto-Discovery:**

The validator automatically searches for config files in this order:
1. Explicit path (if provided)
2. Current directory
3. Parent directories (walks up to root)
4. User home directory

Searched filenames: `iam-validator.yaml`, `iam-validator.yml`, `.iam-validator.yaml`, `.iam-validator.yml`

### 3. Programmatic Configuration

Create configuration dynamically in Python:

```python
import asyncio
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies
from iam_validator.core.config_loader import ValidatorConfig, ConfigLoader
from iam_validator.core.check_registry import create_default_registry
from iam_validator.core.aws_fetcher import AWSServiceFetcher

async def validate_with_programmatic_config():
    # Create configuration dictionary
    config_dict = {
        "settings": {
            "fail_on_severity": ["error", "critical"],
            "cache_enabled": True,
            "cache_ttl_hours": 24,
            "parallel_execution": True,
        },
        "security_best_practices": {
            "enabled": True,
            "severity": "high",
        },
        "action_validation": {
            "enabled": True,
            "severity": "error",
        }
    }

    # Save to temp config file
    import tempfile, yaml
    with tempfile.NamedTemporaryFile(mode='w', suffix='.yaml', delete=False) as f:
        yaml.dump(config_dict, f)
        config_path = f.name

    # Load and validate policies
    loader = PolicyLoader()
    policies = loader.load_from_path("./policies/")

    # Use public API with custom config
    results = await validate_policies(
        policies,
        config_path=config_path,
    )

    return results

asyncio.run(validate_with_programmatic_config())
```

### 4. Custom Checks Integration

Load custom validation checks from a directory:

**Directory structure:**
```
my_project/
├── policies/
│   └── my-policy.json
├── custom_checks/
│   ├── mfa_required_check.py
│   └── region_restriction_check.py
└── validate.py
```

**Create custom check (`custom_checks/mfa_required_check.py`):**
```python
from iam_validator.core.check_registry import PolicyCheck, CheckConfig
from iam_validator.core.models import Statement, ValidationIssue
from iam_validator.core.aws_fetcher import AWSServiceFetcher

class MFARequiredCheck(PolicyCheck):
    @property
    def check_id(self) -> str:
        return "mfa_required"

    @property
    def description(self) -> str:
        return "Ensures sensitive actions require MFA"

    @property
    def default_severity(self) -> str:
        return "error"

    async def execute(
        self,
        statement: Statement,
        statement_idx: int,
        fetcher: AWSServiceFetcher,
        config: CheckConfig,
    ) -> list[ValidationIssue]:
        issues = []

        if statement.effect != "Allow":
            return issues

        # Get config
        require_mfa_for = config.config.get("require_mfa_for", [])

        # Check actions
        actions = statement.get_actions()
        for action in actions:
            if action in require_mfa_for:
                if not self._has_mfa_condition(statement):
                    issues.append(
                        ValidationIssue(
                            severity=self.get_severity(config),
                            statement_sid=statement.sid,
                            statement_index=statement_idx,
                            issue_type="missing_mfa_condition",
                            message=f"Action '{action}' requires MFA",
                            action=action,
                            suggestion="Add: aws:MultiFactorAuthPresent = true",
                            line_number=statement.line_number,
                        )
                    )

        return issues

    def _has_mfa_condition(self, statement: Statement) -> bool:
        if not statement.condition:
            return False
        for operator, conditions in statement.condition.items():
            if "aws:MultiFactorAuthPresent" in conditions:
                value = conditions["aws:MultiFactorAuthPresent"]
                if value is True or str(value).lower() == "true":
                    return True
        return False
```

**Use custom checks in Python:**
```python
import asyncio
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies

async def validate_with_custom_checks():
    loader = PolicyLoader()
    policies = loader.load_from_path("./policies/")

    # Validate with custom checks directory
    results = await validate_policies(
        policies,
        custom_checks_dir="./custom_checks"  # Auto-discovers checks
    )

    return results

asyncio.run(validate_with_custom_checks())
```

**Configure custom checks via YAML:**
```yaml
settings:
  enable_builtin_checks: true

# Specify directory for auto-discovery
custom_checks_dir: "./custom_checks"

# Configure the custom check
mfa_required_check:
  enabled: true
  severity: critical
  require_mfa_for:
    - "iam:DeleteUser"
    - "iam:DeleteRole"
    - "s3:DeleteBucket"
```

## Common Use Cases

### Use Case 1: CI/CD Integration

```python
#!/usr/bin/env python3
"""Validate IAM policies in CI/CD pipeline."""
import asyncio
import sys
from pathlib import Path
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies
from iam_validator.core.formatters.json import JsonFormatter
from iam_validator.core.report import ReportGenerator

async def ci_validation():
    policy_dir = Path("./policies")
    config_file = Path("./iam-validator.yaml")
    output_file = Path("./validation-report.json")

    print(f"🔍 Validating policies in: {policy_dir}")

    # Load policies
    loader = PolicyLoader()
    try:
        policies = loader.load_from_path(str(policy_dir))
        print(f"📄 Found {len(policies)} policies")
    except Exception as e:
        print(f"❌ Error loading policies: {e}", file=sys.stderr)
        return 1

    # Validate
    results = await validate_policies(
        policies,
        config_path=str(config_file) if config_file.exists() else None,
    )

    # Generate JSON report for CI/CD tools
    formatter = JsonFormatter()
    generator = ReportGenerator()
    report = generator.generate_report(results)
    json_output = formatter.format(report)
    output_file.write_text(json_output)
    print(f"📊 Report saved to: {output_file}")

    # Print summary
    total_issues = sum(len(r.issues) for r in results)
    valid_count = sum(1 for r in results if r.is_valid)

    print(f"\n📈 Validation Summary:")
    print(f"  Total Policies: {len(results)}")
    print(f"  ✅ Valid: {valid_count}")
    print(f"  ❌ Invalid: {len(results) - valid_count}")
    print(f"  ⚠️  Total Issues: {total_issues}")

    # Exit with appropriate code for CI/CD
    if all(r.is_valid for r in results):
        print("✅ All policies are valid!")
        return 0
    else:
        print("❌ Validation failed!")
        return 1

if __name__ == "__main__":
    sys.exit(asyncio.run(ci_validation()))
```

### Use Case 2: Batch Processing

```python
import asyncio
from pathlib import Path
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies

async def batch_validate_multiple_dirs():
    """Validate policies from multiple directories."""
    directories = [
        "./iam-policies/",
        "./s3-policies/",
        "./lambda-policies/",
    ]

    all_results = []

    loader = PolicyLoader()

    for directory in directories:
        print(f"Processing {directory}...")

        policies = loader.load_from_path(directory)
        results = await validate_policies(
            policies,
            config_path="./iam-validator.yaml",
        )

        all_results.extend(results)

        # Print summary for this directory
        valid = sum(1 for r in results if r.is_valid)
        print(f"  {directory}: {valid}/{len(results)} valid")

    return all_results

results = asyncio.run(batch_validate_multiple_dirs())
```

### Use Case 3: Custom Report Generation

```python
import asyncio
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies
from iam_validator.core.report import ReportGenerator
from iam_validator.core.formatters.html import HtmlFormatter
from iam_validator.core.formatters.markdown import MarkdownFormatter
from iam_validator.core.formatters.csv import CsvFormatter

async def generate_multiple_reports():
    """Generate reports in multiple formats."""
    loader = PolicyLoader()
    policies = loader.load_from_path("./policies/")

    # Validate
    results = await validate_policies(policies)

    # Generate report
    generator = ReportGenerator()
    report = generator.generate_report(results)

    # Export in different formats
    html_formatter = HtmlFormatter()
    with open("report.html", "w") as f:
        f.write(html_formatter.format(report))
    print("✅ HTML report: report.html")

    md_formatter = MarkdownFormatter()
    with open("report.md", "w") as f:
        f.write(md_formatter.format(report))
    print("✅ Markdown report: report.md")

    csv_formatter = CsvFormatter()
    with open("report.csv", "w") as f:
        f.write(csv_formatter.format(report))
    print("✅ CSV report: report.csv")

    return report

asyncio.run(generate_multiple_reports())
```

### Use Case 4: Filter and Process Issues

```python
import asyncio
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies

async def process_issues_by_severity():
    """Process validation issues by severity."""
    loader = PolicyLoader()
    policies = loader.load_from_path("./policies/")

    results = await validate_policies(policies)

    # Group issues by severity
    issues_by_severity = {
        "critical": [],
        "error": [],
        "warning": [],
        "info": []
    }

    for result in results:
        for issue in result.issues:
            severity = issue.severity.lower()
            if severity in issues_by_severity:
                issues_by_severity[severity].append({
                    "file": result.policy_file,
                    "issue": issue
                })

    # Report critical issues
    if issues_by_severity["critical"]:
        print("🚨 CRITICAL ISSUES:")
        for item in issues_by_severity["critical"]:
            print(f"  {item['file']}: {item['issue'].message}")

    # Report errors
    if issues_by_severity["error"]:
        print("\n❌ ERRORS:")
        for item in issues_by_severity["error"]:
            print(f"  {item['file']}: {item['issue'].message}")

    # Return only critical and error counts
    return {
        "critical": len(issues_by_severity["critical"]),
        "error": len(issues_by_severity["error"]),
        "warning": len(issues_by_severity["warning"]),
    }

summary = asyncio.run(process_issues_by_severity())
print(f"\nSummary: {summary}")
```

## Advanced Usage

### Custom Configuration in Code

For fine-grained control, create configuration programmatically:

```python
import asyncio
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies

async def advanced_validation():
    # Create custom configuration dict
    config_dict = {
        "settings": {
            "parallel_execution": True,
            "enable_builtin_checks": True,
            "fail_on_severity": ["error", "critical"],
        },
        "policy_size": {
            "enabled": False,  # Disable specific check
        },
        "security_best_practices": {
            "enabled": True,
            "severity": "critical",
            "wildcard_action_check": {
                "enabled": True,
                "severity": "high"
            }
        }
    }

    # Save to temp config file
    import tempfile, yaml
    with tempfile.NamedTemporaryFile(mode='w', suffix='.yaml', delete=False) as f:
        yaml.dump(config_dict, f)
        config_path = f.name

    # Load and validate policies
    loader = PolicyLoader()
    policies = loader.load_from_path("./policies/")

    results = await validate_policies(
        policies,
        config_path=config_path,
    )

    return results

asyncio.run(advanced_validation())
```

**Important:** Always use the public `validate_policies()` API. Internal functions (prefixed with `_`) are not part of the stable API.

### Streaming Validation

Process large numbers of policies efficiently:

```python
import asyncio
from pathlib import Path
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies

async def streaming_validation():
    """Validate policies one at a time (memory efficient)."""
    policy_dir = Path("./policies")

    loader = PolicyLoader()

    # Get all policy files
    policy_files = list(policy_dir.glob("**/*.json"))
    policy_files.extend(policy_dir.glob("**/*.yaml"))

    total = len(policy_files)
    processed = 0
    all_valid = True

    for policy_file in policy_files:
        # Load single policy
        policies = loader.load_from_path(str(policy_file))

        # Validate
        results = await validate_policies(policies)

        # Process result immediately
        for result in results:
            processed += 1
            status = "✅" if result.is_valid else "❌"
            print(f"[{processed}/{total}] {status} {policy_file.name}")

            if not result.is_valid:
                all_valid = False
                for issue in result.issues:
                    print(f"  - {issue.severity}: {issue.message}")

    return all_valid

asyncio.run(streaming_validation())
```

### Custom Caching Configuration

Configure caching and offline mode via config file:

```python
import asyncio
from pathlib import Path
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies

async def validate_with_custom_cache():
    # Create config with custom cache settings
    config_dict = {
        "settings": {
            "cache_enabled": True,
            "cache_ttl_hours": 168,  # 7 days
            "cache_directory": str(Path.home() / ".cache" / "iam-validator"),
            "aws_services_dir": "./aws_services",  # For offline mode
        }
    }

    # Save to temp config file
    import tempfile, yaml
    with tempfile.NamedTemporaryFile(mode='w', suffix='.yaml', delete=False) as f:
        yaml.dump(config_dict, f)
        config_path = f.name

    # Load and validate with custom cache config
    loader = PolicyLoader()
    policies = loader.load_from_path("./policies/")

    results = await validate_policies(
        policies,
        config_path=config_path,
    )

    return results

asyncio.run(validate_with_custom_cache())
```

## API Reference

### Core Classes

#### `PolicyLoader`
Loads IAM policies from files or directories.

```python
from iam_validator.core.policy_loader import PolicyLoader

loader = PolicyLoader()

# Load from single file
policies = loader.load_from_path("./policy.json")

# Load from directory (recursive)
policies = loader.load_from_path("./policies/")

# Returns: list[tuple[str, IAMPolicy]]
# - str: file path
# - IAMPolicy: parsed policy object
```

#### `validate_policies()`
Main validation function.

```python
from iam_validator.core.policy_checks import validate_policies

results = await validate_policies(
    policies,                    # list[tuple[str, IAMPolicy]]
    config_path=None,           # Optional: path to config file
    custom_checks_dir=None,     # Optional: directory for custom checks
)

# Returns: list[PolicyValidationResult]
```

#### `ValidatorConfig`
Configuration object.

```python
from iam_validator.core.config_loader import ValidatorConfig

# From dictionary
config = ValidatorConfig(config_dict, use_defaults=True)

# Get check configuration
check_config = config.get_check_config("action_validation")

# Check if enabled
enabled = config.is_check_enabled("security_best_practices")

# Get severity
severity = config.get_check_severity("action_validation")

# Get setting
cache_enabled = config.get_setting("cache_enabled", True)
```

#### `ConfigLoader`
Loads configuration from files.

```python
from iam_validator.core.config_loader import ConfigLoader

# Load with auto-discovery
config = ConfigLoader.load_config(
    explicit_path=None,     # Optional: explicit path
    search_path=None,       # Optional: start search from path
    allow_missing=True,     # Return defaults if not found
)

# Find config file
config_path = ConfigLoader.find_config_file()

# Load YAML
config_dict = ConfigLoader.load_yaml(Path("config.yaml"))

# Apply config to registry
ConfigLoader.apply_config_to_registry(config, registry)
```

#### `CheckRegistry`
Manages validation checks.

```python
from iam_validator.core.check_registry import create_default_registry

# Create registry with built-in checks
registry = create_default_registry(
    enable_parallel=True,
    include_builtin_checks=True
)

# Register custom check
registry.register(my_custom_check)

# Configure check
from iam_validator.core.check_registry import CheckConfig
registry.configure_check(
    "security_best_practices",
    CheckConfig(
        check_id="security_best_practices",
        enabled=True,
        severity="high",
        config={"wildcard_action_check": {"enabled": True}}
    )
)

# Execute checks
issues = await registry.execute_checks_parallel(
    statement, statement_idx, fetcher
)
```

#### `PolicyValidationResult`
Validation result object.

```python
class PolicyValidationResult:
    policy_file: str                    # Path to policy file
    is_valid: bool                      # Overall validation status
    issues: list[ValidationIssue]       # All validation issues
    actions_checked: int                # Number of actions validated
    resources_checked: int              # Number of resources validated
    condition_keys_checked: int         # Number of condition keys validated
```

#### `ValidationIssue`
Individual validation issue.

```python
class ValidationIssue:
    severity: str                       # "critical", "error", "warning", "info"
    statement_sid: str | None           # Statement SID
    statement_index: int                # Statement index in policy
    issue_type: str                     # Type of issue
    message: str                        # Human-readable message
    suggestion: str | None              # Fix suggestion
    action: str | None                  # Related action
    resource: str | None                # Related resource
    condition_key: str | None           # Related condition key
    line_number: int | None             # Line number in file
```

#### `ReportGenerator`
Generates validation reports.

```python
from iam_validator.core.report import ReportGenerator

generator = ReportGenerator()

# Generate report
report = generator.generate_report(results)

# Print console report
generator.print_console_report(report)

# Get statistics
stats = report.get_statistics()
# Returns: dict with counts, severities, etc.
```

### Formatters

All formatters implement the `format(report: ValidationReport) -> str` method.

```python
from iam_validator.core.formatters.json import JsonFormatter
from iam_validator.core.formatters.markdown import MarkdownFormatter
from iam_validator.core.formatters.html import HtmlFormatter
from iam_validator.core.formatters.csv import CsvFormatter
from iam_validator.core.formatters.sarif import SarifFormatter

# Generate different formats
json_output = JsonFormatter().format(report)
markdown_output = MarkdownFormatter().format(report)
html_output = HtmlFormatter().format(report)
csv_output = CsvFormatter().format(report)
sarif_output = SarifFormatter().format(report)
```

## Examples

### Complete Validation Script

```python
#!/usr/bin/env python3
"""Complete validation script with all features."""
import asyncio
import sys
from pathlib import Path
from iam_validator.core.policy_loader import PolicyLoader
from iam_validator.core.policy_checks import validate_policies
from iam_validator.core.config_loader import ConfigLoader
from iam_validator.core.report import ReportGenerator
from iam_validator.core.formatters.json import JsonFormatter
from iam_validator.core.formatters.html import HtmlFormatter

async def main():
    # Configuration
    policy_dir = Path("./policies")
    config_file = Path("./iam-validator.yaml")
    custom_checks_dir = Path("./custom_checks")
    output_dir = Path("./reports")
    output_dir.mkdir(exist_ok=True)

    # Load configuration
    config = ConfigLoader.load_config(
        explicit_path=str(config_file) if config_file.exists() else None,
        allow_missing=True
    )

    print(f"🔍 Validating policies in: {policy_dir}")
    print(f"⚙️  Configuration: {config_file if config_file.exists() else 'defaults'}")

    # Load policies
    loader = PolicyLoader()
    try:
        policies = loader.load_from_path(str(policy_dir))
        print(f"📄 Found {len(policies)} policies\n")
    except Exception as e:
        print(f"❌ Error loading policies: {e}", file=sys.stderr)
        return 1

    # Validate
    print("🔄 Running validation...")
    results = await validate_policies(
        policies,
        config_path=str(config_file) if config_file.exists() else None,
        custom_checks_dir=str(custom_checks_dir) if custom_checks_dir.exists() else None
    )

    # Generate report
    generator = ReportGenerator()
    report = generator.generate_report(results)

    # Console output
    print("\n" + "="*60)
    generator.print_console_report(report)
    print("="*60)

    # Export reports
    json_formatter = JsonFormatter()
    json_file = output_dir / "validation-report.json"
    json_file.write_text(json_formatter.format(report))
    print(f"\n📊 JSON report: {json_file}")

    html_formatter = HtmlFormatter()
    html_file = output_dir / "validation-report.html"
    html_file.write_text(html_formatter.format(report))
    print(f"📊 HTML report: {html_file}")

    # Summary statistics
    stats = report.get_statistics()
    print(f"\n📈 Validation Summary:")
    print(f"  Total Policies: {len(results)}")
    print(f"  ✅ Valid: {stats['valid_policies']}")
    print(f"  ❌ Invalid: {stats['invalid_policies']}")
    print(f"  ⚠️  Total Issues: {stats['total_issues']}")

    # Issue breakdown by severity
    if stats.get('issues_by_severity'):
        print(f"\n  Issues by Severity:")
        for severity, count in stats['issues_by_severity'].items():
            print(f"    {severity.upper()}: {count}")

    # Exit with appropriate code
    if all(r.is_valid for r in results):
        print("\n✅ All policies are valid!")
        return 0
    else:
        print("\n❌ Validation failed!")
        return 1

if __name__ == "__main__":
    try:
        exit_code = asyncio.run(main())
        sys.exit(exit_code)
    except KeyboardInterrupt:
        print("\n⚠️  Interrupted by user")
        sys.exit(130)
    except Exception as e:
        print(f"\n❌ Unexpected error: {e}", file=sys.stderr)
        sys.exit(1)
```

## Configuration Reference

For detailed configuration options, see:
- [Configuration Guide](configuration.md)
- [Custom Checks Guide](custom-checks.md)
- [Examples](../examples/configs/)

## Related Documentation

- **[Complete Documentation (DOCS.md)](../DOCS.md)** - Full user documentation
- **[Custom Checks Guide](custom-checks.md)** - Creating custom validation rules
- **[Configuration Reference](configuration.md)** - YAML configuration options
- **[GitHub Actions Integration](github-actions-workflows.md)** - Using in CI/CD
- **[AWS Services Backup](aws-services-backup.md)** - Offline validation setup

## Support

- **Issues**: [GitHub Issues](https://github.com/boogy/iam-policy-validator/issues)
- **Discussions**: [GitHub Discussions](https://github.com/boogy/iam-policy-validator/discussions)
- **Examples**: [examples/](../examples/) directory
