"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StreamViewType = exports.ProjectionType = exports.BillingMode = exports.AttributeType = exports.Table = exports.TableEncryption = exports.Operation = void 0;
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const dynamodb_canned_metrics_generated_1 = require("./dynamodb-canned-metrics.generated");
const dynamodb_generated_1 = require("./dynamodb.generated");
const perms = require("./perms");
const replica_provider_1 = require("./replica-provider");
const scalable_table_attribute_1 = require("./scalable-table-attribute");
const HASH_KEY_TYPE = 'HASH';
const RANGE_KEY_TYPE = 'RANGE';
// https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
const MAX_LOCAL_SECONDARY_INDEX_COUNT = 5;
/**
 * Supported DynamoDB table operations.
 *
 * @stability stable
 */
var Operation;
(function (Operation) {
    Operation["GET_ITEM"] = "GetItem";
    Operation["BATCH_GET_ITEM"] = "BatchGetItem";
    Operation["SCAN"] = "Scan";
    Operation["QUERY"] = "Query";
    Operation["GET_RECORDS"] = "GetRecords";
    Operation["PUT_ITEM"] = "PutItem";
    Operation["DELETE_ITEM"] = "DeleteItem";
    Operation["UPDATE_ITEM"] = "UpdateItem";
    Operation["BATCH_WRITE_ITEM"] = "BatchWriteItem";
})(Operation = exports.Operation || (exports.Operation = {}));
/**
 * What kind of server-side encryption to apply to this table.
 *
 * @stability stable
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["DEFAULT"] = "AWS_OWNED";
    TableEncryption["CUSTOMER_MANAGED"] = "CUSTOMER_MANAGED";
    TableEncryption["AWS_MANAGED"] = "AWS_MANAGED";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
class TableBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * @stability stable
         */
        this.regionalArns = new Array();
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined).
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...).
     * @stability stable
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                this.tableArn,
                core_1.Lazy.string({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.string({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                })),
            ],
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement associated with this table's stream to an IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined).
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...).
     * @stability stable
     */
    grantStream(grantee, ...actions) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.tableStreamArn],
            scope: this,
        });
    }
    /**
     * Permits an IAM principal all data read operations from this table: BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantReadData(grantee) {
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, tableActions: perms.READ_DATA_ACTIONS });
    }
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined).
     * @stability stable
     */
    grantTableListStreams(grantee) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Permits an IAM principal all stream data read operations for this table's stream: DescribeStream, GetRecords, GetShardIterator, ListStreams.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantStreamRead(grantee) {
        this.grantTableListStreams(grantee);
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, streamActions: perms.READ_STREAM_DATA_ACTIONS });
    }
    /**
     * Permits an IAM principal all data write operations to this table: BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantWriteData(grantee) {
        return this.combinedGrant(grantee, { keyActions: perms.KEY_WRITE_ACTIONS, tableActions: perms.WRITE_DATA_ACTIONS });
    }
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     *
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantReadWriteData(grantee) {
        const tableActions = perms.READ_DATA_ACTIONS.concat(perms.WRITE_DATA_ACTIONS);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions });
    }
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantFullAccess(grantee) {
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions: ['dynamodb:*'] });
    }
    /**
     * Return the given named metric for this Table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/DynamoDB',
            metricName,
            dimensions: {
                TableName: this.tableName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the consumed read capacity units this table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricConsumedReadCapacityUnits(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.consumedReadCapacityUnitsSum, props);
    }
    /**
     * Metric for the consumed write capacity units this table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricConsumedWriteCapacityUnits(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.consumedWriteCapacityUnitsSum, props);
    }
    /**
     * (deprecated) Metric for the system errors this table.
     *
     * @deprecated use `metricSystemErrorsForOperations`.
     */
    metricSystemErrors(props) {
        var _a, _b;
        if (!((_a = props === null || props === void 0 ? void 0 : props.dimensions) === null || _a === void 0 ? void 0 : _a.Operation)) {
            // 'Operation' must be passed because its an operational metric.
            throw new Error("'Operation' dimension must be passed for the 'SystemErrors' metric.");
        }
        const dimensions = {
            TableName: this.tableName,
            ...(_b = props === null || props === void 0 ? void 0 : props.dimensions) !== null && _b !== void 0 ? _b : {},
        };
        return this.metric('SystemErrors', { statistic: 'sum', ...props, dimensions });
    }
    /**
     * Metric for the user errors.
     *
     * Note that this metric reports user errors across all
     * the tables in the account and region the table resides in.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricUserErrors(props) {
        if (props === null || props === void 0 ? void 0 : props.dimensions) {
            throw new Error("'dimensions' is not supported for the 'UserErrors' metric");
        }
        // overriding 'dimensions' here because this metric is an account metric.
        // see 'UserErrors' in https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/metrics-dimensions.html
        return this.metric('UserErrors', { statistic: 'sum', ...props, dimensions: {} });
    }
    /**
     * Metric for the conditional check failed requests this table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricConditionalCheckFailedRequests(props) {
        return this.metric('ConditionalCheckFailedRequests', { statistic: 'sum', ...props });
    }
    /**
     * How many requests are throttled on this table.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricThrottledRequests(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.throttledRequestsSum, props);
    }
    /**
     * Metric for the successful request latency this table.
     *
     * By default, the metric will be calculated as an average over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricSuccessfulRequestLatency(props) {
        var _a;
        if (!((_a = props === null || props === void 0 ? void 0 : props.dimensions) === null || _a === void 0 ? void 0 : _a.Operation)) {
            throw new Error("'Operation' dimension must be passed for the 'SuccessfulRequestLatency' metric.");
        }
        const dimensions = {
            TableName: this.tableName,
            Operation: props.dimensions.Operation,
        };
        return new cloudwatch.Metric({
            ...dynamodb_canned_metrics_generated_1.DynamoDBMetrics.successfulRequestLatencyAverage(dimensions),
            ...props,
            dimensions,
        }).attachTo(this);
    }
    /**
     * Metric for the system errors this table.
     *
     * This will sum errors across all possible operations.
     * Note that by default, each individual metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricSystemErrorsForOperations(props) {
        var _a, _b;
        if ((_a = props === null || props === void 0 ? void 0 : props.dimensions) === null || _a === void 0 ? void 0 : _a.Operation) {
            throw new Error("The Operation dimension is not supported. Use the 'operations' property.");
        }
        const operations = (_b = props === null || props === void 0 ? void 0 : props.operations) !== null && _b !== void 0 ? _b : Object.values(Operation);
        const values = this.createMetricsForOperations('SystemErrors', operations, { statistic: 'sum', ...props });
        const sum = new cloudwatch.MathExpression({
            expression: `${Object.keys(values).join(' + ')}`,
            usingMetrics: { ...values },
            color: props === null || props === void 0 ? void 0 : props.color,
            label: 'Sum of errors across all operations',
            period: props === null || props === void 0 ? void 0 : props.period,
        });
        return sum;
    }
    /**
     * Create a map of metrics that can be used in a math expression.
     *
     * Using the return value of this function as the `usingMetrics` property in `cloudwatch.MathExpression` allows you to
     * use the keys of this map as metric names inside you expression.
     *
     * @param metricName The metric name.
     * @param operations The list of operations to create metrics for.
     * @param props Properties for the individual metrics.
     * @param metricNameMapper Mapper function to allow controlling the individual metric name per operation.
     */
    createMetricsForOperations(metricName, operations, props, metricNameMapper) {
        var _a;
        const metrics = {};
        const mapper = metricNameMapper !== null && metricNameMapper !== void 0 ? metricNameMapper : (op => op.toLowerCase());
        if ((_a = props === null || props === void 0 ? void 0 : props.dimensions) === null || _a === void 0 ? void 0 : _a.Operation) {
            throw new Error('Invalid properties. Operation dimension is not supported when calculating operational metrics');
        }
        for (const operation of operations) {
            const metric = this.metric(metricName, {
                ...props,
                dimensions: {
                    TableName: this.tableName,
                    Operation: operation,
                    ...props === null || props === void 0 ? void 0 : props.dimensions,
                },
            });
            const operationMetricName = mapper(operation);
            const firstChar = operationMetricName.charAt(0);
            if (firstChar === firstChar.toUpperCase()) {
                // https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/using-metric-math.html#metric-math-syntax
                throw new Error(`Mapper generated an illegal operation metric name: ${operationMetricName}. Must start with a lowercase letter`);
            }
            metrics[operationMetricName] = metric;
        }
        return metrics;
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param opts Options for keyActions, tableActions and streamActions
     */
    combinedGrant(grantee, opts) {
        if (opts.tableActions) {
            const resources = [this.tableArn,
                core_1.Lazy.string({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.string({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                }))];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.tableActions,
                resourceArns: resources,
                scope: this,
            });
            if (this.encryptionKey && opts.keyActions) {
                this.encryptionKey.grant(grantee, ...opts.keyActions);
            }
            return ret;
        }
        if (opts.streamActions) {
            if (!this.tableStreamArn) {
                throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
            }
            const resources = [this.tableStreamArn];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.streamActions,
                resourceArns: resources,
                scope: this,
            });
            return ret;
        }
        throw new Error(`Unexpected 'action', ${opts.tableActions || opts.streamActions}`);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ TableName: this.tableName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * Provides a DynamoDB table.
 *
 * @stability stable
 */
class Table extends TableBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.keySchema = new Array();
        this.attributeDefinitions = new Array();
        this.globalSecondaryIndexes = new Array();
        this.localSecondaryIndexes = new Array();
        this.secondaryIndexNames = new Set();
        this.nonKeyAttributes = new Set();
        this.tableScaling = {};
        this.indexScaling = new Map();
        const { sseSpecification, encryptionKey } = this.parseEncryption(props);
        this.billingMode = props.billingMode || BillingMode.PROVISIONED;
        this.validateProvisioning(props);
        let streamSpecification;
        if (props.replicationRegions) {
            if (props.stream && props.stream !== StreamViewType.NEW_AND_OLD_IMAGES) {
                throw new Error('`stream` must be set to `NEW_AND_OLD_IMAGES` when specifying `replicationRegions`');
            }
            streamSpecification = { streamViewType: StreamViewType.NEW_AND_OLD_IMAGES };
            if (props.billingMode && props.billingMode !== BillingMode.PAY_PER_REQUEST) {
                throw new Error('The `PAY_PER_REQUEST` billing mode must be used when specifying `replicationRegions`');
            }
            this.billingMode = BillingMode.PAY_PER_REQUEST;
        }
        else if (props.stream) {
            streamSpecification = { streamViewType: props.stream };
        }
        this.table = new dynamodb_generated_1.CfnTable(this, 'Resource', {
            tableName: this.physicalName,
            keySchema: this.keySchema,
            attributeDefinitions: this.attributeDefinitions,
            globalSecondaryIndexes: core_1.Lazy.any({ produce: () => this.globalSecondaryIndexes }, { omitEmptyArray: true }),
            localSecondaryIndexes: core_1.Lazy.any({ produce: () => this.localSecondaryIndexes }, { omitEmptyArray: true }),
            pointInTimeRecoverySpecification: props.pointInTimeRecovery ? { pointInTimeRecoveryEnabled: props.pointInTimeRecovery } : undefined,
            billingMode: this.billingMode === BillingMode.PAY_PER_REQUEST ? this.billingMode : undefined,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
            sseSpecification,
            streamSpecification,
            timeToLiveSpecification: props.timeToLiveAttribute ? { attributeName: props.timeToLiveAttribute, enabled: true } : undefined,
        });
        this.table.applyRemovalPolicy(props.removalPolicy);
        this.encryptionKey = encryptionKey;
        this.tableArn = this.getResourceArnAttribute(this.table.attrArn, {
            service: 'dynamodb',
            resource: 'table',
            resourceName: this.physicalName,
        });
        this.tableName = this.getResourceNameAttribute(this.table.ref);
        if (props.tableName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', this.tableName);
        }
        this.tableStreamArn = streamSpecification ? this.table.attrStreamArn : undefined;
        this.scalingRole = this.makeScalingRole();
        this.addKey(props.partitionKey, HASH_KEY_TYPE);
        this.tablePartitionKey = props.partitionKey;
        if (props.sortKey) {
            this.addKey(props.sortKey, RANGE_KEY_TYPE);
            this.tableSortKey = props.sortKey;
        }
        if (props.replicationRegions && props.replicationRegions.length > 0) {
            this.createReplicaTables(props.replicationRegions);
        }
    }
    /**
     * (deprecated) Permits an IAM Principal to list all DynamoDB Streams.
     *
     * @param grantee The principal (no-op if undefined).
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     */
    static grantListStreams(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     * @stability stable
     */
    static fromTableName(scope, id, tableName) {
        return Table.fromTableAttributes(scope, id, { tableName });
    }
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     * @stability stable
     */
    static fromTableArn(scope, id, tableArn) {
        return Table.fromTableAttributes(scope, id, { tableArn });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     * @stability stable
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends TableBase {
            constructor(_tableArn, tableName, tableStreamArn) {
                var _a, _b;
                super(scope, id);
                this.hasIndex = ((_a = attrs.globalIndexes) !== null && _a !== void 0 ? _a : []).length > 0 ||
                    ((_b = attrs.localIndexes) !== null && _b !== void 0 ? _b : []).length > 0;
                this.tableArn = _tableArn;
                this.tableName = tableName;
                this.tableStreamArn = tableStreamArn;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        let name;
        let arn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.tableName) {
            if (!attrs.tableArn) {
                throw new Error('One of tableName or tableArn is required!');
            }
            arn = attrs.tableArn;
            const maybeTableName = stack.parseArn(attrs.tableArn).resourceName;
            if (!maybeTableName) {
                throw new Error('ARN for DynamoDB table must be in the form: ...');
            }
            name = maybeTableName;
        }
        else {
            if (attrs.tableArn) {
                throw new Error('Only one of tableArn or tableName can be provided');
            }
            name = attrs.tableName;
            arn = stack.formatArn({
                service: 'dynamodb',
                resource: 'table',
                resourceName: attrs.tableName,
            });
        }
        return new Import(arn, name, attrs.tableStreamArn);
    }
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index.
     * @stability stable
     */
    addGlobalSecondaryIndex(props) {
        this.validateProvisioning(props);
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const gsiKeySchema = this.buildIndexKeySchema(props.partitionKey, props.sortKey);
        const gsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.globalSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: gsiKeySchema,
            projection: gsiProjection,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
        });
        this.indexScaling.set(props.indexName, {});
    }
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index.
     * @stability stable
     */
    addLocalSecondaryIndex(props) {
        // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
        if (this.localSecondaryIndexes.length >= MAX_LOCAL_SECONDARY_INDEX_COUNT) {
            throw new RangeError(`a maximum number of local secondary index per table is ${MAX_LOCAL_SECONDARY_INDEX_COUNT}`);
        }
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const lsiKeySchema = this.buildIndexKeySchema(this.tablePartitionKey, props.sortKey);
        const lsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.localSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: lsiKeySchema,
            projection: lsiProjection,
        });
    }
    /**
     * Enable read capacity scaling for this table.
     *
     * @returns An object to configure additional AutoScaling settings
     * @stability stable
     */
    autoScaleReadCapacity(props) {
        if (this.tableScaling.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'ReadScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for this table.
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     * @stability stable
     */
    autoScaleWriteCapacity(props) {
        if (this.tableScaling.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'WriteScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable read capacity scaling for the given GSI.
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     * @stability stable
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this index');
        }
        return attributePair.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}ReadScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for the given GSI.
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     * @stability stable
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this index');
        }
        return attributePair.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}WriteScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     * @stability stable
     */
    validate() {
        const errors = new Array();
        if (!this.tablePartitionKey) {
            errors.push('a partition key must be specified');
        }
        if (this.localSecondaryIndexes.length > 0 && !this.tableSortKey) {
            errors.push('a sort key of the table must be specified to add local secondary indexes');
        }
        return errors;
    }
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    validateProvisioning(props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            if (props.readCapacity !== undefined || props.writeCapacity !== undefined) {
                throw new Error('you cannot provision read and write capacity for a table with PAY_PER_REQUEST billing mode');
            }
        }
    }
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    validateIndexName(indexName) {
        if (this.secondaryIndexNames.has(indexName)) {
            // a duplicate index name causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`a duplicate index name, ${indexName}, is not allowed`);
        }
        this.secondaryIndexNames.add(indexName);
    }
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    validateNonKeyAttributes(nonKeyAttributes) {
        if (this.nonKeyAttributes.size + nonKeyAttributes.length > 100) {
            // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
            throw new RangeError('a maximum number of nonKeyAttributes across all of secondary indexes is 100');
        }
        // store all non-key attributes
        nonKeyAttributes.forEach(att => this.nonKeyAttributes.add(att));
    }
    buildIndexKeySchema(partitionKey, sortKey) {
        this.registerAttribute(partitionKey);
        const indexKeySchema = [
            { attributeName: partitionKey.name, keyType: HASH_KEY_TYPE },
        ];
        if (sortKey) {
            this.registerAttribute(sortKey);
            indexKeySchema.push({ attributeName: sortKey.name, keyType: RANGE_KEY_TYPE });
        }
        return indexKeySchema;
    }
    buildIndexProjection(props) {
        if (props.projectionType === ProjectionType.INCLUDE && !props.nonKeyAttributes) {
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-dynamodb-projectionobject.html
            throw new Error(`non-key attributes should be specified when using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.projectionType !== ProjectionType.INCLUDE && props.nonKeyAttributes) {
            // this combination causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`non-key attributes should not be specified when not using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.nonKeyAttributes) {
            this.validateNonKeyAttributes(props.nonKeyAttributes);
        }
        return {
            projectionType: props.projectionType ? props.projectionType : ProjectionType.ALL,
            nonKeyAttributes: props.nonKeyAttributes ? props.nonKeyAttributes : undefined,
        };
    }
    findKey(keyType) {
        return this.keySchema.find(prop => prop.keyType === keyType);
    }
    addKey(attribute, keyType) {
        const existingProp = this.findKey(keyType);
        if (existingProp) {
            throw new Error(`Unable to set ${attribute.name} as a ${keyType} key, because ${existingProp.attributeName} is a ${keyType} key`);
        }
        this.registerAttribute(attribute);
        this.keySchema.push({
            attributeName: attribute.name,
            keyType,
        });
        return this;
    }
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    registerAttribute(attribute) {
        const { name, type } = attribute;
        const existingDef = this.attributeDefinitions.find(def => def.attributeName === name);
        if (existingDef && existingDef.attributeType !== type) {
            throw new Error(`Unable to specify ${name} as ${type} because it was already defined as ${existingDef.attributeType}`);
        }
        if (!existingDef) {
            this.attributeDefinitions.push({
                attributeName: name,
                attributeType: type,
            });
        }
    }
    /**
     * Return the role that will be used for AutoScaling
     */
    makeScalingRole() {
        // Use a Service Linked Role.
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_DynamoDBTable',
        }));
    }
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    createReplicaTables(regions) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && regions.includes(stack.region)) {
            throw new Error('`replicationRegions` cannot include the region where this stack is deployed.');
        }
        const provider = replica_provider_1.ReplicaProvider.getOrCreate(this);
        // Documentation at https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
        // is currently incorrect. AWS Support recommends `dynamodb:*` in both source and destination regions
        const onEventHandlerPolicy = new SourceTableAttachedPolicy(this, provider.onEventHandler.role);
        const isCompleteHandlerPolicy = new SourceTableAttachedPolicy(this, provider.isCompleteHandler.role);
        // Permissions in the source region
        this.grant(onEventHandlerPolicy, 'dynamodb:*');
        this.grant(isCompleteHandlerPolicy, 'dynamodb:DescribeTable');
        let previousRegion;
        for (const region of new Set(regions)) { // Remove duplicates
            // Use multiple custom resources because multiple create/delete
            // updates cannot be combined in a single API call.
            const currentRegion = new core_1.CustomResource(this, `Replica${region}`, {
                serviceToken: provider.provider.serviceToken,
                resourceType: 'Custom::DynamoDBReplica',
                properties: {
                    TableName: this.tableName,
                    Region: region,
                },
            });
            currentRegion.node.addDependency(onEventHandlerPolicy.policy, isCompleteHandlerPolicy.policy);
            // Deploy time check to prevent from creating a replica in the region
            // where this stack is deployed. Only needed for environment agnostic
            // stacks.
            if (core_1.Token.isUnresolved(stack.region)) {
                const createReplica = new core_1.CfnCondition(this, `StackRegionNotEquals${region}`, {
                    expression: core_1.Fn.conditionNot(core_1.Fn.conditionEquals(region, core_1.Aws.REGION)),
                });
                const cfnCustomResource = currentRegion.node.defaultChild;
                cfnCustomResource.cfnOptions.condition = createReplica;
            }
            // Save regional arns for grantXxx() methods
            this.regionalArns.push(stack.formatArn({
                region,
                service: 'dynamodb',
                resource: 'table',
                resourceName: this.tableName,
            }));
            // We need to create/delete regions sequentially because we cannot
            // have multiple table updates at the same time. The `isCompleteHandler`
            // of the provider waits until the replica is in an ACTIVE state.
            if (previousRegion) {
                currentRegion.node.addDependency(previousRegion);
            }
            previousRegion = currentRegion;
        }
        // Permissions in the destination regions (outside of the loop to
        // minimize statements in the policy)
        onEventHandlerPolicy.grantPrincipal.addToPolicy(new iam.PolicyStatement({
            actions: ['dynamodb:*'],
            resources: this.regionalArns,
        }));
    }
    /**
     * Whether this table has indexes.
     *
     * @stability stable
     */
    get hasIndex() {
        return this.globalSecondaryIndexes.length + this.localSecondaryIndexes.length > 0;
    }
    /**
     * Set up key properties and return the Table encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _a;
        let encryptionType = props.encryption;
        if (encryptionType != null && props.serverSideEncryption != null) {
            throw new Error('Only one of encryption and serverSideEncryption can be specified, but both were provided');
        }
        if (props.serverSideEncryption && props.encryptionKey) {
            throw new Error('encryptionKey cannot be specified when serverSideEncryption is specified. Use encryption instead');
        }
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey != null
                // If there is a configured encyptionKey, the encryption is implicitly CUSTOMER_MANAGED
                ? TableEncryption.CUSTOMER_MANAGED
                // Otherwise, if severSideEncryption is enabled, it's AWS_MANAGED; else undefined (do not set anything)
                : props.serverSideEncryption ? TableEncryption.AWS_MANAGED : undefined;
        }
        if (encryptionType !== TableEncryption.CUSTOMER_MANAGED && props.encryptionKey) {
            throw new Error('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
        }
        if (encryptionType === TableEncryption.CUSTOMER_MANAGED && props.replicationRegions) {
            throw new Error('TableEncryption.CUSTOMER_MANAGED is not supported by DynamoDB Global Tables (where replicationRegions was set)');
        }
        switch (encryptionType) {
            case TableEncryption.CUSTOMER_MANAGED:
                const encryptionKey = (_a = props.encryptionKey) !== null && _a !== void 0 ? _a : new kms.Key(this, 'Key', {
                    description: `Customer-managed key auto-created for encrypting DynamoDB table at ${this.node.path}`,
                    enableKeyRotation: true,
                });
                return {
                    sseSpecification: { sseEnabled: true, kmsMasterKeyId: encryptionKey.keyArn, sseType: 'KMS' },
                    encryptionKey,
                };
            case TableEncryption.AWS_MANAGED:
                // Not specifying "sseType: 'KMS'" here because it would cause phony changes to existing stacks.
                return { sseSpecification: { sseEnabled: true } };
            case TableEncryption.DEFAULT:
                return { sseSpecification: { sseEnabled: false } };
            case undefined:
                // Not specifying "sseEnabled: false" here because it would cause phony changes to existing stacks.
                return { sseSpecification: undefined };
            default:
                throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
        }
    }
}
exports.Table = Table;
/**
 * Data types for attributes within a table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes
 * @stability stable
 */
var AttributeType;
(function (AttributeType) {
    AttributeType["BINARY"] = "B";
    AttributeType["NUMBER"] = "N";
    AttributeType["STRING"] = "S";
})(AttributeType = exports.AttributeType || (exports.AttributeType = {}));
/**
 * DyanmoDB's Read/Write capacity modes.
 *
 * @stability stable
 */
var BillingMode;
(function (BillingMode) {
    BillingMode["PAY_PER_REQUEST"] = "PAY_PER_REQUEST";
    BillingMode["PROVISIONED"] = "PROVISIONED";
})(BillingMode = exports.BillingMode || (exports.BillingMode = {}));
/**
 * The set of attributes that are projected into the index.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Projection.html
 * @stability stable
 */
var ProjectionType;
(function (ProjectionType) {
    ProjectionType["KEYS_ONLY"] = "KEYS_ONLY";
    ProjectionType["INCLUDE"] = "INCLUDE";
    ProjectionType["ALL"] = "ALL";
})(ProjectionType = exports.ProjectionType || (exports.ProjectionType = {}));
/**
 * When an item in the table is modified, StreamViewType determines what information is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 * @stability stable
 */
var StreamViewType;
(function (StreamViewType) {
    StreamViewType["NEW_IMAGE"] = "NEW_IMAGE";
    StreamViewType["OLD_IMAGE"] = "OLD_IMAGE";
    StreamViewType["NEW_AND_OLD_IMAGES"] = "NEW_AND_OLD_IMAGES";
    StreamViewType["KEYS_ONLY"] = "KEYS_ONLY";
})(StreamViewType = exports.StreamViewType || (exports.StreamViewType = {}));
/**
 * An inline policy that is logically bound to the source table of a DynamoDB Global Tables
 * "cluster". This is here to ensure permissions are removed as part of (and not before) the
 * CleanUp phase of a stack update, when a replica is removed (or the entire "cluster" gets
 * replaced).
 *
 * If statements are added directly to the handler roles (as opposed to in a separate inline
 * policy resource), new permissions are in effect before clean up happens, and so replicas that
 * need to be dropped can no longer be due to lack of permissions.
 */
class SourceTableAttachedPolicy extends core_1.Construct {
    constructor(sourceTable, role) {
        super(sourceTable, `SourceTableAttachedPolicy-${core_1.Names.nodeUniqueId(role.node)}`);
        const policy = new iam.Policy(this, 'Resource', { roles: [role] });
        this.policy = policy;
        this.grantPrincipal = new SourceTableAttachedPrincipal(role, policy);
    }
}
/**
 * An `IPrincipal` entity that can be used as the target of `grant` calls, used by the
 * `SourceTableAttachedPolicy` class so it can act as an `IGrantable`.
 */
class SourceTableAttachedPrincipal extends iam.PrincipalBase {
    constructor(role, policy) {
        super();
        this.role = role;
        this.policy = policy;
    }
    get policyFragment() {
        return this.role.policyFragment;
    }
    addToPrincipalPolicy(statement) {
        this.policy.addStatements(statement);
        return {
            policyDependable: this.policy,
            statementAdded: true,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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