"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.AcceleratorType = exports.AcceleratorClass = exports.Mode = exports.ContainerDefinition = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 * @stability stable
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     * @stability stable
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     * @stability stable
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.JsonPath.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
_a = JSII_RTTI_SYMBOL_1;
S3Location[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.S3Location", version: "1.108.1" };
/**
 * Creates `IDockerImage` instances.
 *
 * @stability stable
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`.
     * @stability stable
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     * @stability stable
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     * @stability stable
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     * @stability stable
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
_b = JSII_RTTI_SYMBOL_1;
DockerImage[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.DockerImage", version: "1.108.1" };
/**
 * S3 Data Type.
 *
 * @stability stable
 */
var S3DataType;
(function (S3DataType) {
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    S3DataType["S3_PREFIX"] = "S3Prefix";
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @stability stable
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @stability stable
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    RecordWrapperType["NONE"] = "None";
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 * Input mode that the algorithm supports.
 *
 * @stability stable
 */
var InputMode;
(function (InputMode) {
    InputMode["PIPE"] = "Pipe";
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @stability stable
 */
var CompressionType;
(function (CompressionType) {
    CompressionType["NONE"] = "None";
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Describes the container, as part of model definition.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 * @stability stable
 */
class ContainerDefinition {
    /**
     * @stability stable
     */
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the ContainerDefinition type configured on Sagemaker Task.
     *
     * @stability stable
     */
    bind(task) {
        var _f, _g, _h;
        return {
            parameters: {
                ContainerHostname: this.options.containerHostName,
                Image: (_f = this.options.image) === null || _f === void 0 ? void 0 : _f.bind(task).imageUri,
                Mode: this.options.mode,
                ModelDataUrl: (_g = this.options.modelS3Location) === null || _g === void 0 ? void 0 : _g.bind(task, { forReading: true }).uri,
                ModelPackageName: this.options.modelPackageName,
                Environment: (_h = this.options.environmentVariables) === null || _h === void 0 ? void 0 : _h.value,
            },
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
_c = JSII_RTTI_SYMBOL_1;
ContainerDefinition[_c] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.ContainerDefinition", version: "1.108.1" };
/**
 * Specifies how many models the container hosts.
 *
 * @stability stable
 */
var Mode;
(function (Mode) {
    Mode["SINGLE_MODEL"] = "SingleModel";
    Mode["MULTI_MODEL"] = "MultiModel";
})(Mode = exports.Mode || (exports.Mode = {}));
/**
 * The generation of Elastic Inference (EI) instance.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 * @stability stable
 */
class AcceleratorClass {
    /**
     * @param version - Elastic Inference accelerator generation
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom AcceleratorType.
     *
     * @param version - Elastic Inference accelerator generation.
     * @stability stable
     */
    static of(version) { return new AcceleratorClass(version); }
}
exports.AcceleratorClass = AcceleratorClass;
_d = JSII_RTTI_SYMBOL_1;
AcceleratorClass[_d] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.AcceleratorClass", version: "1.108.1" };
/**
 * Elastic Inference accelerator 1st generation.
 *
 * @stability stable
 */
AcceleratorClass.EIA1 = AcceleratorClass.of('eia1');
/**
 * Elastic Inference accelerator 2nd generation.
 *
 * @stability stable
 */
AcceleratorClass.EIA2 = AcceleratorClass.of('eia2');
/**
 * The size of the Elastic Inference (EI) instance to use for the production variant.
 *
 * EI instances provide on-demand GPU computing for inference
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 * @stability stable
 */
class AcceleratorType {
    /**
     * @stability stable
     */
    constructor(instanceTypeIdentifier) {
        this.instanceTypeIdentifier = instanceTypeIdentifier;
    }
    /**
     * AcceleratorType.
     *
     * This class takes a combination of a class and size.
     *
     * @stability stable
     */
    static of(acceleratorClass, instanceSize) {
        return new AcceleratorType(`ml.${acceleratorClass}.${instanceSize}`);
    }
    /**
     * Return the accelerator type as a dotted string.
     *
     * @stability stable
     */
    toString() {
        return this.instanceTypeIdentifier;
    }
}
exports.AcceleratorType = AcceleratorType;
_e = JSII_RTTI_SYMBOL_1;
AcceleratorType[_e] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.AcceleratorType", version: "1.108.1" };
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @stability stable
 */
var BatchStrategy;
(function (BatchStrategy) {
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @stability stable
 */
var SplitType;
(function (SplitType) {
    SplitType["NONE"] = "None";
    SplitType["LINE"] = "Line";
    SplitType["RECORD_IO"] = "RecordIO";
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @stability stable
 */
var AssembleWith;
(function (AssembleWith) {
    AssembleWith["NONE"] = "None";
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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