"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetImageCode = exports.EcrImageCode = exports.CfnParametersCode = exports.AssetCode = exports.InlineCode = exports.S3Code = exports.Code = void 0;
const ecr_assets = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
/**
 * Represents the Lambda Handler Code.
 */
class Code {
    /**
     * Lambda handler code as an S3 object.
     *
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromBucket`
     */
    static bucket(bucket, key, objectVersion) {
        return this.fromBucket(bucket, key, objectVersion);
    }
    /**
     * Inline code for Lambda handler.
     *
     * @param code The actual handler code (limited to 4KiB).
     * @returns `LambdaInlineCode` with inline code.
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromInline`
     */
    static inline(code) {
        return this.fromInline(code);
    }
    /**
     * Loads the function code from a local disk path.
     *
     * @param path Either a directory with the Lambda code bundle or a .zip file.
     */
    static fromAsset(path, options) {
        return new AssetCode(path, options);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromAsset`
     */
    static asset(path) {
        return this.fromAsset(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @param props optional construction properties of {@link CfnParametersCode}.
     * @returns a new instance of `CfnParametersCode`
     */
    static fromCfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromCfnParameters`
     */
    static cfnParameters(props) {
        return this.fromCfnParameters(props);
    }
    /**
     * Use an existing ECR image as the Lambda code.
     *
     * @param repository the ECR repository that the image is in.
     * @param props properties to further configure the selected image.
     */
    static fromEcrImage(repository, props) {
        return new EcrImageCode(repository, props);
    }
    /**
     * Create an ECR image from the specified asset and bind it as the Lambda code.
     *
     * @param directory the directory from which the asset must be created.
     * @param props properties to further configure the selected image.
     */
    static fromAssetImage(directory, props = {}) {
        return new AssetImageCode(directory, props);
    }
    /**
     * Called after the CFN function resource has been created to allow the code class to bind to it.
     *
     * Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindToResource(_resource, _options) {
        return;
    }
}
exports.Code = Code;
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    /**
     *
     */
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        /**
         * Determines whether this Code is inline code or not.
         */
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     */
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    /**
     *
     */
    constructor(code) {
        super();
        this.code = code;
        /**
         * Determines whether this Code is inline code or not.
         */
        this.isInline = true;
        if (code.length === 0) {
            throw new Error('Lambda inline code cannot be empty');
        }
        if (code.length > 4096) {
            throw new Error('Lambda source is too large, must be <= 4096 but is ' + code.length);
        }
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     */
    bind(_scope) {
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        /**
         * Determines whether this Code is inline code or not.
         */
        this.isInline = false;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     */
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.path,
                ...this.options,
            });
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Called after the CFN function resource has been created to allow the code class to bind to it.
     *
     * Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindToResource(resource, options = {}) {
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const resourceProperty = options.resourceProperty || 'Code';
        // https://github.com/aws/aws-cdk/issues/1432
        this.asset.addResourceMetadata(resource, resourceProperty);
    }
}
exports.AssetCode = AssetCode;
/**
 * Lambda code defined using 2 CloudFormation parameters.
 *
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
class CfnParametersCode extends Code {
    /**
     *
     */
    constructor(props = {}) {
        super();
        /**
         * Determines whether this Code is inline code or not.
         */
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     */
    bind(scope) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(scope, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(scope, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
        return {
            s3Location: {
                bucketName: this._bucketNameParam.valueAsString,
                objectKey: this._objectKeyParam.valueAsString,
            },
        };
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code.
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    /**
     *
     */
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    /**
     *
     */
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
/**
 * Represents a Docker image in ECR that can be bound as Lambda Code.
 */
class EcrImageCode extends Code {
    /**
     *
     */
    constructor(repository, props = {}) {
        super();
        this.repository = repository;
        this.props = props;
        /**
         * Determines whether this Code is inline code or not.
         */
        this.isInline = false;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     */
    bind(_) {
        var _a, _b;
        this.repository.grantPull(new iam.ServicePrincipal('lambda.amazonaws.com'));
        return {
            image: {
                imageUri: this.repository.repositoryUriForTag((_b = (_a = this.props) === null || _a === void 0 ? void 0 : _a.tag) !== null && _b !== void 0 ? _b : 'latest'),
                cmd: this.props.cmd,
                entrypoint: this.props.entrypoint,
            },
        };
    }
}
exports.EcrImageCode = EcrImageCode;
/**
 * Represents an ECR image that will be constructed from the specified asset and can be bound as Lambda code.
 */
class AssetImageCode extends Code {
    /**
     *
     */
    constructor(directory, props) {
        super();
        this.directory = directory;
        this.props = props;
        /**
         * Determines whether this Code is inline code or not.
         */
        this.isInline = false;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     */
    bind(scope) {
        const asset = new ecr_assets.DockerImageAsset(scope, 'AssetImage', {
            directory: this.directory,
            ...this.props,
        });
        asset.repository.grantPull(new iam.ServicePrincipal('lambda.amazonaws.com'));
        return {
            image: {
                imageUri: asset.imageUri,
                entrypoint: this.props.entrypoint,
                cmd: this.props.cmd,
            },
        };
    }
}
exports.AssetImageCode = AssetImageCode;
//# sourceMappingURL=data:application/json;base64,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