"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const yaml = require("js-yaml");
const path = require("path");
const resourceType = 'Custom::SSM-Document';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * An SSM document
 */
class Document extends cdk.Construct {
    /**
    * Defines a new SSM document
    */
    constructor(scope, id, props) {
        super(scope, id);
        /**
        * Name of the document
        */
        this.name = '';
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::SSM-Document');
        this.tags.setTag('CreatedBy', ID);
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        const name = this.fixDocumentName(props.name);
        if (name.length < 3 || name.length > 128) {
            this.node.addError(`SSM Document name ${name} is invalid. The name must be between 3 and 128 characters.`);
            return;
        }
        let content = props.content;
        if (typeof content === 'string') {
            content = yaml.safeLoad(content);
        }
        const document = new cfn.CustomResource(this, `SSM-Document-${name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                updateDefaultVersion: props.updateDefaultVersion || true,
                name: name,
                content: content,
                documentType: props.documentType || 'Command',
                targetType: props.targetType || '/',
                StackName: stack,
                tags: cdk.Lazy.anyValue({ produce: () => this.tags.renderTags() }),
            }
        });
        this.name = document.getAttString('Name');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'SSM-Document-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'SSM-Document-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'ssm:ListDocuments',
                        'ssm:ListTagsForResource',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'ssm:CreateDocument',
                        'ssm:AddTagsToResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:RequestTag/CreatedBy': ID,
                        }
                    },
                }),
                new iam.PolicyStatement({
                    actions: [
                        'ssm:DeleteDocument',
                        'ssm:DescribeDocument',
                        'ssm:GetDocument',
                        'ssm:ListDocumentVersions',
                        'ssm:ModifyDocumentPermission',
                        'ssm:UpdateDocument',
                        'ssm:UpdateDocumentDefaultVersion',
                        'ssm:AddTagsToResource',
                        'ssm:RemoveTagsFromResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:ResourceTag/CreatedBy': ID,
                        }
                    },
                }),
            ],
        });
        const role = new iam.Role(stack, 'SSM-Document-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ]
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage SSM Documents',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout)
        });
        return fn;
    }
    fixDocumentName(name) {
        return name.replace(/[^a-zA-Z0-9_.-]+/g, '-');
    }
}
exports.Document = Document;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLG1EQUFvRDtBQUNwRCx3Q0FBeUM7QUFDekMsOENBQStDO0FBQy9DLHFDQUFzQztBQUN0QyxnQ0FBaUM7QUFDakMsNkJBQThCO0FBNEk5QixNQUFNLFlBQVksR0FBRyxzQkFBc0IsQ0FBQztBQUM1QyxNQUFNLEVBQUUsR0FBRyxrQkFBa0IsWUFBWSxFQUFFLENBQUM7QUFDNUMsTUFBTSxPQUFPLEdBQUcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsR0FBRyxDQUFDLENBQUM7QUFDdkMsTUFBTSxhQUFhLEdBQUcsQ0FBQyxDQUFDLENBQUMsVUFBVTtBQUVuQzs7R0FFRztBQUNILE1BQWEsUUFBUyxTQUFRLEdBQUcsQ0FBQyxTQUFTO0lBWXZDOztNQUVFO0lBQ0YsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUFvQjtRQUM5RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBZHJCOztVQUVFO1FBQ2MsU0FBSSxHQUFXLEVBQUUsQ0FBQztRQWE5QixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1FBQ3hFLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVsQyxNQUFNLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDM0MsTUFBTSxFQUFFLEdBQUcsSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDO1FBQy9CLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTlDLElBQUksSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksSUFBSSxDQUFDLE1BQU0sR0FBRyxHQUFHLEVBQUU7WUFDdEMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMscUJBQXFCLElBQUksNkRBQTZELENBQUMsQ0FBQztZQUMzRyxPQUFPO1NBQ1Y7UUFFRCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBRTVCLElBQUksT0FBTyxPQUFPLEtBQUssUUFBUSxFQUFFO1lBQzdCLE9BQU8sR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQ3BDO1FBRUQsTUFBTSxRQUFRLEdBQUcsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxnQkFBZ0IsSUFBSSxFQUFFLEVBQUU7WUFDbEUsUUFBUSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDO1lBQ25ELFlBQVksRUFBRSxZQUFZO1lBQzFCLFVBQVUsRUFBRTtnQkFDUixvQkFBb0IsRUFBRSxLQUFLLENBQUMsb0JBQW9CLElBQUksSUFBSTtnQkFDeEQsSUFBSSxFQUFFLElBQUk7Z0JBQ1YsT0FBTyxFQUFFLE9BQU87Z0JBQ2hCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWSxJQUFJLFNBQVM7Z0JBQzdDLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVSxJQUFJLEdBQUc7Z0JBQ25DLFNBQVMsRUFBRSxLQUFLO2dCQUNoQixJQUFJLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDO2FBQ3JFO1NBQ0osQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksR0FBRyxRQUFRLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFTyxZQUFZO1FBQ2hCLE1BQU0sS0FBSyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pDLE1BQU0sYUFBYSxHQUFHLDZCQUE2QixDQUFDO1FBQ3BELE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3hELElBQUksUUFBUSxFQUFFO1lBQ1YsT0FBTyxRQUEyQixDQUFDO1NBQ3RDO1FBRUQsTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSw2QkFBNkIsRUFBRTtZQUN2RSxpQkFBaUIsRUFBRSxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksT0FBTyxFQUFFO1lBQ2xELFdBQVcsRUFBRSxrQkFBa0IsT0FBTywwREFBMEQ7WUFDaEcsVUFBVSxFQUFFO2dCQUNSLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDcEIsT0FBTyxFQUFFO3dCQUNMLG1CQUFtQjt3QkFDbkIseUJBQXlCO3FCQUM1QjtvQkFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7aUJBQ25CLENBQUM7Z0JBQ0YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUNwQixPQUFPLEVBQUU7d0JBQ0wsb0JBQW9CO3dCQUNwQix1QkFBdUI7cUJBQzFCO29CQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztvQkFDaEIsVUFBVSxFQUFFO3dCQUNSLFlBQVksRUFBRTs0QkFDViwwQkFBMEIsRUFBRSxFQUFFO3lCQUNqQztxQkFDSjtpQkFDSixDQUFDO2dCQUNGLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDcEIsT0FBTyxFQUFFO3dCQUNMLG9CQUFvQjt3QkFDcEIsc0JBQXNCO3dCQUN0QixpQkFBaUI7d0JBQ2pCLDBCQUEwQjt3QkFDMUIsOEJBQThCO3dCQUM5QixvQkFBb0I7d0JBQ3BCLGtDQUFrQzt3QkFDbEMsdUJBQXVCO3dCQUN2Qiw0QkFBNEI7cUJBQy9CO29CQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztvQkFDaEIsVUFBVSxFQUFFO3dCQUNSLFlBQVksRUFBRTs0QkFDViwyQkFBMkIsRUFBRSxFQUFFO3lCQUNsQztxQkFDSjtpQkFDSixDQUFDO2FBQ0w7U0FDSixDQUFDLENBQUM7UUFFSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLDJCQUEyQixFQUFFO1lBQzFELFFBQVEsRUFBRSxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksT0FBTyxFQUFFO1lBQ3pDLFdBQVcsRUFBRSxrQkFBa0IsT0FBTywwREFBMEQ7WUFDaEcsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQzNELGVBQWUsRUFBRTtnQkFDYixNQUFNO2dCQUNOLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsMENBQTBDLENBQUM7YUFDekY7U0FDSixDQUFDLENBQUM7UUFFSCxNQUFNLEVBQUUsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtZQUNqRCxZQUFZLEVBQUUsR0FBRyxLQUFLLENBQUMsU0FBUyxJQUFJLE9BQU8sRUFBRTtZQUM3QyxJQUFJLEVBQUUsSUFBSTtZQUNWLFdBQVcsRUFBRSwyQ0FBMkM7WUFDeEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUNuQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztZQUN2RSxPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDO1NBQy9DLENBQUMsQ0FBQztRQUVILE9BQU8sRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVPLGVBQWUsQ0FBQyxJQUFZO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLENBQUMsQ0FBQztJQUNsRCxDQUFDO0NBQ0o7QUFwSUQsNEJBb0lDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IGNmbiA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1jbG91ZGZvcm1hdGlvbicpO1xuaW1wb3J0IGlhbSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1pYW0nKTtcbmltcG9ydCBsYW1iZGEgPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtbGFtYmRhJyk7XG5pbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY29yZScpO1xuaW1wb3J0IHlhbWwgPSByZXF1aXJlKCdqcy15YW1sJyk7XG5pbXBvcnQgcGF0aCA9IHJlcXVpcmUoJ3BhdGgnKTtcblxuLyoqXG4gKiBBbiBTU00gZG9jdW1lbnQgcGFyYW1ldGVyXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRG9jdW1lbnRQYXJhbWV0ZXIge1xuXG4gICAgLyoqXG4gICAgKiAgQWxsb3dlZCB2YWx1ZXMgaW5jbHVkZSB0aGUgZm9sbG93aW5nOiBTdHJpbmcsIFN0cmluZ0xpc3QsIEJvb2xlYW4sIEludGVnZXIsIE1hcExpc3QsIGFuZCBTdHJpbmdNYXAuIFRvIHZpZXcgZXhhbXBsZXMgb2YgZWFjaCB0eXBlLCBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N5c3RlbXMtbWFuYWdlci9sYXRlc3QvdXNlcmd1aWRlL3NzbS1wbHVnaW5zLmh0bWwjdG9wLWxldmVsLXByb3BlcnRpZXMtdHlwZVxuICAgICovXG4gICAgcmVhZG9ubHkgdHlwZTogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgKiBBIGRlc2NyaXB0aW9uIG9mIHRoZSBwYXJhbWV0ZXJcbiAgICAqL1xuICAgIHJlYWRvbmx5IGRlc2NyaXB0aW9uOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAqIFRoZSBkZWZhdWx0IHZhbHVlIG9mIHRoZSBwYXJhbWV0ZXIgb3IgYSByZWZlcmVuY2UgdG8gYSBwYXJhbWV0ZXIgaW4gUGFyYW1ldGVyIFN0b3JlXG4gICAgKi9cbiAgICByZWFkb25seSBkZWZhdWx0PzogYW55O1xuXG4gICAgLyoqXG4gICAgKiBBbGxvd2VkIHZhbHVlcyBmb3IgdGhlIHBhcmFtZXRlclxuICAgICovXG4gICAgcmVhZG9ubHkgYWxsb3dlZFZhbHVlcz86IHN0cmluZ1tdO1xuXG4gICAgLyoqXG4gICAgKiBUaGUgcmVndWxhciBleHByZXNzaW9uIHRoZSBwYXJhbWV0ZXIgbXVzdCBtYXRjaFxuICAgICovXG4gICAgcmVhZG9ubHkgYWxsb3dlZFBhdHRlcm4/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAqICBVc2VkIHRvIGRpc3BsYXkgZWl0aGVyIGEgdGV4dGZpZWxkIG9yIGEgdGV4dGFyZWEgaW4gdGhlIEFXUyBjb25zb2xlLiB0ZXh0ZmllbGQgaXMgYSBzaW5nbGUtbGluZSB0ZXh0IGJveC4gdGV4dGFyZWEgaXMgYSBtdWx0aS1saW5lIHRleHQgYXJlYVxuICAgICovXG4gICAgcmVhZG9ubHkgZGlzcGxheVR5cGU/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAqIFRoZSBtaW5pbXVtIG51bWJlciBvZiBpdGVtcyBhbGxvd2VkXG4gICAgKi9cbiAgICByZWFkb25seSBtaW5JdGVtcz86IG51bWJlcjtcblxuICAgIC8qKlxuICAgICogVGhlIG1heGltdW0gbnVtYmVyIG9mIGl0ZW1zIGFsbG93ZWRcbiAgICAqL1xuICAgIHJlYWRvbmx5IG1heEl0ZW1zPzogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgKiBUaGUgbWluaW11bSBudW1iZXIgb2YgcGFyYW1ldGVyIGNoYXJhY3RlcnMgYWxsb3dlZFxuICAgICovXG4gICAgcmVhZG9ubHkgbWluQ2hhcnM/OiBudW1iZXI7XG5cbiAgICAvKipcbiAgICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiBwYXJhbWV0ZXIgY2hhcmFjdGVycyBhbGxvd2VkXG4gICAgKi9cbiAgICByZWFkb25seSBtYXhDaGFycz86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBTdGVwcyBpbmNsdWRlIG9uZSBvciBtb3JlIGFjdGlvbnMsIGFuIG9wdGlvbmFsIHByZWNvbmRpdGlvbiwgYSB1bmlxdWUgbmFtZSBvZiB0aGUgYWN0aW9uLCBhbmQgaW5wdXRzIChwYXJhbWV0ZXJzKSBmb3IgdGhvc2UgYWN0aW9ucy5cbiAqXG4gKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiBhYm91dCBkb2N1bWVudHMsIGluY2x1ZGluZyBpbmZvcm1hdGlvbiBhYm91dCBjcmVhdGluZyBkb2N1bWVudHMgYW5kIHRoZSBkaWZmZXJlbmNlcyBiZXR3ZWVuIHNjaGVtYSB2ZXJzaW9ucywgc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zeXN0ZW1zLW1hbmFnZXIvbGF0ZXN0L3VzZXJndWlkZS9zc20tcGx1Z2lucy5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRG9jdW1lbnRNYWluU3RlcCB7XG4gICAgW2tleTogc3RyaW5nXTogYW55O1xufVxuXG4vKipcbiAqIFRoZSBjb250ZW50IG9mIHRoZSBTU00gZG9jdW1lbnQuIFRoZSBzeW50YXggb2YgeW91ciBkb2N1bWVudCBpcyBkZWZpbmVkIGJ5IHRoZSBzY2hlbWEgdmVyc2lvbiB1c2VkIHRvIGNyZWF0ZSBpdC5cbiAqXG4gKiBUaGlzIG1vZHVsZSBvbmx5IHN1cHBvcnRzIHNjaGVtYSB2ZXJzaW9uIDIuMlxuICpcbiAqIEZvciBkZXRhaWxzIHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3lzdGVtcy1tYW5hZ2VyL2xhdGVzdC91c2VyZ3VpZGUvc3lzbWFuLWRvYy1zeW50YXguaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIERvY3VtZW50Q29udGVudCB7XG5cbiAgICAvKipcbiAgICAqIFRoZSBzY2hlbWEgdmVyc2lvbiB0byB1c2UuIEN1cnJlbnRseSBvbmx5IHZlcnNpb24gMi4yIGlzIHN1cHBvcnRlZFxuICAgICovXG4gICAgcmVhZG9ubHkgc2NoZW1hVmVyc2lvbjogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgKiBJbmZvcm1hdGlvbiB5b3UgcHJvdmlkZSB0byBkZXNjcmliZSB0aGUgcHVycG9zZSBvZiB0aGUgZG9jdW1lbnRcbiAgICAqL1xuICAgIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgKiBBbiBvYmplY3QgdGhhdCBjYW4gaW5jbHVkZSBtdWx0aXBsZSBzdGVwcyAocGx1Z2lucykuIFN0ZXBzIGluY2x1ZGUgb25lIG9yIG1vcmUgYWN0aW9ucywgYW4gb3B0aW9uYWwgcHJlY29uZGl0aW9uLCBhIHVuaXF1ZSBuYW1lIG9mIHRoZSBhY3Rpb24sIGFuZCBpbnB1dHMgKHBhcmFtZXRlcnMpIGZvciB0aG9zZSBhY3Rpb25zLlxuICAgICpcbiAgICAqIEZvciBtb3JlIGluZm9ybWF0aW9uIGFib3V0IGRvY3VtZW50cywgaW5jbHVkaW5nIGluZm9ybWF0aW9uIGFib3V0IGNyZWF0aW5nIGRvY3VtZW50cyBhbmQgdGhlIGRpZmZlcmVuY2VzIGJldHdlZW4gc2NoZW1hIHZlcnNpb25zLCBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N5c3RlbXMtbWFuYWdlci9sYXRlc3QvdXNlcmd1aWRlL3NzbS1wbHVnaW5zLmh0bWxcbiAgICAqL1xuICAgIHJlYWRvbmx5IG1haW5TdGVwczogRG9jdW1lbnRNYWluU3RlcFtdO1xuXG4gICAgLyoqXG4gICAgKiBUaGUgcGFyYW1ldGVycyB0aGUgZG9jdW1lbnQgYWNjZXB0c1xuICAgICovXG4gICAgcmVhZG9ubHkgcGFyYW1ldGVycz86IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogRG9jdW1lbnRQYXJhbWV0ZXI7XG4gICAgfTtcbn1cblxuLyoqXG4gKiBEZWZpbml0aW9uIG9mIHRoZSBTU00gZG9jdW1lbnRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEb2N1bWVudFByb3BzIGV4dGVuZHMgY2RrLlN0YWNrUHJvcHMge1xuXG4gICAgLyoqXG4gICAgKiBEZWZpbmVzIGlmIHRoZSBkZWZhdWx0IHZlcnNpb24gc2hvdWxkIGJlIHVwZGF0ZWQgdG8gdGhlIGxhdGVzdCB2ZXJzaW9uIG9uIGRvY3VtZW50IHVwZGF0ZXNcbiAgICAqXG4gICAgKiBAZGVmYXVsdCB0cnVlXG4gICAgKi9cbiAgICByZWFkb25seSB1cGRhdGVEZWZhdWx0VmVyc2lvbj86IGJvb2xlYW47XG5cbiAgICAvKipcbiAgICAqIE5hbWUgb2YgdGhlIGRvY3VtZW50XG4gICAgKlxuICAgICogVGhlIG5hbWUgbXVzdCBiZSBiZXR3ZWVuIDMgYW5kIDEyOCBjaGFyYWN0ZXJzLiBWYWxpZCBjaGFyYWN0ZXJzIGFyZSBhLXosIEEtWiwgMC05LCBhbmQgXywgLSwgYW5kIC4gb25seVxuICAgICovXG4gICAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgKiBEb2N1bWVudCB0eXBlIGJhc2VkIG9uIHRoZSBzZXJ2aWNlIHRoYXQgeW91IHdhbnQgdG8gdXNlXG4gICAgKlxuICAgICogQGRlZmF1bHQgQ29tbWFuZFxuICAgICovXG4gICAgcmVhZG9ubHkgZG9jdW1lbnRUeXBlPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgKiBUeXBlcyBvZiByZXNvdXJjZXMgdGhlIGRvY3VtZW50IGNhbiBydW4gb24uIEZvciBleGFtcGxlLCBgL0FXUzo6RUMyOjpJbnN0YW5jZWAgb3IgYC9gIGZvciBhbGwgcmVzb3VyY2UgdHlwZXNcbiAgICAqXG4gICAgKiBAZGVmYXVsdCAvXG4gICAgKi9cbiAgICByZWFkb25seSB0YXJnZXRUeXBlPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgKiBDb250ZW50IG9mIHRoZSBTU00gZG9jdW1lbnQuIENhbiBiZSBwYXNzZWQgYXMgc3RyaW5nIG9yIGFzIG9iamVjdFxuICAgICovXG4gICAgcmVhZG9ubHkgY29udGVudDogc3RyaW5nIHwgRG9jdW1lbnRDb250ZW50O1xufVxuXG5jb25zdCByZXNvdXJjZVR5cGUgPSAnQ3VzdG9tOjpTU00tRG9jdW1lbnQnO1xuY29uc3QgSUQgPSBgQ0ZOOjpSZXNvdXJjZTo6JHtyZXNvdXJjZVR5cGV9YDtcbmNvbnN0IGNsZWFuSUQgPSBJRC5yZXBsYWNlKC86Ky9nLCAnLScpO1xuY29uc3QgbGFtYmRhVGltZW91dCA9IDM7IC8vIG1pbnV0ZXNcblxuLyoqXG4gKiBBbiBTU00gZG9jdW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIERvY3VtZW50IGV4dGVuZHMgY2RrLkNvbnN0cnVjdCBpbXBsZW1lbnRzIGNkay5JVGFnZ2FibGUge1xuXG4gICAgLyoqXG4gICAgKiBOYW1lIG9mIHRoZSBkb2N1bWVudFxuICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZyA9ICcnO1xuXG4gICAgLyoqXG4gICAgKiBSZXNvdXJjZSB0YWdzXG4gICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgdGFnczogY2RrLlRhZ01hbmFnZXI7XG5cbiAgICAvKipcbiAgICAqIERlZmluZXMgYSBuZXcgU1NNIGRvY3VtZW50XG4gICAgKi9cbiAgICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IERvY3VtZW50UHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgICAgICB0aGlzLnRhZ3MgPSBuZXcgY2RrLlRhZ01hbmFnZXIoY2RrLlRhZ1R5cGUuTUFQLCAnQ3VzdG9tOjpTU00tRG9jdW1lbnQnKTtcbiAgICAgICAgdGhpcy50YWdzLnNldFRhZygnQ3JlYXRlZEJ5JywgSUQpO1xuXG4gICAgICAgIGNvbnN0IHN0YWNrID0gY2RrLlN0YWNrLm9mKHRoaXMpLnN0YWNrTmFtZTtcbiAgICAgICAgY29uc3QgZm4gPSB0aGlzLmVuc3VyZUxhbWJkYSgpO1xuICAgICAgICBjb25zdCBuYW1lID0gdGhpcy5maXhEb2N1bWVudE5hbWUocHJvcHMubmFtZSk7XG5cbiAgICAgICAgaWYgKG5hbWUubGVuZ3RoIDwgMyB8fCBuYW1lLmxlbmd0aCA+IDEyOCkge1xuICAgICAgICAgICAgdGhpcy5ub2RlLmFkZEVycm9yKGBTU00gRG9jdW1lbnQgbmFtZSAke25hbWV9IGlzIGludmFsaWQuIFRoZSBuYW1lIG11c3QgYmUgYmV0d2VlbiAzIGFuZCAxMjggY2hhcmFjdGVycy5gKTtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuXG4gICAgICAgIGxldCBjb250ZW50ID0gcHJvcHMuY29udGVudDtcblxuICAgICAgICBpZiAodHlwZW9mIGNvbnRlbnQgPT09ICdzdHJpbmcnKSB7XG4gICAgICAgICAgICBjb250ZW50ID0geWFtbC5zYWZlTG9hZChjb250ZW50KTtcbiAgICAgICAgfVxuXG4gICAgICAgIGNvbnN0IGRvY3VtZW50ID0gbmV3IGNmbi5DdXN0b21SZXNvdXJjZSh0aGlzLCBgU1NNLURvY3VtZW50LSR7bmFtZX1gLCB7XG4gICAgICAgICAgICBwcm92aWRlcjogY2ZuLkN1c3RvbVJlc291cmNlUHJvdmlkZXIuZnJvbUxhbWJkYShmbiksXG4gICAgICAgICAgICByZXNvdXJjZVR5cGU6IHJlc291cmNlVHlwZSxcbiAgICAgICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgICAgICB1cGRhdGVEZWZhdWx0VmVyc2lvbjogcHJvcHMudXBkYXRlRGVmYXVsdFZlcnNpb24gfHwgdHJ1ZSxcbiAgICAgICAgICAgICAgICBuYW1lOiBuYW1lLFxuICAgICAgICAgICAgICAgIGNvbnRlbnQ6IGNvbnRlbnQsXG4gICAgICAgICAgICAgICAgZG9jdW1lbnRUeXBlOiBwcm9wcy5kb2N1bWVudFR5cGUgfHwgJ0NvbW1hbmQnLFxuICAgICAgICAgICAgICAgIHRhcmdldFR5cGU6IHByb3BzLnRhcmdldFR5cGUgfHwgJy8nLFxuICAgICAgICAgICAgICAgIFN0YWNrTmFtZTogc3RhY2ssXG4gICAgICAgICAgICAgICAgdGFnczogY2RrLkxhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnRhZ3MucmVuZGVyVGFncygpIH0pLFxuICAgICAgICAgICAgfVxuICAgICAgICB9KTtcblxuICAgICAgICB0aGlzLm5hbWUgPSBkb2N1bWVudC5nZXRBdHRTdHJpbmcoJ05hbWUnKTtcbiAgICB9XG5cbiAgICBwcml2YXRlIGVuc3VyZUxhbWJkYSgpOiBsYW1iZGEuRnVuY3Rpb24ge1xuICAgICAgICBjb25zdCBzdGFjayA9IGNkay5TdGFjay5vZih0aGlzKTtcbiAgICAgICAgY29uc3QgY29uc3RydWN0TmFtZSA9ICdTU00tRG9jdW1lbnQtTWFuYWdlci1MYW1iZGEnO1xuICAgICAgICBjb25zdCBleGlzdGluZyA9IHN0YWNrLm5vZGUudHJ5RmluZENoaWxkKGNvbnN0cnVjdE5hbWUpO1xuICAgICAgICBpZiAoZXhpc3RpbmcpIHtcbiAgICAgICAgICAgIHJldHVybiBleGlzdGluZyBhcyBsYW1iZGEuRnVuY3Rpb247XG4gICAgICAgIH1cblxuICAgICAgICBjb25zdCBwb2xpY3kgPSBuZXcgaWFtLk1hbmFnZWRQb2xpY3koc3RhY2ssICdTU00tRG9jdW1lbnQtTWFuYWdlci1Qb2xpY3knLCB7XG4gICAgICAgICAgICBtYW5hZ2VkUG9saWN5TmFtZTogYCR7c3RhY2suc3RhY2tOYW1lfS0ke2NsZWFuSUR9YCxcbiAgICAgICAgICAgIGRlc2NyaXB0aW9uOiBgVXNlZCBieSBMYW1iZGEgJHtjbGVhbklEfSwgd2hpY2ggaXMgYSBjdXN0b20gQ0ZOIHJlc291cmNlLCBtYW5hZ2luZyBTU00gZG9jdW1lbnRzYCxcbiAgICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgICdzc206TGlzdERvY3VtZW50cycsXG4gICAgICAgICAgICAgICAgICAgICAgICAnc3NtOkxpc3RUYWdzRm9yUmVzb3VyY2UnLFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgICAgICAgICAgJ3NzbTpDcmVhdGVEb2N1bWVudCcsXG4gICAgICAgICAgICAgICAgICAgICAgICAnc3NtOkFkZFRhZ3NUb1Jlc291cmNlJyxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgICAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgICAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgJ2F3czpSZXF1ZXN0VGFnL0NyZWF0ZWRCeSc6IElELFxuICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgICAgICAgICAgJ3NzbTpEZWxldGVEb2N1bWVudCcsXG4gICAgICAgICAgICAgICAgICAgICAgICAnc3NtOkRlc2NyaWJlRG9jdW1lbnQnLFxuICAgICAgICAgICAgICAgICAgICAgICAgJ3NzbTpHZXREb2N1bWVudCcsXG4gICAgICAgICAgICAgICAgICAgICAgICAnc3NtOkxpc3REb2N1bWVudFZlcnNpb25zJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICdzc206TW9kaWZ5RG9jdW1lbnRQZXJtaXNzaW9uJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICdzc206VXBkYXRlRG9jdW1lbnQnLFxuICAgICAgICAgICAgICAgICAgICAgICAgJ3NzbTpVcGRhdGVEb2N1bWVudERlZmF1bHRWZXJzaW9uJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICdzc206QWRkVGFnc1RvUmVzb3VyY2UnLFxuICAgICAgICAgICAgICAgICAgICAgICAgJ3NzbTpSZW1vdmVUYWdzRnJvbVJlc291cmNlJyxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgICAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgICAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgJ2F3czpSZXNvdXJjZVRhZy9DcmVhdGVkQnknOiBJRCxcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTU00tRG9jdW1lbnQtTWFuYWdlci1Sb2xlJywge1xuICAgICAgICAgICAgcm9sZU5hbWU6IGAke3N0YWNrLnN0YWNrTmFtZX0tJHtjbGVhbklEfWAsXG4gICAgICAgICAgICBkZXNjcmlwdGlvbjogYFVzZWQgYnkgTGFtYmRhICR7Y2xlYW5JRH0sIHdoaWNoIGlzIGEgY3VzdG9tIENGTiByZXNvdXJjZSwgbWFuYWdpbmcgU1NNIGRvY3VtZW50c2AsXG4gICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICAgICAgICAgIHBvbGljeSxcbiAgICAgICAgICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUnKSxcbiAgICAgICAgICAgIF1cbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgZm4gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCBjb25zdHJ1Y3ROYW1lLCB7XG4gICAgICAgICAgICBmdW5jdGlvbk5hbWU6IGAke3N0YWNrLnN0YWNrTmFtZX0tJHtjbGVhbklEfWAsXG4gICAgICAgICAgICByb2xlOiByb2xlLFxuICAgICAgICAgICAgZGVzY3JpcHRpb246ICdDdXN0b20gQ0ZOIHJlc291cmNlOiBNYW5hZ2UgU1NNIERvY3VtZW50cycsXG4gICAgICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICAgICAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgICAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGFtYmRhL2NvZGUuemlwJykpLFxuICAgICAgICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLm1pbnV0ZXMobGFtYmRhVGltZW91dClcbiAgICAgICAgfSk7XG5cbiAgICAgICAgcmV0dXJuIGZuO1xuICAgIH1cblxuICAgIHByaXZhdGUgZml4RG9jdW1lbnROYW1lKG5hbWU6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgICAgIHJldHVybiBuYW1lLnJlcGxhY2UoL1teYS16QS1aMC05Xy4tXSsvZywgJy0nKTtcbiAgICB9XG59XG4iXX0=