import cdk = require('@aws-cdk/core');
/**
 * An SSM document parameter
 */
export interface DocumentParameter {
    /**
    *  Allowed values include the following: String, StringList, Boolean, Integer, MapList, and StringMap. To view examples of each type, see https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#top-level-properties-type
    */
    readonly type: string;
    /**
    * A description of the parameter
    */
    readonly description: string;
    /**
    * The default value of the parameter or a reference to a parameter in Parameter Store
    */
    readonly default?: any;
    /**
    * Allowed values for the parameter
    */
    readonly allowedValues?: string[];
    /**
    * The regular expression the parameter must match
    */
    readonly allowedPattern?: string;
    /**
    *  Used to display either a textfield or a textarea in the AWS console. textfield is a single-line text box. textarea is a multi-line text area
    */
    readonly displayType?: string;
    /**
    * The minimum number of items allowed
    */
    readonly minItems?: number;
    /**
    * The maximum number of items allowed
    */
    readonly maxItems?: number;
    /**
    * The minimum number of parameter characters allowed
    */
    readonly minChars?: number;
    /**
    * The maximum number of parameter characters allowed
    */
    readonly maxChars?: number;
}
/**
 * Steps include one or more actions, an optional precondition, a unique name of the action, and inputs (parameters) for those actions.
 *
 * For more information about documents, including information about creating documents and the differences between schema versions, see https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html
 */
export interface DocumentMainStep {
    [key: string]: any;
}
/**
 * The content of the SSM document. The syntax of your document is defined by the schema version used to create it.
 *
 * This module only supports schema version 2.2
 *
 * For details see https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-doc-syntax.html
 */
export interface DocumentContent {
    /**
    * The schema version to use. Currently only version 2.2 is supported
    */
    readonly schemaVersion: string;
    /**
    * Information you provide to describe the purpose of the document
    */
    readonly description?: string;
    /**
    * An object that can include multiple steps (plugins). Steps include one or more actions, an optional precondition, a unique name of the action, and inputs (parameters) for those actions.
    *
    * For more information about documents, including information about creating documents and the differences between schema versions, see https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html
    */
    readonly mainSteps: DocumentMainStep[];
    /**
    * The parameters the document accepts
    */
    readonly parameters?: {
        [key: string]: DocumentParameter;
    };
}
/**
 * Definition of the SSM document
 */
export interface DocumentProps extends cdk.StackProps {
    /**
    * Defines if the default version should be updated to the latest version on document updates
    *
    * @default true
    */
    readonly updateDefaultVersion?: boolean;
    /**
    * Name of the document
    *
    * The name must be between 3 and 128 characters. Valid characters are a-z, A-Z, 0-9, and _, -, and . only
    */
    readonly name: string;
    /**
    * Document type based on the service that you want to use
    *
    * @default Command
    */
    readonly documentType?: string;
    /**
    * Types of resources the document can run on. For example, `/AWS::EC2::Instance` or `/` for all resource types
    *
    * @default /
    */
    readonly targetType?: string;
    /**
    * Content of the SSM document. Can be passed as string or as object
    */
    readonly content: string | DocumentContent;
}
/**
 * An SSM document
 */
export declare class Document extends cdk.Construct implements cdk.ITaggable {
    /**
    * Name of the document
    */
    readonly name: string;
    /**
    * Resource tags
    */
    readonly tags: cdk.TagManager;
    /**
    * Defines a new SSM document
    */
    constructor(scope: cdk.Construct, id: string, props: DocumentProps);
    private ensureLambda;
    private fixDocumentName;
}
