import os
import pathlib
from typing import List, Tuple, Dict
from collections import defaultdict
from .token import Token

import attr

if os.name == "nt":
    temp = pathlib.PosixPath
    pathlib.PosixPath = pathlib.WindowsPath

LANGUAGE_CODES = defaultdict(lambda: 'other')
LANGUAGE_CODES["ENG"] = "en"

@attr.s
class Borrowing(object):
    """The Borrowing object: a span of text that represents a borrowing.
    A borrowing will be made of several tokens and have a context assigned

    Attributes:
            tokens (List[`pylazaro.token.Token`]): list of Tokens that form the Borrowing
            _language (str): language (the @property language follows iso codes)
            start_pos (int): start position of the borrowing spans (int position refers to
            tokenized sentence)
            end_pos (int): end position of the borrowing spans (int position refers to
            tokenized sentence)
            context_tokens (List[`pylazaro.token.Token`]): list of Tokens that form the sentence
    """
    tokens = attr.ib(type=List[Token])
    _language = attr.ib(type=str)
    start_pos = attr.ib(type=int)
    end_pos = attr.ib(type=int)
    context_tokens = attr.ib(type=List[Token], default=None)

    @property
    def language(self) -> str:
        """

        Returns: language of the borrowing as 2 letter iso code (or other)

        """
        return LANGUAGE_CODES[self._language.upper()]

    @property
    def length(self) -> int:
        """

        Returns: token length of the borrowings (1 token borrowing, 2 token borrowings, etc)

        """
        return len(self.tokens)

    @property
    def text(self) -> str:
        """

        Returns: The borrowing as a string of text

        """

        return " ".join([token.value for token in self.tokens])

    @property
    def context_text(self) -> str:
        """
        Returns:
                The context of the borrowing as a string of text
        """

        return " ".join([token.value for token in self.context_tokens])

    def is_anglicism(self) -> bool:
        """

        Returns: Whether the borrowing is an anglicism

        """
        return self.language == "en"

    def is_other(self) -> bool:
        """

        Returns: Whether the borrowing is of type other (not an anglicism)

        """
        return self.language == "other"




    def to_tuple(self) -> Tuple[str, str]:
        """

        Returns:
            The borrowing formatted as a tuple of form (borrowing, language)

        """

        return (self.text, self.language)


    def to_dict(self) -> Dict:
        """

        Returns:
                The borrowing formatted as a dict

        """
        """
        return {"borrowing": self.text, "language": self.language, "start_pos": self.start_pos,
                "end_pos": self.end_pos}
        """
        attr.asdict(self)