import * as cdk from 'aws-cdk-lib';
import { Construct, IConstruct } from 'constructs';
import { ICluster } from './cluster';
import { DatabaseOptions } from './database-options';
import { IUser } from './user';
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
export declare enum TableAction {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    SELECT = 0,
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    INSERT = 1,
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    UPDATE = 2,
    /**
     * Grants privilege to delete a data row from a table.
     */
    DELETE = 3,
    /**
     * Grants privilege to drop a table.
     */
    DROP = 4,
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    REFERENCES = 5,
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    ALL = 6
}
/**
 * A column in a Redshift table.
 */
export interface Column {
    /**
     * The unique identifier of the column.
     *
     * This is not the name of the column, and renaming this identifier will cause a new column to be created and the old column to be dropped.
     *
     * **NOTE** - This field will be set, however, only by setting the `@aws-cdk/aws-redshift:columnId` feature flag will this field be used.
     *
     * @default - the column name is used as the identifier
     */
    readonly id?: string;
    /**
     * The name of the column. This will appear on Amazon Redshift.
     */
    readonly name: string;
    /**
     * The data type of the column.
     */
    readonly dataType: string;
    /**
     * Boolean value that indicates whether the column is to be configured as DISTKEY.
     *
     * @default - column is not DISTKEY
     */
    readonly distKey?: boolean;
    /**
     * Boolean value that indicates whether the column is to be configured as SORTKEY.
     *
     * @default - column is not a SORTKEY
     */
    readonly sortKey?: boolean;
    /**
     * The encoding to use for the column.
     *
     * @default - Amazon Redshift determines the encoding based on the data type.
     */
    readonly encoding?: ColumnEncoding;
    /**
     * A comment to attach to the column.
     *
     * @default - no comment
     */
    readonly comment?: string;
}
/**
 * Properties for configuring a Redshift table.
 */
export interface TableProps extends DatabaseOptions {
    /**
     * The name of the table.
     *
     * @default - a name is generated
     */
    readonly tableName?: string;
    /**
     * The columns of the table.
     */
    readonly tableColumns: Column[];
    /**
     * The distribution style of the table.
     *
     * @default TableDistStyle.AUTO
     */
    readonly distStyle?: TableDistStyle;
    /**
     * The sort style of the table.
     *
     * @default TableSortStyle.AUTO if no sort key is specified, TableSortStyle.COMPOUND if a sort key is specified
     */
    readonly sortStyle?: TableSortStyle;
    /**
     * The policy to apply when this resource is removed from the application.
     *
     * @default cdk.RemovalPolicy.Retain
     */
    readonly removalPolicy?: cdk.RemovalPolicy;
    /**
       * A comment to attach to the table.
       *
       * @default - no comment
       */
    readonly tableComment?: string;
}
/**
 * Represents a table in a Redshift database.
 */
export interface ITable extends IConstruct {
    /**
     * Name of the table.
     */
    readonly tableName: string;
    /**
     * The columns of the table.
     */
    readonly tableColumns: Column[];
    /**
     * The cluster where the table is located.
     */
    readonly cluster: ICluster;
    /**
     * The name of the database where the table is located.
     */
    readonly databaseName: string;
    /**
     * Grant a user privilege to access this table.
     */
    grant(user: IUser, ...actions: TableAction[]): void;
}
/**
 * A full specification of a Redshift table that can be used to import it fluently into the CDK application.
 */
export interface TableAttributes {
    /**
     * Name of the table.
     */
    readonly tableName: string;
    /**
     * The columns of the table.
     */
    readonly tableColumns: Column[];
    /**
     * The cluster where the table is located.
     */
    readonly cluster: ICluster;
    /**
     * The name of the database where the table is located.
     */
    readonly databaseName: string;
}
declare abstract class TableBase extends Construct implements ITable {
    abstract readonly tableName: string;
    abstract readonly tableColumns: Column[];
    abstract readonly cluster: ICluster;
    abstract readonly databaseName: string;
    grant(user: IUser, ...actions: TableAction[]): void;
}
/**
 * A table in a Redshift cluster.
 */
export declare class Table extends TableBase {
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope: Construct, id: string, attrs: TableAttributes): ITable;
    readonly tableName: string;
    readonly tableColumns: Column[];
    readonly cluster: ICluster;
    readonly databaseName: string;
    private resource;
    constructor(scope: Construct, id: string, props: TableProps);
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy: cdk.RemovalPolicy): void;
    private validateDistKeyColumns;
    private validateDistStyle;
    private validateSortStyle;
    private getDefaultSortStyle;
    private addColumnIds;
}
/**
 * The data distribution style of a table.
 */
export declare enum TableDistStyle {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    AUTO = "AUTO",
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    EVEN = "EVEN",
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    KEY = "KEY",
    /**
     * A copy of the entire table is distributed to every node.
     */
    ALL = "ALL"
}
/**
 * The sort style of a table.
 */
export declare enum TableSortStyle {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    AUTO = "AUTO",
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    COMPOUND = "COMPOUND",
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    INTERLEAVED = "INTERLEAVED"
}
/**
 * The compression encoding of a column.
 *
 * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Compression_encodings.html
 */
export declare enum ColumnEncoding {
    /**
     * Amazon Redshift assigns an optimal encoding based on the column data.
     * This is the default.
     */
    AUTO = "AUTO",
    /**
     * The column is not compressed.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Raw_encoding.html
     */
    RAW = "RAW",
    /**
     * The column is compressed using the AZ64 algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/az64-encoding.html
     */
    AZ64 = "AZ64",
    /**
     * The column is compressed using a separate dictionary for each block column value on disk.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Byte_dictionary_encoding.html
     */
    BYTEDICT = "BYTEDICT",
    /**
     * The column is compressed based on the difference between values in the column.
     * This records differences as 1-byte values.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Delta_encoding.html
     */
    DELTA = "DELTA",
    /**
     * The column is compressed based on the difference between values in the column.
     * This records differences as 2-byte values.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Delta_encoding.html
     */
    DELTA32K = "DELTA32K",
    /**
     * The column is compressed using the LZO algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/lzo-encoding.html
     */
    LZO = "LZO",
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 1 byte.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    MOSTLY8 = "MOSTLY8",
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 2 bytes.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    MOSTLY16 = "MOSTLY16",
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 4 bytes.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    MOSTLY32 = "MOSTLY32",
    /**
     * The column is compressed by recording the number of occurrences of each value in the column.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Runlength_encoding.html
     */
    RUNLENGTH = "RUNLENGTH",
    /**
     * The column is compressed by recording the first 245 unique words and then using a 1-byte index to represent each word.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Text255_encoding.html
     */
    TEXT255 = "TEXT255",
    /**
     * The column is compressed by recording the first 32K unique words and then using a 2-byte index to represent each word.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Text255_encoding.html
     */
    TEXT32K = "TEXT32K",
    /**
     * The column is compressed using the ZSTD algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/zstd-encoding.html
     */
    ZSTD = "ZSTD"
}
export {};
