"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClusterType = exports.NodeType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const aws_redshift_1 = require("aws-cdk-lib/aws-redshift");
const subnet_group_1 = require("./subnet-group");
/**
 * Possible Node Types to use in the cluster
 * used for defining `ClusterProps.nodeType`.
 */
var NodeType;
(function (NodeType) {
    /**
     * ds2.xlarge
     */
    NodeType["DS2_XLARGE"] = "ds2.xlarge";
    /**
     * ds2.8xlarge
     */
    NodeType["DS2_8XLARGE"] = "ds2.8xlarge";
    /**
     * dc1.large
     */
    NodeType["DC1_LARGE"] = "dc1.large";
    /**
     * dc1.8xlarge
     */
    NodeType["DC1_8XLARGE"] = "dc1.8xlarge";
    /**
     * dc2.large
     */
    NodeType["DC2_LARGE"] = "dc2.large";
    /**
     * dc2.8xlarge
     */
    NodeType["DC2_8XLARGE"] = "dc2.8xlarge";
    /**
     * ra3.xlplus
     */
    NodeType["RA3_XLPLUS"] = "ra3.xlplus";
    /**
     * ra3.4xlarge
     */
    NodeType["RA3_4XLARGE"] = "ra3.4xlarge";
    /**
     * ra3.16xlarge
     */
    NodeType["RA3_16XLARGE"] = "ra3.16xlarge";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * What cluster type to use.
 * Used by `ClusterProps.clusterType`
 */
var ClusterType;
(function (ClusterType) {
    /**
     * single-node cluster, the `ClusterProps.numberOfNodes` parameter is not required
     */
    ClusterType["SINGLE_NODE"] = "single-node";
    /**
     * multi-node cluster, set the amount of nodes using `ClusterProps.numberOfNodes` parameter
     */
    ClusterType["MULTI_NODE"] = "multi-node";
})(ClusterType = exports.ClusterType || (exports.ClusterType = {}));
/**
 * A new or imported clustered database.
 */
class ClusterBase extends aws_cdk_lib_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterName,
            targetType: secretsmanager.AttachmentTargetType.REDSHIFT_CLUSTER,
        };
    }
}
/**
 * Create a Redshift cluster a given number of nodes.
 *
 * @resource AWS::Redshift::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ?? {
            subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
        };
        this.parameterGroup = props.parameterGroup;
        this.roles = props?.roles ? [...props.roles] : [];
        const removalPolicy = props.removalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN;
        const subnetGroup = props.subnetGroup ?? new subnet_group_1.ClusterSubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} Redshift cluster`,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            removalPolicy: removalPolicy,
        });
        const securityGroups = props.securityGroups ?? [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Redshift security group',
                vpc: this.vpc,
            })];
        const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
        let secret;
        if (!props.masterUser.masterPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.masterUsername,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        const clusterType = props.clusterType || ClusterType.MULTI_NODE;
        const nodeCount = this.validateNodeCount(clusterType, props.numberOfNodes);
        if (props.encrypted === false && props.encryptionKey !== undefined) {
            throw new Error('Cannot set property encryptionKey without enabling encryption!');
        }
        this.singleUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER;
        this.multiUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER;
        let loggingProperties;
        if (props.loggingProperties) {
            loggingProperties = {
                bucketName: props.loggingProperties.loggingBucket.bucketName,
                s3KeyPrefix: props.loggingProperties.loggingKeyPrefix,
            };
            props.loggingProperties.loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                actions: [
                    's3:GetBucketAcl',
                    's3:PutObject',
                ],
                resources: [
                    props.loggingProperties.loggingBucket.arnForObjects('*'),
                    props.loggingProperties.loggingBucket.bucketArn,
                ],
                principals: [
                    new iam.ServicePrincipal('redshift.amazonaws.com'),
                ],
            }));
        }
        this.cluster = new aws_redshift_1.CfnCluster(this, 'Resource', {
            // Basic
            allowVersionUpgrade: true,
            automatedSnapshotRetentionPeriod: 1,
            clusterType,
            clusterIdentifier: props.clusterName,
            clusterSubnetGroupName: subnetGroup.clusterSubnetGroupName,
            vpcSecurityGroupIds: securityGroupIds,
            port: props.port,
            clusterParameterGroupName: props.parameterGroup && props.parameterGroup.clusterParameterGroupName,
            // Admin (unsafeUnwrap here is safe)
            masterUsername: secret?.secretValueFromJson('username').unsafeUnwrap() ?? props.masterUser.masterUsername,
            masterUserPassword: secret?.secretValueFromJson('password').unsafeUnwrap()
                ?? props.masterUser.masterPassword?.unsafeUnwrap()
                ?? 'default',
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            nodeType: props.nodeType || NodeType.DC2_LARGE,
            numberOfNodes: nodeCount,
            loggingProperties,
            iamRoles: aws_cdk_lib_1.Lazy.list({ produce: () => this.roles.map(role => role.roleArn) }, { omitEmpty: true }),
            dbName: props.defaultDatabaseName || 'default_db',
            publiclyAccessible: props.publiclyAccessible || false,
            // Encryption
            kmsKeyId: props.encryptionKey?.keyId,
            encrypted: props.encrypted ?? true,
            classic: props.classicResizing,
            elasticIp: props.elasticIp,
            enhancedVpcRouting: props.enhancedVpcRouting,
        });
        this.cluster.applyRemovalPolicy(removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterName = this.cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = aws_cdk_lib_1.Token.asNumber(this.cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(this.cluster.attrEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups, defaultPort });
        if (props.rebootForParameterChanges) {
            this.enableRebootForParameterChanges();
        }
        // Add default role if specified and also available in the roles list
        if (props.defaultRole) {
            if (props.roles?.some(x => x === props.defaultRole)) {
                this.addDefaultIamRole(props.defaultRole);
            }
            else {
                throw new Error('Default role must be included in role list.');
            }
        }
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromClusterAttributes);
            }
            throw error;
        }
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: ec2.Port.tcp(attrs.clusterEndpointPort),
                });
                this.clusterName = attrs.clusterName;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.clusterEndpointPort);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_RotationMultiUserOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRotationMultiUser);
            }
            throw error;
        }
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    validateNodeCount(clusterType, numberOfNodes) {
        if (clusterType === ClusterType.SINGLE_NODE) {
            // This property must not be set for single-node clusters; be generous and treat a value of 1 node as undefined.
            if (numberOfNodes !== undefined && numberOfNodes !== 1) {
                throw new Error('Number of nodes must be not be supplied or be 1 for cluster type single-node');
            }
            return undefined;
        }
        else {
            if (aws_cdk_lib_1.Token.isUnresolved(numberOfNodes)) {
                return numberOfNodes;
            }
            const nodeCount = numberOfNodes ?? 2;
            if (nodeCount < 2 || nodeCount > 100) {
                throw new Error('Number of nodes for cluster type multi-node must be at least 2 and no more than 100');
            }
            return nodeCount;
        }
    }
    /**
     * Adds a parameter to the Clusters' parameter group
     *
     * @param name the parameter name
     * @param value the parameter name
     */
    addToParameterGroup(name, value) {
        if (!this.parameterGroup) {
            const param = {};
            param[name] = value;
            this.parameterGroup = new parameter_group_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: this.cluster.clusterIdentifier ? `Parameter Group for the ${this.cluster.clusterIdentifier} Redshift cluster` : 'Cluster parameter group for family redshift-1.0',
                parameters: param,
            });
            this.cluster.clusterParameterGroupName = this.parameterGroup.clusterParameterGroupName;
        }
        else if (this.parameterGroup instanceof parameter_group_1.ClusterParameterGroup) {
            this.parameterGroup.addParameter(name, value);
        }
        else {
            throw new Error('Cannot add a parameter to an imported parameter group.');
        }
    }
    /**
     * Enables automatic cluster rebooting when changes to the cluster's parameter group require a restart to apply.
     */
    enableRebootForParameterChanges() {
        if (this.node.tryFindChild('RedshiftClusterRebooterCustomResource')) {
            return;
        }
        const rebootFunction = new lambda.SingletonFunction(this, 'RedshiftClusterRebooterFunction', {
            uuid: '511e207f-13df-4b8b-b632-c32b30b65ac2',
            runtime: lambda.Runtime.NODEJS_16_X,
            code: lambda.Code.fromAsset(path.join(__dirname, 'cluster-parameter-change-reboot-handler')),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(900),
        });
        rebootFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift:DescribeClusters'],
            resources: ['*'],
        }));
        rebootFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift:RebootCluster'],
            resources: [
                aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'redshift',
                    resource: 'cluster',
                    resourceName: this.clusterName,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                }),
            ],
        }));
        const provider = new custom_resources_1.Provider(this, 'ResourceProvider', {
            onEventHandler: rebootFunction,
        });
        const customResource = new aws_cdk_lib_1.CustomResource(this, 'RedshiftClusterRebooterCustomResource', {
            resourceType: 'Custom::RedshiftClusterRebooter',
            serviceToken: provider.serviceToken,
            properties: {
                ClusterId: this.clusterName,
                ParameterGroupName: aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        if (!this.parameterGroup) {
                            throw new Error('Cannot enable reboot for parameter changes when there is no associated ClusterParameterGroup.');
                        }
                        return this.parameterGroup.clusterParameterGroupName;
                    },
                }),
                ParametersString: aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        if (!(this.parameterGroup instanceof parameter_group_1.ClusterParameterGroup)) {
                            throw new Error('Cannot enable reboot for parameter changes when using an imported parameter group.');
                        }
                        return JSON.stringify(this.parameterGroup.parameters);
                    },
                }),
            },
        });
        aws_cdk_lib_1.Lazy.any({
            produce: () => {
                if (!this.parameterGroup) {
                    throw new Error('Cannot enable reboot for parameter changes when there is no associated ClusterParameterGroup.');
                }
                customResource.node.addDependency(this, this.parameterGroup);
            },
        });
    }
    /**
     * Adds default IAM role to cluster. The default IAM role must be already associated to the cluster to be added as the default role.
     *
     * @param defaultIamRole the IAM role to be set as the default role
     */
    addDefaultIamRole(defaultIamRole) {
        // Get list of IAM roles attached to cluster
        const clusterRoleList = this.roles ?? [];
        // Check to see if default role is included in list of cluster IAM roles
        var roleAlreadyOnCluster = false;
        for (var i = 0; i < clusterRoleList.length; i++) {
            if (clusterRoleList[i] === defaultIamRole) {
                roleAlreadyOnCluster = true;
                break;
            }
        }
        if (!roleAlreadyOnCluster) {
            throw new Error('Default role must be associated to the Redshift cluster to be set as the default role.');
        }
        // On UPDATE or CREATE define the default IAM role. On DELETE, remove the default IAM role
        const defaultRoleCustomResource = new custom_resources_1.AwsCustomResource(this, 'default-role', {
            onUpdate: {
                service: 'Redshift',
                action: 'modifyClusterIamRoles',
                parameters: {
                    ClusterIdentifier: this.cluster.ref,
                    DefaultIamRoleArn: defaultIamRole.roleArn,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${defaultIamRole.roleArn}-${this.cluster.ref}`),
            },
            onDelete: {
                service: 'Redshift',
                action: 'modifyClusterIamRoles',
                parameters: {
                    ClusterIdentifier: this.cluster.ref,
                    DefaultIamRoleArn: '',
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${defaultIamRole.roleArn}-${this.cluster.ref}`),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
            installLatestAwsSdk: false,
        });
        defaultIamRole.grantPassRole(defaultRoleCustomResource.grantPrincipal);
    }
    /**
     * Adds a role to the cluster
     *
     * @param role the role to add
     */
    addIamRole(role) {
        const clusterRoleList = this.roles;
        if (clusterRoleList.includes(role)) {
            throw new Error(`Role '${role.roleArn}' is already attached to the cluster`);
        }
        clusterRoleList.push(role);
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Cluster", version: "2.71.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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