# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['MIP',
 'MIP.geom',
 'MIP.geom.grammars',
 'MIP.mip',
 't4_geom_convert',
 't4_geom_convert.Kernel',
 't4_geom_convert.Kernel.BoundaryCondition',
 't4_geom_convert.Kernel.Composition',
 't4_geom_convert.Kernel.Configuration',
 't4_geom_convert.Kernel.Exceptions',
 't4_geom_convert.Kernel.FileHandlers',
 't4_geom_convert.Kernel.FileHandlers.Parser',
 't4_geom_convert.Kernel.FileHandlers.Writer',
 't4_geom_convert.Kernel.GeomComp',
 't4_geom_convert.Kernel.Surface',
 't4_geom_convert.Kernel.Transformation',
 't4_geom_convert.Kernel.Volume']

package_data = \
{'': ['*'],
 't4_geom_convert': ['.hypothesis/examples/7d04b6175f19883b/*',
                     '.hypothesis/examples/b5818b9694c7412d/*',
                     '.hypothesis/examples/c02acd8530982b40/*',
                     '.hypothesis/examples/f5595e3a381d5483/*',
                     '.hypothesis/unicode_data/12.1.0/*',
                     '.hypothesis/unicode_data/8.0.0/*']}

install_requires = \
['importlib-metadata>=4.4,<5.0', 'numpy>=1.23,<2.0', 'tatsu>=5.6,<6.0']

entry_points = \
{'console_scripts': ['t4_geom_convert = t4_geom_convert.main:main']}

setup_kwargs = {
    'name': 't4-geom-convert',
    'version': '0.6.1',
    'description': 'A tool to convert MCNP geometries into the TRIPOLI-4 format',
    'long_description': "t4_geom_convert\n===============\n\n[![PyPI version](https://badge.fury.io/py/t4-geom-convert.svg)](https://badge.fury.io/py/t4-geom-convert)\n\nThis repository contains the development version of `t4_geom_convert`, a Python\ntool to convert [MCNP] geometries into the [TRIPOLI-4®] format.\n\nFeatures\n--------\n\nHere is a list of features of the MCNP modelling engine that are\nat least partially supported and converted by `t4_geom_convert`:\n\n* All surface types\n* Boolean cell operators\n* `LIKE n BUT` syntax\n* Affine transformations on surfaces and on cells (see the [Current\n  limitations](#current-limitations) section though)\n* Boundary conditions (reflection, white surfaces)\n* Isotopic compositions and cell densities\n* Universes and fills, even nested, possibly with affine transformations\n* [Lattices](#lattice-conversion)\n\n\nInstallation\n------------\n\n`t4_geom_convert` is available on [PyPI][pypi]. The suggested way to install it\nis to use [the `pip`/`pip3` package manager][pip] and [the `venv`\nmodule][venv]. If `pip3` and `venv` are not available on your machine, you can\nuse your package manager to install them:\n\n```\n# sudo apt install python3-pip python3-venv  # on Debian/Ubuntu\n# sudo yum install python3-pip python3-libs  # on Fedora/CentOS\n```\n\nYou can then create a virtual Python environment and install the latest stable\nversion of `t4_geom_convert` there with\n\n```\n$ python3 -m venv /path/to/some/folder\n$ source /path/to/some/folder/bin/activate\n$ pip3 install -U pip setuptools\n$ pip3 install t4_geom_convert\n```\n\nThis will install the `t4_geom_convert` executable in\n`/path/to/some/folder/bin/t4_geom_convert`. Sourcing\n`/path/to/some/folder/bin/activate` will put `t4_geom_convert` in your `PATH`.\n\nYou can also install the latest development version with\n\n```\n$ pip3 install git+https://github.com/arekfu/t4_geom_convert.git@next\n```\n\nYou can find more information about [`pip3`][pip] and [`venv`][venv] on the\nweb.\n\n\n### Dependencies\n\nThe MCNP input file is parsed by [MIP]. We use a slightly modified version of\nMIP, which is shipped along with `t4_geom_convert`. MIP depends on [TatSu].\n\n`t4_geom_convert` also depends on [NumPy].\n\n\nUsage\n-----\n\nThe basic usage is simply\n\n```\n$ t4_geom_convert <mcnp_input>\n```\n\nThis will create a TRIPOLI-4 output file called `<mcnp_input>.t4` containing\nthe converted geometry.  You can also choose a different name for the output\nfile using the `-o` option.\n\nUse the `-h` option for a list of all available options.\n\n### Lattice conversion\n\n`t4_geom_convert` is capable of handling the conversion of repeated structures\n(lattices). Hexahedral lattices (`LAT=1`) and hexagonal (`LAT=2`) are\nsupported.\n\nThe cell declared as `LAT=1` or `LAT=2` represents the unit cell of the\nlattice, which is assumed to repeat in all directions up to the boundaries of\nthe enclosing cell. Due to limitations of the TRIPOLI-4 representation of\nlattices, we have chosen to represent lattices using a purely surface-based\napproach.  This means that `t4_geom_convert` will actually emit separate cell\ndefinitions for each cell of the lattice that is visible through the enclosing\ncell. The ranges of cell definitions to be emitted must be specified by the\nuser via the `--lattice` command-line option. For instance, consider the\nfollowing MCNP input:\n\n```\nA lattice example\n1 0  -2 1 -4 3 IMP:N=1 U=2 LAT=1\n10 1 -1. -10 IMP:N=1 FILL=2\n1000 0 10 IMP:N=0\n\n1 PX -1.5\n2 PX 1.5\n3 PY -0.5\n4 PY 0.5\n10 SO 4\n\nm1 13027 1.\n```\n\nHere the unit cell is two-dimensional. The lattice fills a sphere of radius\n4.  Assuming the unit cell is indexed as (0,0), the visible lattice cells are\n* (-1, -4) to (-1, 4)\n* (0, -4) to (0, 4)\n* (1, -4) to (1, 4)\n\nThis can be confirmed by visual inspection of the MCNP geometry or by\ngeometrical considerations. Once the index bounds are determined, the\n`--lattice` option must be specified as\n\n```\n$ t4_geom_convert --lattice 1,-1:1,-4:4 <mcnp_input>\n                            ↑  ↑ ↑  ↑ ↑\n             cell number ───┘  │ │  │ └ j-range upper bound\n        i-range lower bound ───┘ │  └─── j-range lower bound\n          i-range upper bound ───┘\n```\n\nThis results in the following TRIPOLI-4 geometry (X-Y cut), where a few cell\nindices have been annotated:\n![example of converted geometry with lattices][lattice_example]\n\nThe syntax for one-dimensional lattices is\n\n```\n--lattice <cell>,<i-from>:<i-to>\n```\n\nand for three-dimensional lattices it is\n\n```\n--lattice <cell>,<i-from>:<i-to>,<j-from>:<j-to>,<k-from>:<k-to>\n```\n\nNote that the `ijk` axes are not necessarily the same as the coordinate axes.\nThe orientation of the lattice axes is specified by MCNP (see the *Lattice\nindexing* paragraph in the User's Manual) and it is determined by the order in\nwhich the surfaces of the unit cell appear specified. In our example, the first\nsurface appearing in the definition of the unit cell is surface `2`; therefore,\nsurface `2` separates the unit cell (0, 0) from cell (1, 0); the next surface\n(`1`) separates the unit cell from cell (-1, 0); the following surfaces, `4`\nand `3`, separate the unit cell from cells (0, 1) and (0, -1), respectively.\n\nFor hexagonal lattices, `t4_geom_convert` follows the convention described in\nthe MCNP manual. The lattice axes are defined by the first and the third plane\nappearing in the definition of the base cell. The positive direction of the\nthird axis (if present) is defined by the normal to the seventh plane appearing\nin the definition.\n\nA lattice unit cell may appear as a fill pattern in several enclosing cells. It\nis currently not possible to specify different fill ranges for each of them.\n\n\n### Fully-specified lattices\n\nMCNP provides a syntax for the specification of lattice with heterogeneous\ncells. An example is\n\n```\nA lattice example\nc cells\n2 0  -21 u=2 imp:n=1\n21 0  21 u=2 imp:n=1\n3 0  -31 u=3 imp:n=1\n31 0  31 u=3 imp:n=1\n10 0  -2 1 3 -4  lat=1 u=20 IMP:N=1\n        FILL=-1:1 -4:4\nc       i=-1   i=0   i=1\n        2      3     2   $ j=-4\n        2      3     2   $ j=-3\n        2      3     2   $ j=-2\n        2      3     2   $ j=-1\n        2      3     2   $ j=0\n        3      3     2   $ j=1\n        3      3     2   $ j=2\n        3      3     2   $ j=3\n        3      3     2   $ j=4\n100 1 -1. -10 IMP:N=1 FILL=20\n1000 0 10 IMP:N=0\n\n1 PX -1.5\n2 PX 1.5\n3 PY -0.5\n4 PY 0.5\n10 SO 4\n21 SO 0.4\n31 SO 0.1\n\nm1 13027 1.\n```\n\nThe `FILL=-1:1 -4:4` option indicates the range of indices where cells will be\nspecified. The universes filling the cells are detailed in the table below,\nwhich by convention loops over the leftmost (`i`) axis first. This syntax is\nsupported by `t4_geom_convert` and does *not* require a `--lattice` option. One\nrestriction applies: none of the subcells may be filled with the universe of\nthe lattice cell itself. This syntax indicates to MCNP that the cell should be\nfilled with the material of the lattice cell. Using `0` as a universe number\nfor the subcells is supported and results in no subcell being generated.\n\nHere is how the example above is converted and rendered in TRIPOLI-4:\n![example of converted geometry with fully-specified\nlattice][lattice_fully_specified]\n\nCurrent limitations\n-------------------\n\nHere is a list of some things that `t4_geom_convert` cannot currently do, but\nmay be able to do in the future (in roughly decreasing order of likelihood):\n\n- [ ] Import the title of the MCNP input file (tracked in [issue\n  #5](https://github.com/arekfu/t4_geom_convert/issues/5))\n- [ ] Handle affine transformations with `m=-1` (the last parameter of the\n  affine transformation) (tracked in [issue\n  #12](https://github.com/arekfu/t4_geom_convert/issues/12))\n- [ ] Optimize fills with negative universes (do not intersect with the\n  enclosing cell) (tracked in [issue\n  #13](https://github.com/arekfu/t4_geom_convert/issues/13))\n- [ ] Warn about isotopes that are missing from the TRIPOLI-4 dictionary\n  (currently you need to edit the converted file by hand and remove the\n  occurrences of the missing isotopes)\n- [ ] Convert cell temperatures\n- [ ] Import comments describing the MCNP cells/surfaces (tracked in [issue\n  #9](https://github.com/arekfu/t4_geom_convert/issues/9))\n- [ ] Provide a way to specify lattice fill ranges per enclosing cell(s) (this\n  needs to be specified in such a way that it works with nested lattices, too)\n- [ ] Deduplicate repeated cell definitions (this is a bit harder than\n  deduplicating surfaces)\n- [ ] Produce a TRIPOLI-4 connectivity map for as many cells as possible\n  (mostly lattices)\n- [ ] Recognize and automatically suppress empty cells (they may be generated\n  by lattice development or fill development)\n  - Use a linear programming solver for cells bounded by planes?\n  - Use a SAT solver in the general case?\n- [ ] Convert (some) MCNP source definitions\n- [ ] Convert (some) MCNP tally definitions\n\nYour help is welcome! Feel free to open an issue if you would like to implement\na new feature or contribute to the project in any way.\n\nThe full changelog is [here](changelog.rst).\n\n\nTesting\n-------\n\nThe correctness of `t4_geom_convert` is tested using [a specific test\noracle](Oracle/README.md), which is included in this repository.\n\n\nReporting bugs\n--------------\n\nPlease report any bug/feature request on [the GitHub issues page][bugs].\n\n\nLicence and acknowledgments\n---------------------------\n\nThe development of `t4_geom_convert` was partially financed by the [EUROfusion]\nconsortium. `t4_geom_convert` is released under the terms of the  [GNU General\nPublic Licence, version 3](COPYING).\n\n\n[bugs]: https://github.com/arekfu/t4_geom_convert/issues\n[EUROfusion]: https://www.euro-fusion.org/\n[lattice_example]: pics/lattice_example.png\n[lattice_fully_specified]: pics/lattice_fully_specified.png\n[MCNP]: https://mcnp.lanl.gov/\n[MIP]: https://github.com/travleev/mip\n[NumPy]: https://numpy.org/\n[pip]: https://packaging.python.org/tutorials/installing-packages/\n[pypi]: https://pypi.org/project/t4-geom-convert/\n[TatSu]: https://tatsu.readthedocs.io/en/stable/\n[TRIPOLI-4®]: https://www.cea.fr/nucleaire/tripoli-4\n[venv]: https://docs.python.org/3/tutorial/venv.html\n",
    'author': 'Davide Mancusi',
    'author_email': 'davide.mancusi@cea.fr',
    'maintainer': 'Davide Mancusi',
    'maintainer_email': 'davide.mancusi@cea.fr',
    'url': 'https://www.cea.fr/energies/tripoli-4/tripoli-4/pre_post_tools/t4_geom_convert/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4',
}


setup(**setup_kwargs)
