# AUTOGENERATED! DO NOT EDIT! File to edit: ../../nbs/distributed.fugue.ipynb.

# %% auto 0
__all__ = ['FugueBackend']

# %% ../../nbs/distributed.fugue.ipynb 4
from typing import Any, Dict, List

import numpy as np
import pandas as pd
try:
    from fugue import transform
except ModuleNotFoundError as e:
    msg = (
        f'{e}. To use fugue you have to install it.'
        'Please run `pip install fugue`. '
    )
    raise ModuleNotFoundError(msg) from e
from ..core import StatsForecast
from .core import ParallelBackend
from triad import Schema

# %% ../../nbs/distributed.fugue.ipynb 5
class FugueBackend(ParallelBackend):
    """FugueBackend for Distributed Computation.
    [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/distributed/fugue.py).

    This class uses [Fugue](https://github.com/fugue-project/fugue) backend capable of distributing 
    computation on Spark and Dask without any rewrites.

    **Parameters:**<br>
    `engine`: fugue.ExecutionEngine, a selection between spark and dask.<br>
    `conf`: fugue.Config, engine configuration.<br>
    `**transform_kwargs`: additional kwargs for Fugue's transform method.<br>

    **Notes:**<br>
    A short introduction to Fugue, with examples on how to scale pandas code to scale pandas 
    based code to Spark or Dask is available [here](https://fugue-tutorials.readthedocs.io/tutorials/quick_look/ten_minutes.html).
    """
    def __init__(
            self, 
            engine: Any = None,
            conf: Any = None,
            **transform_kwargs: Any
        ):        
        self._engine = engine
        self._conf = conf
        self._transform_kwargs = dict(transform_kwargs)

    def __getstate__(self) -> Dict[str, Any]:
        return {}

    def forecast(
            self, 
            df,
            models,
            freq,
            fallback_model = None,
            **kwargs: Any,
        ) -> Any:
        """Memory Efficient core.StatsForecast predictions with FugueBackend.

        This method uses Fugue's transform function, in combination with 
        `core.StatsForecast`'s forecast to efficiently fit a list of StatsForecast models.

        **Parameters:**<br>
        `df`: pandas.DataFrame, with columns [`unique_id`, `ds`, `y`] and exogenous.<br>
        `freq`: str, frequency of the data, [panda's available frequencies](https://pandas.pydata.org/pandas-docs/stable/user_guide/timeseries.html#offset-aliases).<br>
        `models`: List[typing.Any], list of instantiated objects `StatsForecast.models`.<br>
        `fallback_model`: Any, Model to be used if a model fails. Only works with the `forecast` method.<br>
        `**kwargs`: Additional `core.StatsForecast` parameters. Example forecast horizon `h`.<br>

        **Returns:**<br>
        `fcsts_df`: pandas.DataFrame, with `models` columns for point predictions and probabilistic
        predictions for all fitted `models`.<br>
        
        **References:**<br>
        For more information check the 
        [Fugue's transform](https://fugue-tutorials.readthedocs.io/tutorials/beginner/introduction.html#fugue-transform)
        tutorial.<br>
        The [core.StatsForecast's forecast](https://nixtla.github.io/statsforecast/core.html#statsforecast.forecast)
        method documentation.<br>
        Or the list of available [StatsForecast's models](https://nixtla.github.io/statsforecast/models.html).
        """
        schema = "*-y+" + str(self._get_output_schema(models))
        return transform(
            df,
            self._forecast_series,
            params=dict(models=models, freq=freq, 
                        kwargs=kwargs, fallback_model=fallback_model),
            schema=schema,
            partition={"by": "unique_id"},
            engine=self._engine,
            engine_conf=self._conf,
            **self._transform_kwargs,
        )

    def cross_validation(
            self, 
            df,
            models,
            freq,
            **kwargs: Any, 
        ) -> Any:
        """Temporal Cross-Validation with core.StatsForecast and FugueBackend.

        This method uses Fugue's transform function, in combination with 
        `core.StatsForecast`'s cross-validation to efficiently fit a list of StatsForecast 
        models through multiple training windows, in either chained or rolled manner.

        `StatsForecast.models`' speed along with Fugue's distributed computation allow to 
        overcome this evaluation technique high computational costs. Temporal cross-validation 
        provides better model's generalization measurements by increasing the test's length 
        and diversity.

        **Parameters:**<br>
        `df`: pandas.DataFrame, with columns [`unique_id`, `ds`, `y`] and exogenous.<br>
        `freq`: str, frequency of the data, [panda's available frequencies](https://pandas.pydata.org/pandas-docs/stable/user_guide/timeseries.html#offset-aliases).<br>
        `models`: List[typing.Any], list of instantiated objects `StatsForecast.models`.<br>

        **Returns:**<br>
        `fcsts_df`: pandas.DataFrame, with `models` columns for point predictions and probabilistic
        predictions for all fitted `models`.<br>
        
        **References:**<br>
        The [core.StatsForecast's cross validation](https://nixtla.github.io/statsforecast/core.html#statsforecast.cross_validation)
        method documentation.<br>
        [Rob J. Hyndman and George Athanasopoulos (2018). "Forecasting principles and practice, Temporal Cross-Validation"](https://otexts.com/fpp3/tscv.html).
        """
        schema = "*-y+" + str(self._get_output_schema(models, mode="cv"))
        return transform(
            df,
            self._cv,
            params=dict(models=models, freq=freq, 
                        kwargs=kwargs),
            schema=schema,
            partition={"by": "unique_id"},
            engine=self._engine,
            engine_conf=self._conf,
            **self._transform_kwargs,
        )

    def _forecast_series(self, df: pd.DataFrame, models, freq, fallback_model, kwargs) -> pd.DataFrame:
        model = StatsForecast(df=df, models=models, freq=freq, 
                              fallback_model=fallback_model, n_jobs=1)
        return model.forecast(**kwargs).reset_index()

    def _cv(self, df: pd.DataFrame, models, freq, kwargs) -> pd.DataFrame:
        model = StatsForecast(df=df, models=models, freq=freq, n_jobs=1)
        return model.cross_validation(**kwargs).reset_index()

    def _get_output_schema(self, models, mode="forecast") -> Schema:
        cols: List[Any]
        cols = [(repr(model), np.float32) for model in models]
        if mode == "cv":
            cols = [("cutoff", "datetime"), ("y", np.float32)] + cols
        return Schema(cols)
