# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/models.ipynb.

# %% auto 0
__all__ = ['AutoARIMA', 'ETS', 'AutoCES', 'SimpleExponentialSmoothing', 'SimpleExponentialSmoothingOptimized',
           'SeasonalExponentialSmoothing', 'SeasonalExponentialSmoothingOptimized', 'Holt', 'HoltWinters',
           'HistoricAverage', 'Naive', 'RandomWalkWithDrift', 'SeasonalNaive', 'WindowAverage', 'SeasonalWindowAverage',
           'ADIDA', 'CrostonClassic', 'CrostonOptimized', 'CrostonSBA', 'IMAPA', 'TSB', 'MSTL']

# %% ../nbs/models.ipynb 4
from inspect import signature
from math import trunc
from typing import Any, Dict, List, Optional, Sequence, Tuple, Union

import numpy as np
from numba import njit
from scipy.optimize import minimize
from scipy.stats import norm

from .arima import auto_arima_f, forecast_arima, fitted_arima
from .ces import auto_ces, forecast_ces
from .ets import ets_f, forecast_ets
from .mstl import mstl

# %% ../nbs/models.ipynb 6
class _TS:
    
    def new(self):
        b = type(self).__new__(type(self))
        b.__dict__.update(self.__dict__)
        return b

# %% ../nbs/models.ipynb 7
# Functions used for calculating prediction intervals 
def _quantiles(level): 
    z = norm.ppf(0.5+level/200)   
    return z

def _calculate_intervals(out, level, h, sigmah):
    z = _quantiles(np.asarray(level))
    zz = np.repeat(z, h)
    zz = zz.reshape(z.shape[0], h)
    lower = out['mean'] - zz * sigmah
    upper = out['mean'] + zz * sigmah
    pred_int = {**{f'lo-{lv}': lower[i] for i, lv in enumerate(level)}, 
                **{f'hi-{lv}': upper[i] for i, lv in enumerate(level)}}    
    return pred_int

def _calculate_sigma(residuals, n): 
    sigma = np.nansum(residuals ** 2) 
    sigma = sigma / n
    sigma = np.sqrt(sigma)
    return sigma

# %% ../nbs/models.ipynb 10
class AutoARIMA(_TS):
    """AutoARIMA model.
    [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/arima.py).

    Automatically selects the best ARIMA (AutoRegressive Integrated Moving Average) 
    model using an information criterion. Default is Akaike Information Criterion (AICc). 
    
    **Parameters:**<br>
    `d`: int, order of first-differencing.<br>
    `D`: int, order of seasonal-differencing.<br>
    `max_p`: int, max autorregresives p.<br>
    `max_q`: int, max moving averages q.<br>
    `max_P`: int, max seasonal autorregresives P.<br>
    `max_Q`: int, max seasonal moving averages Q.<br>
    `max_order`: int, max p+q+P+Q value if not stepwise selection.<br>
    `max_d`: int, max non-seasonal differences.<br>
    `max_D`: int, max seasonal differences.<br>
    `start_p`: int, starting value of p in stepwise procedure.<br>
    `start_q`: int, starting value of q in stepwise procedure.<br>
    `start_P`: int, starting value of P in stepwise procedure.<br>
    `start_Q`: int, starting value of Q in stepwise procedure.<br>
    `stationary`: bool, if True, restricts search to stationary models.<br>
    `seasonal`: bool, if False, restricts search to non-seasonal models.<br>
    `ic`: str, information criterion to be used in model selection.<br>
    `stepwise`: bool, if True, will do stepwise selection (faster).<br>
    `nmodels`: int, number of models considered in stepwise search.<br>
    `trace`: bool, if True, the searched ARIMA models is reported.<br>
    `approximation`: bool, if True, conditional sums-of-squares estimation, final MLE.<br>
    `method`: str, fitting method between maximum likelihood or sums-of-squares.<br>
    `truncate`: int, observations truncated series used in model selection.<br>
    `test`: str (default 'kpss'), unit root test to use. See `ndiffs` for details.<br>
    `test_kwargs`: str optional (default None), unit root test additional arguments.<br>
    `seasonal_test`: str (default 'seas'), selection method for seasonal differences.<br>
    `seasonal_test_kwargs`: dict (optional), seasonal unit root test arguments.<br>
    `allowdrift`: bool (default True), If True, drift models terms considered.<br>
    `allowmean`: bool (default True), If True, non-zero mean models considered.<br>
    `blambda`: float optional (default None), Box-Cox transformation parameter.<br>
    `biasadj`: bool (default False), Use adjusted back-transformed mean Box-Cox.<br>
    `parallel`: bool, If True and stepwise=False, then parallel search.<br>
    `num_cores`: int, amount of parallel processes to be used if parallel=True.<br>
    `season_length`: int, number of observations per unit of time. Ex: 24 Hourly data.<br>
    
    **Note:**<br>
    This implementation is a mirror of Hyndman's [forecast::auto.arima](https://github.com/robjhyndman/forecast).
    
    **References:**<br>
    [Rob J. Hyndman, Yeasmin Khandakar (2008). "Automatic Time Series Forecasting: The forecast package for R"](https://www.jstatsoft.org/article/view/v027i03).
    """
    def __init__(
        self,
        d: Optional[int] = None,
        D: Optional[int] = None,
        max_p: int = 5,
        max_q: int = 5,
        max_P: int = 2,
        max_Q: int = 2,
        max_order: int = 5,
        max_d: int = 2,
        max_D: int = 1,
        start_p: int = 2,
        start_q: int = 2,
        start_P: int = 1,
        start_Q: int = 1,
        stationary: bool = False,
        seasonal: bool = True,
        ic: str = 'aicc',
        stepwise: bool = True,
        nmodels: int = 94,
        trace: bool = False,
        approximation: Optional[bool] = False,
        method: Optional[str] = None,
        truncate: Optional[bool] = None,
        test: str = 'kpss',
        test_kwargs: Optional[str] = None,
        seasonal_test: str = 'seas',
        seasonal_test_kwargs: Optional[Dict] = None,
        allowdrift: bool = False,
        allowmean: bool = False,
        blambda: Optional[float] = None,
        biasadj: bool = False,
        parallel: bool = False,
        num_cores: int = 2,
        season_length: int = 1
    ):
        self.d=d
        self.D=D
        self.max_p=max_p
        self.max_q=max_q
        self.max_P=max_P
        self.max_Q=max_Q
        self.max_order=max_order
        self.max_d=max_d
        self.max_D=max_D
        self.start_p=start_p
        self.start_q=start_q
        self.start_P=start_P
        self.start_Q=start_Q
        self.stationary=stationary
        self.seasonal=seasonal
        self.ic=ic
        self.stepwise=stepwise
        self.nmodels=nmodels
        self.trace=trace
        self.approximation=approximation
        self.method=method
        self.truncate=truncate
        self.test=test
        self.test_kwargs=test_kwargs
        self.seasonal_test=seasonal_test
        self.seasonal_test_kwargs=seasonal_test_kwargs
        self.allowdrift=allowdrift
        self.allowmean=allowmean
        self.blambda=blambda
        self.biasadj=biasadj
        self.parallel=parallel
        self.num_cores=num_cores
        self.season_length=season_length
        
    def __repr__(self):
        return 'AutoARIMA'
    
    def fit(
            self, 
            y: np.ndarray,
            X: Optional[np.ndarray] = None,
        ):
        """Fit the AutoARIMA model.

        Fit an AutoARIMA to a time series (numpy array) `y`
        and optionally exogenous variables (numpy array) `X`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: AutoARIMA fitted model.
        """
        with np.errstate(invalid='ignore'):
            self.model_ = auto_arima_f(
                x=y,
                d=self.d,
                D=self.D,
                max_p=self.max_p,
                max_q=self.max_q,
                max_P=self.max_P,
                max_Q=self.max_Q,
                max_order=self.max_order,
                max_d=self.max_d,
                max_D=self.max_D,
                start_p=self.start_p,
                start_q=self.start_q,
                start_P=self.start_P,
                start_Q=self.start_Q,
                stationary=self.stationary,
                seasonal=self.seasonal,
                ic=self.ic,
                stepwise=self.stepwise,
                nmodels=self.nmodels,
                trace=self.trace,
                approximation=self.approximation,
                method=self.method,
                truncate=self.truncate,
                xreg=X,
                test=self.test,
                test_kwargs=self.test_kwargs,
                seasonal_test=self.seasonal_test,
                seasonal_test_kwargs=self.seasonal_test_kwargs,
                allowdrift=self.allowdrift,
                allowmean=self.allowmean,
                blambda=self.blambda,
                biasadj=self.biasadj,
                parallel=self.parallel,
                num_cores=self.num_cores,
                period=self.season_length
            )
        return self
    
    def predict(
            self, 
            h: int,
            X: np.ndarray = None,
            level: Optional[Tuple[int]] = None,
        ):
        """Predict with fitted AutoArima.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>
        `X`: array-like of shape (h, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        fcst = forecast_arima(self.model_, h=h, xreg=X, level=level)
        mean = fcst['mean']
        if level is None:
            return {'mean': mean}
        return {
            'mean': mean,
            **{f'lo-{l}': fcst['lower'][f'{l}%'] for l in reversed(level)},
            **{f'hi-{l}': fcst['upper'][f'{l}%'] for l in level},
        }
    
    def predict_in_sample(self, level: Optional[Tuple[int]] = None):
        """Access fitted AutoArima insample predictions.

        **Parameters:**<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        if level is not None:
            raise NotImplementedError
        mean = fitted_arima(self.model_)
        return {'mean': mean}
    
    def forecast(
            self,
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            level: Optional[List[int]] = None,
            fitted: bool = False,
        ):
        """Memory Efficient AutoARIMA predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `X`: array-like of shape (t, n_x) optional insample exogenous (default=None).<br>
        `X_future`: array-like of shape (h, n_x) optional exogenous (default=None).<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        with np.errstate(invalid='ignore'):
            mod = auto_arima_f(
                x=y,
                d=self.d,
                D=self.D,
                max_p=self.max_p,
                max_q=self.max_q,
                max_P=self.max_P,
                max_Q=self.max_Q,
                max_order=self.max_order,
                max_d=self.max_d,
                max_D=self.max_D,
                start_p=self.start_p,
                start_q=self.start_q,
                start_P=self.start_P,
                start_Q=self.start_Q,
                stationary=self.stationary,
                seasonal=self.seasonal,
                ic=self.ic,
                stepwise=self.stepwise,
                nmodels=self.nmodels,
                trace=self.trace,
                approximation=self.approximation,
                method=self.method,
                truncate=self.truncate,
                xreg=X,
                test=self.test,
                test_kwargs=self.test_kwargs,
                seasonal_test=self.seasonal_test,
                seasonal_test_kwargs=self.seasonal_test_kwargs,
                allowdrift=self.allowdrift,
                allowmean=self.allowmean,
                blambda=self.blambda,
                biasadj=self.biasadj,
                parallel=self.parallel,
                num_cores=self.num_cores,
                period=self.season_length
            )
        fcst = forecast_arima(mod, h, xreg=X_future, level=level)
        res = {'mean': fcst['mean']}
        if fitted:
            res['fitted'] = fitted_arima(mod)
        if level is not None:
            level = sorted(level)
            res = {
                **res,
                **{f'lo-{l}': fcst['lower'][f'{l}%'] for l in reversed(level)},
                **{f'hi-{l}': fcst['upper'][f'{l}%'] for l in level},
            }
            if fitted:
                # add prediction intervals for fitted values
                arr_level = np.asarray(level) 
                se = np.sqrt(mod['sigma2'])
                quantiles = norm.ppf(0.5 * (1 + arr_level / 100))

                lo = res['fitted'].reshape(-1, 1) - quantiles * se.reshape(-1, 1)
                hi = res['fitted'].reshape(-1, 1) + quantiles * se.reshape(-1, 1)
                lo = lo[:, ::-1]
                lo = {f'fitted-lo-{l}': lo[:, i] for i, l in enumerate(reversed(level))}
                hi = {f'fitted-hi-{l}': hi[:, i] for i, l in enumerate(level)}
                res = {**res, **lo, **hi}
        return res

# %% ../nbs/models.ipynb 20
class ETS(_TS):
    """Exponential Smoothing model.
    [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/ets.py).

    Automatically selects the best ETS (Error, Trend, Seasonality) 
    model using an information criterion. Default is Akaike Information Criterion (AICc), while particular models are estimated using maximum likelihood.
    The state-space equations can be determined based on their $M$ multiplicative, $A$ additive, 
    $Z$ optimized or $N$ ommited components. The `model` string parameter defines the ETS equations: 
    E in [$M, A, Z$], T in [$N, A, M, Z$], and S in [$N, A, M, Z$].
    
    For example when model='ANN' (additive error, no trend, and no seasonality), ETS will 
    explore only a simple exponential smoothing.
    
    If the component is selected as 'Z', it operates as a placeholder to ask the AutoETS model
    to figure out the best parameter.
    
    **Parameters:**<br>
    `model`: str, controlling state-space-equations.<br>
    `season_length`: int, number of observations per unit of time. Ex: 24 Hourly data.<br>
    
    **Note:**<br>
    This implementation is a mirror of Hyndman's [forecast::ets](https://github.com/robjhyndman/forecast).
    
    **References:**<br>
    [Rob J. Hyndman, Yeasmin Khandakar (2008). "Automatic Time Series Forecasting: The forecast package for R"](https://www.jstatsoft.org/article/view/v027i03).
    
    [Hyndman, Rob, et al (2008). "Forecasting with exponential smoothing: the state space approach"](https://robjhyndman.com/expsmooth/).
    """
    def __init__(
            self, 
            season_length: int = 1,
            model: str = 'ZZZ'
        ):
        self.season_length = season_length
        self.model = model
    
    def __repr__(self):
        return 'ETS'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the Exponential Smoothing model.

        Fit an Exponential Smoothing model to a time series (numpy array) `y`
        and optionally exogenous variables (numpy array) `X`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: Exponential Smoothing fitted model.
        """
        self.model_ = ets_f(y, m=self.season_length, model=self.model)
        return self
    
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted Exponential Smoothing.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>
        `X`: array-like of shape (h, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = forecast_ets(self.model_, h=h)['mean']
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted Exponential Smoothing insample predictions.

        **Parameters:**<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        res = {'mean': self.model_['fitted']}
        return res
    
    def forecast(
            self,
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient Exponential Smoothing predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `X`: array-like of shape (t, n_x) optional insample exogenous (default=None).<br>
        `X_future`: array-like of shape (h, n_x) optional exogenous (default=None).<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mod = ets_f(y, m=self.season_length, model=self.model)
        fcst = forecast_ets(mod, h)
        keys = ['mean']
        if fitted:
            keys.append('fitted')
        return {key: fcst[key] for key in keys}

# %% ../nbs/models.ipynb 29
class AutoCES(_TS):
    """Complex Exponential Smoothing model.
    [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/ces.py).

    Automatically selects the best Complex Exponential Smoothing
    model using an information criterion. Default is Akaike Information Criterion (AICc), while particular 
    models are estimated using maximum likelihood.
    The state-space equations can be determined based on their $S$ simple, $P$ parial, 
    $Z$ optimized or $N$ ommited components. The `model` string parameter defines the 
    kind of CES model: $N$ for simple CES (withous seasonality), $S$ for simple seasonality (lagged CES),
    $P$ for partial seasonality (without complex part), $F$ for full seasonality (lagged CES
    with real and complex seasonal parts).
    
    If the component is selected as 'Z', it operates as a placeholder to ask the AutoCES model
    to figure out the best parameter.
    
    **Parameters:**<br>
    `model`: str, controlling state-space-equations.<br>
    `season_length`: int, number of observations per unit of time. Ex: 24 Hourly data.<br>
    
    **References:**<br>
    [Svetunkov, Ivan & Kourentzes, Nikolaos. (2015). "Complex Exponential Smoothing". 10.13140/RG.2.1.3757.2562. ](https://onlinelibrary.wiley.com/doi/full/10.1002/nav.22074).
    """
    
    def __init__(
            self, 
            season_length: int = 1,
            model: str = 'Z'
        ):
        self.season_length = season_length
        self.model = model
    
    def __repr__(self):
        return 'CES'  

    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the Complex Exponential Smoothing model.

        Fit the Comples Exponential Smoothing model to a time series (numpy array) `y`
        and optionally exogenous variables (numpy array) `X`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: Complex Exponential Smoothing fitted model.
        """
        self.model_ = auto_ces(y, m=self.season_length, model=self.model)
        return self
    
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted Exponential Smoothing.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>
        `X`: array-like of shape (h, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = forecast_ces(self.model_, h=h)['mean']
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted Exponential Smoothing insample predictions.

        **Parameters:**<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        raise NotImplementedError('fitted CES values')
        res = {'mean': self.model_['fitted']}
        return res
    
    def forecast(
            self,
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient Complex Exponential Smoothing predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `X`: array-like of shape (t, n_x) optional insample exogenous (default=None).<br>
        `X_future`: array-like of shape (h, n_x) optional exogenous (default=None).<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mod = auto_ces(y, m=self.season_length, model=self.model)
        fcst = forecast_ces(mod, h)
        keys = ['mean']
        if fitted:
            raise NotImplementedError('ces insample predictions')
        return {key: fcst[key] for key in keys}

# %% ../nbs/models.ipynb 39
@njit
def _ses_fcst_mse(x: np.ndarray, alpha: float) -> Tuple[float, float, np.ndarray]:
    """Perform simple exponential smoothing on a series.

    This function returns the one step ahead prediction
    as well as the mean squared error of the fit.
    """
    smoothed = x[0]
    n = x.size
    mse = 0.
    fitted = np.full(n, np.nan, np.float32)

    for i in range(1, n):
        smoothed = (alpha * x[i - 1] + (1 - alpha) * smoothed).item()
        error = x[i] - smoothed
        mse += error * error
        fitted[i] = smoothed

    mse /= n
    forecast = alpha * x[-1] + (1 - alpha) * smoothed
    return forecast, mse, fitted


def _ses_mse(alpha: float, x: np.ndarray) -> float:
    """Compute the mean squared error of a simple exponential smoothing fit."""
    _, mse, _ = _ses_fcst_mse(x, alpha)
    return mse


@njit
def _ses_forecast(x: np.ndarray, alpha: float) -> Tuple[float, np.ndarray]:
    """One step ahead forecast with simple exponential smoothing."""
    forecast, _, fitted = _ses_fcst_mse(x, alpha)
    return forecast, fitted


@njit
def _demand(x: np.ndarray) -> np.ndarray:
    """Extract the positive elements of a vector."""
    return x[x > 0]


@njit
def _intervals(x: np.ndarray) -> np.ndarray:
    """Compute the intervals between non zero elements of a vector."""
    y = []

    ctr = 1
    for val in x:
        if val == 0:
            ctr += 1
        else:
            y.append(ctr)
            ctr = 1

    return np.array(y)


@njit
def _probability(x: np.ndarray) -> np.ndarray:
    """Compute the element probabilities of being non zero."""
    return (x != 0).astype(np.int32)


def _optimized_ses_forecast(
        x: np.ndarray,
        bounds: Sequence[Tuple[float, float]] = [(0.1, 0.3)]
    ) -> Tuple[float, np.ndarray]:
    """Searches for the optimal alpha and computes SES one step forecast."""
    alpha = minimize(
        fun=_ses_mse,
        x0=(0,),
        args=(x,),
        bounds=bounds,
        method='L-BFGS-B'
    ).x[0]
    forecast, fitted = _ses_forecast(x, alpha)
    return forecast, fitted


@njit
def _chunk_sums(array: np.ndarray, chunk_size: int) -> np.ndarray:
    """Splits an array into chunks and returns the sum of each chunk."""
    n = array.size
    n_chunks = n // chunk_size
    sums = np.empty(n_chunks)
    for i, start in enumerate(range(0, n, chunk_size)):
        sums[i] = array[start : start + chunk_size].sum()
    return sums

@njit
def _repeat_val(val: float, h: int):
    return np.full(h, val, np.float32)

@njit
def _repeat_val_seas(season_vals: np.ndarray, h: int, season_length: int):
    out = np.empty(h, np.float32)
    for i in range(h):
        out[i] = season_vals[i % season_length]
    return out

# %% ../nbs/models.ipynb 40
@njit
def _ses(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
        alpha: float, # smoothing parameter
    ): 
    fcst, _, fitted_vals = _ses_fcst_mse(y, alpha)
    mean = _repeat_val(val=fcst, h=h)
    fcst = {'mean': mean}
    if fitted:
        fcst['fitted'] = fitted_vals
    return fcst

# %% ../nbs/models.ipynb 41
class SimpleExponentialSmoothing(_TS):
    """SimpleExponentialSmoothing model.
    [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/smoothing.py).

    Uses a weighted average of all past observations where the weights decrease exponentially into the past. 
    Suitable for data with no clear trend or seasonality. 
    Assuming there are $t$ observations, the one-step forecast is given by: $\hat{y}_{t+1} = \\alpha y_t + (1-\\alpha) \hat{y}_{t-1}$

    The rate $0 \leq \\alpha \leq 1$ at which the weights decrease is called the smoothing parameter. When $\\alpha = 1$, SES is equal to the naive method.

    **Parameters:**<br>
    `alpha`: float, smoothing parameter.<br>

    **References:**<br>
    [Charles C Holt (1957). “Forecasting seasonals and trends by exponentially weighted moving averages”](https://doi.org/10.1016/j.ijforecast).

    """
    def __init__(
            self, 
            alpha: float
        ):
        self.alpha = alpha
        
    def __repr__(self):
        return 'SES'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the SimpleExponentialSmoothing model.

        Fit an SimpleExponentialSmoothing to a time series (numpy array) `y`
        and optionally exogenous variables (numpy array) `X`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: SimpleExponentialSmoothing fitted model.
        """
        mod = _ses(y=y, alpha=self.alpha, h=1, fitted=True)
        self.model_ = dict(mod)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted SimpleExponentialSmoothing.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val(val=self.model_['mean'][0], h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted SimpleExponentialSmoothing insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        res = {'mean': self.model_['fitted']}
        return res
    
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient SimpleExponentialSmoothing predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        out = _ses(y=y, h=h, fitted=fitted, alpha=self.alpha)
        return out

# %% ../nbs/models.ipynb 50
def _ses_optimized(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
    ):
    fcst_, fitted_vals = _optimized_ses_forecast(y, [(0.01, 0.99)])
    mean = _repeat_val(val=fcst_, h=h)
    fcst = {'mean': mean}
    if fitted:
        fcst['fitted'] = fitted_vals
    return fcst

# %% ../nbs/models.ipynb 51
class SimpleExponentialSmoothingOptimized(_TS):
    """SimpleExponentialSmoothing model.
    [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/smoothing.py).

    Uses a weighted average of all past observations where the weights decrease exponentially into the past. 
    Suitable for data with no clear trend or seasonality. 
    Assuming there are $t$ observations, the one-step forecast is given by: $\hat{y}_{t+1} = \\alpha y_t + (1-\\alpha) \hat{y}_{t-1}$

    The smoothing parameter $\\alpha^*$ is optimized by square error minimization.

    **Parameters:**<br>

    **References:**<br>
    [Charles C Holt (1957). “Forecasting seasonals and trends by exponentially weighted moving averages”](https://doi.org/10.1016/j.ijforecast).

    """
    def __init__(self):
        pass
    
    def __repr__(self):
        return 'SESOpt'

    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the SimpleExponentialSmoothingOptimized model.

        Fit an SimpleExponentialSmoothingOptimized to a time series (numpy array) `y`
        and optionally exogenous variables (numpy array) `X`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: SimpleExponentialSmoothingOptimized fitted model.
        """
        mod = _ses_optimized(y=y, h=1, fitted=True)
        self.model_ = dict(mod)
        return self

    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted SimpleExponentialSmoothingOptimized.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val(val=self.model_['mean'][0], h=h)
        res = {'mean': mean}
        return res

    def predict_in_sample(self):
        """Access fitted SimpleExponentialSmoothingOptimized insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        res = {'mean': self.model_['fitted']}
        return res

    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient SimpleExponentialSmoothingOptimized predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        out = _ses_optimized(y=y, h=h, fitted=fitted)
        return out

# %% ../nbs/models.ipynb 60
@njit
def _seasonal_exponential_smoothing(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
        season_length: int, # length of season
        alpha: float, # smoothing parameter
    ):
    if y.size < season_length:
        return {'mean': np.full(h, np.nan, np.float32)}
    season_vals = np.empty(season_length, np.float32)
    fitted_vals = np.full(y.size, np.nan, np.float32)
    for i in range(season_length):
        season_vals[i], fitted_vals[i::season_length] = _ses_forecast(y[i::season_length], alpha)
    out = _repeat_val_seas(season_vals=season_vals, h=h, season_length=season_length)
    fcst = {'mean': out}
    if fitted:
        fcst['fitted'] = fitted_vals
    return fcst

# %% ../nbs/models.ipynb 61
class SeasonalExponentialSmoothing(_TS):
    """SeasonalExponentialSmoothing model.
    [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/smoothing.py).

    Uses a weighted average of all past observations where the weights decrease exponentially into the past. 
    Suitable for data with no clear trend or seasonality. 
    Assuming there are $t$ observations and season $s$, the one-step forecast is given by: 
    $\hat{y}_{t+1,s} = \\alpha y_t + (1-\\alpha) \hat{y}_{t-1,s}$

    **Note:**<br>
    This method is an extremely simplified of Holt-Winter's method where the trend and level are set to zero.
    And a single seasonal smoothing parameter $\\alpha$ is shared across seasons.

    **Parameters:**<br>
    `alpha`: float, smoothing parameter.<br>
    `season_length`: int, number of observations per unit of time. Ex: 24 Hourly data.<br>    

    **References:**<br>
    [Charles. C. Holt (1957). "Forecasting seasonals and trends by exponentially weighted moving averages", ONR Research Memorandum, Carnegie Institute of Technology 52.](https://www.sciencedirect.com/science/article/abs/pii/S0169207003001134).

    [Peter R. Winters (1960). "Forecasting sales by exponentially weighted moving averages". Management Science](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.6.3.324).
    """
    def __init__(
            self, 
            season_length: int,
            alpha: float,
        ):
        self.season_length = season_length
        self.alpha = alpha
    
    def __repr__(self):
        return 'SeasonalES'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the SeasonalExponentialSmoothing model.

        Fit an SeasonalExponentialSmoothing to a time series (numpy array) `y`
        and optionally exogenous variables (numpy array) `X`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: SeasonalExponentialSmoothing fitted model.
        """
        mod = _seasonal_exponential_smoothing(
            y=y, 
            season_length=self.season_length, 
            alpha=self.alpha,
            fitted=True,
            h=self.season_length,
        )
        self.model_ = dict(mod)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted SeasonalExponentialSmoothing.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val_seas(self.model_['mean'], season_length=self.season_length, h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted SeasonalExponentialSmoothing insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        res = {'mean': self.model_['fitted']}
        return res
    
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient SeasonalExponentialSmoothing predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        out = _seasonal_exponential_smoothing(
            y=y, h=h, fitted=fitted, 
            alpha=self.alpha,
            season_length=self.season_length
        )
        return out

# %% ../nbs/models.ipynb 70
def _seasonal_ses_optimized(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool , # fitted values
        season_length: int, # season length
    ): 
    if y.size < season_length:
        return {'mean': np.full(h, np.nan, np.float32)}
    season_vals = np.empty(season_length, np.float32)
    fitted_vals = np.full(y.size, np.nan, np.float32)
    for i in range(season_length):
        season_vals[i], fitted_vals[i::season_length] = _optimized_ses_forecast(y[i::season_length], [(0.01, 0.99)])
    out = _repeat_val_seas(season_vals=season_vals, h=h, season_length=season_length)
    fcst = {'mean': out}
    if fitted:
        fcst['fitted'] = fitted_vals
    return fcst

# %% ../nbs/models.ipynb 71
class SeasonalExponentialSmoothingOptimized(_TS):
    
    def __init__(
            self, 
            season_length: int,
        ):
        """SeasonalExponentialSmoothingOptimized model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/smoothing.py).

        Uses a weighted average of all past observations where the weights decrease exponentially into the past. 
        Suitable for data with no clear trend or seasonality. 
        Assuming there are $t$ observations and season $s$, the one-step forecast is given by: 
        $\hat{y}_{t+1,s} = \\alpha y_t + (1-\\alpha) \hat{y}_{t-1,s}$
        
        The smoothing parameter $\\alpha^*$ is optimized by square error minimization.        

        **Note:**<br>
        This method is an extremely simplified of Holt-Winter's method where the trend and level are set to zero.
        And a single seasonal smoothing parameter $\\alpha$ is shared across seasons.

        **Parameters:**<br>
        `season_length`: int, number of observations per unit of time. Ex: 24 Hourly data.<br>

        **References:**<br>
        [Charles. C. Holt (1957). "Forecasting seasonals and trends by exponentially weighted moving averages", ONR Research Memorandum, Carnegie Institute of Technology 52.](https://www.sciencedirect.com/science/article/abs/pii/S0169207003001134).

        [Peter R. Winters (1960). "Forecasting sales by exponentially weighted moving averages". Management Science](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.6.3.324).
        """
        self.season_length = season_length

    def __repr__(self):
        return 'SeasESOpt'

    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the SeasonalExponentialSmoothingOptimized model.

        Fit an SeasonalExponentialSmoothingOptimized to a time series (numpy array) `y`
        and optionally exogenous variables (numpy array) `X`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: SeasonalExponentialSmoothingOptimized fitted model.
        """
        mod = _seasonal_ses_optimized(
            y=y, 
            season_length=self.season_length, 
            fitted=True,
            h=self.season_length,
        )
        self.model_ = dict(mod)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted SeasonalExponentialSmoothingOptimized.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val_seas(self.model_['mean'], season_length=self.season_length, h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted SeasonalExponentialSmoothingOptimized insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        res = {'mean': self.model_['fitted']}
        return res
    
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient SeasonalExponentialSmoothingOptimized predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        out = _seasonal_ses_optimized(
            y=y, h=h, fitted=fitted, 
            season_length=self.season_length
        )
        return out

# %% ../nbs/models.ipynb 80
class Holt(ETS): 
    """ Holt's method. 

    Also known as double exponential smoothing, Holt's method is an extension of exponential smoothing for series with a trend.
    This implementation returns the corresponding `ETS` model with additive (A) or multiplicative (M) errors (so either 'AAN' or 'MAN'). 

    **Parameters:**<br>
     `season_length`: int, number of observations per unit of time. Ex: 12 Monthly data. <br>  
     `error_type`: The type of error of the ETS model. Can be additive (A) or multiplicative (M). <br> 

    **References:**<br>
    - [Rob J. Hyndman and George Athanasopoulos (2018). "Forecasting principles and practice, Methods with trend"](https://otexts.com/fpp3/holt.html).
    """

    def __init__(
            self, 
            season_length: int = 1, 
            error_type: str = 'A'
        ): 

        self.season_length = season_length
        self.error_type = error_type
        model = error_type + 'AN'
        super().__init__(season_length, model)
        
    def __rep__(self):
        return 'Holt'

# %% ../nbs/models.ipynb 90
class HoltWinters(ETS): 
    """ Holt-Winters' method. 
    
    Also known as triple exponential smoothing, Holt-Winters' method is an extension of exponential smoothing for series that contain both trend and seasonality.
    This implementation returns the corresponding `ETS` model with additive (A) or multiplicative (M) errors (so either 'AAA' or 'MAM'). 
    
    **Parameters:**<br>
     `season_length`: int, number of observations per unit of time. Ex: 12 Monthly data. <br>  
     `error_type`: The type of error of the ETS model. Can be additive (A) or multiplicative (M). <br> 

    **References:**<br>
    - [Rob J. Hyndman and George Athanasopoulos (2018). "Forecasting principles and practice, Methods with seasonality"](https://otexts.com/fpp3/holt-winters.html).
    """
        
    def __init__(
            self, 
            season_length: int = 1, # season length
            error_type: str = 'A' # error type
        ): 
        self.season_length = season_length
        self.error_type = error_type
        model = error_type + 'A' + error_type
        super().__init__(season_length, model)
        
    def __rep__(self):
        return 'HoltWinters'

# %% ../nbs/models.ipynb 101
@njit
def _historic_average(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
    ):
    mean = _repeat_val(val=y.mean(), h=h)
    fcst = {'mean': mean}
    if fitted:
        #fitted_vals = np.full(y.size, np.nan, np.float32) # one-step ahead
        #fitted_vals[1:] = y.cumsum()[:-1] / np.arange(1, y.size) 
        fitted_vals = _repeat_val(val=y.mean(), h=len(y))
        fcst['fitted'] = fitted_vals
    return fcst

# %% ../nbs/models.ipynb 102
class HistoricAverage(_TS):

    def __init__(self):
        """HistoricAverage model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/simple_methods.py).

        Also known as mean method. Uses a simple average of all past observations. 
        Assuming there are $t$ observations, the one-step forecast is given by: 
        $$ \hat{y}_{t+1} = \\frac{1}{t} \sum_{j=1}^t y_j $$

        **Parameters:**<br>

        **References:**<br>
        [Rob J. Hyndman and George Athanasopoulos (2018). "Forecasting principles and practice, Simple Methods"](https://otexts.com/fpp3/simple-methods.html).
        """
        pass
    
    def __repr__(self):
        return 'HistoricAverage'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the HistoricAverage model.

        Fit an HistoricAverage to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: HistoricAverage fitted model.
        """        
        mod = _historic_average(y, h=1, fitted=True)
        mod = dict(mod) 
        residuals = y - mod['fitted']
        mod['sigma'] = _calculate_sigma(residuals, len(residuals) - 1)
        mod['n'] = len(y)
        self.model_ = mod
        return self
        
    def predict(
            self, 
            h: int,
            X: np.ndarray = None,
            level: Optional[Tuple[int]] = None,
        ):
        """Predict with fitted HistoricAverage.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val(val=self.model_['mean'][0], h=h)
        res = {'mean': mean}
        
        if level is not None: 
            sigma = self.model_['sigma']
            sigmah = sigma * np.sqrt(1 + (1 / self.model_['n']))
            pred_int = _calculate_intervals(res, level, h, sigmah)
            res = {**res, **pred_int}
        
        return res
    
    def predict_in_sample(self):
        """Access fitted HistoricAverage insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        res = {'mean': self.model_['fitted']}
        return res
    
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
            level: Optional[Tuple[int]] = None,
        ):
        """Memory Efficient HistoricAverage predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        fitted = fitted or (level is not None)
        out = _historic_average(y=y, h=h, fitted=fitted)
        res = {'mean': out['mean']}
        
        if fitted:
            res['fitted'] = out['fitted']
        
        if level is not None: 
            residuals = y - out['fitted']
            sigma = _calculate_sigma(residuals, len(residuals) - 1)
            sigmah = sigma * np.sqrt(1 + (1 / len(y)))
            pred_int = _calculate_intervals(out, level, h, sigmah)
            res = {**res, **pred_int}
        
        return res

# %% ../nbs/models.ipynb 114
@njit
def _naive(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
    ): 
    mean = _repeat_val(val=y[-1], h=h)
    if fitted:
        fitted_vals = np.full(y.size, np.nan, np.float32)
        fitted_vals[1:] = np.roll(y, 1)[1:]
        return {'mean': mean, 'fitted': fitted_vals}
    return {'mean': mean}

# %% ../nbs/models.ipynb 115
class Naive(_TS):
    
    def __init__(self):
        """Naive model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/simple_methods.py).

        Also known as mean method. Uses a simple average of all past observations. 
        Assuming there are $t$ observations, the one-step forecast is given by: 
        $$ \hat{y}_{t+1} = y_t $$

        **Parameters:**<br>

        **References:**<br>
        [Rob J. Hyndman and George Athanasopoulos (2018). "forecasting principles and practice, Simple Methods"](https://otexts.com/fpp3/simple-methods.html).
        """
        pass
    
    def __repr__(self):
        return 'Naive'
    
    def fit(
            self, 
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the Naive model.

        Fit an Naive to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: Naive fitted model.
        """
        mod = _naive(y, h=1, fitted=True)
        mod = dict(mod) 
        residuals = y - mod['fitted']
        sigma = _calculate_sigma(residuals, len(residuals) - 1)
        mod['sigma'] = sigma
        self.model_ = mod
        return self
    
    def predict(
            self, 
            h: int, # forecasting horizon 
            X: np.ndarray = None, # exogenous regressors
            level: Optional[Tuple[int]] = None # confidence level
        ):
        """Predict with fitted Naive.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val(self.model_['mean'][0], h=h)
        res = {'mean': mean}
        
        if level is not None: 
            steps = np.arange(1,h+1)
            sigma = self.model_['sigma']
            sigmah = sigma * np.sqrt(steps)
            pred_int = _calculate_intervals(res, level, h, sigmah)
            res = {**res, **pred_int}
        
        return res
    
    def predict_in_sample(self):
        """Access fitted Naive insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        res = {'mean': self.model_['fitted']}
        return res
    
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
            level: Optional[Tuple[int]] = None,
        ):
        """Memory Efficient Naive predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        fitted = fitted or (level is not None)
        out = _naive(y=y, h=h, fitted=fitted)
        res = {'mean': out['mean']}
        
        if fitted:
            res['fitted'] = out['fitted']
        
        if level is not None: 
            steps = np.arange(1,h+1)
            residuals = y - out['fitted']
            sigma = _calculate_sigma(residuals, len(residuals) - 1)
            sigmah = sigma * np.sqrt(steps)
            pred_int = _calculate_intervals(out, level, h, sigmah)
            res = {**res, **pred_int}
        
        return res

# %% ../nbs/models.ipynb 127
@njit
def _random_walk_with_drift(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
    ): 
    slope = (y[-1] - y[0]) / (y.size - 1)
    mean = slope * (1 + np.arange(h)) + y[-1]
    fcst = {'mean': mean.astype(np.float32), 
            'slope': np.array([slope], dtype=np.float32), 
            'last_y': np.array([y[-1]], dtype=np.float32)}
    if fitted:
        fitted_vals = np.full(y.size, np.nan, dtype=np.float32)
        fitted_vals[1:] = (slope + y[:-1]).astype(np.float32)
        fcst['fitted'] = fitted_vals
    return fcst

# %% ../nbs/models.ipynb 128
class RandomWalkWithDrift(_TS):
    
    def __init__(self):
        """RandomWalkWithDrift model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/simple_methods.py).

        A variation of the naive method allows the forecasts to change over time. 
        The amout of change, called drift, is the average change seen in the historical data. 

        $$ \hat{y}_{t+1} = y_t+\\frac{1}{t-1}\sum_{j=1}^t (y_j-y_{j-1}) = y_t+ \\frac{y_t-y_1}{t-1} $$

        From the previous equation, we can see that this is equivalent to extrapolating a line between 
        the first and the last observation. 

        **Parameters:**<br>

        **References:**<br>
        [Rob J. Hyndman and George Athanasopoulos (2018). "forecasting principles and practice, Simple Methods"](https://otexts.com/fpp3/simple-methods.html).
        """
        pass
    
    def __repr__(self):
        return 'RWD'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the RandomWalkWithDrift model.

        Fit an RandomWalkWithDrift to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: RandomWalkWithDrift fitted model.
        """        
        mod = _random_walk_with_drift(y, h=1, fitted=True)
        mod = dict(mod) 
        residuals = y - mod['fitted']
        sigma = _calculate_sigma(residuals, len(residuals) - 1)
        mod['sigma'] = sigma
        mod['n'] = len(y)
        self.model_ = mod
        return self
        
    def predict(
            self,
            h: int, 
            X: np.ndarray = None, 
            level: Optional[Tuple[int]] = None
        ):
        """Predict with fitted RandomWalkWithDrift.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        hrange = np.arange(h, dtype=np.float32)
        mean = self.model_['slope'] * (1 + hrange) + self.model_['last_y']
        res = {'mean': mean}
        
        if level is not None: 
            steps = np.arange(1, h + 1)
            sigma = self.model_['sigma']
            sigmah = sigma * np.sqrt(steps * (1 + steps / (self.model_['n'] - 1)))
            pred_int = _calculate_intervals(res, level, h, sigmah)
            res = {**res, **pred_int}
        
        return res
    
    def predict_in_sample(self):
        """Access fitted RandomWalkWithDrift insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        res = {'mean': self.model_['fitted']}
        return res

    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
            level: Optional[Tuple[int]] = None,
        ):
        """Memory Efficient RandomWalkWithDrift predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        fitted = fitted or (level is not None)
        out = _random_walk_with_drift(y=y, h=h, fitted=fitted)
        res = {'mean': out['mean']}
        
        if fitted:
            res['fitted'] = out['fitted']
        
        if level is not None: 
            steps = np.arange(1, h + 1)
            residuals = y - out['fitted']
            sigma = _calculate_sigma(residuals, len(residuals) - 1)
            sigmah = sigma * np.sqrt(steps * (1 + steps / (len(y) - 1)))
            pred_int = _calculate_intervals(out, level, h, sigmah)
            res = {**res, **pred_int}

        return res 

# %% ../nbs/models.ipynb 140
@njit
def _seasonal_naive(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, #fitted values
        season_length: int, # season length
    ): 
    if y.size < season_length:
        return {'mean': np.full(h, np.nan, np.float32)}
    season_vals = np.empty(season_length, np.float32)
    fitted_vals = np.full(y.size, np.nan, np.float32)
    for i in range(season_length):
        s_naive = _naive(y[i::season_length], h=1, fitted=fitted)
        season_vals[i] = s_naive['mean'].item()
        if fitted:
            fitted_vals[i::season_length] = s_naive['fitted']
    out = _repeat_val_seas(season_vals=season_vals, h=h, season_length=season_length)
    fcst = {'mean': out}
    if fitted:
        fcst['fitted'] = fitted_vals
    return fcst

# %% ../nbs/models.ipynb 141
class SeasonalNaive(_TS):
    
    def __init__(self, season_length: int):
        self.season_length = season_length

    def __repr__(self):
        return 'SeasonalNaive'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the SeasonalNaive model.

        Fit an SeasonalNaive to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: SeasonalNaive fitted model.
        """        
        mod = _seasonal_naive(
            y=y, 
            season_length=self.season_length, 
            h=self.season_length, 
            fitted=True,
        )
        mod = dict(mod) 
        residuals = y - mod['fitted']
        mod['sigma'] = _calculate_sigma(residuals, 
                                        len(y) - self.season_length)
        self.model_ = mod
        return self
        
        
    def predict(
            self,
            h: int,  
            X: np.ndarray = None,
            level: Optional[Tuple[int]] = None, 
        ):
        """Predict with fitted Naive.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val_seas(season_vals=self.model_['mean'], 
                                season_length=self.season_length, h=h)
        res = {'mean': mean}
        
        if level is not None: 
            k = np.floor((h - 1) / self.season_length)
            sigma = self.model_['sigma']
            sigmah = sigma * np.sqrt(k + 1)
            pred_int = _calculate_intervals(res, level, h, sigmah)
            res = {**res, **pred_int}
        
        return res
        
    def predict_in_sample(self):
        """Access fitted SeasonalNaive insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """        
        res = {'mean': self.model_['fitted']}
        return res
    
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
            level: Optional[Tuple[int]] = None,
        ):
        """Memory Efficient SeasonalNaive predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        fitted = fitted or (level is not None)
        out = _seasonal_naive(
            y=y, h=h, fitted=fitted, 
            season_length=self.season_length
        )
        res = {'mean': out['mean']}
        
        if fitted:
            res['fitted'] = out['fitted']
        
        if level is not None: 
            k = np.floor((h - 1) / self.season_length)
            residuals = y - out['fitted']
            sigma = _calculate_sigma(residuals, len(y) - self.season_length)
            sigmah = sigma * np.sqrt(k + 1)
            pred_int = _calculate_intervals(out, level, h, sigmah)
            res = {**res, **pred_int}
            
        return res    

# %% ../nbs/models.ipynb 153
@njit
def _window_average(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
        window_size: int, # window size
    ): 
    if fitted:
        raise NotImplementedError('return fitted')
    if y.size < window_size:
        return {'mean': np.full(h, np.nan, np.float32)}
    wavg = y[-window_size:].mean()
    mean = _repeat_val(val=wavg, h=h)
    return {'mean': mean}

# %% ../nbs/models.ipynb 154
class WindowAverage(_TS):
    
    def __init__(
            self, 
            window_size: int
        ):
        """WindowAverage model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/simple_methods.py).

        Uses the average of the last $k$ observations, with $k$ the length of the window.
        Wider windows will capture global trends, while narrow windows will reveal local trends.
        The length of the window selected should take into account the importance of past
        observations and how fast the series changes.

        **Parameters:**<br>
        `window_size`: int, size of truncated series on which average is estimated.

        **References:**<br>
        [Rob J. Hyndman and George Athanasopoulos (2018). "forecasting principles and practice, Simple Methods"](https://otexts.com/fpp3/simple-methods.html).
        """        
        self.window_size = window_size

    def __repr__(self):
        return 'WindowAverage'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the WindowAverage model.

        Fit an WindowAverage to a time series (numpy array) `y`
        and optionally exogenous variables (numpy array) `X`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: WindowAverage fitted model.
        """
        mod = _window_average(y=y, h=1, window_size=self.window_size, fitted=False)
        self.model_ = dict(mod)
        return self
        
    def predict(
            self, 
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted WindowAverage.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val(self.model_['mean'][0], h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted WindowAverage insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        raise NotImplementedError
        
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient WindowAverage predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        out = _window_average(y=y, h=h, fitted=fitted, window_size=self.window_size)
        return out

# %% ../nbs/models.ipynb 163
@njit
def _seasonal_window_average(
        y: np.ndarray,
        h: int,
        fitted: bool,
        season_length: int,
        window_size: int,
    ):
    if fitted:
        raise NotImplementedError('return fitted')
    min_samples = season_length * window_size
    if y.size < min_samples:
        return {'mean': np.full(h, np.nan, np.float32)}
    season_avgs = np.zeros(season_length, np.float32)
    for i, value in enumerate(y[-min_samples:]):
        season = i % season_length
        season_avgs[season] += value / window_size
    out = _repeat_val_seas(season_vals=season_avgs, h=h, season_length=season_length)
    return {'mean': out}

# %% ../nbs/models.ipynb 164
class SeasonalWindowAverage(_TS):
    
    def __init__(
            self, 
            season_length: int,
            window_size: int
        ):
        """SeasonalWindowAverage model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/simple_methods.py).

        An average of the last $k$ observations of the same period, with $k$ the length of the window.

        **Parameters:**<br>
        `window_size`: int, size of truncated series on which average is estimated.
        `seasonal_length`: int, number of observations per cycle.

        **References:**<br>
        [Rob J. Hyndman and George Athanasopoulos (2018). "forecasting principles and practice, Simple Methods"](https://otexts.com/fpp3/simple-methods.html).
        """        
        self.season_length = season_length
        self.window_size = window_size

    def __repr__(self):
        return 'SeasWA'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the SeasonalWindowAverage model.

        Fit an SeasonalWindowAverage to a time series (numpy array) `y`
        and optionally exogenous variables (numpy array) `X`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: SeasonalWindowAverage fitted model.
        """        
        mod = _seasonal_window_average(
            y=y, 
            h=self.season_length,
            fitted=False,
            season_length=self.season_length, 
            window_size=self.window_size,
        )
        self.model_ = dict(mod)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted SeasonalWindowAverage.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val_seas(season_vals=self.model_['mean'], 
                                season_length=self.season_length, h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted SeasonalWindowAverage insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        raise NotImplementedError

    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient SeasonalWindowAverage predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """        
        out = _seasonal_window_average(
            y=y, h=h, fitted=fitted, 
            season_length=self.season_length,
            window_size=self.window_size
        )
        return out

# %% ../nbs/models.ipynb 174
def _adida(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
    ):
    if fitted:
        raise NotImplementedError('return fitted')
    if (y == 0).all():
        return {'mean': np.repeat(np.float32(0), h)}
    y_intervals = _intervals(y)
    mean_interval = y_intervals.mean()
    aggregation_level = round(mean_interval)
    lost_remainder_data = len(y) % aggregation_level
    y_cut = y[lost_remainder_data:]
    aggregation_sums = _chunk_sums(y_cut, aggregation_level)
    sums_forecast, _ = _optimized_ses_forecast(aggregation_sums)
    forecast = sums_forecast / aggregation_level
    mean = _repeat_val(val=forecast, h=h)
    return {'mean': mean}

# %% ../nbs/models.ipynb 175
class ADIDA(_TS):

    def __init__(self):
        """ADIDA model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/sparse.py).

        Aggregate-Dissagregate Intermittent Demand Approach: Uses temporal aggregation to reduce the 
        number of zero observations. Once the data has been agregated, it uses the optimized SES to 
        generate the forecasts at the new level. It then breaks down the forecast to the original 
        level using equal weights.

        ADIDA specializes on sparse or intermittent series are series with very few non-zero observations. 
        They are notoriously hard to forecast, and so, different methods have been developed 
        especifically for them.

        **Parameters:**<br>

        **References:**<br>
        [Nikolopoulos, K., Syntetos, A. A., Boylan, J. E., Petropoulos, F., & Assimakopoulos, V. (2011). An aggregate–disaggregate intermittent demand approach (ADIDA) to forecasting: an empirical proposition and analysis. Journal of the Operational Research Society, 62(3), 544-554.](https://researchportal.bath.ac.uk/en/publications/an-aggregate-disaggregate-intermittent-demand-approach-adida-to-f).
        """        
        pass

    def __repr__(self):
        return 'ADIDA'

    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the ADIDA model.

        Fit an ADIDA to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: ADIDA fitted model.
        """        
        mod = _adida(y=y, h=1, fitted=False)
        self.model_ = dict(mod)
        return self

    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted ADIDA.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """        
        mean = _repeat_val(val=self.model_['mean'][0], h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted ADIDA insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        raise NotImplementedError
        
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient ADIDA predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        out = _adida(y=y, h=h, fitted=fitted)
        return out

# %% ../nbs/models.ipynb 184
@njit
def _croston_classic(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
    ): 
    if fitted:
        raise NotImplementedError('return fitted')
    yd = _demand(y)
    yi = _intervals(y)
    ydp, _ = _ses_forecast(yd, 0.1)
    yip, _ = _ses_forecast(yi, 0.1)
    if yip == 0.:
        mean = ydp / yip
    else:
        mean = ydp
    mean = _repeat_val(val=mean, h=h)
    return {'mean': mean}

# %% ../nbs/models.ipynb 185
class CrostonClassic(_TS):
    
    def __init__(self):
        """CrostonClassic model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/sparse.py).

        A method to forecast time series that exhibit intermittent demand.
        It decomposes the original time series into a non-zero demand size $z_t$ and 
        inter-demand intervals $p_t$. Then the forecast is given by:
        $$ \hat{y}_t = \\frac{\hat{z}_t}{\hat{p}_t} $$ 

        where $\hat{z}_t$ and $\hat{p}_t$ are forecasted using SES. The smoothing parameter 
        of both components is set equal to 0.1

        **Parameters:**<br>

        **References:**<br>
        [Croston, J. D. (1972). Forecasting and stock control for intermittent demands. Journal of the Operational Research Society, 23(3), 289-303.](https://link.springer.com/article/10.1057/jors.1972.50)
        """        
        pass

    def __repr__(self):
        return 'CrostonClassic'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the CrostonClassic model.

        Fit an CrostonClassic to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: CrostonClassic fitted model.
        """        
        mod = _croston_classic(y=y, h=1, fitted=False)
        self.model_ = dict(mod)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted CrostonClassic.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val(val=self.model_['mean'][0], h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self, level):
        """Access fitted CrostonClassic insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        raise NotImplementedError
        
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient CrostonClassic predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """        
        out = _croston_classic(y=y, h=h, fitted=fitted)
        return out

# %% ../nbs/models.ipynb 194
def _croston_optimized(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
    ): 
    if fitted:
        raise NotImplementedError('return fitted')
    yd = _demand(y)
    yi = _intervals(y)
    ydp, _ = _optimized_ses_forecast(yd)
    yip, _ = _optimized_ses_forecast(yi)
    if yip == 0.:
        mean = ydp / yip
    else:
        mean = ydp
    mean = _repeat_val(val=mean, h=h)
    return {'mean': mean}

# %% ../nbs/models.ipynb 195
class CrostonOptimized(_TS):
    
    def __init__(self):
        """CrostonOptimized model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/sparse.py).

        A method to forecast time series that exhibit intermittent demand.
        It decomposes the original time series into a non-zero demand size $z_t$ and 
        inter-demand intervals $p_t$. Then the forecast is given by:
        $$ \hat{y}_t = \\frac{\hat{z}_t}{\hat{p}_t} $$

        A variation of the classic Croston's method where the smooting paramater is optimally 
        selected from the range $[0.1,0.3]$. Both the non-zero demand $z_t$ and the inter-demand 
        intervals $p_t$ are smoothed separately, so their smoothing parameters can be different.

        **Parameters:**<br>

        **References:**<br>
        [Croston, J. D. (1972). Forecasting and stock control for intermittent demands. Journal of the Operational Research Society, 23(3), 289-303.](https://link.springer.com/article/10.1057/jors.1972.50).
        """        
        pass
    
    def __repr__(self):
        return 'CrostonOptimized'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the CrostonOptimized model.

        Fit an CrostonOptimized to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: CrostonOptimized fitted model.
        """        
        mod = _croston_optimized(y=y, h=1, fitted=False)
        self.model_ = dict(mod)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted CrostonOptimized.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val(val=self.model_['mean'][0], h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted CrostonOptimized insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """        
        raise NotImplementedError

    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient CrostonOptimized predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """        
        out = _croston_optimized(y=y, h=h, fitted=fitted)
        return out

# %% ../nbs/models.ipynb 205
@njit
def _croston_sba(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool,  # fitted values
    ):
    if fitted:
        raise NotImplementedError('return fitted')
    mean = _croston_classic(y, h, fitted)
    mean['mean'] *= 0.95
    return mean

# %% ../nbs/models.ipynb 206
class CrostonSBA(_TS):
    
    def __init__(self):
        """CrostonSBA model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/sparse.py).

        A method to forecast time series that exhibit intermittent demand.
        It decomposes the original time series into a non-zero demand size $z_t$ and 
        inter-demand intervals $p_t$. Then the forecast is given by:
        $$ \hat{y}_t = \\frac{\hat{z}_t}{\hat{p}_t} $$

        A variation of the classic Croston's method that uses a debiasing factor, so that the 
        forecast is given by:
        $$ \hat{y}_t = 0.95  \\frac{\hat{z}_t}{\hat{p}_t} $$

        **Parameters:**<br>

        **References:**<br>
        [Croston, J. D. (1972). Forecasting and stock control for intermittent demands. Journal of the Operational Research Society, 23(3), 289-303.](https://link.springer.com/article/10.1057/jors.1972.50).
        """        
        pass

    def __repr__(self):
        return 'CrostonSBA'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the CrostonSBA model.

        Fit an CrostonSBA to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: CrostonSBA fitted model.
        """
        mod = _croston_sba(y=y, h=1, fitted=False)
        self.model_ = dict(mod)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted CrostonSBA.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val(val=self.model_['mean'][0], h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted CrostonSBA insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        raise NotImplementedError
        
    def forecast(
            self,
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient CrostonSBA predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """        
        out = _croston_sba(y=y, h=h, fitted=fitted)
        return out

# %% ../nbs/models.ipynb 215
def _imapa(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: bool, # fitted values
    ): 
    if fitted:
        raise NotImplementedError('return fitted')
    if (y == 0).all():
        return {'mean': np.repeat(np.float32(0), h)}
    y_intervals = _intervals(y)
    mean_interval = y_intervals.mean().item()
    max_aggregation_level = round(mean_interval)
    forecasts = np.empty(max_aggregation_level, np.float32)
    for aggregation_level in range(1, max_aggregation_level + 1):
        lost_remainder_data = len(y) % aggregation_level
        y_cut = y[lost_remainder_data:]
        aggregation_sums = _chunk_sums(y_cut, aggregation_level)
        forecast, _ = _optimized_ses_forecast(aggregation_sums)
        forecasts[aggregation_level - 1] = (forecast / aggregation_level)
    forecast = forecasts.mean()
    mean = _repeat_val(val=forecast, h=h)
    return {'mean': mean}

# %% ../nbs/models.ipynb 216
class IMAPA(_TS):
    
    def __init__(self):
        """IMAPA model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/sparse.py).

        Intermittent Multiple Aggregation Prediction Algorithm: Similar to ADIDA, but instead of
        using a single aggregation level, it considers multiple in order to capture different
        dynamics of the data. Uses the optimized SES to generate the forecasts at the new levels
        and then combines them using a simple average.

        **Parameters:**<br>

        **References:**<br>
        - [Syntetos, A. A., & Boylan, J. E. (2021). Intermittent demand forecasting: Context, methods and applications. John Wiley & Sons.](https://www.ifors.org/intermittent-demand-forecasting-context-methods-and-applications/).
        """
        pass

    def __repr__(self):
        return 'IMAPA'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the IMAPA model.

        Fit an IMAPA to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: IMAPA fitted model.
        """
        mod = _imapa(y=y, h=1, fitted=False)
        self.model_ = dict(mod)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted IMAPA.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        mean = _repeat_val(val=self.model_['mean'][0], h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted IMAPA insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        raise NotImplementedError
        
    def forecast(
        self, 
        y: np.ndarray,
        h: int,
        X: np.ndarray = None,
        X_future: np.ndarray = None,
        fitted: bool = False,
    ):
        """Memory Efficient IMAPA predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        out = _imapa(y=y, h=h, fitted=fitted)
        return out

# %% ../nbs/models.ipynb 225
@njit
def _tsb(
        y: np.ndarray, # time series
        h: int, # forecasting horizon
        fitted: int, # fitted values
        alpha_d: float,
        alpha_p: float,
    ):
    if fitted:
        raise NotImplementedError('return fitted')
    if (y == 0).all():
        return {'mean': np.repeat(np.float32(0), h)}
    yd = _demand(y)
    yp = _probability(y)
    ypf, _ = _ses_forecast(yp, alpha_p)
    ydf, _ = _ses_forecast(yd, alpha_d)
    forecast = np.float32(ypf * ydf)
    mean = _repeat_val(val=forecast, h=h)
    return {'mean': mean}

# %% ../nbs/models.ipynb 226
class TSB(_TS):
    
    def __init__(
            self, 
            alpha_d: float,
            alpha_p: float
        ):
        """TSB model.
        [Source code](https://github.com/Nixtla/statsforecast/blob/main/statsforecast/sparse.py).

        Teunter-Syntetos-Babai: A modification of Croston's method that replaces the inter-demand 
        intervals with the demand probability $d_t$, which is defined as follows.

        $$
        d_t = \\begin{cases}
            1  & \\text{if demand occurs at time t} \\\ 
            0  & \\text{otherwise.}
        \\end{cases}
        $$

        Hence, the forecast is given by 

        $$\hat{y}_t= \hat{d}_t\hat{z_t}$$

        Both $d_t$ and $z_t$ are forecasted using SES. The smooting paramaters of each may differ, 
        like in the optimized Croston's method.

        **Parameters:**<br>
        `alpha_d`: float, smoothing parameter for demand<br>
        `alpha_p`: float, smoothing parameter for probability<br>

        **References:**<br>
        - [Teunter, R. H., Syntetos, A. A., & Babai, M. Z. (2011). Intermittent demand: Linking forecasting to inventory obsolescence. European Journal of Operational Research, 214(3), 606-615.](https://www.sciencedirect.com/science/article/abs/pii/S0377221711004437)
        """
        self.alpha_d = alpha_d
        self.alpha_p = alpha_p
        
    def __repr__(self):
        return 'TSB'
    
    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the TSB model.

        Fit an TSB to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>

        **Returns:**<br>
        `self`: TSB fitted model.
        """
        mod = _tsb(
            y=y, h=1, 
            fitted=False, 
            alpha_d=self.alpha_d, 
            alpha_p=self.alpha_p
        )
        self.model_ = dict(mod)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None
        ):
        """Predict with fitted TSB.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """        
        mean = _repeat_val(self.model_['mean'][0], h=h)
        res = {'mean': mean}
        return res
    
    def predict_in_sample(self):
        """Access fitted TSB insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        raise NotImplementedError
        
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            fitted: bool = False,
        ):
        """Memory Efficient TSB predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        out = _tsb(
            y=y, h=h, 
            fitted=fitted, 
            alpha_d=self.alpha_d, 
            alpha_p=self.alpha_p
        )
        return out

# %% ../nbs/models.ipynb 235
def _predict_mstl_seas(mstl_ob, h, season_length):
    seasoncolumns = mstl_ob.filter(regex='seasonal*').columns
    nseasons = len(seasoncolumns)
    seascomp = np.full((h, nseasons), np.nan)
    seasonal_periods = [season_length] if isinstance(season_length, int) else season_length
    for i in range(nseasons):
        mp = seasonal_periods[i]
        colname = seasoncolumns[i]
        seascomp[:, i] = np.tile(mstl_ob[colname].values[-mp:], trunc(1 + (h-1)/mp))[:h]
    lastseas = seascomp.sum(axis=1)
    return lastseas

# %% ../nbs/models.ipynb 236
class MSTL(_TS):
    """MSTL model.
    
    The MSTL (Multiple Seasonal-Trend decomposition using LOESS) decomposes the time series
    in multiple seasonalities using LOESS. Then forecasts the trend using 
    a custom non-seaonal model and each seasonality using a SeasonalNaive model.
    
    **Parameters:**<br>
    `season_length`: Union[int, List[int], number of observations per unit of time. For multiple seasonalities use a list.<br>
    `trend_forecaster`: StatsForecast model used to forecast the trend component.<br>
    
    **References:**<br>
    [Bandara, Kasun & Hyndman, Rob & Bergmeir, Christoph. (2021). "MSTL: A Seasonal-Trend Decomposition Algorithm for Time Series with Multiple Seasonal Patterns".](https://arxiv.org/abs/2107.13462).
    """
    
    def __init__(
        self, 
        season_length: Union[int, List[int]],
        trend_forecaster = ETS(model='ZZN')
    ):
        
        # check ETS model doesnt have seasonality
        if repr(trend_forecaster) == 'ETS':
            if trend_forecaster.model[2] != 'N':
                raise Exception(
                    'Trend forecaster should not adjust '
                    'seasonal models.'
                )
        if hasattr(trend_forecaster, 'season_length'):
            if trend_forecaster.season_length != 1:
                raise Exception(
                    'Trend forecaster should not adjust '
                    'seasonal models. Please pass `season_length=1` '
                    'to your trend forecaster'
                )
        self.season_length = season_length
        self.trend_forecaster = trend_forecaster
        
        # check if trend forecaster has season_length=1
        
    def __repr__(self):
        return 'MSTL'

    def fit(
            self,
            y: np.ndarray,
            X: np.ndarray = None
        ):
        """Fit the MSTL model.

        Fit MSTL to a time series (numpy array) `y`.

        **Parameters:**<br>
        `y`: numpy array of shape (t, ), clean time series.<br>
        `X`: array-like of shape (t, n_x) optional exogenous (default=None).<br>

        **Returns:**<br>
        `self`: MSTL fitted model.
        """
        self.model_ = mstl(
            x=y, 
            period=self.season_length
        )
        x_sa = self.model_[['trend', 'remainder']].sum(axis=1).values
        self.trend_forecaster = self.trend_forecaster.fit(y=x_sa, X=X)
        return self
        
    def predict(
            self,
            h: int,
            X: np.ndarray = None,
            level: Optional[Tuple[int]] = None,
        ):
        """Predict with fitted MSTL.

        **Parameters:**<br>
        `h`: int, forecast horizon.<br>
        `X`: array-like of shape (h, n_x) optional exogenous (default=None).<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        kwargs: Dict[str, Any] = {'h': h, 'X': X}
        if 'level' in signature(self.trend_forecaster.predict).parameters:
            kwargs['level'] = level
        res = self.trend_forecaster.predict(**kwargs)
        seas = _predict_mstl_seas(self.model_, h=h, season_length=self.season_length)
        res = {key: val + seas for key, val in res.items()}
        return res
    
    def predict_in_sample(self, level: Optional[Tuple[int]] = None):
        """Access fitted MSTL insample predictions.

        **Parameters:**<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        kwargs = {}
        if 'level' in signature(self.trend_forecaster.predict_in_sample).parameters:
            kwargs['level'] = level
        
        res = self.trend_forecaster.predict_in_sample(**kwargs)
        seas = self.model_.filter(regex='seasonal*').sum(axis=1).values
        res = {key: val + seas for key, val in res.items()}
        return res
        
    def forecast(
            self, 
            y: np.ndarray,
            h: int,
            X: np.ndarray = None,
            X_future: np.ndarray = None,
            level: Optional[List[int]] = None,
            fitted: bool = False,
        ):
        """Memory Efficient MSTL predictions.

        This method avoids memory burden due from object storage.
        It is analogous to `fit_predict` without storing information.
        It assumes you know the forecast horizon in advance.

        **Parameters:**<br>
        `y`: numpy array of shape (n,), clean time series.<br>
        `h`: int, forecast horizon.<br>
        `X`: array-like of shape (t, n_x) optional insample exogenous (default=None).<br>
        `X_future`: array-like of shape (h, n_x) optional exogenous (default=None).<br>
        `level`: float list 0-100, confidence levels for prediction intervals.<br>
        `fitted`: bool, wether or not returns insample predictions.<br>

        **Returns:**<br>
        `forecasts`: dictionary, with entries 'mean' for point predictions and
            'level_*' for probabilistic predictions.<br>
        """
        model_ = mstl(
            x=y, 
            period=self.season_length
        )
        x_sa = model_[['trend', 'remainder']].sum(axis=1).values
        kwargs = {
            'y': x_sa,
            'h': h,
            'X': X,
            'X_future': X_future,
            'fitted': fitted
        }
        if 'level' in signature(self.trend_forecaster.forecast).parameters:
            kwargs['level'] = level
        res = self.trend_forecaster.forecast(**kwargs)
        #reseasonalize results
        seas_h = _predict_mstl_seas(model_, h=h, season_length=self.season_length)
        seas_insample = model_.filter(regex='seasonal*').sum(axis=1).values
        res = {
            key: val + (seas_insample if 'fitted' in key else seas_h) \
            for key, val in res.items()
        }
        return res
