from CompNeuroPy import (
    opt_neuron,
    Experiment,
    Monitors,
    generate_simulation,
    current_step,
)
import numpy as np
from ANNarchy import Neuron, dt


### in this example we want to fit a ANNarchy neuron model to some data from another ANNarchy neuron model
### one could for example fit a simpler model to replicate the dynamics of a more complex model
### for this example, we have the following two neuron models
complex_neuron = Neuron(
    parameters="""
        I_app = 0
        f = 6.0542364610842572e-002 : population
        e = 3.7144041714209490e+000 : population
        d = -4.9446336126026436e-001 : population
        c = 9.0909599124334911e-002 : population
        b = -4.4497411506061648e-003 : population
        a = -6.2239117460540167e-005 : population
    """,
    equations="""
        r = a*I_app**5 + b*I_app**4 + c*I_app**3 + d*I_app**2 + e*I_app**1 + f
    """,
)
simple_neuron = Neuron(
    parameters="""
        I_app = 0
        a = 0 : population
        b = 0 : population
    """,
    equations="""
        r = a*I_app + b
    """,
)


### define the experiment which is conducted with both simulated neurons
### for the opt_neuron class we need to provide an experiment class containing the "run" function
class my_exp(Experiment):
    """
    parent class Experiment provides the variables:
        self.mon = self.cnp.Monitors() --> a CompNeuroPy Monitors object to do recordings
        self.data = {}                 --> a dictionary with any optional data
    and the functions:
        self.reset()   --> resets the model and monitors
        self.results() --> returns a results object (with recordings and optional data from self.data)
    """

    def run(self, population_name):
        """
        the function which defines the experiment

        here recordings have to be defined and simulations have to be run

        to use the experiment for the opt_neuron class, the arguments have to be:
            self - the experiment object
            population_name - the name of the population which contains a single neuron. this will be automatically provided by opt_neuron
        """

        ### define recordings
        self.mon = Monitors({"pop;" + population_name: ["r"]})

        ### define simulation
        ### you don't have to use the CompNeuroPy generate_simulation object
        sim_step = generate_simulation(
            simulation_function=current_step,
            simulation_kwargs={
                "pop": population_name,
                "t1": 500,
                "t2": 500,
                "a1": 0,
                "a2": 5,
            },
            kwargs_warning=False,
            name="test",
            monitor_object=self.mon,
        )

        ### run simulation/recordings
        self.mon.start()
        sim_step.run()
        self.reset()  # if you want to reset the model, use the object's self.reset() function, monitors are automatically resetted too
        sim_step.run({"a2": 10})
        ### SIMULATION END

        ### optional: store anything you want in the data dict, for example information about the simulations
        self.data["sim"] = sim_step.simulation_info()
        self.data["population_name"] = population_name
        self.data["time_step"] = dt()
        self.data["recording_times"] = self.mon.get_recording_times()

        ### return results, use the object's self.results() function which automatically returns an object with "recordings", "monDict", and "data"
        return self.results()


### next, the opt_neuron class needs a function to calculate the loss
### this function has to contain two arguments
### the first argument (results_ist) = results returned by the simulated experiment (just defined above), with the neuron model provided in the argument "neuron_model"
### the second argument (results_soll) = results either provided directly to the opt_neuron class (argument "results_soll") or, as in this example, simulated with the neuron provided in the argument "target_neuron_model"
def get_loss(results_ist, results_soll):
    """
    results_soll: any
        contains the target data
        the results object generated by the experiment
    results_ist: object
        contains the simulated data for each run
        the results object generated by the experiment
    """

    ### get the recordings and other important things from the results_ist (results generated during the optimization using the defrined experiment from above)
    rec_ist = results_ist.recordings
    pop_ist = results_ist.data[
        "population_name"
    ]  # we could also access "sim" or "time_step" but not needed for analyses
    rec_soll = results_soll.recordings
    pop_soll = results_soll.data["population_name"]
    neuron = 0

    ### get the important data for calculating the loss from the results_soll (target data directly provided to opt_neuron)
    v_soll_0 = rec_soll[0][pop_soll + ";r"][:, neuron]
    v_soll_1 = rec_soll[1][pop_soll + ";r"][:, neuron]

    ### get the important data for calculating the loss from the recordings
    v_ist_0 = rec_ist[0][pop_ist + ";r"][:, neuron]
    v_ist_1 = rec_ist[1][pop_ist + ";r"][:, neuron]

    ### calculate the loss
    rmse1 = np.sqrt(np.mean((v_soll_0 - v_ist_0) ** 2))
    rmse2 = np.sqrt(np.mean((v_soll_1 - v_ist_1) ** 2))

    ### return the loss, one can return a singel value or a list of values (which will be summed)
    return [rmse1, rmse2]


### now we need to define which variables should be optimized and between which bounds (min and max values)
variables_bounds = {"a": [-10, 10], "b": [-10, 10]}


### run optimization
opt = opt_neuron(
    experiment=my_exp,
    get_loss_function=get_loss,
    variables_bounds=variables_bounds,
    time_step=1,
    compile_folder_name="annarchy_opt_neuron_example",
    neuron_model=simple_neuron,
    target_neuron_model=complex_neuron,
    method="hyperopt",
)

### run the optimization, define how often the experiment should be repeated
opt.run(max_evals=1000, results_file_name="best_with_exp.npy")

### we should get around a=2.8 and b=0.28
print("\nresults:\n")
for key, val in opt.results.items():
    if key in ["std", "results", "results_soll"]:
        print(key, "\n")
    else:
        print(key, val, "\n")
