"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalizeStatistic = exports.parseStatistic = void 0;
const metric_types_1 = require("../metric-types");
/**
 * Parse a statistic, returning the type of metric that was used (simple or percentile)
 */
function parseStatistic(stat) {
    const lowerStat = stat.toLowerCase();
    // Simple statistics
    const statMap = {
        average: metric_types_1.Statistic.AVERAGE,
        avg: metric_types_1.Statistic.AVERAGE,
        minimum: metric_types_1.Statistic.MINIMUM,
        min: metric_types_1.Statistic.MINIMUM,
        maximum: metric_types_1.Statistic.MAXIMUM,
        max: metric_types_1.Statistic.MAXIMUM,
        samplecount: metric_types_1.Statistic.SAMPLE_COUNT,
        n: metric_types_1.Statistic.SAMPLE_COUNT,
        sum: metric_types_1.Statistic.SUM,
    };
    if (lowerStat in statMap) {
        return {
            type: 'simple',
            statistic: statMap[lowerStat],
        };
    }
    // Percentile statistics
    const re = /^p([\d.]+)$/;
    const m = re.exec(lowerStat);
    if (m) {
        return {
            type: 'percentile',
            percentile: parseFloat(m[1]),
        };
    }
    throw new Error(`Not a valid statistic: '${stat}', must be one of Average | Minimum | Maximum | SampleCount | Sum | pNN.NN`);
}
exports.parseStatistic = parseStatistic;
function normalizeStatistic(stat) {
    const parsed = parseStatistic(stat);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    else {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return stat.toLowerCase();
    }
}
exports.normalizeStatistic = normalizeStatistic;
//# sourceMappingURL=data:application/json;base64,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